/*
 * fbgrab - takes screenshots using the framebuffer.
 *
 * (C) Gunnar Monell <gmo@linux.nu> 2002
 *
 * This program is free Software, see the COPYING file
 * and is based on Stephan Beyer's <fbshot@s-beyer.de> FBShot
 * (C) 2000.
 *
 * For features and differences, read the manual page.
 *
 * This program has been checked with "splint +posixlib" without
 * warnings. Splint is available from http://www.splint.org/ .
 * Patches and enhancements of fbgrab have to fulfill this too.
 *
 * $Id: fbgrab.c 12157 2015-08-14 18:35:41Z NiLuJe $
 * $Revision: 12157 $
 *
 */

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include <string.h>
#include <getopt.h>
#include <sys/vt.h>   /* to handle vt changing */
#include <png.h>      /* PNG lib */
#include <zlib.h>
#include <linux/fb.h> /* to handle framebuffer ioctls */

// Make sure we have a fallback version if the Makefile wasn't used.
#ifndef VERSION
#define VERSION "1.3.N"
#endif
#define DEFAULT_FB "/dev/fb0"
#define MAX_LEN 512
#define UNDEFINED -1

static uint32_t srcBlue = 0;
static uint32_t srcGreen = 1;
static uint32_t srcRed = 2;
static uint32_t srcAlpha = 3;
static uint32_t srcAlphaIsUsed = 0;

static const uint32_t Blue = 0;
static const uint32_t Green = 1;
static const uint32_t Red = 2;
static const uint32_t Alpha = 3;

static void fatal_error(char *message)
{
	fprintf(stderr, "%s\n", message);
	exit(EXIT_FAILURE);
}

static void usage(char *binary)
{
	fprintf(stderr, "Usage:   %s\t[-hi] [-{C|c} vt] [-d dev] [-s n] [-z n]\n"
		"\t\t[-f fromfile -w n -h n -b n] filename.png\n", binary);
}

static void help(char *binary)
{
	fprintf(stderr, "fbgrab - takes screenshots using the framebuffer, v%s\n", VERSION);

	usage(binary);

	fprintf(stderr, "\nPossible options:\n");
	/* please keep this list alphabetical */
	fprintf(stderr, "\t-b n  \tforce use of n bits/pixel, required when reading from file\n");
	fprintf(stderr, "\t-C n  \tgrab from console n, for slower framebuffers\n");
	fprintf(stderr, "\t-c n  \tgrab from console n\n");
	fprintf(stderr, "\t-d dev\tuse framebuffer device dev instead of default\n");
	fprintf(stderr, "\t-f file\t read from file instead of framebuffer\n");
	fprintf(stderr, "\t-h n  \tset height to n pixels, required when reading from file\n"
		"\t\tcan be used to force height when reading from framebuffer\n");
	fprintf(stderr, "\t-i    \tturns on interlacing in PNG\n");
	fprintf(stderr, "\t-k    \tdisables automagic cropping (Kindle/Kobo)\n");
	fprintf(stderr, "\t-s n  \tsleep n seconds before making screenshot\n");
	fprintf(stderr, "\t-v    \tverbose, print debug information.\n");
	fprintf(stderr, "\t-w n  \tset width to n pixels, required when reading from file\n"
		"\t\tcan be used to force width when reading from framebuffer\n");
	fprintf(stderr, "\t-z n  \tPNG compression level: 0 (fast) .. 9 (best)\n");
	fprintf(stderr, "\t-?    \tprint this usage information\n");
}


static void chvt(int num)
{
	int fd;

	if (-1 == (fd = open("/dev/console", O_RDWR)))
		fatal_error("Cannot open /dev/console");

	if (ioctl(fd, VT_ACTIVATE, num) != 0)
		fatal_error("ioctl VT_ACTIVATE");

	if (ioctl(fd, VT_WAITACTIVE, num) != 0)
		fatal_error("ioctl VT_WAITACTIVE");

	(void) close(fd);
}

static unsigned short int change_to_vt(unsigned short int vt_num)
{
	int fd;
	unsigned short int old_vt;
	struct vt_stat vt_info;

	memset(&vt_info, 0, sizeof(struct vt_stat));

	if (-1 == (fd=open("/dev/console", O_RDONLY)))
		fatal_error("Couldn't open /dev/console");

	if (ioctl(fd, VT_GETSTATE,  &vt_info) != 0)
		fatal_error("ioctl VT_GETSTATE");

	(void) close (fd);

	old_vt = vt_info.v_active;

	chvt((int) vt_num); /* go there for information */

	return old_vt;
}

static void get_framebufferdata(char *device, struct fb_fix_screeninfo *fb_fixedinfo_p, struct fb_var_screeninfo *fb_varinfo_p, int verbose)
{
	int fd;

	/* now open framebuffer device */
	if (-1 == (fd=open(device, O_RDONLY)))
	{
		fprintf(stderr, "Error: Couldn't open %s.\n", device);
		exit(EXIT_FAILURE);
	}

	if (ioctl(fd, FBIOGET_VSCREENINFO, fb_varinfo_p) != 0)
		fatal_error("ioctl FBIOGET_VSCREENINFO");

	if (ioctl(fd, FBIOGET_FSCREENINFO, fb_fixedinfo_p) != 0)
		fatal_error("ioctl FBIOGET_FSCREENINFO");

	if (verbose)
	{
		fprintf(stderr, "Fixed framebuffer info:\n");
		fprintf(stderr, "id: \"%s\"\n", fb_fixedinfo_p->id);
		fprintf(stderr, "start of fb mem: %#lx\n", fb_fixedinfo_p->smem_start);
		fprintf(stderr, "length of fb mem: %u\n", fb_fixedinfo_p->smem_len);
		switch (fb_fixedinfo_p->type)
		{
			case FB_TYPE_PACKED_PIXELS:
				fprintf(stderr, "type: packed pixels\n");
				break;
			case FB_TYPE_PLANES:
				fprintf(stderr, "type: non interleaved planes\n");
				break;
			case FB_TYPE_INTERLEAVED_PLANES:
				fprintf(stderr, "type: interleaved planes\n");
				break;
			case FB_TYPE_TEXT:
				fprintf(stderr, "type: text/attributes\n");
				break;
			case FB_TYPE_VGA_PLANES:
				fprintf(stderr, "type: EGA/VGA planes\n");
				break;
			/*
			case FB_TYPE_FOURCC:
				fprintf(stderr, "type: identified by a V4L2 FOURCC\n");
				break;
			*/
			default:
				fprintf(stderr, "type: undefined!\n");
				break;
		}
		fprintf(stderr, "interleave for interleaved planes: %u\n", fb_fixedinfo_p->type_aux);
		switch (fb_fixedinfo_p->visual)
		{
			case FB_VISUAL_MONO01:
				fprintf(stderr, "visual: monochrome, B=1 W=0\n");
				break;
			case FB_VISUAL_MONO10:
				fprintf(stderr, "visual: monochrome, W=1 B=0\n");
				break;
			case FB_VISUAL_TRUECOLOR:
				fprintf(stderr, "visual: true color\n");
				break;
			case FB_VISUAL_PSEUDOCOLOR:
				fprintf(stderr, "visual: pseudo color\n");
				break;
			case FB_VISUAL_DIRECTCOLOR:
				fprintf(stderr, "visual: direct color\n");
				break;
			case FB_VISUAL_STATIC_PSEUDOCOLOR:
				fprintf(stderr, "visual: pseudo color readonly\n");
				break;
			/*
			case FB_VISUAL_FOURCC:
				fprintf(stderr, "visual: identified by a V4L2 FOURCC\n");
				break;
			*/
			default:
				fprintf(stderr, "visual: undefined!\n");
				break;
		}
		fprintf(stderr, "hw panning x step: %hu\n", fb_fixedinfo_p->xpanstep);
		fprintf(stderr, "hw panning y step: %hu\n", fb_fixedinfo_p->ypanstep);
		fprintf(stderr, "hw panning y wrap: %hu\n", fb_fixedinfo_p->ywrapstep);
		// Avoid a division by zero on < 8bpp fbs, do the maths w/ a FP dividend...
		fprintf(stderr, "line length: %u bytes (%.0f pixels)\n", fb_fixedinfo_p->line_length, fb_fixedinfo_p->line_length/((double)fb_varinfo_p->bits_per_pixel/8));
		fprintf(stderr, "start of mmio: %#lx\n", fb_fixedinfo_p->mmio_start);
		fprintf(stderr, "length of mmio: %u\n", fb_fixedinfo_p->mmio_len);
		fprintf(stderr, "accel chip/card: %u\n", fb_fixedinfo_p->accel);
		/*
		switch (fb_fixedinfo_p->capabilities)
		{
			case FB_CAP_FOURCC:
				fprintf(stderr, "capabilities: supports FOURCC-based formats\n");
				break;
			default:
				fprintf(stderr, "capabilities: undefined!\n");
				break;
		}
		*/


		fprintf(stderr, "\nVariable framebuffer info:\n");
		fprintf(stderr, "visible resolution: %ux%u\n", fb_varinfo_p->xres, fb_varinfo_p->yres);
		fprintf(stderr, "virtual resolution: %ux%u\n", fb_varinfo_p->xres_virtual, fb_varinfo_p->yres_virtual);
		fprintf(stderr, "offset from virtual to visible resolution: %ux%u\n", fb_varinfo_p->xoffset, fb_varinfo_p->yoffset);
		fprintf(stderr, "bits per pixel: %u\n", fb_varinfo_p->bits_per_pixel);
		fprintf(stderr, "grayscale: %s\n", fb_varinfo_p->grayscale ? "true" : "false");		// NOTE: >1 = FOURCC
		fprintf(stderr, "red:   bitfield offset: %u, bitfield length: %u, MSB is right: %s\n", fb_varinfo_p->red.offset, fb_varinfo_p->red.length, fb_varinfo_p->red.msb_right == 0 ? "no" : "yes");
		fprintf(stderr, "green: bitfield offset: %u, bitfield length: %u, MSB is right: %s\n", fb_varinfo_p->green.offset, fb_varinfo_p->green.length, fb_varinfo_p->green.msb_right == 0 ? "no" : "yes");
		fprintf(stderr, "blue:  bitfield offset: %u, bitfield length: %u, MSB is right: %s\n", fb_varinfo_p->blue.offset, fb_varinfo_p->blue.length, fb_varinfo_p->blue.msb_right == 0 ? "no" : "yes");
		fprintf(stderr, "alpha: bitfield offset: %u, bitfield length: %u, MSB is right: %s\n", fb_varinfo_p->transp.offset, fb_varinfo_p->transp.length, fb_varinfo_p->transp.msb_right == 0 ? "no" : "yes");
		fprintf(stderr, "pixel format: %s\n", fb_varinfo_p->nonstd == 0 ? "standard" : "non-standard");
		fprintf(stderr, "activate: %u\n", fb_varinfo_p->activate);
		fprintf(stderr, "height: %i mm\n", fb_varinfo_p->height);
		fprintf(stderr, "width: %i mm\n", fb_varinfo_p->width);
		fprintf(stderr, "obsolete accel_flags: %u\n", fb_varinfo_p->accel_flags);
		fprintf(stderr, "pixel clock: %u ps\n", fb_varinfo_p->pixclock);
		fprintf(stderr, "time from sync to picture (left_margin): %u pixclocks\n", fb_varinfo_p->left_margin);
		fprintf(stderr, "time from picture to sync (right_margin): %u pixclocks\n", fb_varinfo_p->right_margin);
		fprintf(stderr, "time from sync to picture (upper_margin): %u pixclocks\n", fb_varinfo_p->upper_margin);
		fprintf(stderr, "lower_margin: %u pixclocks\n", fb_varinfo_p->lower_margin);
		fprintf(stderr, "length of horizontal sync: %u pixclocks\n", fb_varinfo_p->hsync_len);
		fprintf(stderr, "length of vertical sync: %u pixclocks\n", fb_varinfo_p->vsync_len);
		switch (fb_varinfo_p->sync)
		{
			case FB_SYNC_HOR_HIGH_ACT:
				fprintf(stderr, "sync: horizontal sync high active\n");
				break;
			case FB_SYNC_VERT_HIGH_ACT:
				fprintf(stderr, "sync: vertical sync high active\n");
				break;
			case FB_SYNC_EXT:
				fprintf(stderr, "sync: external sync\n");
				break;
			case FB_SYNC_COMP_HIGH_ACT:
				fprintf(stderr, "sync: composite sync high active\n");
				break;
			case FB_SYNC_BROADCAST:
				fprintf(stderr, "sync: broadcast video timings\n");
				break;
			case FB_SYNC_ON_GREEN:
				fprintf(stderr, "sync: sync on green\n");
				break;
			default:
				fprintf(stderr, "sync: undefined!\n");
				break;
		}
		switch (fb_varinfo_p->vmode)
		{
			case FB_VMODE_NONINTERLACED:
				fprintf(stderr, "vmode: non interlaced\n");
				break;
			case FB_VMODE_INTERLACED:
				fprintf(stderr, "vmode: interlaced\n");
				break;
			case FB_VMODE_DOUBLE:
				fprintf(stderr, "vmode: double scan\n");
				break;
			case FB_VMODE_ODD_FLD_FIRST:
				fprintf(stderr, "vmode: interlaced: top line first\n");
				break;
			default:
				fprintf(stderr, "vmode: undefined or mask (%u)!\n", fb_varinfo_p->vmode);
				break;
		}
		fprintf(stderr, "rotate: %u\n", fb_varinfo_p->rotate);
		/*
		fprintf(stderr, "colorspace for FOURCC-based modes: %u\n", fb_varinfo_p->colorspace);
		*/
	}
	srcBlue = fb_varinfo_p->blue.offset >> 3;
	srcGreen = fb_varinfo_p->green.offset >> 3;
	srcRed = fb_varinfo_p->red.offset >> 3;

	if (fb_varinfo_p->transp.length > 0) {
		srcAlpha = fb_varinfo_p->transp.offset >> 3;
		srcAlphaIsUsed = 1;
	} else {
		srcAlphaIsUsed = 0; // not used
	}

	if (verbose)
	{
		fprintf(stderr, "\nFBGrab info:\n");
		fprintf(stderr, "srcBlue: %u, srcGreen: %u, srcRed: %u, srcAlpha: %d\n", srcBlue, srcGreen, srcRed, srcAlpha);
		fprintf(stderr, "\n");
	}

	(void) close(fd);
}

static void read_framebuffer(char *device, size_t bytes, unsigned char *buf_p, off_t skip_bytes)
{
	int fd;
	ssize_t read_bytes;

	if (-1 == (fd=open(device, O_RDONLY)))
	{
		fprintf(stderr, "Error: Couldn't open %s.\n", device);
		exit(EXIT_FAILURE);
	}

	if (skip_bytes) {
		lseek(fd, skip_bytes, SEEK_SET);
	}

	read_bytes = read(fd, buf_p, bytes);

	if (buf_p == NULL || read_bytes != (ssize_t) bytes)
	{
		fprintf(stderr, "Warning: only read %d of %d bytes\n", read_bytes, bytes);
		fatal_error("Error: Not enough memory or data");
	}
}

static void convert1555to32(unsigned int width, unsigned int height,
				unsigned char *inbuffer,
				unsigned char *outbuffer)
{
	unsigned int i;

	for (i=0; i < height*width*2; i+=2)
	{
		outbuffer[(i<<1)+Blue] = (unsigned char) ((inbuffer[i+1] & 0x7C) << 1);		// B
		outbuffer[(i<<1)+Green] = (unsigned char) ((((inbuffer[i+1] & 0x3) << 3) |
						((inbuffer[i] & 0xE0) >> 5)) << 3);		// G
		outbuffer[(i<<1)+Red] = (unsigned char) ((inbuffer[i] & 0x1F) << 3);		// R
		outbuffer[(i<<1)+Alpha] = (inbuffer[i] & 0x80) ? 0xFF : 0;			// A
	}
}

static void convert565to32(unsigned int width, unsigned int height,
				unsigned char *inbuffer,
				unsigned char *outbuffer)
{
	unsigned int i;
	uint16_t v, b, g, r;
	for (i=0; i < height*width*2; i+=2)
	{
#ifdef __BIG_ENDIAN__
		v = (uint16_t) ((inbuffer[i] << 8) + inbuffer[i+1]);
#else
		v = (uint16_t) ((inbuffer[i+1] << 8) + inbuffer[i]);
#endif

		// NOTE: Once again, ported from KOReader ;)
		//	cf. https://github.com/koreader/koreader-base/blob/master/ffi/blitbuffer.lua#L330
		b = v & 0x001F;
		outbuffer[(i<<1)+Blue] = (unsigned char) ((b << 3) + (b >> 2));		// B
		g = (v >> 5) & 0x3F;
		outbuffer[(i<<1)+Green] = (unsigned char) ((g << 2) + (g >> 4));	// G
		r = (uint16_t) (v >> 11);
		outbuffer[(i<<1)+Red] = (unsigned char) ((r << 3) + (r >> 2));		// R

		// NOTE: Here's FBGrab's original implementation,
		//	which appears to be ever-so-slightly off (white @ 248,252,248), at least for Kobo fbs...
		/*
		outbuffer[(i<<1)+Blue] = (unsigned char) ((v << 3) & 0xF8);		// B
		outbuffer[(i<<1)+Green] = (unsigned char) ((v >> 3) & 0xFC);		// G
		outbuffer[(i<<1)+Red] = (unsigned char) ((v >> 8) & 0xF8);		// R
		*/

		// NOTE: And the perl script implementation, which is as off as FBGrab's original implementation...
		//	cf. http://www.mobileread.com/forums/showpost.php?p=2366353&postcount=1
		/*
		outbuffer[(i<<1)+Blue] = (unsigned char) ((v & 0x001F) << 3);		// B
		outbuffer[(i<<1)+Green] = (unsigned char) ((v & 0x07E0) >> 3);		// G
		outbuffer[(i<<1)+Red] = (unsigned char) ((v & 0xF800) >> 8);		// R
		*/

		outbuffer[(i<<1)+Alpha] = 0xFF;						// A
	}
}

static void convert888to32(unsigned int width, unsigned int height,
				unsigned char *inbuffer,
				unsigned char *outbuffer)
{
	unsigned int i;

	for (i=0; i < height*width; i++)
	{
		outbuffer[(i<<2)+Blue] = inbuffer[i*3+srcBlue];		// B
		outbuffer[(i<<2)+Green] = inbuffer[i*3+srcGreen];	// G
		outbuffer[(i<<2)+Red] = inbuffer[i*3+srcRed];		// R
		outbuffer[(i<<2)+Alpha] = 0xFF;				// A
	}
}

static void convert8888to32(unsigned int width, unsigned int height,
				unsigned char *inbuffer,
				unsigned char *outbuffer)
{
	unsigned int i;

	for (i=0; i < height*width; i++)
	{
		outbuffer[(i<<2)+Blue] = inbuffer[i*4+srcBlue];				// B
		outbuffer[(i<<2)+Green] = inbuffer[i*4+srcGreen];			// G
		outbuffer[(i<<2)+Red] = inbuffer[i*4+srcRed];				// R
		outbuffer[(i<<2)+Alpha] = srcAlphaIsUsed? inbuffer[i*4+srcAlpha] : 0;	// A
	}
}

static void convertgray8to32(unsigned int width, unsigned int height,
				unsigned char *inbuffer,
				unsigned char *outbuffer,
				unsigned int invert)
{
	unsigned int i;

	for (i=0; i < height*width; i++)
	{
		// NOTE: Gray means B = G = R == i, and RGBA offsets == 0.
		// Needs to be inversed on the K4 for some reason (weird driver shims mixing eink_fb & mxc_epdc_fb)...
		outbuffer[(i<<2)+Blue] = outbuffer[(i<<2)+Green] = outbuffer[(i<<2)+Red] = invert? (inbuffer[i] ^ 0xFF) : inbuffer[i];	// BGR
		outbuffer[(i<<2)+Alpha] = 0xFF;											// A
	}
}

static void convertgray4to32(unsigned int width, unsigned int height,
				unsigned char *inbuffer,
				unsigned char *outbuffer)
{
	unsigned int i;

	for (i=0; i < (height*width); i+=2)
	{
		// NOTE: Still grayscale, so B = G = R, and the RGBA offsets are useless.
		// AFAIK, always needs to be inverted on the Kindle devices sporting that kind of fb.
		// Ported from KOReader's https://github.com/koreader/koreader-base/blob/master/ffi/blitbuffer.lua#L245
		// See also https://github.com/dpavlin/k3libre/blob/master/fb2pgm.pl for an alternative (w/ a slight contrast issue [too light]),
		// and https://github.com/koreader/koreader/wiki/porting & https://github.com/koreader/koreader-base/blob/c109f02680ca55ba2515bf11d28dd928b19df416/einkfb.c
		// for more background info.
		// First pixel
		outbuffer[(i<<2)+Blue] = outbuffer[(i<<2)+Green] = outbuffer[(i<<2)+Red] = (unsigned char) (((inbuffer[i/2] & 0xF0) | ((inbuffer[i/2] & 0xF0) >> 4)) ^ 0xFF);	// BGR
		outbuffer[(i<<2)+Alpha] = 0xFF;																// A

		// Second pixel
		outbuffer[((i+1)<<2)+Blue] = outbuffer[((i+1)<<2)+Green] = outbuffer[((i+1)<<2)+Red] = (unsigned char) (((inbuffer[i/2] & 0x0F) * 0x11) ^ 0xFF);			// BGR
		outbuffer[((i+1)<<2)+Alpha] = 0xFF;																// A
	}
}


static void write_PNG(unsigned char *outbuffer, char *filename,
			unsigned int width, unsigned int height, int interlace, int compression, int x_offset, int y_offset)
{
	unsigned int i;
	png_bytep row_pointers[height];
	png_structp png_ptr;
	png_infop info_ptr;
	FILE *outfile;

	if (strcmp(filename, "-") == 0)
		outfile = stdout;
	else {
		outfile = fopen(filename, "wb");
		if (!outfile)
		{
			fprintf(stderr, "Error: Couldn't fopen %s.\n", filename);
			exit(EXIT_FAILURE);
		}
	}

	// NOTE: Kobo hack: crop what's hidden by the bezel (on the top).
	if (y_offset > 0)
	{
		for (i=0; i<height; i++)
			row_pointers[i] = outbuffer + (i + (unsigned int) y_offset) * 4 * width;
	}
	// Otherwise, go as planned.
	else
	{
		for (i=0; i<height; i++)
			row_pointers[i] = outbuffer + i * 4 * width;
	}

	// NOTE: Kindle/Kobo hack: crop the garbage on the (right) side.
	// Note that we need to have read outbuffer with its original width before tweaking width itself.
	width -= (unsigned int) abs(x_offset);
	// FIXME: Note that we don't handle cropping the *left* pixels. Good thing we don't need to ;p.
	if (x_offset > 0)
		fprintf(stderr, "Unsupported positive x_offset (%i)! Data *will* be lost!\n", x_offset);


	png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
		(png_voidp) NULL, (png_error_ptr) NULL, (png_error_ptr) NULL);

	if (!png_ptr)
		fatal_error("Error: Couldn't create PNG write struct.");

	info_ptr = png_create_info_struct(png_ptr);
	if (!info_ptr)
	{
		png_destroy_write_struct(&png_ptr, (png_infopp) NULL);
		fatal_error("Error: Couldn't create PNG info struct.");
	}

	png_init_io(png_ptr, outfile);

	png_set_IHDR(png_ptr, info_ptr, width, height,
			8, PNG_COLOR_TYPE_RGB_ALPHA, interlace,
			PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

	png_set_bgr(png_ptr);

	png_set_compression_level(png_ptr, compression);

	png_write_info(png_ptr, info_ptr);

	fprintf(stderr, "Now writing PNG file (compression %d)\n", compression);

	png_write_image(png_ptr, row_pointers);

	png_write_end(png_ptr, info_ptr);
	/* puh, done, now freeing memory... */
	png_destroy_write_struct(&png_ptr, &info_ptr);

	if (outfile != NULL)
		(void) fclose(outfile);
}

static void convert_and_write(unsigned char *inbuffer, char *filename,
				unsigned int width, unsigned int height, unsigned int bits, int interlace, int compression, unsigned int invert, int x_offset, int y_offset)
{
	size_t bufsize = (size_t) width * height * 4;

	unsigned char *outbuffer = malloc(bufsize);

	if (outbuffer == NULL)
		fatal_error("Not enough memory");

	memset(outbuffer, 0, bufsize);

	fprintf(stderr, "Converting image from %ubpp\n", bits);

	// NOTE: Tweak height according to the offsets between the two steps to handle the Kobo cropping.
	unsigned int cropped_height = height - (unsigned int) abs(y_offset);

	switch(bits)
	{
		case 4:
			convertgray4to32(width, height, inbuffer, outbuffer);
			write_PNG(outbuffer, filename, width, cropped_height, interlace, compression, x_offset, y_offset);
			break;
		case 8:
			convertgray8to32(width, height, inbuffer, outbuffer, invert);
			write_PNG(outbuffer, filename, width, cropped_height, interlace, compression, x_offset, y_offset);
			break;
		case 15:
			convert1555to32(width, height, inbuffer, outbuffer);
			write_PNG(outbuffer, filename, width, cropped_height, interlace, compression, x_offset, y_offset);
			break;
		case 16:
			convert565to32(width, height, inbuffer, outbuffer);
			write_PNG(outbuffer, filename, width, cropped_height, interlace, compression, x_offset, y_offset);
			break;
		case 24:
			convert888to32(width, height, inbuffer, outbuffer);
			write_PNG(outbuffer, filename, width, cropped_height, interlace, compression, x_offset, y_offset);
			break;
		case 32:
			convert8888to32(width, height, inbuffer, outbuffer);
			write_PNG(outbuffer, filename, width, cropped_height, interlace, compression, x_offset, y_offset);
			break;
		default:
			fprintf(stderr, "%d bits per pixel are not supported!\n", bits);
			exit(EXIT_FAILURE);
	}

	(void) free(outbuffer);
}


/********
 * MAIN *
 ********/

int main(int argc, char **argv)
{
	unsigned char *buf_p;
	char *device = NULL;
	char *outfile = argv[argc-1];
	int optc;
	int vt_num=UNDEFINED;
	unsigned int bitdepth=0, height=0, width=0;
	int old_vt=UNDEFINED;
	size_t buf_size;
	char infile[MAX_LEN];
	struct fb_fix_screeninfo fb_fixedinfo;
	struct fb_var_screeninfo fb_varinfo;
	int waitbfg=0; /* wait before grabbing (for -C )... */
	int interlace = PNG_INTERLACE_NONE;
	int noautocrop = 0;
	int verbose = 0;
	int png_compression = Z_DEFAULT_COMPRESSION;
	off_t skip_bytes = 0;
	unsigned int invert = 0;
	int x_offset=0, y_offset = 0;

	memset(infile, 0, MAX_LEN);
	memset(&fb_varinfo, 0, sizeof(struct fb_var_screeninfo));


	for(;;)
	{
		optc = getopt(argc, argv, "f:z:w:b:h:iC:c:d:s:?vk");
		if (optc == -1)
			break;
		switch (optc)
		{
			/* please keep this list alphabetical */
			case 'b':
				bitdepth = (unsigned int) atoi(optarg);
				break;
			case 'C':
				waitbfg = 1;
				/*@fallthrough@*/
			case 'c':
				vt_num = atoi(optarg);
				break;
			case 'd':
				device = optarg;
				break;
			case 'f':
				strncpy(infile, optarg, MAX_LEN);
				break;
			case 'h':
				height = (unsigned int) atoi(optarg);
				break;
			case '?':
				help(argv[0]);
				return 1;
			case 'i':
				interlace = PNG_INTERLACE_ADAM7;
				break;
			case 'k':
				noautocrop = 1;
				break;
			case 'v':
				verbose = 1;
				break;
			case 's':
				(void) sleep((unsigned int) atoi(optarg));
				break;
			case 'w':
				width = (unsigned int) atoi(optarg);
				break;
			case 'z':
				png_compression = atoi(optarg);
				break;
			default:
				usage(argv[0]);
		}
	}

	if ((optind == argc) || (1 != argc-optind))
	{
		usage(argv[0]);
		return 1;
	}

	if (UNDEFINED != vt_num)
	{
		old_vt = (int) change_to_vt((unsigned short int) vt_num);
		if (waitbfg != 0) (void) sleep(3);
	}

	if (strlen(infile) > 0)
	{
		if (!bitdepth || !width || !height)
		{
			fprintf(stderr, "Width, height and bitdepth are mandatory when reading from file\n");
			exit(EXIT_FAILURE);
		}
	}
	else
	{
		if (NULL == device)
		{
			device = getenv("FRAMEBUFFER");
			if (NULL == device)
			{
				device = DEFAULT_FB;
			}
		}

		get_framebufferdata(device, &fb_fixedinfo, &fb_varinfo, verbose);

		if (!bitdepth)
			bitdepth = fb_varinfo.bits_per_pixel;

		if (!width)
		{
			// NOTE: Kindle/Kobo hack (Freescale iMX5 & iMX6). Use the virtual xres if driver is mxc_epdc_fb @ 8bpp (Kindle) or @ 16bpp (Kobo) to get a correct dump on those devices ...
			if (strcmp(fb_fixedinfo.id, "mxc_epdc_fb") == 0 && (bitdepth == 8 || bitdepth == 16))
			{
				width = fb_varinfo.xres_virtual;
				// ... and store the x offset in order to crop the 8~10px of garbage when writing the PNG ...
				if (fb_varinfo.xres_virtual != fb_varinfo.xres)
					x_offset = (int) -(fb_varinfo.xres_virtual - fb_varinfo.xres);

				// NOTE: Kobo hack. Crop the pixels hidden behind the bezel on some models... Unfortunately we don't have a better way to detect the model or the amount/location of the hidden pixels...
				if (bitdepth == 16)
				{
					// Get the model from Kobo's script...
					FILE *fp = popen("/bin/kobo_config.sh 2>/dev/null", "r");
					if (!fp) {
						fprintf(stderr, "** Couldn't detect Kobo model (not running on a Kobo?)! **\n");
					}
					else
					{
						char kobo_model[16];
						while (fgets(kobo_model, sizeof(kobo_model)-1, fp))
						{
							// Since we don't have a way to autodetect this, hardcode it...
							if (strncmp(kobo_model, "dahlia", 6) == 0)
								y_offset = 11;	// Aura H2O: top 11 pixels.
							/*
							 * FIXME: To be confirmed, I don't have an Aura. Kobo says 10, KOReader uses 12? Can't find a goddamn screenshot to confirm...
							else if (strncmp(kobo_model, "phoenix", 7) == 0)
								y_offset = -10;	// Aura: bottom 10 pixels.
							*/
						}
						pclose(fp);
					}
				}
			}
			else
			{
				width = fb_varinfo.xres;
			}
		}

		if (!height)
			height = fb_varinfo.yres;

		skip_bytes = (off_t) ((fb_varinfo.yoffset * fb_varinfo.xres) * (fb_varinfo.bits_per_pixel >> 3));
		fprintf(stderr, "Skipping %jd bytes\n", (intmax_t) skip_bytes);

		// Disable automagic cropping on request
		if (noautocrop)
		{
			x_offset = 0;
			y_offset = 0;
		}

		// NOTE: Kindle hack. On the K4, we need to invert the fb like on legacy devices. Crappy detection based on bpp & xres/yres vs. virtual inconsistencies.
		// We might also be able to get away with only the id (eink_fb), but I vaguely remember that being unreliable on some devices between portrait and landscape mode...
		// Cf. http://www.mobileread.com/forums/showpost.php?p=2045889&postcount=1 & http://www.mobileread.com/forums/showpost.php?p=2832686&postcount=10 for more background...
		// The id inconsistencies appear to have been fixed on FW 4.1.1, so, let's handle the rest... :).
		if (strcmp(fb_fixedinfo.id, "eink_fb") == 0 && bitdepth == 8 && fb_varinfo.xres_virtual == fb_varinfo.xres && fb_varinfo.yres == fb_varinfo.yres_virtual)
			invert = 1;

		fprintf(stderr, "Resolution: %ux%u", width, height);
		// Only print the cropping status when one is applied
		if (x_offset != 0 || y_offset != 0)
		{
			fprintf(stderr, " -> %ux%u (cropping:", width - (unsigned int) abs(x_offset), height - (unsigned int) abs(y_offset));
			if (x_offset != 0)
				fprintf(stderr, " %i %s pixels", abs(x_offset), (x_offset > 0)? "leftmost" : "rightmost");
			if (x_offset != 0 && y_offset != 0)
				fprintf(stderr, " &");
			if (y_offset != 0)
				fprintf(stderr, " %s %i pixels", (y_offset > 0)? "top" : "bottom", abs(y_offset));
			fprintf(stderr, ")");
		}
		fprintf(stderr, " @ depth %u", bitdepth);
		// Only print the invert status when it's actually needed (8bpp, to differentiate between the K4 and K5 devices)
		if (bitdepth == 8)
			fprintf(stderr, " (invert: %s)\n", invert? "yes" : "no");
		else
			fprintf(stderr, "\n");

		strncpy(infile, device, MAX_LEN - 1);
	}

	buf_size = width * height * ((bitdepth + 7) >> 3);
	// NOTE: 4bpp hack!
	if (bitdepth == 4)
		buf_size /= 2;

	buf_p = malloc(buf_size);

	if(buf_p == NULL)
		fatal_error("Not enough memory");

	memset(buf_p, 0, buf_size);

	read_framebuffer(infile, buf_size, buf_p, skip_bytes);

	if (UNDEFINED != old_vt)
		(void) change_to_vt((unsigned short int) old_vt);

	convert_and_write(buf_p, outfile, width, height, bitdepth, interlace, png_compression, invert, x_offset, y_offset);

	(void) free(buf_p);

	return 0;
}
