#!/usr/bin/env python
"Add metadata and thumbnails to files for Irex Iliad."
import optparse, sys, os, shutil, xml.dom.minidom, ImageOps, datetime, configobj

#-------- Option parsing --------
USAGE = "usage: %prog [options] file1 file2"
PARSER = optparse.OptionParser(usage=USAGE)
PARSER.add_option("-o", "--out", dest="out", \
  help="Output directory for prepared files. Defaults to current directory.")
PARSER.add_option("-m", "--move", action="store_true", dest="move", \
  help="Move files to destination, rather copying it.")
PARSER.add_option("-a", "--archive", dest="archive", \
  help="Place a copy of the original file in the specified directory.")
PARSER.add_option("-f", "--fullscreen", action="store_true", dest="fullscreen", \
  help="Tell Iliad to open the file in fullscreen mode, if possible.")
PARSER.add_option("-c", "--crop", action="store_true", dest="crop", \
  help="Crop file margins, if possible.")


#-------- Generic file class --------
class FileInfo():
    "store file metadata"
    def __init__(self, filename=None):
        self.file = filename
        self.title = None
        self.author = None
        self.pages = None
        self.cover = None
        self.date = None
    def fullscreen(self, manifest):
        return manifest
    def crop(self, out_file):
        return False

#-------- Module loading --------
def load_modules():
    "Attempt to load parsing modules."
    modulepath = ("/usr/share/sendtoiliad", "/usr/local/share/sendtoiliad", \
      os.path.join(os.path.expanduser("~"), ".sendtoiliad/modules"))
    modules = {}
    filetypes = {}
    for location in modulepath:
        if os.path.exists(location):
            sys.path.append(location)
            files = os.listdir(location)
            files = [i for i in files if os.path.splitext(i)[1] == ".py"]
            filename_to_module_name = lambda f: os.path.splitext(f)[0]
            module_names = map(filename_to_module_name, files)
            for module in module_names:
                try:
                    modules[module] = __import__(module)
                    filetypes[module.upper()] = modules[module].init(FileInfo)
                    print module.upper() + " support loaded."
                except:
                    print module + " module could not be loaded. "
                    continue
    return filetypes


#-------- Progress indicator --------


#-------- Create thumbnail --------
def make_cover(cover, out):
    """Takes a PIL image and returns the image transformed
    to be used as a thumbnail on the Iliad"""
    if cover:
        try:
            coversize = (69, 93)
            out = os.path.join(out, "cover.png")
            cover = ImageOps.grayscale(cover)
            cover.thumbnail(coversize)
            cover.save(out)
            return True
        except: return False
    else:
        return False


#-------- Create manifest.xml --------
def write_manifest(file_info, out):
    """Creates a manifest.xml file using the metadata from file_info
    at the specified location"""
    manifest = xml.dom.minidom.Document()
    package = manifest.createElement("package")
    manifest.appendChild(package)
    metadata = manifest.createElement("metadata")
    package.appendChild(metadata)
    dcmetadata = manifest.createElement("dc-metadata")
    metadata.appendChild(dcmetadata)

    #Title
    title = manifest.createElement("Title")
    if file_info.title:
        title.appendChild(manifest.createTextNode(file_info.title))
    else:
        title.appendChild(manifest.createTextNode( \
        os.path.splitext(os.path.split(file_info.file)[1])[0]))


    dcmetadata.appendChild(title)

    #Description
    description = manifest.createElement("Description")
    if file_info.author:
        file_description = file_info.author
    else:
        file_description = ""
    if file_info.pages:
        file_description += "  (%s pages)" % str(file_info.pages)
    description.appendChild(manifest.createTextNode(file_description))
    dcmetadata.appendChild(description)

    #Date
    date = manifest.createElement("Date")
    if file_info.date:
        date.appendChild(manifest.createTextNode( \
        file_info.date.strftime("%Y-%m-%dT00:00:00")))
    else:
        date.appendChild(manifest.createTextNode( \
        datetime.date.today().strftime("%Y-%m-%dT00:00:00")))
    dcmetadata.appendChild(date)

    #Unused attributes
    dcmetadata.appendChild(manifest.createElement("Format"))
    dcmetadata.appendChild(manifest.createElement("Identifier"))
    dcmetadata.appendChild(manifest.createElement("Language"))
    dcmetadata.appendChild(manifest.createElement("Type"))

    #Y-Metadata section
    ymetadata = manifest.createElement("y-metadata")
    metadata.appendChild(ymetadata)

    startpage = manifest.createElement("startpage")
    startpage.appendChild(manifest.createTextNode( \
        os.path.split(file_info.file)[1]))
    ymetadata.appendChild(startpage)

    if file_info.cover:
        cover = manifest.createElement("image")
        cover.appendChild(manifest.createTextNode("cover.png"))
        ymetadata.appendChild(cover)

    version = manifest.createElement("version")
    version.appendChild(manifest.createTextNode("000"))
    ymetadata.appendChild(version)

    if options.fullscreen:
        manifest = file_info.fullscreen(manifest)

    manifest_file = open(os.path.join(out, "manifest.xml"),"w")
    manifest.writexml(manifest_file, "", "", "", "UTF-8")
    manifest_file.close()


#-------- File scanner --------

(options, args) = PARSER.parse_args()

SETTINGS_FILE = os.path.join(os.path.expanduser("~"), ".sendtoiliad/settings")
try:
    config = configobj.ConfigObj(SETTINGS_FILE)
    if 'out' in config:
        options.out = config['out']
    if options.archive in config:
        option.archive = config['archive']
    if 'move' in config:
        options.move = config['move']
    if 'fullscreen' in config:
        options.fullscreen = config['fullscreen']
    if 'crop' in config:
        options.crop = config['crop']
except:
    pass
finally:
    options.out = options.out or './'
    options.move = options.move or False
    options.fullscreen = options.fullscreen or False
    options.crop = options.crop or False

def main():
    "Main scanning loop."
    filetypes = load_modules()

    filelist = [os.path.normcase(f) for f in args]

    for f in filelist:
        filename = os.path.split(f)[1]
        filetype = os.path.splitext(f)[1][1:].upper()
        try:
            file_info = filetypes[filetype](f)
            if not os.path.exists(f): raise IOError
        except KeyError:
            print "Skipping %s: File type is not supported." % filename
            continue
        except IOError:
            if os.path.isdir(f): pass
            else: print "Skipping %s: Could not open file." % filename
            continue

        print "Processing %s   " % filename,
        out = os.path.join(options.out, os.path.splitext(os.path.split(f)[1])[0])
        if not os.path.exists(out):
            try:
            	os.mkdir(out)
            	shutil.copyfile(f, os.path.join(out, filename))
            	make_cover(file_info.cover, out)
            	write_manifest(file_info, out)
            	if options.archive:
            	    shutil.copy(f, os.path.join(options.archive, filename))
            	if options.move: os.remove(f)
            	print " done."
            except:
                print " error."
                if os.path.exists(os.path.join(out, 'cover.png')):
                    os.remove(os.path.join(out, 'cover.png'))
                if os.path.exists(os.path.join(out, 'manifest.xml')):
                    os.remove(os.path.join(out, 'manifest.xml'))
                if os.path.exists(os.path.join(out, filename)):
                    os.remove(os.path.join(out, filename))
                if os.path.exists(out):
                    os.rmdir(out)
        else:
            print "Output file already exists"

main()
