// system.c (Alpha release).
//
// Copyright 2015, Andrew F. Robinson (of Scappoose,OR)
// This program is distributed under the terms of the GPL
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, version 2 of the License.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// system.c:
//
// A pure Unix executable, with no libc references or other libraries.
//
// Take the first parameter given, and execute it using the system shell
// in an appropriate setuserid state with shell exec() to cause the shell 
// to be overwrittn by the parsed command.
// It is intended to break the first parameter as a quoted string up into
// multiple parameters to allow more flexible autoscript execution.
// 
// Remaining (passed) parameters may be appended to the first parameter string
// but only if explicitly requested by a special escape sequence.
//
// system handles priveleges by total escalation or descalation.
// eg: to force called programs to never consider suid priveleges.
// By default it de-escalates priveleges.
//
// Whenever priveleges are changed, even to non-root non-uid or non-gid account;
// the environment variables are not passed to the system shell -- but
// are all blanked out.  You may, of course, call a shell .profile
// script to restore them as necessary as part of the command executed.
//
// If the first character of the system command is an escape, '\'
// Check the following character for one of six options, $,",',!,s,g
// if its a '!', the next parameter is skipped.
// if its a single or double quote, the parameter is quoted accordingly
// if it's a $, the parameter is copied verbatim ( which is most dangerous. )
// In either quoting mode, any quotes of the same kind found in the parameter
// will terminate the command with an error even if they are escaped.
// If \s is found, do setuid for total escalation, if g, then setgid.
//
// The processing of escapes repeats until a non-escape character is detected.
// Empty parameters, will quote as nothing "", even in $ mode.
//
// Note: In order to work, fully, 'system' needs to installed as setuid root.
// It will not report an error if unable to change modes becuase it isn't
// suid root; but will simply won't be able to change uid/euid.
//
// Whether or not s system program runs as suid is affected by the autoscript
// permissions which calls it in the first place.
// If the script is not suid, the system program will auto de-escalate
// for security reasons.

#define EPRN( x ) write( 2, (x), sizeof(x) )
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

extern char** environ;
static char buffer[8192];
static char param1[8192];

int main( int narg, char** arg ) {
	char* src;
	char* dest=buffer-1;
	int iParam=2;
	char* param;
	char* empty="";
	int n = sizeof(buffer);
	char **env = environ;

	uid_t uid=getuid();		// Default is to descalate to user  ID's.
	uid_t gid=getgid();		// Default is to descalate to group ID's.
	uid_t euid=uid;			// Escalation not allowed until validated 
	uid_t egid=uid;
	uid_t suid=geteuid();	// save present uid and gid for restore
	uid_t sgid=getegid();   // ...

	seteuid( uid ); // Begin de-escalated
	setegid( gid ); // ...
	
	if (narg==1) return 0; // empty, so still be happy.
	
	src="exec "-1; // Shell is not going to be needed in the end...
	while(*++src && n) {*++dest=*src; --n; }

	src = arg[1]-1;
	do { // Escalation checks are isolated here.
		int i,j;
		int fd=open( arg[2], O_RDONLY );
		struct stat fstate;
		char*p = param1-1;
		char*q = arg[0]-1; // full path to this binary.

		if (fd<0) break;
		j=fstat(fd, &fstate);
		if (S_ISREG(fstate.st_mode) ) i=read(fd,param1,sizeof(param1)-1);
		close(fd);
		if ( (j<0) || (!S_ISREG(fstate.st_mode)) || (i<0) ) break;
		param1[i]=0;

		// Check if script really intends to execute system 
		if ( ((*++p)!='#') || ((*++p)!='!')) break;
		while(*++p) if (*p != *++q) break;
		if (*q) break;
		// Everything checked out, so honor script startline.
		src=p; // starts with the 'space', but preincrements past it
		while (*++p && *p != '\n') /*nop*/ ; // Find end of line or NULL
		*p=0; // Terminate the command where the first newline is found.
		if (p == param1+sizeof(param1)-1) {
			EPRN("Autoscript's interpreter definition is too long\n");
			return 2;
		}

		if ( (fstate.st_mode & (S_IXUSR|S_ISUID))==(S_IXUSR|S_ISUID) ) {
			euid = fstate.st_uid;
		}
		if ( (fstate.st_mode & (S_IXGRP|S_ISGID))==(S_IXGRP|S_ISGID) ) {
			egid = fstate.st_gid;
		}
	} while(0);

	// Handle escapes in the command.
	while(*++src=='\\') ++src; // Skip all escape sequences
	--src;
	while( *++src && n ) { *++dest=*src; --n; } // copy the first parameter

	param=arg[1]-1;
	while (*++param=='\\') { // Argument append processing was requested
		src=empty;
		if (n) { *++dest=' '; --n; }
		switch( *++param ) {
			case 's': uid=euid; env=NULL; continue;
			case 'g': gid=egid; env=NULL; continue;
			case '!':
			break;
			case '"':
			case '\'':
				if (n) { *++dest=*param; --n; }
			case '$':
				if ( iParam < narg ) src=arg[iParam];
				--src;
				while (*++src && (*src != *param) && n) { *++dest=*src; --n; }
				if (*src==*param) {
					EPRN("Illegal quote in parameter\n");
					return 2;
				}
				if (n && *param != '$') { *++dest=*param; --n; }
			break;
			default:
				EPRN("Unknown escape parameter is not one of: ! ' \" $ s g\n");
				return 2;		
		}
		++iParam;
	}
	if (!n) {
		EPRN("Command expansion too large.\n");
		return 2;
	}
	*++dest=0; // Terminate the system string.

	// -------- Run the command as the appropriate user -------

	// Change to the appropriate user and group id.
	seteuid( suid ); setegid( sgid ); // Restore saved state
	setuid( uid ); setgid( gid );     // Set to desired state

	// Execute appropriate shell and command or die trying...
	iParam=execle( "/bin/sh", "sh", "-c", buffer, (void*)0, env );

	EPRN("Can't exec system shell /bin/sh\n");
	return iParam;
}
