#!/usr/bin/env python2
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__ = 'GPL v3'
__copyright__ = '2015, David Forrester <davidfor@internode.on.net>'
__docformat__ = 'markdown en'

import os

from calibre.devices.kobo.driver import KOBOTOUCH
from calibre.devices.usbms.driver import debug_print

class KOBOTOUCH148(KOBOTOUCH):
    '''
    Driver to enable new Kobo devices in calibre 1.48.

    NOTE: The only reason this driver exists is to allow use of the newer Kobo devices with calibre 1.48.
    And it will only be supported on platforms that calibre v2 cannot run on. So, don't ask if about it 
    if you can install calibre v2.
    '''

    name = 'KOBOTOUCH148'
    author = 'David Forrester'
    description = 'Communicate with the Kobo ereaders releases since calibre 2.0.0 was released. Currently supports the Aura H2O, Glo HD and Touch 2. DO NOT use with anything other than calibre v1.48'

    minimum_calibre_version = (1, 48, 0)
    version = (1, 0, 1)

    supported_dbversion             = 125
    max_supported_fwversion         = (3, 19, 5613)
    min_aurah2o_fwversion           = (3,  7, 0)
    min_reviews_fwversion           = (3, 12, 0)
    min_glohd_fwversion             = (3, 14, 0)

    AURA_H2O_PRODUCT_ID = [0x4213]
    GLO_HD_PRODUCT_ID   = [0x4223]
    TOUCH2_PRODUCT_ID   = [0x4224]
    PRODUCT_ID          = AURA_H2O_PRODUCT_ID + GLO_HD_PRODUCT_ID + TOUCH2_PRODUCT_ID + KOBOTOUCH.PRODUCT_ID
    debug_print("KOBOTOUCH148: PRODUCT_ID= [%s]" % (", ".join(hex(n) for n in PRODUCT_ID)))
    
    def isAuraH2O(self):
        return self.detected_device.idProduct in self.AURA_H2O_PRODUCT_ID

    def isGloHD(self):
        return self.detected_device.idProduct in self.GLO_HD_PRODUCT_ID
    
    def isTouch2(self):
        return self.detected_device.idProduct in self.TOUCH2_PRODUCT_ID

    def cover_file_endings(self):
        if self.isAuraH2O() or self.isGloHD():
            cover_endings = self.AURA_HD_COVER_FILE_ENDINGS
        else:
            cover_endings = super(KOBOTOUCH148, self).cover_file_endings()

        return cover_endings
    
    def set_device_name(self):
        if self.isAuraH2O():
            device_name = 'Kobo Aura H2O'
        elif self.isGloHD():
            device_name = 'Kobo Glo HD'
        elif self.isTouch2():
            device_name = 'Kobo Touch 2'
        else:
            device_name = super(KOBOTOUCH148, self).set_device_name()
        self.__class__.gui_name = device_name
        return device_name
