﻿using System;
using System.Collections.Generic;
using System.Data.SQLite;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Windows.Forms;

namespace XRayBuilderGUI
{
    public partial class frmMain : Form
    {
        public bool checkInternet = false;
        public bool exiting = false;
        public bool bookFoundShelfari = false;

        private string currentLog = Environment.CurrentDirectory + @"\log\" +
                                    string.Format("{0:yyyy.MM.dd.hh.mm.ss}.txt", DateTime.Now);

        private string newPath = "";
        private string OutFolder = "";
        private Properties.Settings settings = XRayBuilderGUI.Properties.Settings.Default;

        public frmMain()
        {
            InitializeComponent();
        }

        public void Log(string message)
        {
            if (exiting) return;
            txtOutput.AppendText(message + "\r\n");
        }

        private void btnBrowseMobi_Click(object sender, EventArgs e)
        {
            txtMobi.Text = Functions.GetBook(txtMobi.Text);
        }

        private void btnBrowseOutput_Click(object sender, EventArgs e)
        {
            if (!Directory.Exists(settings.outDir))
            {
                MessageBox.Show("Specified output directory does not exist. Please review the settings page.",
                    "Output Directory Not found");
                return;
            }
            else
                Process.Start(settings.outDir);
            //if (!cbAuthorProfile.Checked &&
            //    !cbEndAction.Checked &&
            //    !cbXray.Checked)
            //    if (MessageBox.Show("This book has not been processed yet.\r\nDo you want to open the Output directory?", "Unprocessed book", MessageBoxButtons.YesNo) == DialogResult.Yes)
            //        Process.Start(settings.outDir);
            //    else
            //        return;
            //Process.Start(settings.outDir);
        }

        private void btnBrowseXML_Click(object sender, EventArgs e)
        {
            txtXMLFile.Text = Functions.GetFile(txtXMLFile.Text,
                "XML files (*.xml)|*.xml|TXT files (*.txt)|*.txt");
        }

        private void btnBuild_Click(object sender, EventArgs e)
        {
            //Check current settings
            if (!File.Exists(txtMobi.Text))
            {
                MessageBox.Show("Specified book was not found.", "Book Not Found");
                return;
            }
            if (rdoShelfari.Checked && txtShelfari.Text == "")
            {
                MessageBox.Show("No Shelfari link was specified.", "Missing Shelfari Link");
                return;
            }
            if (!File.Exists(settings.mobi_unpack))
            {
                MessageBox.Show("Kindleunpack was not found. Please review the settings page.", "Kindleunpack Not Found");
                return;
            }
            if (!Directory.Exists(settings.outDir))
            {
                MessageBox.Show("Specified output directory does not exist. Please review the settings page.",
                    "Output Directory Not found");
                return;
            }
            if (XRayBuilderGUI.Properties.Settings.Default.realName.Trim().Length == 0 |
                XRayBuilderGUI.Properties.Settings.Default.penName.Trim().Length == 0)
            {
                MessageBox.Show(
                    "Both Real and Pen names are required for End Action\r\nfile creation. Please review the settings page.",
                    "Amazon Customer Details Not found");
                return;
            }
            //Create temp dir and ensure it exists
            string randomFile = Functions.GetTempDirectory();
            if (!Directory.Exists(randomFile))
            {
                MessageBox.Show("Temporary path not accessible for some reason.", "Temporary Directory Error");
                return;
            }

            cbAuthorProfile.Checked = false;
            cbEndAction.Checked = false;
            cbXray.Checked = false;
            //txtOutput.Clear();
            //txtOutput.Refresh();
            prgBar.Value = 0;

            Log("Running Kindleunpack to get metadata...");

            //0 = asin, 1 = uniqid, 2 = databasename, 3 = rawML, 4 = author, 5 = title
            //this.TopMost = true;
            List<string> results = Functions.GetMetaData(txtMobi.Text, settings.outDir, randomFile, settings.mobi_unpack);
            if (results.Count != 6)
            {
                Log(results[0]);
                return;
            }

            if (settings.saverawml)
            {
                Log("Saving rawML to output directory...");
                File.Copy(results[3], Path.Combine(settings.outDir, Path.GetFileName(results[3])), true);
            }
            // Added author name to log output
            Log(
                String.Format(
                    "Got metadata!\r\nDatabase Name: {0}\r\nASIN: {1}\r\nAuthor: {2}\r\nTitle: {3}\r\nUniqueID: {4}",
                    results[2], results[0], results[4], results[5], results[1]));

            if (rdoShelfari.Checked && txtShelfari.Text.Trim().Length == 0)
            {
                //Get Shelfari Search URL
                var shelfariSearchUrl = @"http://www.shelfari.com/search/books?Author=" +
                                        Regex.Replace(results[4], " ", "%20", RegexOptions.IgnoreCase) +
                                        "&Title=" +
                                        Regex.Replace(results[5], " ", "%20", RegexOptions.IgnoreCase) +
                                        "&Binding=Hardcover";
                Log("Searching for book on Shelfari...");

                // Search book on Shelfari
                HtmlAgilityPack.HtmlDocument shelfariHtmlDoc = new HtmlAgilityPack.HtmlDocument();
                using (var webClient = new WebClient())
                {
                    try
                    {
                        using (var stream = webClient.OpenRead(shelfariSearchUrl))
                        {
                            shelfariHtmlDoc.Load(stream);
                        }
                        webClient.Dispose();
                    }
                    catch (Exception ex)
                    {
                        Log("Exception caught: " + ex.Message);
                        return;
                    }
                }

                // Try to find book's page from Shelfari search
                string shelfariBookUrl = "";
                string bookSeries = "";
                int index = 0;
                List<string> listofthings = new List<string>();
                List<string> listoflinks = new List<string>();
                foreach (
                    var bookItems in shelfariHtmlDoc.DocumentNode.SelectNodes("//li[@class='item']/div[@class='text']"))
                    if (bookItems != null)
                    {
                        listofthings.Clear();
                        listoflinks.Clear();
                        for (var i = 1; i < bookItems.ChildNodes.Count; i++)
                        {
                            listofthings.Add(bookItems.ChildNodes[i].InnerText.Trim());
                            listoflinks.Add(bookItems.ChildNodes[i].InnerHtml);
                        }
                        index = 0;
                        foreach (string line in listofthings)
                        {
                            if (listofthings.Contains("(Author)") &&
                                line.StartsWith(results[5], StringComparison.OrdinalIgnoreCase) &&
                                listofthings.Contains(results[4]))
                            {
                                shelfariBookUrl = listoflinks[index].ToString();
                                bookSeries = listofthings[0].ToString();
                                shelfariBookUrl = Regex.Replace(shelfariBookUrl, "<a href=\"", "", RegexOptions.None);
                                shelfariBookUrl = Regex.Replace(shelfariBookUrl, "\">.*?</a>", "", RegexOptions.None);
                                Log("Book found on Shelfari!");
                                Log(results[5] + " by " + results[4] + " (" + bookSeries + ")");
                                Log(shelfariBookUrl);
                                txtShelfari.Text = shelfariBookUrl;
                                //rtbOutput.Text = shelfariBookUrl;
                                txtShelfari.Refresh();
                                txtShelfari.Text = shelfariBookUrl;
                                Log("Shelfari link updated! Click this link to open the Shelfari web page.");
                                bookFoundShelfari = true;
                                break;
                            }
                            index++;
                        }
                    }
                if (!bookFoundShelfari)
                    Log("Unable to find a Hardcover edition of this book on Shelfari!");
            }

            //Create Author Profile and attempt to create the base file
            try
            {
                AuthorProfile aa;
                aa = new AuthorProfile(results[5], results[4], results[0], results[1], results[2], randomFile, this);
            }
            catch (Exception ex)
            {
                Log("An error occurred while creating the new Author Profile and End Action files: " + ex.Message);
            }

            Log(String.Format("Attempting to build X-ray...\r\nSpoilers: {0}",
                settings.spoilers ? "Enabled" : "Disabled"));
            Log("Offset: " + settings.offset.ToString());

            //Create X-ray and attempt to create the base file (essentially the same as the site)
            XRay ss;
            try
            {
                if (rdoShelfari.Checked)
                    ss = new XRay(txtShelfari.Text, results[2], results[1], results[0], this, settings.spoilers,
                        settings.offset, "", false);
                else
                    ss = new XRay(txtXMLFile.Text, results[2], results[1], results[0], this, settings.spoilers,
                        settings.offset, "");
                if (ss.createXRAY() > 0)
                {
                    Log("Error while processing.");
                    return;
                }
                Log("Initial X-ray built, adding locations and chapters...");
                //Expand the X-ray file from the unpacked mobi
                if (ss.expandFromRawML(results[3], settings.ignoresofthyphen, !settings.useNewVersion) > 0)
                {
                    Log("Error while processing locations and chapters.");
                    return;
                }
            }
            catch (Exception ex)
            {
                Log("An error occurred while creating the new X-ray database. Is it opened in another program?\n" +
                    ex.Message);
                return;
            }
            Log("Saving X-ray to file...");

            if (settings.useSubDirectories)
            {
                OutFolder = Functions.GetBookOutputDirectory(results[4], results[5]);
                newPath = OutFolder + "\\" + ss.getXRayName(settings.android);
            }
            else
            {
                newPath = settings.outDir + "\\" + ss.getXRayName(settings.android);
            }

            if (settings.useNewVersion)
            {
                try
                {
                    SQLiteConnection.CreateFile(newPath);
                }
                catch (Exception ex)
                {
                    Log("An error occurred while creating the new X-ray database. Is it opened in another program?\n" +
                        ex.Message);
                    return;
                }
                SQLiteConnection m_dbConnection;
                m_dbConnection = new SQLiteConnection(("Data Source=" + newPath + ";Version=3;"));
                m_dbConnection.Open();
                string sql;
                try
                {
                    using (StreamReader streamReader = new StreamReader("BaseDB.sql", Encoding.UTF8))
                    {
                        sql = streamReader.ReadToEnd();
                    }
                }
                catch (Exception ex)
                {
                    Log(
                        "An error occurred while opening the BaseDB.sql file. Ensure you extracted it to the same directory as the program.\n" +
                        ex.Message);
                    m_dbConnection.Close();
                    return;
                }
                SQLiteCommand command = new SQLiteCommand("BEGIN; " + sql + " COMMIT;", m_dbConnection);
                Log("\nBuilding new X-ray database. May take a few minutes...");
                command.ExecuteNonQuery();
                command = new SQLiteCommand("PRAGMA user_version = 1; PRAGMA encoding = utf8; BEGIN;", m_dbConnection);
                command.ExecuteNonQuery();
                Console.WriteLine("Done building initial database. Populating with info from source X-ray...");
                try
                {
                    ss.PopulateDB(m_dbConnection);
                }
                catch (Exception ex)
                {
                    Log("An error occurred while creating the new X-ray database. Is it opened in another program?\n" +
                        ex.Message);
                    m_dbConnection.Close();
                    return;
                }
                Console.WriteLine("Updating indices...");
                sql = "CREATE INDEX idx_occurrence_start ON occurrence(start ASC);\n"
                      + "CREATE INDEX idx_entity_type ON entity(type ASC);\n"
                      + "CREATE INDEX idx_entity_excerpt ON entity_excerpt(entity ASC); COMMIT;";
                command = new SQLiteCommand(sql, m_dbConnection);
                command.ExecuteNonQuery();
                m_dbConnection.Close();
            }
            else
            {
                using (
                    StreamWriter streamWriter = new StreamWriter(newPath, false,
                        settings.utf8 ? Encoding.UTF8 : Encoding.Default))
                {
                    streamWriter.Write(ss.ToString());
                }
            }
            cbXray.Checked = true;
            Log("X-ray file created successfully!\r\nSaved to " + newPath);
            Directory.Delete(randomFile, true);
            //this.TopMost = false;
        }

        private void btnKindleExtras_Click(object sender, EventArgs e)
        {
            //Check current settings
            if (!File.Exists(txtMobi.Text))
            {
                MessageBox.Show("Specified book was not found.", "Book Not Found");
                return;
            }
            if (rdoShelfari.Checked && txtShelfari.Text == "")
            {
                MessageBox.Show("No Shelfari link was specified.", "Missing Shelfari Link");
                return;
            }
            if (!File.Exists(settings.mobi_unpack))
            {
                MessageBox.Show("Kindleunpack was not found. Please review the settings page.", "Kindleunpack Not Found");
                return;
            }
            if (XRayBuilderGUI.Properties.Settings.Default.realName.Trim().Length == 0 |
                XRayBuilderGUI.Properties.Settings.Default.penName.Trim().Length == 0)
            {
                MessageBox.Show(
                    "Both Real and Pen names are required for End Action\r\nfile creation. Please review the settings page.",
                    "Amazon Customer Details Not found");
                return;
            }
            //Create temp dir and ensure it exists
            string randomFile = Functions.GetTempDirectory();
            if (!Directory.Exists(randomFile))
            {
                MessageBox.Show("Temporary path not accessible for some reason.", "Temporary Directory Error");
                return;
            }

            cbAuthorProfile.Checked = false;
            cbEndAction.Checked = false;
            cbXray.Checked = false;
            //txtOutput.Clear();
            //txtOutput.Refresh();

            Log("Running Kindleunpack to get metadata...");

            //0 = asin, 1 = uniqid, 2 = databasename, 3 = rawML, 4 = author, 5 = title
            //this.TopMost = true;
            List<string> results = Functions.GetMetaData(txtMobi.Text, settings.outDir, randomFile, settings.mobi_unpack);
            if (results.Count != 6)
            {
                Log(results[0]);
                return;
            }

            // Added author name to log output
            Log(
                String.Format(
                    "Got metadata!\r\nDatabase Name: {0}\r\nASIN: {1}\r\nAuthor: {2}\r\nTitle: {3}\r\nUniqueID: {4}",
                    results[2], results[0], results[4], results[5], results[1]));

            try
            {
                AuthorProfile aa;
                aa = new AuthorProfile(results[5], results[4], results[0], results[1], results[2], randomFile, this);
            }
            catch (Exception ex)
            {
                Log("An error occurred while creating the new Author Profile and End Action files: " + ex.Message);
            }
        }

        private void btnLink_Click(object sender, EventArgs e)
        {
            if (txtShelfari.Text.Trim().Length == 0)
                MessageBox.Show("No Shelfari link was specified.", "Missing Shelfari Link");
            else
                Process.Start(txtShelfari.Text);
        }

        private void btnSaveShelfari_Click(object sender, EventArgs e)
        {
            if (txtShelfari.Text == "")
            {
                MessageBox.Show("No Shelfari link was specified.", "Missing Shelfari Link");
                return;
            }
            if (!File.Exists(txtMobi.Text))
            {
                MessageBox.Show("Specified book was not found.", "Book Not Found");
                return;
            }
            if (!Directory.Exists(Environment.CurrentDirectory + @"\xml\"))
                Directory.CreateDirectory(Environment.CurrentDirectory + @"\xml\");
            string path = Environment.CurrentDirectory + @"\xml\" + Path.GetFileNameWithoutExtension(txtMobi.Text) +
                          ".xml";
            try
            {
                txtXMLFile.Text = path;

                //txtOutput.Clear();
                //txtOutput.Refresh();

                XRay xray = new XRay(txtShelfari.Text, this, settings.spoilers);
                if (xray.saveXML(path) > 0)
                {
                    Log("Error while processing.");
                    return;
                }
                Log("Shelfari info has been saved to: " + path);
            }
            catch (Exception)
            {
                Log("Error while saving Shelfari data to XML. Path was: " + path);
                return;
            }
        }

        private void btnSearchShelfari_Click(object sender, EventArgs e)
        {
            //Check internet connection
            //Log("Checking internet connection...");
            //if (Functions.CheckForInternetConnection() != true)
            //    Log("No Internet connection detected!");
            //else
            //    Log("Internet connection ok!");

            //Check current settings
            if (!File.Exists(txtMobi.Text))
            {
                MessageBox.Show("Specified book was not found.", "Book Not Found");
                return;
            }
            if (!File.Exists(settings.mobi_unpack))
            {
                MessageBox.Show("Kindleunpack was not found. Please review the settings page.", "Kindleunpack Not Found");
                return;
            }
            if (!Directory.Exists(settings.outDir))
            {
                MessageBox.Show("Specified output directory does not exist. Please review the settings page.",
                    "Output Directory Not found");
                return;
            }
            //Create temp dir and ensure it exists
            string randomFile = Functions.GetTempDirectory();
            if (!Directory.Exists(randomFile))
            {
                MessageBox.Show("Temporary path not accessible for some reason.", "Temporary Directory Error");
                return;
            }
            //txtOutput.Clear();
            //txtOutput.Refresh();
            Log("Running Kindleunpack to get metadata...");

            //0 = asin, 1 = uniqid, 2 = databasename, 3 = rawML, 4 = author, 5 = title
            //this.TopMost = true;
            List<string> results = Functions.GetMetaData(txtMobi.Text, settings.outDir, randomFile, settings.mobi_unpack);
            if (results.Count != 6)
            {
                Log(results[0]);
                return;
            }

            if (settings.saverawml)
            {
                Log("Saving rawML to output directory...");
                File.Copy(results[3], Path.Combine(settings.outDir, Path.GetFileName(results[3])), true);
            }
            // Added author name to log output
            Log(
                String.Format(
                    "Got metadata!\r\nDatabase Name: {0}\r\nASIN: {1}\r\nAuthor: {2}\r\nTitle: {3}\r\nUniqueID: {4}",
                    results[2], results[0], results[4], results[5], results[1]));

            //Get Shelfari Search URL
            var shelfariSearchUrl = @"http://www.shelfari.com/search/books?Author=" +
                                    Regex.Replace(results[4], " ", "%20", RegexOptions.IgnoreCase) +
                                    "&Title=" +
                                    Regex.Replace(results[5], " ", "%20", RegexOptions.IgnoreCase) +
                                    "&Binding=Hardcover";
            Log("Searching for book on Shelfari...");
            Directory.Delete(randomFile, true);
            
            // Search book on Shelfari
            HtmlAgilityPack.HtmlDocument shelfariHtmlDoc = new HtmlAgilityPack.HtmlDocument();
            using (var webClient = new WebClient())
            {
                try
                {
                    using (var stream = webClient.OpenRead(shelfariSearchUrl))
                    {
                        shelfariHtmlDoc.Load(stream);
                    }
                    webClient.Dispose();
                }
                catch (Exception ex)
                {
                    Log("Exception caught: " + ex.Message);
                    return;
                }
            }

            // Try to find book's page from Shelfari search
            string shelfariBookUrl = "";
            string bookSeries = "";
            int index = 0;
            List<string> listofthings = new List<string>();
            List<string> listoflinks = new List<string>();
            foreach (var bookItems in shelfariHtmlDoc.DocumentNode.SelectNodes("//li[@class='item']/div[@class='text']")
                )
                if (bookItems != null)
                {
                    listofthings.Clear();
                    listoflinks.Clear();
                    for (var i = 1; i < bookItems.ChildNodes.Count; i++)
                    {
                        listofthings.Add(bookItems.ChildNodes[i].InnerText.Trim());
                        listoflinks.Add(bookItems.ChildNodes[i].InnerHtml);
                    }
                    index = 0;
                    foreach (string line in listofthings)
                    {
                        if (listofthings.Contains("(Author)") &&
                            line.StartsWith(results[5], StringComparison.OrdinalIgnoreCase) &&
                            listofthings.Contains(results[4]))
                        {
                            shelfariBookUrl = listoflinks[index].ToString();
                            bookSeries = listofthings[0].ToString();
                            shelfariBookUrl = Regex.Replace(shelfariBookUrl, "<a href=\"", "", RegexOptions.None);
                            shelfariBookUrl = Regex.Replace(shelfariBookUrl, "\">.*?</a>", "", RegexOptions.None);
                            Log("Book found on Shelfari!");
                            Log(results[5] + " by " + results[4] + " (" + bookSeries + ")");
                            txtShelfari.Text = shelfariBookUrl;
                            txtShelfari.Refresh();
                            Log("Shelfari URL updated!");
                            bookFoundShelfari = true;
                            break;
                        }
                        index++;
                    }
                }
            if (!bookFoundShelfari)
                Log("Unable to find a Hardcover edition of this book on Shelfari!");
        }

        private void btnSettings_Click(object sender, EventArgs e)
        {
            //this.TopMost = false;
            Form frm = new frmSettings();
            frm.ShowDialog();
            //this.TopMost = true;
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            XRayBuilderGUI.Properties.Settings.Default.mobiFile = txtMobi.Text;
            XRayBuilderGUI.Properties.Settings.Default.xmlFile = txtXMLFile.Text;
            XRayBuilderGUI.Properties.Settings.Default.shelfari = txtShelfari.Text;
            if (rdoShelfari.Checked)
                XRayBuilderGUI.Properties.Settings.Default.buildSource = "Shelfari";
            else
                XRayBuilderGUI.Properties.Settings.Default.buildSource = "XML";
            XRayBuilderGUI.Properties.Settings.Default.Save();
            if (txtOutput.Text.Trim().Length != 0)
                File.WriteAllText(currentLog, txtOutput.Text);
            exiting = true;
            Application.Exit();
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            ToolTip ToolTip1 = new ToolTip();
            ToolTip1.SetToolTip(btnBrowseMobi, "Open a Kindle book.");
            ToolTip1.SetToolTip(btnBrowseOutput, "Open the default output directory.");
            ToolTip1.SetToolTip(btnLink, "Open the Shelfari link in your default web browser.");
            ToolTip1.SetToolTip(btnBrowseXML, "Open a supported alias file containg Characters and Topics.");
            ToolTip1.SetToolTip(btnSearchShelfari, "Try to search for this book on Shelfari.");
            ToolTip1.SetToolTip(btnSaveShelfari, "Save Shelfari info to an XML file.");
            ToolTip1.SetToolTip(btnKindleExtras, "Try to build the Author Profile and End Action files for this book.");
            
            this.DragEnter += frmMain_DragEnter;
            this.DragDrop += frmMain_DragDrop;

            string[] args = Environment.GetCommandLineArgs();
            if (args.Length > 1)
            {
                for (int i = 0; i < args.Length; i++)
                {
                    if (File.Exists(args[i]))
                    {
                        txtMobi.Text = Path.GetFullPath(args[i]);
                    }
                }
            }
            if (txtMobi.Text == "") txtMobi.Text = XRayBuilderGUI.Properties.Settings.Default.mobiFile;
            if (txtXMLFile.Text == "") txtXMLFile.Text = XRayBuilderGUI.Properties.Settings.Default.xmlFile;
            if (XRayBuilderGUI.Properties.Settings.Default.outDir == "")
            {
                XRayBuilderGUI.Properties.Settings.Default.outDir = Environment.CurrentDirectory + @"\out";
                if (!Directory.Exists(XRayBuilderGUI.Properties.Settings.Default.outDir))
                    Directory.CreateDirectory(XRayBuilderGUI.Properties.Settings.Default.outDir);
            }
            if (!Directory.Exists(Environment.CurrentDirectory + @"\log"))
                Directory.CreateDirectory(Environment.CurrentDirectory + @"\log");
            if (XRayBuilderGUI.Properties.Settings.Default.mobi_unpack == "")
            {
                XRayBuilderGUI.Properties.Settings.Default.mobi_unpack = Environment.CurrentDirectory +
                                                                         @"\dist\kindleunpack.exe";
            }
            txtShelfari.Text = XRayBuilderGUI.Properties.Settings.Default.shelfari;
            if (XRayBuilderGUI.Properties.Settings.Default.buildSource == "Shelfari")
                rdoShelfari.Checked = true;
            else
                rdoFile.Checked = true;
        }

        private void frmMain_DragDrop(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                string[] filePaths = (string[]) (e.Data.GetData(DataFormats.FileDrop));
                foreach (string fileLoc in filePaths)
                {
                    if (File.Exists(fileLoc))
                    {
                        txtMobi.Text = fileLoc;
                        return;
                    }
                }
            }
        }

        private void frmMain_DragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                e.Effect = DragDropEffects.Copy;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        private void rdoSource_CheckedChanged(object sender, EventArgs e)
        {
            if (((RadioButton) sender).Text == "Shelfari")
            {
                lblShelfari.Visible = !lblShelfari.Visible;
                txtShelfari.Visible = !txtShelfari.Visible;
                lblXMLFile.Visible = !lblXMLFile.Visible;
                txtXMLFile.Visible = !txtXMLFile.Visible;
                txtShelfari.Visible = !txtShelfari.Visible;
                btnBrowseXML.Visible = !btnBrowseXML.Visible;
                btnSaveShelfari.Enabled = !btnSaveShelfari.Enabled;
                btnLink.Visible = !btnLink.Visible;
            }
        }

        private void txtMobi_TextChanged(object sender, EventArgs e)
        {
            txtShelfari.Text = "";
            cbAuthorProfile.Checked = false;
            cbEndAction.Checked = false;
            cbXray.Checked = false;
        }
    }
}