#!/bin/sh
# 
#                             xshell launcher
#            (standardized windowed shell launcher for the iLiad)
#
version=050 #                     v0.5.0    
#
#                    05/Jun/2008, Ben Mares (aka bobj)
#
#     For help, type:
#        ./launch --help
#
#     License: Anyone may modify and distribute freely 
#

# Obtain the current directory
currentdir=`pwd`

# Find the (relative) path of this script
launchdir=`/usr/bin/dirname "$0"`

# Find the filename of this script
launchbase=`basename "$0"`

# Find the full path of this script
cd "$launchdir"
launchdir=`pwd`

# Make a human-readable version string 
versionstring="xshell launcher v`echo $version | sed "s/\(.\)\(.\)\(.\)/\1.\2.\3/"`"

printhelp() {
  # Print the standard help message
  echo
  echo $versionstring
  echo 
  echo "Launches a windowed command line terminal"
  echo
  echo "Usage: $launchbase [--location [XSHELLPATH]] [ARGS...]"
  echo
  echo 
  echo "  --location               Print the default location of xshell and exit"
  echo "                           (this can be changed by putting the new"
  echo "                           location as the contents of the file"
  echo "                           ~/.xstart)"
  echo
  echo "  --location XSHELLPATH    Specify a new location for xshell"
  echo 
  echo "  ARGS...                  Arguments to pass on to the xshell "
  echo
  echo "  --help                   Print this help message and exit"
  echo
  echo "  --xshellhelp             Prints help for the xshell arguments"
  echo
  echo "  --launcherversion        Print version of this launcher and exit"
  echo "                           (use --xshellversion to get the version of xshell)"


}

# ----------------------------
# Parse command line arguments
# ----------------------------

# If a command line option defines a variable, that variable name will be in ac_prev.
ac_prev=

# While the number of unprocessed arguments is nonzero,
while test $# -ne 0 ; do

  # The current argument is ac_option
  ac_option="$1"

  # Set the variable named in ac_prev to ac_option.
  if test -n "$ac_prev" ; then
    eval "$ac_prev=\$ac_option"
    ac_prev=
    shift
    continue
  fi

  case $ac_option in

    --launcherversion)
      # Print the version in human-readable format 
      echo $versionstring
      exit 0 ;;

    --launcherversionnum)
      # Print the version as an integer
      echo $version
      exit 0 ;;

    --location)
      # Prepare to either set $location to the following argument, or print out the default location
      ac_prev=location ;;

    --help)
      printhelp
      exit 0 ;;

    --preserve-manifest)
      # Don't disturb manifest.xml
      preservemanifest=1 ;;

    *)
      # Pass on any unrecognized arguments
      passargs="$@"
      break ;;

  esac

  # Advance to the next argument
  shift

done


# ----------------------------
# Find the xshell start script
# ----------------------------

validatelocation () {
  # Takes a location as an argument.  Checks to ensure that the necessary
  # files are indeed there and are executable.
  location=$1
  if test ! -n $location ; then
    echo "Error: no location specified to validatelocation." >&2
	return 1
  elif test ! -d "$location" ; then
    echo "Not a directory: $location" >&2
	return 1
  elif test ! -f "$location/prerun" ; then
    echo "Not a file: $location/prerun" >&2
	return 1
  elif test ! -x "$location/prerun" ; then 
    echo "Not executable: $location/prerun" >&2  
	return 1
  elif test ! -f "$location/startxshell" ; then
    echo "Not a file: $location/startxshell" >&2
	return 1
  elif test ! -x "$location/startxshell" ; then 
    echo "Not executable: $location/startxshell" >&2  
	return 1
  fi
  return 0
}

runfromlocation () {

  # If possible, execute using the following location for an X terminal
  location=$1

  # If $ac_prev = "location" then no location was specified, so we'll print the defualt.
  if test x$ac_prev = x"location" ; then
    echo "$location"

  else 
    # Execute the prerun script.  Provides an opportunity to upgrade this launcher.
    "$location/prerun" "$version" "$launchdir/$launchbase"
    result=$?  # Return value of prerun

    if test $result -ne 0; then
      # If prerun doesn't exit with code zero, then abort.
      echo "Prerun not passed.  Will not execute." >&2 
      return 1

    else 
      # Run the terminal script and pass arguments.
      "$location/startxshell" $passargs
      return $?
    fi
  fi

}


# If manifest.xml is displaying an error message, switch back to the original.
if test -f manifest.xml.orig && test "$preservemanifest" -ne 1 ; then
  mv manifest.xml.orig manifest.xml
fi
cd "$currentdir"


# If we were given a location on the command line,
if test -n "$location"; then 

  # validate that location
  validatelocation "$location"

  if test $? -eq 0 ; then
    # If it's valid, then run from that location.
    runfromlocation "$location"

  else
    echo "Specified location not valid: $location" >&2
    exit 1
  fi

# Look for the .xshell file which specified where to find the xshell.
elif test -f "$HOME/.xshell" && test -x "$HOME/.xshell" ; then
  # Set the location to the contents of the .xshell file
  . "$HOME/.xshell"

  # Try to validate and run from this location
  validatelocation "$location"
  if test $? -eq 0 ; then
    runfromlocation "$location"
  else
    echo "Location provided in ~/.xshell is not valid: $location" >&2
  fi  

# We don't know where xshell is, so we have to search for it.
else

  # Highest version number found so far
  latestversion=0

  # Search in some standard locations such as /mnt/cf/Programs/_xshell
  for fs in /mnt/free /usr/local/programs /mnt/card /mnt/cf /mnt/usb ; do
    for subdir in /_xshell /Programs/_xshell ; do

      # Validate the location
      location=$fs$subdir
      validatelocation "$location"

      # If it's valid and we found a newer version,
      if test $? -eq 0 ; then
        foundversion=`$location/startxshell --xshellversionnum`
        if test $foundversion -gt $latestversion ; then
          # Update the latest version and remember its location
          latestversion="$foundversion"
          latestlocation="$location"
        fi 
      fi
    done
  done

  # If we couldn't find a valid version
  if test $latestversion -eq 0 ; then
    echo "Could not find xshell in the usual locations.  Please run" >&2
    echo "the xshell / mrxvt installer, version 0.5.0 or greater." >&2
    echo "http://wiki.mobileread.com/wiki/Iliad_Software" >&2

    # Make a new manifest.xml file
    cd "$launchdir"
    # Poor man's XML parser
    title=`sed -n "s/.*<title>\(.*\)<\/title>.*/\1/Ip" < manifest.xml`
    startpage=`sed -n "s/.*<startpage>\(.*\)<\/startpage>.*/\1/Ip" < manifest.xml`
    image=`sed -n "s/.*<image>\(.*\)<\/image>.*/\1/Ip" < manifest.xml`

    # Show an error message through manifest.xml if allowed
    if test "$preservemanifest" -ne 1 ; then
      # Move existing manifest
      mv manifest.xml manifest.xml.orig

      # Write new manifest with error message
      cat > manifest.xml <<END
<?xml version="1.0" encoding="utf-8"?>
        <package><metadata><dc-metadata><Title>$title</Title>
        <Description>ERROR: Unable to find xshell/mrxvt v0.5.0 or greater.
See http://wiki.mobileread.com/wiki/Iliad_Software</Description>
        </dc-metadata><y-metadata><startpage>$startpage</startpage><image>$image</image></y-metadata></metadata></package>
END
    fi

    exit 1  # Exit with an error

  # Otherwise, run from it.
  else
    location="$latestlocation"
    runfromlocation "$location"
  fi
fi

# Exit with the return value from the script we ran.
exit $?
