#!/usr/bin/env python
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3' #Based on B&N plugin by Grant Drake
__copyright__ = '2011, Rodrigo Coin Curvo <rodrigoccurvo@gmail.com>'
__docformat__ = 'en'

try:
    from PyQt5.Qt import QLabel, QGridLayout, Qt, QGroupBox, QCheckBox, QSpinBox
except:
    from PyQt4.Qt import QLabel, QGridLayout, Qt, QGroupBox, QCheckBox, QSpinBox
from calibre.gui2.metadata.config import ConfigWidget as DefaultConfigWidget
from calibre.utils.config import JSONConfig

STORE_NAME = 'Options'
KEY_MAX_PAGES = 'maxPages'
KEY_MAX_DOWNLOADS = 'maxDownloads'
KEY_GET_ADDITIONAL_INFO = 'getAdditionalInfo'
KEY_THRESHOLD = 'threshold'
KEY_TRY_EXCHANGING = 'tryExchanging'

DEFAULT_STORE_VALUES = {
    KEY_MAX_PAGES: 1,
    KEY_MAX_DOWNLOADS: 1,
    KEY_THRESHOLD: 0,
    KEY_GET_ADDITIONAL_INFO: False,
    KEY_TRY_EXCHANGING: False
}

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Skoob')

# Set defaults
plugin_prefs.defaults[STORE_NAME] = DEFAULT_STORE_VALUES

def getOption(option_key):
    default_value = DEFAULT_STORE_VALUES[option_key]
    return plugin_prefs[STORE_NAME].get(option_key, default_value)

class ConfigWidget(DefaultConfigWidget):

    def __init__(self, plugin):
        DefaultConfigWidget.__init__(self, plugin)
        c = plugin_prefs[STORE_NAME]

        other_group_box = QGroupBox('Other options', self)
        self.l.addWidget(other_group_box, self.l.rowCount(), 0, 1, 2)
        other_group_box_layout = QGridLayout()
        other_group_box.setLayout(other_group_box_layout)


        max_pages_label = QLabel('Maximum search result pages to analyse (1 = fastest):', self)
        max_pages_label.setToolTip('Usually one page is enough, but\n'
                                   'if the search is too broad the best\n'
                                   'result may not be in the first page.\n')
        other_group_box_layout.addWidget(max_pages_label, 0, 0, 1, 1)
        self.max_pages_spin = QSpinBox(self)
        self.max_pages_spin.setMinimum(1)
        self.max_pages_spin.setMaximum(5)
        self.max_pages_spin.setProperty('value', c.get(KEY_MAX_PAGES, DEFAULT_STORE_VALUES[KEY_MAX_PAGES]))
        other_group_box_layout.addWidget(self.max_pages_spin, 0, 1, 1, 1)
        other_group_box_layout.setColumnStretch(2, 1)

        max_downloads_label = QLabel('Maximum title/author search matches to download/evaluate (1 = fastest):', self)
        max_downloads_label.setToolTip('More matches means higher chance of better\n'
                             'metadata (but not necessarily).\n')
        other_group_box_layout.addWidget(max_downloads_label, 1, 0, 1, 1)
        self.max_downloads_spin = QSpinBox(self)
        self.max_downloads_spin.setMinimum(1)
        self.max_downloads_spin.setMaximum(5)
        self.max_downloads_spin.setProperty('value', c.get(KEY_MAX_DOWNLOADS, DEFAULT_STORE_VALUES[KEY_MAX_DOWNLOADS]))
        other_group_box_layout.addWidget(self.max_downloads_spin, 1, 1, 1, 1)
        other_group_box_layout.setColumnStretch(2, 1)

        threshold_label = QLabel('[Experimental] Minimum similarity threshold (%):', self)
        threshold_label.setToolTip('Try to eliminate results that are not\n'
                                   'very similar to the search based on a\n'
                                   'composition of Title and Author similarities.\n'
                                   'This is not quite precise, but 20-30 seems\n'
                                   'to be a good guess. Default is no discarding (0).\n')
        other_group_box_layout.addWidget(threshold_label, 2, 0, 1, 1)
        self.threshold_spin = QSpinBox(self)
        self.threshold_spin.setMinimum(0)
        self.threshold_spin.setMaximum(100)
        self.threshold_spin.setProperty('value', c.get(KEY_THRESHOLD, DEFAULT_STORE_VALUES[KEY_THRESHOLD]))
        other_group_box_layout.addWidget(self.threshold_spin, 2, 1, 1, 1)
        other_group_box_layout.setColumnStretch(2, 1)

        self.additional_info_checkbox = QCheckBox('Get additional info', self)
        self.additional_info_checkbox.setToolTip(
                'Skoob maintains additional info (ISBN, publisher,\n'
                'pubdate, editions, etc) in separate pages. This will\n'
                'define whether or not to download and check those pages.\n')
        self.additional_info_checkbox.setChecked(c.get(KEY_GET_ADDITIONAL_INFO, DEFAULT_STORE_VALUES[KEY_GET_ADDITIONAL_INFO]))
        other_group_box_layout.addWidget(self.additional_info_checkbox, 3, 0, 1, 3)

        self.try_exchanging_checkbox = QCheckBox('Try exchanging title and authors if nothing found', self)
        self.try_exchanging_checkbox.setToolTip(
                'If no result is found, try searching again with\n'
                'title and authors exchanged.\n')
        self.try_exchanging_checkbox.setChecked(c.get(KEY_TRY_EXCHANGING, DEFAULT_STORE_VALUES[KEY_TRY_EXCHANGING]))
        other_group_box_layout.addWidget(self.try_exchanging_checkbox, 4, 0, 1, 3)

    def commit(self):
        DefaultConfigWidget.commit(self)
        new_prefs = {}
        new_prefs[KEY_MAX_PAGES] = int(unicode(self.max_pages_spin.value()))
        new_prefs[KEY_MAX_DOWNLOADS] = int(unicode(self.max_downloads_spin.value()))
        new_prefs[KEY_THRESHOLD] = int(unicode(self.threshold_spin.value()))
        new_prefs[KEY_GET_ADDITIONAL_INFO] = self.additional_info_checkbox.checkState() == Qt.Checked
        new_prefs[KEY_TRY_EXCHANGING] = self.try_exchanging_checkbox.checkState() == Qt.Checked
        plugin_prefs[STORE_NAME] = new_prefs

