# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011-2014, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# Main menu
#####################################################################

try:
    from PyQt5.Qt import QMenu, QToolButton, QIcon
except:
    from PyQt4.Qt import QMenu, QToolButton, QIcon

from calibre.gui2.actions import InterfaceAction
from calibre.gui2.device import device_signals
from calibre.gui2 import FunctionDispatcher


import calibre_plugins.kindleCollectionsLeefa.messages as msg
import calibre_plugins.kindleCollectionsLeefa.config as cfg
import calibre_plugins.kindleCollectionsLeefa.calibre_info as calibre_info
import calibre_plugins.kindleCollectionsLeefa.create_preview as create_preview
import calibre_plugins.kindleCollectionsLeefa.kindle_view_collections as kindle_view
import calibre_plugins.kindleCollectionsLeefa.kindle_edit_collections as kindle_edit
import calibre_plugins.kindleCollectionsLeefa.kindle_modify_settings as kindle_settings
import calibre_plugins.kindleCollectionsLeefa.kindle_restore_collections as kindle_restore
import calibre_plugins.kindleCollectionsLeefa.import_collections as import_collections
import calibre_plugins.kindleCollectionsLeefa.help as help
import calibre_plugins.kindleCollectionsLeefa.about as about
from calibre_plugins.kindleCollectionsLeefa.__init__ import PLUGIN_NAME, PLUGIN_VERSION
from calibre_plugins.kindleCollectionsLeefa.utilities import debug_print, get_icon, set_plugin_icon_resources, get_pref

PLUGIN_ICONS = [ 'images/kindleCollectionsLeefa.png' ]

#####################################################################

class InterfacePlugin(InterfaceAction):

    name = PLUGIN_NAME
    action_spec = (_(PLUGIN_NAME), None, PLUGIN_NAME, None)
    action_type = 'current'
    dont_add_to = frozenset(['toolbar'])

    # Run when Calibre starts up - not when selecting the icon
    # Do not put print statements here - they would be seen even before plugin is run
    def genesis(self):

        # Define menu used on toolbar icon
        m = QMenu(self.gui)
        self.menu = m
        self.qaction.setMenu(m)

        # Load the plugin icons
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(PLUGIN_NAME, icon_resources)

        # Set signals so we know when device metadata is available and good to read
        device_signals.device_metadata_available.connect(calibre_info.device_metadata_available_signalled)
        device_signals.device_connection_changed.connect(calibre_info.device_connection_changed_signalled)

    # Routines to run once calibre/gui is fully configured
    def initialization_complete(self):
        self.rebuild_menus()
        self.set_icon_style()

    # Build the menu list attached to the toolbar plugin icon/right-click menu
    # Do not put print statements here - they would be seen even before plugin is run
    def rebuild_menus(self):
        m = self.menu
        m.clear()

        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))

        m.addAction(get_icon(PLUGIN_ICONS[0]), _('&Create collections on the Kindle from Calibre'), self.create_kindleCollectionsLeefa)
        m.addAction( _('&Preview collections from Calibre without saving'), self.preview_kindleCollectionsLeefa)
        m.addAction(QIcon(I('config.png')), _('C&ustomize collections to create from Calibre...'), self.customize)
        m.addSeparator()
        m.addAction( _('&View report of Kindle Collections Leefa and books'), self.view_kindleCollectionsLeefa)
        m.addSeparator()
        m.addAction( _('&Edit Kindle Collections Leefa manually...'), self.edit_kindleCollectionsLeefa)
        m.addAction( _('&Import Kindle Collections Leefa into Calibre...'), self.import_kindleCollectionsLeefa)
        m.addSeparator()
        m.addAction( _('&Restore previous Kindle Collections Leefa file...'), self.restore_previous_kindleCollectionsLeefa)
        m.addSeparator()
        m.addAction( _('&Modify Kindle Settings...'), self.modify_kindle_settings)
        # Completely disable this on the Touch/PaperWhite, once we know we have one... [We're initialized too early to be able to do this :/]
        #if 'kindle_model_version' in cfg.config_settings and cfg.config_settings['kindle_model_version'] >= 500:
        #    self.modify_kindle_settings.setDisabled(True)
        m.addSeparator()
        m.addAction(QIcon(I('help.png')), _('&Help'), self.help)
        m.addAction( _('&About'), self.about)

    def set_icon_style(self):
        # Get the current menu style/default
        click_action = get_pref(cfg.MENU_CLICK_STYLE)
        if click_action not in cfg.MENU_OPTIONS:
            click_action = cfg.MENU_DEFAULT

        # Setup the actions for the icon/menu
        if click_action ==  cfg.MENU_DEFAULT:
            self.qaction.setToolTip(PLUGIN_NAME)
            self.popup_type = QToolButton.InstantPopup
        else:
            self.popup_type = QToolButton.MenuButtonPopup
            if click_action ==  cfg.MENU_CREATE:
                self.qaction.triggered.connect(self.create_kindleCollectionsLeefa)
                self.qaction.setToolTip(PLUGIN_NAME + ': Create collections')
            elif click_action ==  cfg.MENU_PREVIEW:
                self.qaction.triggered.connect(self.preview_kindleCollectionsLeefa)
                self.qaction.setToolTip(PLUGIN_NAME + ': Preview')
            elif click_action ==  cfg.MENU_CUSTOMIZE:
                self.qaction.triggered.connect(self.customize)
                self.qaction.setToolTip(PLUGIN_NAME + ': Customize')
            elif click_action ==  cfg.MENU_VIEW:
                self.qaction.triggered.connect(self.view_kindleCollectionsLeefa)
                self.qaction.setToolTip(PLUGIN_NAME + ': View Report')
            elif click_action ==  cfg.MENU_EDIT:
                self.qaction.triggered.connect(self.edit_kindleCollectionsLeefa)
                self.qaction.setToolTip(PLUGIN_NAME + ': Edit collections')
            elif click_action ==  cfg.MENU_IMPORT:
                self.qaction.triggered.connect(self.import_kindleCollectionsLeefa)
                self.qaction.setToolTip(PLUGIN_NAME + ': Import collections')
            elif click_action ==  cfg.MENU_MODIFY:
                self.qaction.triggered.connect(self.modify_kindle_settings)
                self.qaction.setToolTip(PLUGIN_NAME + ': Modify settings')

        self.change_toolbar_popup_type(self.popup_type)

    def change_toolbar_popup_type(self, popup_type):
        self.popup_type = popup_type
        for bar in self.gui.bars_manager.bars:
            if hasattr(bar, 'setup_tool_button'):
                if self.qaction in bar.added_actions:
                    bar.setup_tool_button(bar, self.qaction, self.popup_type)

    # Initialize customization (called from __init__.py Base class)
    def is_configurable(self):
        if not cfg.init(self):
            (info, error, detail) = msg.message.get_all()
            return (error, detail)
        return None

    # Menu item Create
    def create_kindleCollectionsLeefa(self):
        self.show_version()
        self._queue_job_create_collections(preview=False)

    # Menu item Preview
    def preview_kindleCollectionsLeefa(self):
        self.show_version()
        self._queue_job_create_collections(preview=True)

    # Menu item Customize (call __init__.py function to allow Preferences->Plugin->Customize to work too)
    def customize(self):
        self.show_version()
        self.interface_action_base_plugin.do_user_config(self.gui)

    # Menu item View
    def view_kindleCollectionsLeefa(self):
        self.show_version()
        self._queue_job_view_collections()

    # Menu item Edit
    def edit_kindleCollectionsLeefa(self):
        self.show_version()
        kindle_edit.run(self)

    # Menu item Import
    def import_kindleCollectionsLeefa(self):
        self.show_version()
        import_collections.run(self)

    # Menu item Restore
    def restore_previous_kindleCollectionsLeefa(self):
        self.show_version()
        kindle_restore.run(self)

    # Menu item Modify
    def modify_kindle_settings(self):
        self.show_version()
        kindle_settings.run(self)

    # Menu item Help
    def help(self):
        self.show_version()
        help.run()

    # Menu item About
    def about(self):
        self.show_version()
        about.run(self)

    def show_version(self):
        debug_print(PLUGIN_NAME + ' ' + PLUGIN_VERSION)

    # Execute create/preview collections as a device job in the background
    def _queue_job_create_collections(self, preview):
        debug_print('Starting queue create job')
        if preview:
            desc = _('Preview Kindle Collections Leefa')
        else:
            desc = _('Create Kindle Collections Leefa')
        # fix split preview so it doesn't run get books
        # Use Dispatcher so _completed function does not block jobs while waiting for user
        self.gui.device_manager.create_job(create_preview.create, self.Dispatcher(self._create_collections_completed), desc, args=[ self, preview ])
        if not preview:
            # Refresh the device view with collection information
            #self.gui.device_manager.get_device_information(FunctionDispatcher(self.gui.info_read))
            #self.gui.device_manager.books(self.Dispatcher(self.gui.metadata_downloaded))
            self.gui.device_manager.books(FunctionDispatcher(self.gui.metadata_downloaded))

        self.gui.status_bar.show_message(desc, 5000)
        debug_print('Done queueing create job')

    # Display results from create/preview collections job
    def _create_collections_completed(self, job):
        debug_print('Job _create_collections_completed')
        if job.failed:
            return self.gui.job_exception(job, dialog_title='Failed to create/preview collections')
        messages = job.result
        messages.display()

    # Execute view report collections as a device job in the background
    def _queue_job_view_collections(self):
        debug_print('Starting queue view job')
        self.gui.device_manager.create_job(kindle_view.run, self.Dispatcher(self._view_collections_completed), _('View Kindle Collections Leefa'), args=[self]),
        self.gui.status_bar.show_message('Viewing Kindle Collections Leefa', 5000)
        debug_print('Done queue view job')

    # Display results from view report job
    def _view_collections_completed(self, job):
        debug_print('Job _view_collections_completed')
        if job.failed:
            return self.gui.job_exception(job, dialog_title='Failed to view collections')
        messages = job.result
        messages.display()

