# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011-2014, meme'
__docformat__ = 'restructuredtext en'

import os, sys, re, time
from collections import defaultdict

import calibre_plugins.kindleCollectionsLeefa.config as cfg
import calibre_plugins.kindleCollectionsLeefa.messages as msg
import calibre_plugins.kindleCollectionsLeefa.calibre_info as calibre_info
import calibre_plugins.kindleCollectionsLeefa.kindle_device as kindle_device
import calibre_plugins.kindleCollectionsLeefa.kindle_books as kindle_books
import calibre_plugins.kindleCollectionsLeefa.kindleCollectionsLeefa as kindleCollectionsLeefa
from calibre_plugins.kindleCollectionsLeefa.__init__ import PLUGIN_NAME, PLUGIN_VERSION
from calibre_plugins.kindleCollectionsLeefa.utilities import debug_print, array_to_csv, wording, max_string_len, quote, get_pref

##########################################################################
# Preview and Create Report
##########################################################################

def generate_report_all(preview, status):
    debug_print('BEGIN Generating Report')
    if preview:
        msg.message.header('\n=== %s %s PREVIEW Report\n' % (PLUGIN_NAME, PLUGIN_VERSION))
        msg.message.header('Preview only - no collections have actually been modified on the Kindle.\n')
    else:
        msg.message.header('\n=== %s Report\n' % PLUGIN_NAME)

    generate_report_collections(kindleCollectionsLeefa.kc.get_sorted_names(), status, create=True)
    generate_report_collections(kindleCollectionsLeefa.kc.get_deleted_names(), status, create=False)

    msg.message.report('\n\n=== Settings:')
    for label in sorted(calibre_info.ci.column_labels.keys(), key=lambda x: calibre_info.ci.column_labels[x].lower()):
        action = cfg.config_table[label]['action']
        if action != cfg.OPT_NONE:
            msg.message.report('\nColumn: %s' % calibre_info.ci.column_labels[label])
            msg.message.report('        Action:      %s' % action)
            msg.message.report('        Prefix:      %s' % quote(cfg.config_table[label]['prefix']))
            msg.message.report('        Suffix:      %s' % quote(cfg.config_table[label]['suffix']))
            msg.message.report('        Minimum:     %s' % quote(cfg.config_table[label]['minimum']))
            msg.message.report('        Ignore:      %s' % quote(pattern_message(array_to_csv(cfg.config_table[label]['ignore']), 17)))
            msg.message.report('        Include:     %s' % quote(pattern_message(array_to_csv(cfg.config_table[label]['include']), 17)))
            msg.message.report('        Rename from: %s' % quote(cfg.config_table[label]['rename_from']))
            msg.message.report('        Rename to:   %s' % quote(cfg.config_table[label]['rename_to']))
            msg.message.report('        Split on:    %s' % quote(cfg.config_table[label]['split_char']))

    msg.message.report('')
    msg.message.report('Never delete/modify these Kindle Collections Leefa:      %s' % pattern_message(cfg.config_settings['ignore_all'], 51))
    msg.message.report('Action when toolbar icon is clicked:               %s' % get_pref(cfg.MENU_CLICK_STYLE))
    msg.message.report('Preserve existing Kindle-only collections:         %s' % cfg.config_settings['keep_kindle_only'])
    msg.message.report('Ignore prefix/suffix when comparing Calibre names: %s' % cfg.config_settings['ignore_prefix_suffix'])
    msg.message.report('Ignore case when pattern matching:                 %s' % cfg.config_settings['ignore_case'])
    msg.message.report('Reset collection times:                            %s' % cfg.config_settings['reset_collection_times'])
    msg.message.report('Ignore exported collections db:                    %s' % cfg.config_settings['ignore_json_db'])
    msg.message.report('Only build a diff database:                        %s' % cfg.config_settings['diff_db_only'])

    reboot_file = kindle_device.kdevice.get_kindle_hack_reboot_trigger_file()
    if reboot_file:
        msg.message.report('Fast reboot using hacks:                           %s' % cfg.config_settings['fast_reboot'])
        msg.message.report('Fast reboot trigger file:                          %s' % reboot_file)

    msg.message.report('Kindle Model:                                      %d' % cfg.config_settings['kindle_model_version'])
    msg.message.report('Calibre Personal doc tag:                          %s' % quote(calibre_info.ci.get_personal_doc_tag()))
    msg.message.report('Kindle Collections Leefa file:                           %s' % kindle_device.kdevice.get_collections_filename())
    msg.message.report('Plugin customization file:                         %s' % cfg.store.get_calibre_plugin_store_absolute_path())
    msg.message.report('System Encoding:                                   %s' % sys.getfilesystemencoding())

    if not preview:
        msg.message.report('\nRun View Report to see collection and book details.')

    debug_print('END Generating Report')

def generate_report_collections(collections, status, create):
    if collections:
        maxlen = max_string_len(collections)
        if maxlen < 10:
            maxlen = 10

        if create:
            msg.message.report('\nCollections saved to the Kindle:\n')
        else:
            msg.message.report('\nCollections deleted from the Kindle:\n')
        msg.message.report('    %-*s  %-10s  %-5s  %s' % (maxlen, 'Collection', 'Managed By', 'Books' , 'Change'))
        msg.message.report('    %-*s  %-10s  %-5s  %s' % (maxlen, '----------', '----------', '-----', '------'))

        for collection in collections:
            text = status.get(collection)
            if not msg:
                text = 'STATUS NOT FOUND'

            if kindleCollectionsLeefa.kc.is_kindle_managed(collection):
                managed_by = 'Kindle'
            else:
                managed_by = 'Calibre'
            msg.message.report('    %-*s  %-10s  %-5s  %s' % (maxlen, quote(collection), managed_by, kindle_books.kbooks.get_visible_book_count(collection), text))
    else:
        if create:
            msg.message.report('\nNo collections selected to create on the Kindle.\n')
        else:
            msg.message.report('\nNo collections selected to delete from the Kindle.\n')

def pattern_message(patterns, indent):
    SPECIAL_CHARACTERS = '.^$*+?{[(|\\'
    pattern = ''
    chars = ''
    if patterns:
        if type(patterns) == list:
            pattern = array_to_csv(patterns)
        else:
            pattern = patterns
        for c in SPECIAL_CHARACTERS:
            if c in pattern:
                chars += c
    if chars:
        text = '%s\n%*sContains special characters "%s" - check if "\\" is needed' % (pattern, indent, '', chars)
    else:
        text= '%s' % pattern
    return text

def report_book_details():

    msg.message.report('\n\n=== Book Details:')

    for path in kindle_books.kbooks.get_sorted_paths(kindle_books.kbooks.path_info.keys(), 'title', False):
#    for path in kindle_books.kbooks.get_paths_sorted_by_title():

        b = kindle_books.kbooks.path_info[path]
        collection_names = kindleCollectionsLeefa.kc.get_collections_for_code(b['code'])
        if b['in_calibre']:
            in_calibre_status = 'In Calibre'
        else:
            in_calibre_status = 'Not in Calibre'

        fmt = '\n%s\n' + \
                '    Path:          %s\n' + \
                '    Status:        %s\n' + \
                '    Collections:   %s'
        msg.message.report(fmt % (b['title'], path, in_calibre_status, '; '.join(collection_names)))

        fmt = \
                '    Modified:      %s\n' + \
                '    Size:          %s\n' + \
                '    Code:          %s\n' + \
                '    Mobi Type:     %s\n' + \
                '    cdeType:       %s\n' + \
                '    Path Author:   %s\n' + \
                '    Kindle Author: %s\n' + \
                '    Kindle Date:   %s\n' + \
                '    Sort Date:     %s'
        debug_print(fmt % (time.ctime(b['time']), b['path_size'], \
                    b['code'], b['mobi_type'], b['cdetype'], b['path_author'], b['author'], \
                    b['date'], b['sort_date']))

        if path in kindle_books.kbooks.uncollectable_paths:
            tags = []
            if path in calibre_info.ci.lpath_info:
                tags = calibre_info.ci.lpath_info[path]['tags']
            pdoc = calibre_info.ci.get_personal_doc_tag()
            msg.message.report(    '    WARNING:       The Kindle will not display this Mobi book in a collection due to an invalid code: "%s"' % b['code'][len(b['code'])-3:])
            msg.message.report(    '    Calibre tags:  %s' % array_to_csv(tags))
            if pdoc not in tags:
                msg.message.report('    Note:          Your Personal doc tag "%s" is not one of the Calibre tags for this book' % pdoc)
            else:
                msg.message.report('    Note:          Your Personal doc tag "%s" correctly appears in the Calibre tags for this book' % pdoc)
                msg.message.report('                   Try resending the book to the device if you recently updated your Personal doc tag')
                msg.message.report('                   Check your Metadata plugboards - if you modify tags this may prevent Calibre setting the type correctly')

