# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011-2014, meme'
__docformat__ = 'restructuredtext en'

############################################################
# Kindle Edit window
############################################################

from collections import defaultdict
from functools import partial

try:
    from PyQt5 import Qt as QtGui
    from PyQt5.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                         QLineEdit, QTableWidget, QTableWidgetItem, \
                         QAbstractItemView, QToolButton, QDialog, QDialogButtonBox
except:
    from PyQt4 import QtGui
    from PyQt4.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                         QLineEdit, QTableWidget, QTableWidgetItem, \
                         QAbstractItemView, QToolButton, QDialog, QDialogButtonBox

from calibre.gui2 import error_dialog

import calibre_plugins.kindleCollectionsLeefa.config as cfg
import calibre_plugins.kindleCollectionsLeefa.messages as msg
import calibre_plugins.kindleCollectionsLeefa.kindle_device as kindle_device
import calibre_plugins.kindleCollectionsLeefa.calibre_info as calibre_info
import calibre_plugins.kindleCollectionsLeefa.kindleCollectionsLeefa as kindleCollectionsLeefa
import calibre_plugins.kindleCollectionsLeefa.kindle_books as kindle_books
import calibre_plugins.kindleCollectionsLeefa.kindle_view_collections as view
import calibre_plugins.kindleCollectionsLeefa.save as save
from calibre_plugins.kindleCollectionsLeefa.__init__ import PLUGIN_NAME
from calibre_plugins.kindleCollectionsLeefa.utilities import debug_print, SizePersistedDialog, get_icon, \
        ComboTableWidgetItem, CheckableTableWidgetItem, CheckableBoxWidgetItem, ReadOnlyTableWidgetItem

EDIT_COLUMNS = {
    'include':     { 'pos': 0, 'title': 'In Collection' },
    'title':       { 'pos': 1, 'title': 'Title' },
    'author':      { 'pos': 2, 'title': 'Author' },
    'date':        { 'pos': 3, 'title': 'Date' },
    'in_calibre':  { 'pos': 4, 'title': 'In Calibre' },
    'visible':     { 'pos': 5, 'title': 'Visible' },
    'collections': { 'pos': 6, 'title': 'Collections' }
}
COL_INCLUDE = EDIT_COLUMNS['include']['pos']
COL_TITLE = EDIT_COLUMNS['title']['pos']
COL_AUTHOR = EDIT_COLUMNS['author']['pos']
COL_DATE = EDIT_COLUMNS['date']['pos']
COL_IN_CALIBRE = EDIT_COLUMNS['in_calibre']['pos']
COL_VISIBLE = EDIT_COLUMNS['visible']['pos']
COL_COLLECTIONS = EDIT_COLUMNS['collections']['pos']

PREFS_DIALOG_GEOMETRY = 'kindleCollectionsLeefa plugin:edit collections dialog: geometry'

############################################################


def run(parent):
        debug_print('BEGIN Editing Collections')
        gui = parent.gui

        if not cfg.init(parent):
            msg.message.display()
            debug_print('END Editing Collections - Initialization failed')
            return

        # Make sure the settings are up to date, we rely on the model being accurate here...
        if not cfg.store.is_current_store_format():
            msg.message.info('Please run "Customize collections" and select OK to save your customizations before running Edit Collections.')
            msg.message.display()
            return

        calibre_info.ci.load_column_book_info()
        kindle_device.init(calibre_info.ci.device_path)
        kindleCollectionsLeefa.init(kindle_device.kdevice.get_collections(), cfg.config_settings['reset_collection_times'])
        kindle_paths = kindle_device.kdevice.get_file_paths()
        kindle_books.init(kindle_paths)

        # Check that books were found on the Kindle
        if len(kindle_paths) < 1:
            error_dialog(gui, _(PLUGIN_NAME), _('No books were found on the Kindle.<P>If you know your books are on the device, try disconnecting/reconnecting your Kindle or restarting Calibre.'), '', show=True)
            return

        d = KindleEditDialog(gui, kindle_books.kbooks.get_paths_from_codes())
        d.exec_()
        if d.result() == d.Accepted:
            save.save_kindleCollectionsLeefa(compare=False)
            msg.message.report('\n\nRun View report to see collection and book details.')
            msg.message.display()

        debug_print('END Editing Collections')

class KindleEditDialog(SizePersistedDialog):

    def __init__(self, parent, book_paths):
        SizePersistedDialog.__init__(self, parent, PREFS_DIALOG_GEOMETRY)

        self.book_paths = book_paths
        # If there are no collections on Kindle need to init now
        self.collection = ''
        self.init_sort_order()

        self.setWindowTitle(_('Edit ' + PLUGIN_NAME))

        button_box = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        v = QVBoxLayout(self)

        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)

        # Add the window for editing
        v.addWidget(self.edit_widget())
        v.addWidget(button_box)

        self.resize_dialog()

    def init_sort_order(self):
        self.sort_order = { 'include': 1, 'title': 0, 'author': 0, 'date': 0, 'in_calibre': 0, 'visible': 0, 'collections': 0}

        # Initial sort is include->title so sort by title before automatically sorting by include
        self.sorted_paths = self.get_sorted_paths('title')
        self.sort_field = COL_INCLUDE

    def edit_widget(self):
        self.edit_w = QWidget()
        layout = QVBoxLayout(self.edit_w)
        self.edit_w.setLayout(layout)

        # Add a comment describing the window
        heading_label = QLabel('Edit collections on your Kindle.<P>If you run Create Collections to automatically create collections from Calibre, any changes you make to Calibre-managed collections will be over-written.<P>', self.edit_w)
        heading_label.setToolTip('<dt>Books:</dt><dd>Select individual books to include in the collection by checking the Include column for the book.<P>To select multiple books click on one entry then use shift-click or ctrl-click to select multiple books.<BR>Then click the "+" or "-" buttons on the right side of the table to add or remove the books from the collection.<P>To sort the columns, click on the column headers.  To sort multiple columns, click on one column header then another)<P>Authors are sorted using Calibre\'s Author Sort field if its available, otherwise by the Author defined in the book if its a Mobi book.  The name is always displayed as firstname lastname</dd><dt>Collections:</dt><dd>Create a new collection by entering the name in the box and clicking the "+" button.<BR>Rename a collection by entering the new name in the box and clicking the "->" button.<BR>Delete a collection by clicking the "-" button next to the collection name.</dd>')

        heading_layout = QHBoxLayout()
        heading_layout.addWidget(heading_label)
        layout.addLayout(heading_layout)

        # Add Collection row at top
        button_layout = QHBoxLayout()
        layout.addLayout(button_layout)

        c_label = QLabel('Collection:', self.edit_w)
        c_label.setToolTip('Click the collection drop down box to select a collection on the Kindle')
        button_layout.addWidget(c_label)

        # Add the Collection combobox - will be populated later
        self.edit_w.collection_combo = ComboTableWidgetItem(self.edit_w,'',[])
        self.edit_w.collection_combo.setToolTip('Click this drop down box to select a collection on the Kindle')
        button_layout.addWidget(self.edit_w.collection_combo)
        self.edit_w.collection_combo.setMinimumSize(200,20)
        self.edit_w.collection_combo.currentIndexChanged.connect(partial(self.collection_combo_index_changed, self.edit_w.collection_combo))

        # Add a delete button
        button = QtGui.QToolButton(self.edit_w)
        button.setToolTip('Delete the collection name listed to the left')
        button.setIcon(get_icon('minus.png'))
        button_layout.addWidget(button)
        button.clicked.connect(partial(self.delete_collection, self.edit_w))

        # Add some space
        button_layout.addSpacing(50)

        # Add a edit box button
        self.edit_w._new_name_ledit = QLineEdit('', self.edit_w)
        button_layout.addWidget(self.edit_w._new_name_ledit)
        self.edit_w._new_name_ledit.setMinimumSize(200,20)

        # Add an add button
        button = QtGui.QToolButton(self.edit_w)
        button.setToolTip('Create a new collection using the name in the edit box')
        button.setIcon(get_icon('plus.png'))
        button_layout.addWidget(button)
        button.clicked.connect(partial(self.add_collection, self.edit_w))

        # Add a rename button
        button = QtGui.QToolButton(self.edit_w)
        button.setToolTip('Rename collection to the name in the edit box')
        button.setIcon(get_icon('edit-redo.png'))
        button_layout.addWidget(button)
        button.clicked.connect(partial(self.rename_collection, self.edit_w))

        # Add space at the end so items are left aligned
        button_layout.addStretch(1)


        # Add table vertical layout containing the table and the buttons next to it
        table_layout = QHBoxLayout()
        layout.addLayout(table_layout)

        # Add the table
        self.edit_w._table = EditTableWidget(self.edit_w)
        # Handle clicks on the table
        self.edit_w._table.cellClicked.connect(self.cell_clicked)
        self.edit_w._table.horizontalHeader().sectionClicked.connect(self.header_clicked)
        table_layout.addWidget(self.edit_w._table)


        # Side row of table buttons
        side_layout = QVBoxLayout()
        table_layout.addLayout(side_layout)

        # Add button to add selected books to collection
        button = QtGui.QToolButton(self.edit_w)
        button.setToolTip('Add selected books to collection')
        button.setIcon(get_icon('plus.png'))
        button.clicked.connect(partial(self.add_selected_books_to_collection, self.edit_w))
        side_layout.addWidget(button)

        # Add button to remove selected books from collection
        button = QtGui.QToolButton(self.edit_w)
        button.setToolTip('Remove selected books from collection')
        button.setIcon(get_icon('minus.png'))
        button.clicked.connect(partial(self.remove_selected_books_from_collection, self.edit_w))
        side_layout.addWidget(button)

        # Refresh to current values
        self.refresh_all()
        return self.edit_w

    # Save which collections each book is in
    def add_collection_to_book(self, book_code, collection, row):
        if book_code in self.book_collections:
            if collection not in self.book_collections[book_code]:
                self.book_collections[book_code].append(collection)
        else:
            self.book_collections[book_code] = [ collection ]
        self.edit_w._table.table_set_collections(row, self.book_collections[book_code])

    def delete_collection_from_book(self, book_code, collection, row):
        if book_code in self.book_collections:
            if collection in self.book_collections[book_code]:
                self.book_collections[book_code].remove(collection)
        collections = []
        if book_code in self.book_collections:
            collections = self.book_collections[book_code]
        self.edit_w._table.table_set_collections(row, collections)

    def rebuild_collections_for_books(self):
        debug_print('BEGIN Rebuild collections for books')
        self.book_collections = defaultdict()
        for collection in self.get_editable_collection_names():
            codes = kindleCollectionsLeefa.kc.get_book_codes(collection)
            if codes:
                for code in codes:
                    if code in self.book_collections:
                        if collection not in self.book_collections[code]:
                            self.book_collections[code].append(collection)
                    else:
                        self.book_collections[code] = [ collection ]
        debug_print('END Rebuild collections for books')

    # Build a list of sorted book codes based on sorted titles split into 3 groups (in, out-0, out>0)
    def rebuild_list_of_books(self, collection):
        debug_print('BEGIN Rebuild list of books for collection: %s, sort=%s' % (collection, self.sort_field))
        a = []
        b = []
        c = []
        self.book_details = {}

        # Get the book codes this collection contains
        book_codes = kindleCollectionsLeefa.kc.get_book_codes(collection)

        # Get information on every path and save per book code
        for path in kindle_books.kbooks.path_info.keys():
            fullpath = kindle_device.kdevice.get_fullpath(path)
            code = kindle_books.kbooks.path_info[path]['code']
            author = kindle_books.kbooks.path_info[path]['author']
            author_sort = kindle_books.kbooks.path_info[path]['author_sort']
            title = kindle_books.kbooks.path_info[path]['title']
            sort_date = kindle_books.kbooks.path_info[path]['sort_date']
            in_calibre = kindle_books.kbooks.path_info[path]['in_calibre']
            mobi_type = kindle_books.kbooks.path_info[path]['mobi_type']
            cdetype = kindle_books.kbooks.path_info[path]['cdetype']
            visible = kindle_books.kbooks.path_info[path]['visible']

            # Get list of collections this code is in
            include = code in book_codes
            collection_list = self.book_collections[code] if code in self.book_collections else []

            # Save details for the table
            self.book_details[code] =  { 'include': include, 'path': fullpath, 'code': code, 'title': title, 'author': author, 'author_sort': author_sort, 'date': sort_date, 'in_calibre': in_calibre, 'collections': collection_list, 'mobi_type': mobi_type, 'cdetype': cdetype, 'visible': visible }

            # Update the book info to include collection related information - only used by Edit
            kindle_books.kbooks.set_path_info(path, 'include', include)
            kindle_books.kbooks.set_path_info(path, 'collection_count', len(collection_list))

        # Choose the field to sort on based on the sort field (column of header recently clicked)
        if self.sort_field == COL_TITLE:
            field = 'title'
        elif self.sort_field == COL_INCLUDE:
            field = 'include'
        elif self.sort_field == COL_AUTHOR:
            field = 'author'
        elif self.sort_field == COL_DATE:
            field = 'date'
        elif self.sort_field == COL_IN_CALIBRE:
            field = 'in_calibre'
        elif self.sort_field == COL_VISIBLE:
            field = 'visible'
        elif self.sort_field == COL_COLLECTIONS:
            field = 'collections'
        else:
            field = ''

        # Sort paths based on field
        self.sorted_paths = self.get_sorted_paths(field)

        # Sort codes based on sorted path
        self.sorted_book_codes = []
        for path in self.sorted_paths:
            code = kindle_books.kbooks.path_info[path]['code']
            self.sorted_book_codes.append(code)
        debug_print('END Rebuild list of books, %d paths, %d codes' % (len(self.sorted_paths), len(self.sorted_book_codes)))

    def get_sorted_paths(self, field):
        reverse_sort_order = self.sort_order[field] < 0 if field else False
        return kindle_books.kbooks.get_sorted_paths(self.book_paths, field, reverse_sort_order)

    def collection_combo_index_changed(self, combo, row):
        idx = combo.currentIndex()
        #debug_print('Combobox changed to index: %s' % idx)
        if idx >= 0:
            collection = self.get_editable_collection_names()[idx]
            if not collection:
                collection = ''
            self.collection = collection
            #debug_print('Combobox select collection "%s"' % collection)
            self.refresh_table(collection)

    def add_collection(self, w):
        collection = unicode(self.edit_w._new_name_ledit.text())
        if collection and not collection.isspace():
            if kindleCollectionsLeefa.kc.is_collection(collection):
                error_dialog(self.edit_w, _('Error'), _('Collection "%s" exists' % collection), '', show=True)
            else:
                debug_print('Add collection "%s"' % collection)
                kindleCollectionsLeefa.kc.add(collection, [], kindle_managed=True)
                # Refresh since book order might change as new collection will have no books
                self.refresh_all(collection)
        else:
            error_dialog(self.edit_w, _('Error'), _('Enter the name of the new collection to create'), '', show=True)

    def delete_collection(self, w):
        idx = w.collection_combo.currentIndex()
        if idx >= 0:
            collection = self.get_editable_collection_names()[idx]
            kindleCollectionsLeefa.kc.delete(collection)
            self.collection = ''
            debug_print('Delete collection "%s"' % collection)
            # Refresh since book order might change based on collections changed
            self.refresh_all()

    def rename_collection(self, w):
        idx = w.collection_combo.currentIndex()
        if idx >= 0:
            oldcollection = self.get_editable_collection_names()[idx]
            newcollection = unicode(self.edit_w._new_name_ledit.text())
            if newcollection and not newcollection.isspace():
                if kindleCollectionsLeefa.kc.is_collection(newcollection):
                    error_dialog(self.edit_w, _('Error'), _('Collection "%s" exists' % newcollection), '', show=True)
                else:
                    debug_print('Rename collection "%s" to "%s"' % (oldcollection, newcollection))
                    kindleCollectionsLeefa.kc.rename(oldcollection, newcollection)
                    # Refresh since its easier than coding special case
                    self.refresh_all(newcollection)
            else:
                error_dialog(self.edit_w, _('Error'), _('Enter the new name for the current collection'), '', show=True)

    def refresh_all(self, collection=''):
        debug_print('BEGIN Refresh all, collection "%s"' % collection)

        self.current_collection = ''
        # Update the collection drop-down box
        self.edit_w.collection_combo.clear()
        debug_print('Get editable collection names')
        self.edit_w.collection_combo.addItems(self.get_editable_collection_names())

        if len(self.get_editable_collection_names()) == 0:
            self.refresh_table()
        else:
            idx = self.get_editable_collection_names().index(collection) if collection != '' else 0
            self.edit_w.collection_combo.setCurrentIndex(idx)
            # Automatically refreshes table once the combobox index is changed
        debug_print('END Refresh all')

    def refresh_table(self, collection=''):
        debug_print('BEGIN Refresh table from collection "%s" to "%s"' % (self.current_collection, collection))
        if collection == '':
            collection = self.current_collection
        if collection != '' and collection != self.current_collection:
            self.current_collection = collection
            self.init_sort_order()
        if collection == '':
            if len(self.get_editable_collection_names()) > 0:
                collection = self.get_editable_collection_names()[0]
        # Update the list of collections per book, then list of books
        self.rebuild_collections_for_books()
        self.rebuild_list_of_books(collection)

        # Repaint the table for the current collection
        self.edit_w._table.populate_table(self.sorted_book_codes, self.book_details, self.sort_field, self.sort_order)
        debug_print('END Refresh table to collection "%s"' % collection)

    def header_clicked(self, col):
        #debug_print('Header clicked: %s' % (col))
        # Re-sort only if a different column or the include column (in case items were checked/unchecked)
        if col == COL_INCLUDE:
            self.set_sort_order(COL_INCLUDE, 'include')
        elif col == COL_TITLE:
            self.set_sort_order(COL_TITLE, 'title')
        elif col == COL_AUTHOR:
            self.set_sort_order(COL_AUTHOR, 'author')
        elif col == COL_DATE:
            self.set_sort_order(COL_DATE, 'date')
        elif col == COL_IN_CALIBRE:
            self.set_sort_order(COL_IN_CALIBRE, 'in_calibre')
        elif col == COL_VISIBLE:
            self.set_sort_order(COL_VISIBLE, 'visible')
        elif col == COL_COLLECTIONS:
            self.set_sort_order(COL_COLLECTIONS, 'collections')
        self.sort_field = col
        self.refresh_table()

    def set_sort_order(self, field, fieldname):
        if self.sort_order[fieldname] == 0 or self.sort_field != field:
            self.sort_order[fieldname] = 1
        else:
            self.sort_order[fieldname] *= -1

    # Add/remove the book from the current collection
    def cell_clicked(self, row, col):
        #debug_print('\nCell clicked: %s, %s' % (row, col))

        if self.collection == '':
            self.edit_w._table.item(row, COL_INCLUDE).setCheckState(Qt.Unchecked)
        elif col == COL_INCLUDE:
            old_state = self.edit_w._table.include_state[row]
            new_state = self.edit_w._table.item(row, COL_INCLUDE).checkState() == Qt.Checked
            self.edit_w._table.include_state[row] = new_state
            #debug_print('Old/New Include State: %s, %s' % (old_state, new_state))
            if old_state != new_state:
                self.add_remove_clicked_book(row)

    def add_selected_books_to_collection(self, w):
        if self.collection != '':
            # Use the range information to select multiple books
            row_ranges = w._table.selectedRanges()
            for row_range in row_ranges:
                for row in range(row_range.topRow(), row_range.bottomRow() + 1):
                    self.set_row_include(row, True)

    def remove_selected_books_from_collection(self, w):
        if self.collection != '':
            # Use the range information to select multiple books
            row_ranges = w._table.selectedRanges()
            for row_range in row_ranges:
                for row in range(row_range.topRow(), row_range.bottomRow() + 1):
                    self.set_row_include(row, False)

    def set_row_include(self, row, checked):
        check_state = Qt.Checked if checked else Qt.Unchecked
        self.edit_w._table.item(row, COL_INCLUDE).setCheckState(check_state)
        self.add_remove_clicked_book(row)

    def add_remove_clicked_book(self, row):
        #debug_print('Add_remove book row %d' % row)
        book_code = self.sorted_book_codes[row]
        name = self.book_details[book_code]['title']

        # Change count and collection list - but not sort order (confusing to user, extra work)
        if self.edit_w._table.item(row, COL_INCLUDE).checkState() == Qt.Checked:
            #debug_print('Checked so add "%s" to collection "%s"' % (name, self.collection))
            kindleCollectionsLeefa.kc.add_book(self.collection, book_code)
            self.add_collection_to_book(book_code, self.collection, row)
        else:
            #debug_print('Unchecked so delete "%s" from collection "%s"' % (name, self.collection))
            kindleCollectionsLeefa.kc.delete_book(self.collection, book_code)
            self.delete_collection_from_book(book_code, self.collection, row)

        # Set sort so if include is clicked it always sorts in forward order after inclusion changed
        self.sort_order['include'] = 0

    def get_editable_collection_names(self):
        return kindleCollectionsLeefa.kc.get_sorted_names()

############################################################
class EditTableWidget(QTableWidget):

    def __init__(self, *args):
        QTableWidget.__init__(self, *args)
        self.already_resized = False

    # Extract the column header labels in the right order from the main dictionary
    def get_header_labels(self, sort_field, sort_order):
        header_labels = [ '' ] * len(EDIT_COLUMNS)
        for k in EDIT_COLUMNS.keys():
            s = '   '
            if EDIT_COLUMNS[k]['pos'] == sort_field:
                if sort_order[k] > 0:
                    s = u'\u2193' # down arrow
                elif sort_order[k] < 0:
                    s = u'\u2191' # up arrow
            header_labels[EDIT_COLUMNS[k]['pos']] = '   ' + EDIT_COLUMNS[k]['title']  + ' ' + s
        return header_labels

    # Fill the table with the given data
    def populate_table(self, row_sorted_indexes, row_data, sort_field, sort_order):
        self.clear()
        self.row_sorted_indexes = row_sorted_indexes
        self.row_data = row_data

        # Define the row/columns before putting any data in table
        h = self.horizontalHeader()
        h.setMinimumSectionSize(50)
        self.setRowCount(len(self.row_sorted_indexes))
        header_labels = self.get_header_labels(sort_field, sort_order)
        self.setColumnCount(len(header_labels))

        # Fill the table, saving each column label for each row number for easier lookup when saving
        #debug_print('Populate Edit table, rows = %s' % self.rowCount())
        self.revrow = range(self.rowCount())
        self.include_state = {}
        for row, index in enumerate(row_sorted_indexes):
            self.revrow[row] = index
            self.populate_table_row(row, row_data[index])

        # Set style of table, and set selection to first row
        self.setAlternatingRowColors(True)
        self.setSortingEnabled(False)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.selectRow(0)

        # Set the column headers
        self.verticalHeader().setDefaultSectionSize(24)
        self.setHorizontalHeaderLabels(header_labels)

        if not self.already_resized:
            self.resizeColumnsToContents()
            self.already_resized = True

    # Return the column in the row for the given label
    def pos(self, column):
        return EDIT_COLUMNS[column]['pos']

    def populate_table_row(self, row, data):
        self.blockSignals(True)

        # Checkbox for included in collection or not
        item = CheckableTableWidgetItem(data['include'])
        self.setItem(row, self.pos('include'), item)
        self.include_state[row] = data['include']
        if data['include']:
            item.setToolTip('This book is included in the current Collection.<P>Click on the checkbox to remove it from the collection.<P>To select multiple books at once use the buttons on the right.')
        else:
            item.setToolTip('This book is not included in current collection.<P>Click on the checkbox to add it to the collection.<P>To select multiple books at once use the buttons on the right.')

        # Title of book
        item = ReadOnlyTableWidgetItem(data['title'])
        self.setItem(row, self.pos('title'), item)
        item.setToolTip(data['path'])

        # Author
        item = ReadOnlyTableWidgetItem(data['author'])
        self.setItem(row, self.pos('author'), item)
        item.setToolTip('Sort by: %s' % data['author_sort'])

        # Date
        item = ReadOnlyTableWidgetItem(data['date'])
        self.setItem(row, self.pos('date'), item)
        item.setToolTip(data['path'])

        # In Calibre
        item = ReadOnlyTableWidgetItem('')
        self.setItem(row, self.pos('in_calibre'), item)
        if data['in_calibre']:
            item.setIcon(get_icon('ok.png'))
            item.setToolTip('Calibre manages this book')
        else:
            item.setToolTip('This book is not managed by Calibre.<P>The pathname does not match any books sent to the device by Calibre.')

        # Visible on Kindle in collections
        item = ReadOnlyTableWidgetItem('')
        self.setItem(row, self.pos('visible'), item)
        if data['visible']:
            item.setIcon(get_icon('ok.png'))
            item.setToolTip('The Kindle can display this title in a collection.<P>Code is %s<BR>Mobi type is "%s"<BR>cdeType is "%s"' % (data['code'], data['mobi_type'], data['cdetype']))
        else:
            item.setToolTip('The Kindle cannot display this title in a collection due to the book\'s code.<P>Code is %s<BR>Mobi type is "%s", cdeType is "%s"<P>Run View report for more details.' % (data['code'], data['mobi_type'], data['cdetype']))

        # Collection drop-down list for book
        item = ComboTableWidgetItem(self, '', [ str(len(data['collections']))] + data['collections'])
        self.setCellWidget(row, self.pos('collections'), item)
        item.setCurrentIndex(0)
        item.setToolTip('Number and list of collections for this book<P>Field is view-only.')
        item.currentIndexChanged.connect(partial(self.collection_combo_table_index_changed, item))

        self.blockSignals(False)

    # Get all the data from the updated customization window to save for later
    def get_data(self):
        row_data = {}
        for row in range(self.rowCount()):
            row_data[self.revrow[row]] = self.convert_row_to_data(row)
        return row_data

    # Get data from one row in the Customization window to save for later
    def convert_row_to_data(self, row):
        # Blank data since we'll populate it with specific data, leave spaces in prefix
        data = {}
        data['include'] = self.item(row, self.pos('include')).checkState() == Qt.Checked
        # Don't care about the other data as its read-only
        return data

    def collection_combo_table_index_changed(self, combo, row):
        # Always show first entry (count of collections) to avoid making it appear you can select a collection
        combo.setCurrentIndex(0)

    def table_set_collections(self, row, collections):
        collections_item = self.cellWidget(row, self.pos('collections'))
        collections_item.clear()
        collections_item.addItems([str(len(collections)) ] + collections)
        collections_item.setCurrentIndex(0)

