# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011-2014, meme'
__docformat__ = 'restructuredtext en'

############################################################
# Kindle Edit window
############################################################

import re
from collections import defaultdict

from calibre.gui2 import question_dialog, error_dialog

try:
    from PyQt5 import Qt as QtGui
    from PyQt5.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                         QDialogButtonBox
except:
    from PyQt4 import QtGui
    from PyQt4.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                         QDialogButtonBox

import calibre_plugins.kindleCollectionsLeefa.config as cfg
import calibre_plugins.kindleCollectionsLeefa.messages as msg
import calibre_plugins.kindleCollectionsLeefa.kindle_device as kindle_device
import calibre_plugins.kindleCollectionsLeefa.calibre_info as calibre_info
import calibre_plugins.kindleCollectionsLeefa.kindleCollectionsLeefa as kindleCollectionsLeefa
import calibre_plugins.kindleCollectionsLeefa.kindle_books as kindle_books
from calibre_plugins.kindleCollectionsLeefa.__init__ import PLUGIN_NAME
from calibre_plugins.kindleCollectionsLeefa.utilities import debug_print, SizePersistedDialog, \
        ComboTableWidgetItem, get_pref, set_pref

PREFS_COLUMN_DEFAULT  = 'kindleCollectionsLeefa plugin:import collections dialog: selected column'
PREFS_DIALOG_GEOMETRY = 'kindleCollectionsLeefa plugin:import collections dialog: geometry'

############################################################

def run(parent):
    debug_print('BEGIN Import Kindle Collections Leefa ')
    gui = parent.gui


    if not cfg.init(parent):
        msg.message.display()
        debug_print('END Import Kindle Collections Leefa - Initialization failed')
        return


    calibre_info.ci.load_column_book_info()
    kindle_device.init(calibre_info.ci.device_path)
    kindleCollectionsLeefa.init(kindle_device.kdevice.get_collections())
    kindle_paths = kindle_device.kdevice.get_file_paths()

    # Check that books were found on the Kindle
    if len(kindle_paths) < 1:
        error_dialog(gui, _(PLUGIN_NAME), _('No books were found on the Kindle.<P>If you know your books are on the device, try disconnecting/reconnecting your Kindle or restarting Calibre.'), '', show=True)
        return

    kindle_books.init(kindle_paths)
    useable_columns = get_useable_custom_columns()

    # bypass all prompts
    #leefa begin
    default_field = get_pref(PREFS_COLUMN_DEFAULT)
    if not default_field:
        default_field = column_names[0]
    debug_print('**********default_field:'+default_field)
    defCol=get_default_columns(default_field)
    print(defCol)
    if defCol==None:
        msg.message.info('No default column selected, nothing imported.')
        msg.message.display()
        return
    column_name = default_field
    column_label = str(defCol['label'])
    column_type = defCol['type']
    column_is_multiple = defCol['is_multiple']
    debug_print('column_name = ' + str(default_field) + ' column_label = ' + str(defCol['label'] ) + ' column_type = ' + str(defCol['type'])+ ' column_is_multiple = ' + str(defCol['is_multiple']))
    #leefa end

    debug_print('Continuing import to column "%s"' % column_name)
    debug_print('Building list from %d books in Calibre on device' % len(calibre_info.ci.ids_on_device))
    set_pref(PREFS_COLUMN_DEFAULT, column_name)
    id_collections = {}
    for id in calibre_info.ci.ids_on_device:
        lpath = calibre_info.ci.id_map[id].lpath
        collections = []
        if lpath in kindle_books.kbooks.path_info:
            code = kindle_books.kbooks.path_info[lpath]['code']
            collections = kindleCollectionsLeefa.kc.get_collections_for_code(code)
        if column_type == 'bool':
            id_collections[id] = True if len(collections) > 0 else None
        else:
            # If column contains ",", convert to ";" - hack to allow Author Sort names to be imported/exported
            new_collections = [ collection.replace(',', ';') for collection in collections ]
            id_collections[id] = ','.join(new_collections)
    debug_print('%d book entries to update' % len(calibre_info.ci.ids_on_device))

    debug_print('Importing to column %s' % column_label)
    for id in id_collections:
        debug_print('    id %s, collections=%s' % (id, id_collections[id]))

    if len(id_collections) < 1:
        msg.message.error('No collections for books found on the Kindle - import aborted.<P>If you know there are books on your Kindle and in Calibre, try disconnecting/reconnecting your Kindle or restarting Calibre.')
    else:
        calibre_info.ci.clear_custom_column(column_label)
        calibre_info.ci.set_custom_values(id_collections, column_label)
        msg.message.info('Collections imported into custom column "%s".' % column_name)
    msg.message.display()
    debug_print('END Import Kindle Collections Leefa')

def get_useable_custom_columns():
    useable_columns = defaultdict()
    all_custom_columns = calibre_info.ci.get_custom_columns()
    for column in all_custom_columns:
        col = all_custom_columns[column]
        col_name = col['name']
        typ = col['datatype']
        is_multiple = col['is_multiple'] != None
        debug_print('Checking column=%s, Name="%s", Type=%s, Multiple=%s' % (column, col_name, typ, is_multiple))
        if typ in ('text', 'comments', 'bool'):
            # Strip '#' at start of custom column labels since we know its custom
            column = column[1:]
            useable_columns[col_name] = { 'label': column, 'type': typ, 'is_multiple': is_multiple }
            debug_print('    OK for import')
        else:
            debug_print('    Cannot be used for import')
    return useable_columns

class KindleImportDialog(SizePersistedDialog):

    def __init__(self, parent, fields, default_field):
        SizePersistedDialog.__init__(self, parent, PREFS_DIALOG_GEOMETRY)

        self.setWindowTitle(_('Import ' + PLUGIN_NAME))

        v = QVBoxLayout(self)

        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)

        self.w = QWidget()
        layout = QVBoxLayout(self.w)
        self.w.setLayout(layout)

        # Add a comment describing the window
        heading_label = QLabel('Import all Kindle collection names into the Calibre Custom Column listed below:', self.w)
        heading_label.setToolTip('Select the custom column you created to import to.  All existing entries will be overwritten.<P>You can use custom columns of type text, comma separated text, long text, or yes/no.')

        heading_layout = QHBoxLayout()
        heading_layout.addWidget(heading_label)
        layout.addLayout(heading_layout)

        # Add Collection row at top
        button_layout = QHBoxLayout()
        layout.addLayout(button_layout)

        c_label = QLabel('Custom Column:', self.w)
        c_label.setToolTip('Click the collection drop down box to select a collection on the Kindle')
        button_layout.addWidget(c_label)

        # Add the Collection combobox - will be populated later
        self.w.collection_combo = ComboTableWidgetItem(self.w,default_field, fields)
        self.w.collection_combo.setToolTip('Click this drop down box to select a collection on the Kindle')
        self.w.collection_combo.setMinimumSize(200, 20)
        button_layout.addWidget(self.w.collection_combo)
        button_layout.addStretch(1)
        layout.addStretch(1)

        # Add the top piece
        v.addWidget(self.w)

        # Add the selection buttons
        v.addWidget(button_box)

        self.resize_dialog()

def get_default_columns(default_field):
    all_custom_columns = calibre_info.ci.get_custom_columns()
    print default_field
    for column in all_custom_columns:
        col = all_custom_columns[column]
        col_name = col['name']
        typ = col['datatype']
        is_multiple = col['is_multiple'] != None
        debug_print('Checking column=%s, Name="%s", Type=%s, Multiple=%s' % (column, col_name, typ, is_multiple))
        if col_name == default_field:
            if typ in ('text', 'comments', 'bool'):
                # Strip '#' at start of custom column labels since we know its custom
                column = column[1:]
                return  { 'label': column, 'type': typ, 'is_multiple': is_multiple }
                debug_print('    OK for import')
            else:
                debug_print('    Cannot be used for import')
    return None
