# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011-2014, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# Manage collections in Kindle format
#####################################################################

import os, re, time
from collections import defaultdict

import calibre_plugins.kindleCollectionsLeefa.messages as msg
from calibre_plugins.kindleCollectionsLeefa.kindle_sort import KindleSort
from calibre_plugins.kindleCollectionsLeefa.utilities import debug_print

kc = None
kcd = None

#####################################################################

def init(collections, reset_collection_times = False):
    global kc, kcd
    kc = KindleCollectionsLeefa(collections, reset_collection_times)
    kcd = KindleCollectionsLeefa(None, reset_collection_times)

class KindleCollectionsLeefa():

    # Some time in the past so collections appear at end of lists sorted by time
    LAST_ACCESS_TIME = '1234567890000'
    # Default locale if none defined in existing collections
    KINDLE_LOCALE = '@en-US'

    def __init__(self, collections, reset_collection_times = False):
        if not collections:
            collections = {}
        self.reset_collection_times = reset_collection_times
        self.initialize(collections)

    def initialize(self, collections):
        if collections:
            debug_print('BEGIN Initializing KindleCollectionsLeefa, %d existing collections:' % len(collections))
        else:
            debug_print('BEGIN Initializing KindleCollectionsLeefa, no collections')
        self.locale = self.get_locale(collections)
        self.collections = self.strip_locale(collections)
        self.collections = self.check_collections(self.collections)
        debug_print('%d cleaned up existing collections' % len(self.collections))
        self.notkindle_managed_names = []
        self.deleted_names = defaultdict()
        self.sorted_names = None
        self.time_sorted_names = None
        self.ksort = KindleSort()
        self.all_collected_codes = []
        self.code_collections = defaultdict()
        debug_print('END Initializing KindleCollectionsLeefa')

    def check_collections(self, collections):
        for c in collections:
            debug_print('    "%s"' % c)
            if 'items' not in collections[c] or type (collections[c]['items']) != list:
                msg.message.warning('Collection "%s" has an incorrect format - items are missing or not in a list.  Fixed by setting items to none.\n' % c)
                collections[c]['items'] = []

            if 'lastAccess' not in collections[c] or (type(collections[c]['lastAccess']) != long and type(collections[c]['lastAccess']) != int):
                msg.message.warning('Collection "%s" has an incorrect format - lastAccess time missing or not a number.  Fixed by setting lastAccess time to 0.\n' % c)
                collections[c]['lastAccess'] = 0
        return collections

    # Use locale from existing collections or use default
    def get_locale(self, collections):
        if collections and len(collections) > 0:
            keyname = collections.keys()[0]
            locale = re.sub(r'^.*(@[^@]+)$',r'\1', keyname)
        else:
            locale = self.KINDLE_LOCALE
        return locale

    # Strip ending locale from collection names to make working with them externally easier
    def strip_locale(self, collections):
        newcollections = {}
        if collections:
            #pat = re.compile(r'^(.*)' + self.locale +  r'$')
            pat = re.compile(r'^(.*)@[^@]+$')
            for oldname in collections.keys():
                newname = pat.sub(r'\1', oldname)
                newcollections[newname] = collections[oldname]
        return newcollections

    def add_locale(self, collections):
        newcollections = {}
        if collections:
            for oldname in collections.keys():
                newname = oldname + self.locale
                newcollections[newname] = collections[oldname]
        return newcollections

    def is_kindle_managed(self, collection):
        return collection not in self.notkindle_managed_names

    def is_collection(self, collection):
        return collection in self.collections.keys()

    def get_collections_for_code(self, book_code):
        if not self.code_collections:
            debug_print('Rebuilding get_collections code list %s' % book_code)
            code_collections = defaultdict()
            for collection in self.collections:
                codes = self.get_book_codes(collection)
                if codes:
                    for code in codes:
                        if code in self.code_collections:
                            if collection not in self.code_collections[code]:
                                self.code_collections[code].append(collection)
                        else:
                            self.code_collections[code] = [ collection ]
        if book_code in self.code_collections:
            c = self.code_collections[book_code]
        else:
            c = []
        return c

    def is_code_in_collection(self, code):
        if not self.all_collected_codes:
            self.all_collected_codes = []
            for collection in self.collections:
                for item in self.collections[collection]['items']:
                    self.all_collected_codes.append(item)
        return code in self.all_collected_codes

    def get_len(self):
        length = 0
        if self.collections:
            length = len(self.collections)
        return length

    def get_book_codes(self, collection):
        codes = []
        if collection in self.collections:
            codes = self.collections[collection]['items']
        return codes

    def get_all_book_codes(self):
        codes = []
        for collection in self.collections:
            codes += self.get_book_codes(collection)
        return codes

    def get_current_access_time(self):
        return time.time() * 1000

    # Add one book to a collection and update collection timestamp if requested
    def add_book(self, collection, book_code, multiple=False):
        if collection in self.collections:
            lastaccess = self.get_current_access_time() if self.reset_collection_times else self.collections[collection]['lastAccess']
            book_codes = self.get_book_codes(collection)
            if book_codes:
                if multiple or book_code not in book_codes:
                    self.collections[collection] = {'items':book_codes+[ book_code ], 'lastAccess':long(lastaccess)}
            else:
                self.collections[collection] = {'items':[book_code], 'lastAccess':long(lastaccess)}

    # Delete one book to a collection and update collection timestamp if requested
    def delete_book(self, collection, book_code, lastaccess=None):
        if collection in self.collections:
            lastaccess = self.get_current_access_time() if self.reset_collection_times else self.collections[collection]['lastAccess']
            book_codes = self.get_book_codes(collection)
            if book_codes and book_code in book_codes:
                book_codes.remove(book_code)
                self.collections[collection] = {'items':book_codes,'lastAccess':long(lastaccess)}

    def get_book_count(self, collection):
        length = 0
        if collection in self.collections:
            length = len(self.collections[collection]['items'])
        elif collection in self.deleted_names:
            length = self.deleted_names[collection]
        return length

    def get_unsorted_names(self):
        names = []
        if self.collections:
            names = self.collections.keys()
        return names

    # Only sort if not sorted
    def get_time_sorted_names(self,init=False):
        if init:
            self.time_sorted_names = []
        if not self.time_sorted_names:
            if self.collections:
                if self.reset_collection_times:
                    self.set_last_access_times()
                self.time_sorted_names = self.collections.keys()
                self.time_sorted_names.sort(key=lambda x: self.collections[x]['lastAccess'], reverse=True)
            else:
                self.time_sorted_names = []
        return self.time_sorted_names

    # Return a list of names sorted Kindle style
    def get_sorted_names(self, init=False):
        if init:
            self.sorted_names = []
        if not self.sorted_names:
            if self.collections:
                self.sorted_names = self.ksort.sort_names(self.collections.keys())
            else:
                self.sorted_names = []
        return self.sorted_names

    # Update lastAccess time for all non-Kindle-only collections so the order matches alphabetical sorting
    def set_last_access_times(self):
        if self.collections:
            last_access_time = long(self.LAST_ACCESS_TIME)
            for collection in self.get_sorted_names():
                if not self.is_kindle_managed(collection):
                    self.collections[collection]['lastAccess'] = last_access_time
                    last_access_time -= 1

    def get_collection(self, collection):
        found = None
        if collection in self.collections:
            found = self.collections[collection]
        return found

    def get_raw_collections(self):
        # Only update the collection times to sort like alphabetically if requested
        if self.reset_collection_times:
            self.set_last_access_times()
        return self.add_locale(self.collections)

    def get_deleted_names(self):
        names = ''
        if self.deleted_names:
            names = self.ksort.sort_names(self.deleted_names.keys())
        return names

    # Remove whole collection or just a book (book not used yet) without adding to delete list
    def delete_quiet(self, collection):
        if collection in self.collections:
            del self.collections[collection]
            if self.sorted_names:
                self.sorted_names.remove(collection)
            if self.time_sorted_names:
                self.time_sorted_names.remove(collection)

    # Remove whole collection or just a book (book not used yet)
    def delete(self, collection, kindle_managed=True):
        if collection in self.collections:
            self.deleted_names[collection] = self.get_book_count(collection)
            self.delete_quiet(collection)
            if not kindle_managed:
                self.notkindle_managed_names.append(collection)

    def rename(self, old, new):
        if old in self.collections:
            self.collections[new] = self.collections[old]
            del self.collections[old]
        if self.sorted_names:
            self.get_sorted_names(init=True)
        if self.time_sorted_names:
            self.get_time_sorted_names(init=True)

    # Add a collection (if it already exists, check timestamp option)
    def add(self, collection, book_codes, kindle_managed=False):
        if not kindle_managed:
            self.notkindle_managed_names.append(collection)
        lastaccess = self.get_current_access_time()
        if collection in self.collections and not self.reset_collection_times:
            lastaccess = self.collections[collection]['lastAccess']
        self.collections[collection] = { 'items': book_codes, 'lastAccess': long(lastaccess) }

        if self.sorted_names:
            self.get_sorted_names(init=True)
        if self.time_sorted_names:
            self.get_time_sorted_names(init=True)

