#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (division, absolute_import, print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, David Forrester <davidfor@internode.on.net>'
__docformat__ = 'restructuredtext en'

import ConfigParser
import calendar
import os, threading, time, shutil
from datetime import datetime, timedelta
from contextlib import closing
from collections import OrderedDict

try:
    from PyQt5.Qt import QUrl, pyqtSignal, QTimer
    from PyQt5.Qt import (Qt, QApplication, QMenu, QToolButton, QStandardItemModel, QStandardItem, QUrl,
                          QModelIndex, QFileDialog, QIcon)
except ImportError:
    from PyQt4.Qt import QUrl, pyqtSignal, QTimer
    from PyQt4.Qt import (Qt, QApplication, QMenu, QToolButton, QStandardItemModel, QStandardItem, QUrl, 
                          QModelIndex, QFileDialog, QIcon)

from calibre import strftime
from calibre.constants import numeric_version as calibre_version
from calibre.gui2.dialogs.message_box import MessageBox
from calibre.gui2 import error_dialog, info_dialog, open_url, question_dialog, FileDialog, warning_dialog
from calibre.gui2.actions import InterfaceAction
from calibre.ptempfile import PersistentTemporaryDirectory, remove_dir
from calibre.gui2.dialogs.message_box import ViewLog
from calibre.gui2.library.views import DeviceBooksView
from calibre.utils.icu import sort_key
from calibre.utils.config import config_dir
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2.device import device_signals

from calibre.devices.kobo.driver import KOBO, KOBOTOUCH
from calibre.devices.kobo.books import Book
from calibre.devices.usbms.driver import USBMS

from calibre_plugins.koboutilities.dialogs import (
                    ReaderOptionsDialog, CoverUploadOptionsDialog, RemoveCoverOptionsDialog, AboutDialog, 
                    UpdateMetadataOptionsDialog, ChangeReadingStatusOptionsDialog, ShowBooksNotInDeviceDatabaseDialog, 
                    ManageSeriesDeviceDialog, BookmarkOptionsDialog, BackupAnnotationsOptionsDialog, DismissTilesOptionsDialog,
                    QueueProgressDialog, CleanImagesDirOptionsDialog, BlockAnalyticsOptionsDialog,
                    FixDuplicateShelvesDialog, OrderSeriesShelvesDialog, ShowReadingPositionChangesDialog,
                    GetShelvesFromDeviceDialog
                    )
from calibre_plugins.koboutilities.common_utils import (set_plugin_icon_resources, get_icon, ProgressBar,
                                         create_menu_action_unique, get_library_uuid, debug_print)
from calibre_plugins.koboutilities.book import SeriesBook
import calibre_plugins.koboutilities.config as cfg

PLUGIN_ICONS = ['images/icon.png', 'images/logo_kobo.png', 'images/manage_series.png', 'images/lock.png',
                'images/lock32.png','images/lock_delete.png', 'images/lock_open.png', 'images/sort.png',
                'images/ms_ff.png', 'images/device_connected.png']

MIMETYPE_KOBO = 'application/x-kobo-epub+zip'

BOOKMARK_SEPARATOR = '|@ @|'       # Spaces are included to allow wrapping in the details panel

EPUB_FETCH_QUERY = 'SELECT c1.ChapterIDBookmarked, ' \
                        'c2.adobe_location, '      \
                        'c1.ReadStatus, '          \
                        'c1.___PercentRead, '      \
                        'c1.Attribution, '         \
                        'c1.DateLastRead, '        \
                        'c1.Title, '               \
                        'c1.MimeType, '            \
                        'r.rating, '               \
                        'c1.contentId '            \
                    'FROM content c1 LEFT OUTER JOIN content c2 ON c1.ChapterIDBookmarked = c2.ContentID ' \
                        'LEFT OUTER JOIN ratings r ON c1.ContentID = r.ContentID '  \
                    'WHERE c1.ContentID = ?'

EPUB_FETCH_QUERY_NORATING = 'SELECT c1.ChapterIDBookmarked, ' \
                        'c2.adobe_location, '      \
                        'c1.ReadStatus, '          \
                        'c1.___PercentRead, '      \
                        'c1.Attribution, '         \
                        'c1.DateLastRead, '        \
                        'c1.Title, '               \
                        'c1.MimeType, '            \
                        'NULL as rating, '         \
                        'c1.contentId '            \
                    'FROM content c1 LEFT OUTER JOIN content c2 ON c1.ChapterIDBookmarked = c2.ContentID ' \
                    'WHERE c1.ContentID = ?'

KEPUB_FETCH_QUERY = 'SELECT c1.ChapterIDBookmarked, ' \
                        'c1.adobe_location, '      \
                        'c1.ReadStatus, '          \
                        'c1.___PercentRead, '      \
                        'c1.Attribution, '         \
                        'c1.DateLastRead, '        \
                        'c1.Title, '               \
                        'c1.MimeType, '            \
                        'r.rating, '               \
                        'c1.contentId '            \
                    'FROM content c1 LEFT OUTER JOIN ratings r ON c1.ContentID = r.ContentID '  \
                    'WHERE c1.ContentID = ?'

KEPUB_FETCH_QUERY_NORATING = 'SELECT c1.ChapterIDBookmarked, ' \
                        'c1.adobe_location, '      \
                        'c1.ReadStatus, '          \
                        'c1.___PercentRead, '      \
                        'c1.Attribution, '         \
                        'c1.DateLastRead, '        \
                        'c1.Title, '               \
                        'c1.MimeType, '            \
                        'NULL as rating, '         \
                        'c1.contentId '            \
                    'FROM content c1 '             \
                    'WHERE c1.ContentID = ?'

KOBO_FIRMWARE_UPDATE_CHECK_URL      = "https://api.kobobooks.com/1.0/UpgradeCheck/Device/{0}/{1}/{2}/{3}"
KOBO_FIRMWARE_DL_LOCATION           = ".kobo"
KOBO_FIRMWARE_UPDATE_CHECK_INTERVAL = 86400

try:
    debug_print("KoboUtilites::action.py - loading translations")
    load_translations()
except NameError:
    debug_print("KoboUtilites::action.py - exception when loading translations")
    pass # load_translations() added in calibre 1.9

def store_current_bookmark(log, book_id, contentIDs, options):
    start_time = time.time()
    modifier = BookModifier(log)
    new_book_path = modifier.process_book(title, epub_path, calibre_opf_path,
                                          cover_path, options)
    if new_book_path:
        log('ePub updated in %.2f seconds'%(time.time() - start_time))
    else:
        log('ePub not changed after %.2f seconds'%(time.time() - start_time))
    return new_book_path

# Implementation of QtQHash for strings. This doesn't seem to be in the Python implemention. 
def qhash (inputstr):
    instr = ""
    if isinstance (inputstr, str):
        instr = inputstr 
    elif isinstance (inputstr, unicode):
        instr = inputstr.encode ("utf8")
    else:
        return -1

    h = 0x00000000
    for i in range (0, len (instr)):
        h = (h << 4) + ord(instr[i])
        h ^= (h & 0xf0000000) >> 23
        h &= 0x0fffffff

    return h


class KoboUtilitiesAction(InterfaceAction):

    name = 'KoboUtilities'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ( "KoboUtilities", None, _("Utilities to use with Kobo ereaders"), ())
    action_type = 'current'

    timestamp_string = None
    CONTENTTYPE = 6

    plugin_device_connection_changed = pyqtSignal(object);
    plugin_device_metadata_available = pyqtSignal();

    def genesis(self):
        base = self.interface_action_base_plugin
        self.version = base.name+" v%d.%d.%d"%base.version

        self.menu = QMenu(self.gui)
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)
        self.device_actions_map     = {}
        self.library_actions_map    = {}
        self.menu_actions           = []
        
        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.toolbar_button_clicked)
        self.menu.aboutToShow.connect(self.about_to_show_menu)
        self.menus_lock = threading.RLock()
        self.current_device_profile = None
        self.version_info           = None

    def initialization_complete(self):
        # otherwise configured hot keys won't work until the menu's
        # been displayed once.
        self.rebuild_menus()
        # Subscribe to device connection events
        device_signals.device_connection_changed.connect(self._on_device_connection_changed)
        device_signals.device_metadata_available.connect(self._on_device_metadata_available)

        self.connected_device_info = None

    def about_to_show_menu(self):
        self.rebuild_menus()


    def haveKobo(self):
        return self.device is not None and isinstance(self.device, KOBO)


    def haveKoboTouch(self):
        try:
            from calibre_plugins.kobotouchbeta_plugin import KOBOTOUCHBETA
            haveKoboTouch = isinstance(self.device, KOBOTOUCHBETA)
        except:
            haveKoboTouch = False
        haveKoboTouch = haveKoboTouch or isinstance(self.device, KOBOTOUCH)
        haveKoboTouch = haveKoboTouch and self.haveKobo() 
        return haveKoboTouch


    def library_changed(self, db):
        # We need to reset our menus after switching libraries
        self.device = self.get_device()

        self.rebuild_menus()
        if self.haveKobo() and self.current_device_profile:
            if self.current_device_profile[cfg.STORE_OPTIONS_STORE_NAME][cfg.KEY_STORE_ON_CONNECT]:
                debug_print('KoboUtilites:library_changed - About to do auto store')
                QTimer.singleShot(1000, self.auto_store_current_bookmark)


    def _on_device_connection_changed(self, is_connected):
        debug_print("KoboUtilities:_on_device_connection_changed - self.plugin_device_connection_changed.__class__: ", self.plugin_device_connection_changed.__class__)
        debug_print("Methods for self.plugin_device_connection_changed: ", dir(self.plugin_device_connection_changed))
        self.plugin_device_connection_changed.emit(is_connected)
        if not is_connected:
            debug_print('KoboUtilites:_on_device_connection_changed - Device disconnected')
            self.connected_device_info = None
            self.current_device_profile = None
            self.rebuild_menus()


    def _on_device_metadata_available(self):
        debug_print('KoboUtilites:_on_device_metadata_available - Start')
        self.plugin_device_metadata_available.emit()
        self.device = self.get_device()

        if self.current_device_profile:
            debug_print('KoboUtilites:_on_device_metadata_available - self.current_device_profile:', self.current_device_profile)
            debug_print('KoboUtilites:_on_device_metadata_available - self.current_backup_config:', self.current_backup_config)
            if self.haveKobo() and (self.current_backup_config[cfg.KEY_DO_DAILY_BACKUP]
                                    or self.current_backup_config[cfg.KEY_BACKUP_EACH_CONNECTION]):
                debug_print('KoboUtilites:_on_device_metadata_available - About to start auto backup')
                self.auto_backup_device_database()

            if self.haveKobo() and self.current_device_profile[cfg.STORE_OPTIONS_STORE_NAME][cfg.KEY_STORE_ON_CONNECT]:
                debug_print('KoboUtilites:_on_device_metadata_available - About to start auto store')
                self.auto_store_current_bookmark()

            if self.haveKoboTouch() and self.current_firmware_check_config[cfg.KEY_DO_UPDATE_CHECK]:
                debug_print('KoboUtilities:_on_device_metadata_available - About to do firmware check')
                self.auto_firmware_update_check()

        self.rebuild_menus()


    def rebuild_menus(self):
        with self.menus_lock:
            # Show the config dialog
            # The config dialog can also be shown from within
            # Preferences->Plugins, which is why the do_user_config
            # method is defined on the base plugin class
            do_user_config = self.interface_action_base_plugin.do_user_config
            self.menu.clear()
#            debug_print("rebuild_menus - self.menu_actions: ", self.menu_actions)
            for action in self.menu_actions:
                self.gui.keyboard.unregister_shortcut(action.calibre_shortcut_unique_name)
                # starting in calibre 2.10.0, actions are registers at
                # the top gui level for OSX' benefit.
                if calibre_version >= (2,10,0):
                    self.gui.removeAction(action)
            self.menu_actions = []

            self.device   = self.get_device()
            haveKobo      = self.haveKobo()
            haveKoboTouch = self.haveKoboTouch()
#            self.supports_series  = haveKoboTouch and "supports_series" in dir(self.device) and self.device.supports_series()
#            self.supports_ratings = haveKoboTouch and self.device.dbversion > 36
            debug_print("rebuild_menus - self.supports_ratings=%s" % self.supports_ratings)

            if haveKoboTouch:
                debug_print("rebuild_menus - have device.")
                self.set_reader_fonts_action = self.create_menu_item_ex(self.menu,  _("&Set Reader Font for Selected Books"), 
                                                              unique_name='Set Reader Font for Selected Books',
                                                              shortcut_name= _("Set Reader Font for Selected Books"),
                                                              triggered=self.set_reader_fonts,
                                                              enabled=haveKoboTouch, 
                                                              is_library_action=True, 
                                                              is_device_action=True)
                self.library_actions_map['Set Reader Font for Selected Books'] = self.set_reader_fonts_action
                self.device_actions_map['Set Reader Font for Selected Books']  = self.set_reader_fonts_action
    
                self.remove_reader_fonts_action = self.create_menu_item_ex(self.menu,  _("&Remove Reader Font for Selected Books"), 
                                                              unique_name='Remove Reader Font for Selected Books',
                                                              shortcut_name= _("Remove Reader Font for Selected Books"),
                                                              triggered=self.remove_reader_fonts,
                                                              enabled=haveKoboTouch, 
                                                              is_library_action=True, 
                                                              is_device_action=True)
                self.library_actions_map['Remove Reader Font for Selected Books'] = self.remove_reader_fonts_action
                self.device_actions_map['Remove Reader Font for Selected Books']  = self.remove_reader_fonts_action
    
                self.dismiss_tiles_action = self.create_menu_item_ex(self.menu,  _("&Dismiss tiles from new home screen"), 
                                                              unique_name='Dismiss tiles from new home screen',
                                                              shortcut_name= _("Dismiss tiles from new home screen"),
                                                              triggered=self.dismiss_tiles,
                                                              enabled=haveKoboTouch, 
                                                              is_library_action=True, 
                                                              is_device_action=True)
                self.library_actions_map['Dismiss tiles from new home screen'] = self.dismiss_tiles_action
                self.device_actions_map['Dismiss tiles from new home screen']  = self.dismiss_tiles_action
    
                self.menu.addSeparator()

            self.update_metadata_action = self.create_menu_item_ex(self.menu,  _("Update &metadata in device library"), 
                                                          unique_name='Update metadata in device library',
                                                          shortcut_name= _("Update metadata in device library"),
                                                          triggered=self.update_metadata,
                                                          enabled=not self.isDeviceView() and haveKobo, 
                                                          is_library_action=True)

            self.change_reading_status_action = self.create_menu_item_ex(self.menu,  _("&Change Reading Status in device library"), 
                                                          unique_name='Change Reading Status in device library',
                                                          shortcut_name= _("Change Reading Status in device library"),
                                                          triggered=self.change_reading_status,
                                                          enabled=self.isDeviceView() and haveKobo, 
                                                          is_device_action=True)

#            self.mark_not_interested_action = self.create_menu_item_ex(self.menu,  _("&Mark as "Not Interested""), 
#                                                          unique_name='Mark as "Not Interested"',
#                                                          shortcut_name= _("Mark as "Not Interested""),
#                                                          triggered=self.mark_not_interested,
#                                                          enabled=self.isDeviceView() and haveKoboTouch, 
#                                                          is_device_action=True)

            if self.supports_series:
                self.manage_series_on_device_action = self.create_menu_item_ex(self.menu,  _("&Manage Series Information in device library"), 
                                                              unique_name='Manage Series Information in device library',
                                                              shortcut_name= _("Manage Series Information in device library"),
                                                              triggered=self.manage_series_on_device,
                                                              enabled=self.isDeviceView() and haveKoboTouch and self.supports_series, 
                                                              is_device_action=True)

            self.handle_bookmarks_action = self.create_menu_item_ex(self.menu,  _("&Store/Restore current bookmark"), 
                                                          unique_name='Store/Restore current bookmark',
                                                          shortcut_name= _("Store/Restore current bookmark"),
                                                          triggered=self.handle_bookmarks,
                                                          enabled=not self.isDeviceView() and haveKobo, 
                                                          is_library_action=True)

            self.menu.addSeparator()
            self.upload_covers_action = self.create_menu_item_ex(self.menu,  _("&Upload covers for Selected Books"), 
                                                          unique_name='Upload/covers for Selected Books',
                                                          shortcut_name= _("Upload covers for Selected Books"),
                                                          triggered=self.upload_covers,
                                                          enabled=not self.isDeviceView() and haveKobo, 
                                                          is_library_action=True)
            if haveKoboTouch:
                self.remove_covers_action = self.create_menu_item_ex(self.menu,  _("&Remove covers for Selected Books"), 
                                                              unique_name='Remove covers for Selected Books',
                                                              shortcut_name= _("Remove covers for Selected Books"),
                                                              triggered=self.remove_covers,
                                                              enabled=haveKoboTouch, 
                                                              is_library_action=True, 
                                                              is_device_action=True)

            self.clean_images_dir_action = self.create_menu_item_ex(self.menu,  _("&Clean images directory of extra cover images"), 
                                                          unique_name='Clean images directory of extra cover images',
                                                          shortcut_name= _("Clean images directory of extra cover images"),
                                                          triggered=self.clean_images_dir,
                                                          enabled=haveKobo, 
                                                          is_library_action=True, 
                                                          is_device_action=True)
#            self.test_covers_action = self.create_menu_item_ex(self.menu, '&Test Covers - generate hash', 
#                                                          unique_name='Test Covers - generate hash',
#                                                          shortcut_name='Test Covers - generate hash',
#                                                          triggered=self.test_covers,
#                                                          enabled=haveKoboTouch, 
#                                                          is_library_action=True, 
#                                                          is_device_action=False)

            if haveKoboTouch:
                self.order_series_shelves_action = self.create_menu_item_ex(self.menu,  _("Order Series Shelves"),
                                                                unique_name='Order Series  Shelves',
                                                                shortcut_name= _("Order Series  Shelves"),
                                                                triggered=self.order_series_shelves,
                                                                enabled=haveKoboTouch and self.supports_series,
                                                                is_library_action=True,
                                                                is_device_action=True)
            if haveKoboTouch:
                self.get_shelves_from_device_action = self.create_menu_item_ex(self.menu,  _("Get Shelves From Device"),
                                                                unique_name='Get Shelves From Device',
                                                                shortcut_name= _("Get Shelves From Device"),
                                                                triggered=self.get_shelves_from_device,
                                                                enabled=haveKoboTouch,
                                                                is_library_action=True,
                                                                is_device_action=False)
            self.menu.addSeparator()
            self.getAnnotationForSelected_action = self.create_menu_item_ex(self.menu,  _("Copy annotation for Selected Book"), image='bookmarks.png',
                                                            unique_name='Copy annotation for Selected Book',
                                                            shortcut_name= _("Copy annotation for Selected Book"),
                                                            triggered=self.getAnnotationForSelected,
                                                            enabled=not self.isDeviceView() and haveKobo, 
                                                            is_library_action=True)
            self.backup_annotation_files_action = self.create_menu_item_ex(self.menu,  _("Backup Annotation File"),
                                                            unique_name='Backup Annotation File',
                                                            shortcut_name= _("Backup Annotation File"),
                                                            triggered=self.backup_annotation_files,
                                                            enabled=not self.isDeviceView() and haveKobo, 
                                                            is_library_action=True)

            self.menu.addSeparator()

            self.show_books_not_in_database_action = self.create_menu_item_ex(self.menu,  _("Show books not in the device database"),
                                                            unique_name='Show books not in the device database',
                                                            shortcut_name= _("Show books not in the device database"),
                                                            triggered=self.show_books_not_in_database,
                                                            enabled=self.isDeviceView() and haveKobo, 
                                                            is_device_action=True)

            self.refresh_device_books_action = self.create_menu_item_ex(self.menu,  _("Refresh the list of books on the device"),
                                                            unique_name='Refresh the list of books on the device',
                                                            shortcut_name= _("Refresh the list of books on the device"),
                                                            triggered=self.refresh_device_books,
                                                            enabled=haveKobo, 
                                                            is_library_action=True, 
                                                            is_device_action=True)
            self.databaseMenu = self.menu.addMenu(_("Database"))
            if haveKoboTouch:
                self.block_analytics_action = self.create_menu_item_ex(self.databaseMenu,  _("Block Analytics Events"),
                                                                unique_name='Block Analytics Events',
                                                                shortcut_name= _("Block Analytics Events"),
                                                                triggered=self.block_analytics,
                                                                enabled=haveKoboTouch,
                                                                is_library_action=True,
                                                                is_device_action=True)
                self.databaseMenu.addSeparator()
                self.fix_duplicate_shelves_action = self.create_menu_item_ex(self.databaseMenu,  _("Fix Duplicate Shelves"),
                                                                unique_name='Fix Duplicate Shelves',
                                                                shortcut_name= _("Fix Duplicate Shelves"),
                                                                triggered=self.fix_duplicate_shelves,
                                                                enabled=haveKoboTouch,
                                                                is_library_action=True,
                                                                is_device_action=True)
            self.check_device_database_action = self.create_menu_item_ex(self.databaseMenu,  _("Check the device database"),
                                                            unique_name='Check the device database',
                                                            shortcut_name= _("Check the device database"),
                                                            triggered=self.check_device_database,
                                                            enabled=haveKobo, 
                                                            is_library_action=True, 
                                                            is_device_action=True)
            self.vacuum_device_database_action = self.create_menu_item_ex(self.databaseMenu,  _("Compress the device database"),
                                                            unique_name='Compress the device database',
                                                            shortcut_name= _("Compress the device database"),
                                                            triggered=self.vacuum_device_database,
                                                            enabled=haveKobo, 
                                                            is_library_action=True, 
                                                            is_device_action=True)
            self.backup_device_database_action = self.create_menu_item_ex(self.databaseMenu,  _("Backup device database"),
                                                            unique_name='Backup device database',
                                                            shortcut_name= _("Backup device database"),
                                                            triggered=self.backup_device_database,
                                                            enabled=haveKobo, 
                                                            is_library_action=True, 
                                                            is_device_action=True)

#            self.menu.addSeparator()
#            self.get_list_action = self.create_menu_item_ex(self.menu, 'Update TOC for Selected Book',
#                                                            unique_name='Update TOC for Selected Book',
#                                                            shortcut_name='Update TOC for Selected Book',
#                                                            triggered=self.updateTOCForSelected)


            self.menu.addSeparator()
            self.firmware_update_action = self.create_menu_item_ex(self.menu, _('Check for Kobo Updates') + '...', #shortcut=False,
                                                                   unique_name='Check for Kobo Updates',
                                                                   shortcut_name=_('Check for Kobo Updates'),
                                                                   triggered=self.menu_firmware_update_check,
                                                                   enabled=haveKobo,
                                                                   is_library_action=True,
                                                                   is_device_action=True)

#            self.backup_device_database_action = self.create_menu_item_ex(self.menu, _('Do Auto Database Backup'), shortcut=False,
#                                                                   unique_name='Do Auto Database Backup',
#                                                                   shortcut_name=_('Do Auto Database Backup'),
#                                                                   triggered=self.menu_backup_device_database,
#                                                                   enabled=haveKobo,
#                                                                   is_library_action=True,
#                                                                   is_device_action=True)

            self.menu.addSeparator()
#            self.config_action = create_menu_action_unique(self, self.menu, _('&Customize plugin')+'...', shortcut=False,
#                                                           image= 'config.png',
#                                                           triggered=self.show_configuration)
            self.config_action = self.create_menu_item_ex(self.menu, _('&Customize plugin')+'...',# shortcut=False,
                                                            unique_name='Customize plugin',
                                                            shortcut_name= _("Customize plugin"),
                                                            image= 'config.png',
                                                            triggered=self.show_configuration,
                                                            enabled=True,  
                                                            is_library_action=True, 
                                                            is_device_action=True)
            
            self.about_action = create_menu_action_unique(self, self.menu,  _("&About Plugin"), #shortcut=False,
                                                           image= 'images/icon.png',
                                                           unique_name='About KoboUtilities',
                                                           shortcut_name= _("About KoboUtilities"),
                                                           triggered=self.about)

#            self.help_action = create_menu_action_unique(self, self.menu, _('&Help'), 'help.png',
#                                                         shortcut=False, 
#                                                         triggered=self.show_help1)

#            self.store_current_bookmark_action.setEnabled( not self.isDeviceView() and haveKoboTouch)
#            self.restore_current_bookmark_action.setEnabled( not self.isDeviceView() and haveKoboTouch)
            # self.get_list_action.setEnabled( len(self.gui.library_view.get_selected_ids()) > 0 )

            self.gui.keyboard.finalize()

    def about(self):
        # Get the about text from a file inside the plugin zip file
        # The get_resources function is a builtin function defined for all your
        # plugin code. It loads files from the plugin zip file. It returns
        # the bytes from the specified file.
        #
        # Note that if you are loading more than one file, for performance, you
        # should pass a list of names to get_resources. In this case,
        # get_resources will return a dictionary mapping names to bytes. Names that
        # are not found in the zip file will not be in the returned dictionary.
        
        self.about_text = get_resources('about.txt')
        AboutDialog(self.gui, self.qaction.icon(), self.version + self.about_text).exec_()
        
    def create_menu_item_ex(self, parent_menu, menu_text, image=None, tooltip=None,
                           shortcut=None, triggered=None, is_checked=None, shortcut_name=None,
                           unique_name=None, enabled=False, is_library_action=False, is_device_action=False):
        ac = create_menu_action_unique(self, parent_menu, menu_text, image, tooltip,
                                       shortcut, triggered, is_checked, shortcut_name, unique_name)

        ac.setEnabled(enabled)

        if is_library_action:
            self.library_actions_map[shortcut_name] = ac
        if is_device_action:
            self.device_actions_map[shortcut_name] = ac
#        debug_print("create_menu_item_ex - adding action to menu_actions: ", ac.calibre_shortcut_unique_name)
        self.menu_actions.append(ac)

        return ac

    def toolbar_button_clicked(self):
        self.rebuild_menus()

        if not self.haveKobo():
            self.show_configuration()
        elif len(self.gui.current_view().selectionModel().selectedRows()) >= 0:
            self.device     = self.get_device()
#            haveKoboTouch   = self.device is not None
#            self.supports_series  = haveKoboTouch and "supports_series" in dir(self.device) and self.device.supports_series()
#            self.supports_ratings = haveKoboTouch and self.device.dbversion > 36

            if self.isDeviceView():
                if self.supports_series:
                    button_action = cfg.get_plugin_pref(cfg.COMMON_OPTIONS_STORE_NAME, cfg.KEY_BUTTON_ACTION_DEVICE)
                    if button_action == '':
                        self.show_configuration()
                    else:
                        self.device_actions_map[button_action].trigger()
#                    self.manage_series_on_device()
#                    self.show_books_not_in_database()
#                    self.mark_not_interested()
                else:
                    self.change_reading_status()
            else:
                button_action = cfg.get_plugin_pref(cfg.COMMON_OPTIONS_STORE_NAME, cfg.KEY_BUTTON_ACTION_LIBRARY)
                if button_action == '':
                    debug_print("toolbar_button_clicked - no button action")
                    self.show_configuration()
                else:
                    try:
                        self.library_actions_map[button_action].trigger()
                    except:
                        debug_print("toolbar_button_clicked - exception running button action")
                        self.show_configuration()
#                self.library_actions_map.values()[0].trigger()
#                self.handle_bookmarks()
#                self.upload_covers()
#                self.update_metadata()
#                self.set_reader_fonts_action.trigger()
#                self.backup_annotation_files()
#                self.show_configuration()

    def isDeviceView(self):
        view = self.gui.current_view()
        return isinstance(view, DeviceBooksView)

    def _get_contentIDs_for_selected(self):
        view = self.gui.current_view()
        if self.isDeviceView():
            rows = view.selectionModel().selectedRows()
            books = [view.model().db[view.model().map[r.row()]] for r in rows]
            contentIDs = [book.contentID for book in books]
#            debug_print("_get_contentIDs_for_selected - book.ImageID=", book.ImageID)
        else:
            book_ids = view.get_selected_ids()
            contentIDs = self.get_contentIDs_for_books(book_ids)
            
        return contentIDs

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def set_reader_fonts(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot set reader font settings."),
                                 _("No device connected."),
                                show=True)
        self.device_path = self.get_device_path()
        self.singleSelected = len(self.gui.current_view().selectionModel().selectedRows()) == 1

        contentIDs = self._get_contentIDs_for_selected()

        debug_print('set_reader_fonts - contentIDs', contentIDs)

        #print("update books:%s"%books)

        if len(contentIDs) == 0:
            return

        if len(contentIDs) == 1:
            self.single_contentID = contentIDs[0]

        dlg = ReaderOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.prefs

        if self.options[cfg.KEY_UPDATE_CONFIG_FILE]:
            self._update_config_reader_settings(self.options)

        updated_fonts, added_fonts, deleted_fonts, count_books = self._set_reader_fonts(contentIDs)
#        result_message =  _("Change summary:\n\tFont settings updated=%d\n\tFont settings added=%d\n\tTotal books=%d") % (updated_fonts, added_fonts, count_books)
        result_message =  _("Change summary:") + "\n\t" + _("Font settings updated={0}\n\tFont settings added={1}\n\tTotal books={2}").format(updated_fonts, added_fonts, count_books)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Device library updated"),
                    result_message,
                    show=True)


    def remove_reader_fonts(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot remove reader font settings"),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        contentIDs = self._get_contentIDs_for_selected()
        
        if len(contentIDs) == 0:
            return

        mb = question_dialog(self.gui,  _("Remove Reader settings"),  _("Do you want to remove the reader settings for the selected books?"), show_copy_button=False)
        if not mb:
            return

        
        updated_fonts, added_fonts, deleted_fonts, count_books = self._set_reader_fonts(contentIDs, delete=True)
        result_message = _("Change summary:") + "\n\t" + _("Font settings deleted={0}").format(deleted_fonts)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Device library updated"),
                    result_message,
                    show=True)

    def update_metadata(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot update metadata in device library."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return
        debug_print("update_metadata - selectedIDs:", selectedIDs)
        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
        for book in books:
            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            debug_print("update_metadata - device_book_paths:", device_book_paths)
            book.paths = device_book_paths
            book.contentIDs = [self.contentid_from_path(path, self.CONTENTTYPE) for path in device_book_paths]
            book.series_index_string = None
        
        dlg = UpdateMetadataOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.new_prefs

        updated_books, unchanged_books, not_on_device_books, count_books = self._update_metadata(books)
        result_message = _("Update summary:") + "\n\t" + _("Books updated={0}\n\tUnchanged books={1}\n\tBooks not on device={2}\n\tTotal books={3}").format(updated_books, unchanged_books, not_on_device_books, count_books)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Device library updated"),
                    result_message,
                    show=True)


    def dismiss_tiles(self):
        #debug_print("dismiss_tiles - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot mark tiles to be dismissed."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        
        dlg = DismissTilesOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options
        debug_print("dismiss_tiles - self.options=", self.options)
        result = self._dismiss_tiles()
        result_message = _("Update successful") if result == 1 else _("Update unsuccessful")
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Dismiss Tiles from Home Screen"),
                    result_message,
                    show=True)


    def handle_bookmarks(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot store or restore current reading position."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return

        dlg = BookmarkOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options

        if self.options['storeBookmarks']:
            self.store_current_bookmark()
        else:
            self.restore_current_bookmark()


    def auto_store_current_bookmark(self):
        debug_print("auto_store_current_bookmark - start")
        self.device = self.get_device()
        self.device_path = self.get_device_path()

        db = self.gui.current_db

        self.options = {}
        self.options[cfg.KEY_STORE_BOOKMARK]    = True
        self.options[cfg.KEY_READING_STATUS]    = False
        self.options[cfg.KEY_DATE_TO_NOW]       = False
        self.options[cfg.KEY_SET_RATING]        = False
        self.options[cfg.KEY_CLEAR_IF_UNREAD]   = False
        self.options[cfg.KEY_BACKGROUND_JOB]    = True
        self.options[cfg.KEY_PROMPT_TO_STORE]          = self.current_device_profile[cfg.STORE_OPTIONS_STORE_NAME][cfg.KEY_PROMPT_TO_STORE]
        self.options[cfg.KEY_STORE_IF_MORE_RECENT]     = self.current_device_profile[cfg.STORE_OPTIONS_STORE_NAME][cfg.KEY_STORE_IF_MORE_RECENT]
        self.options[cfg.KEY_DO_NOT_STORE_IF_REOPENED] = self.current_device_profile[cfg.STORE_OPTIONS_STORE_NAME][cfg.KEY_DO_NOT_STORE_IF_REOPENED]

        self.options["device_database_path"] = self.device_database_path()
        self.options["job_function"]         = 'store_current_bookmark'
        self.options["supports_ratings"]     = self.supports_ratings
        self.options['allOnDevice']          = True

        QueueProgressDialog(self.gui, [], None, self.options, self._store_queue_job, db, plugin_action=self)


    def should_check_kobo_updates(self):
        last_check = self.current_firmware_check_config[cfg.KEY_LAST_FIRMWARE_CHECK_TIME]
        now = calendar.timegm(time.gmtime())
        debug_print("Delta since last update check: %s" % (now - last_check))
        return KOBO_FIRMWARE_UPDATE_CHECK_INTERVAL < (now - last_check)


    def menu_firmware_update_check(self):
        debug_print('menu_firmware_update_check - start')
        if not self.should_check_kobo_updates() and not question_dialog(self.gui, _("Kobo Firmware Update"), _('You last checked for a Kobo update less than a day ago. Do you want to check again now?'), show_copy_button=False):
            return
        else:
            options = self.current_firmware_check_config
            options[cfg.KEY_LAST_FIRMWARE_CHECK_TIME] = 0
            cfg.plugin_prefs[cfg.UPDATE_OPTIONS_STORE_NAME] = options
        self.auto_firmware_update_check(from_menu=True)


    def auto_firmware_update_check(self, from_menu=False):
        debug_print('auto_firmware_update_check - start')

        if self.should_check_kobo_updates():
            options = cfg.get_plugin_prefs(cfg.UPDATE_OPTIONS_STORE_NAME)
            options[cfg.KEY_LAST_FIRMWARE_CHECK_TIME] = calendar.timegm(time.gmtime())
            cfg.plugin_prefs[cfg.UPDATE_OPTIONS_STORE_NAME] = options
            self.device_path = self.get_device_path()

            kobo_update_file = os.path.join(self.device_path, KOBO_FIRMWARE_DL_LOCATION, "KoboRoot.tgz")
            kobo_manifest_sums_file = os.path.join(self.device_path, KOBO_FIRMWARE_DL_LOCATION, "manifest.md5sum")
            kobo_upgrade_dir = os.path.join(self.device_path, KOBO_FIRMWARE_DL_LOCATION, "upgrade")
            if os.path.exists(kobo_update_file):
                if not question_dialog(self.gui, _("Kobo Firmware Update"), _("The KoboRoot.tgz file is already in place for an upgrade. Do you want to check for updates anyway?"), show_copy_button=False):
                    return
                else:
                    os.unlink(kobo_update_file)
            if os.path.exists(kobo_manifest_sums_file):
                os.unlink(kobo_manifest_sums_file)
            if os.path.exists(kobo_upgrade_dir):
                shutil.rmtree(kobo_upgrade_dir)

            from ConfigParser import NoOptionError
            kobo_config, config_file_path = self.get_config_file()
            try:
                is_beta_user = kobo_config.has_section("FeatureSettings") and kobo_config.getboolean("FeatureSettings", "AcceptPreviewUpgrades")
            except ValueError:
                is_beta_user = False
            except NoOptionError:
                is_beta_user = False
            if is_beta_user:
                beta_continue_anyway = question_dialog(self.gui, _("Kobo Firmware Update"), _("You appear to be a beta tester. Upgrade checking currently only works for official firmware releases. Do you wish to continue checking for official release firmware updates?"), show_copy_button=False)
                if not beta_continue_anyway:
                    debug_print("do_check_firmware_update - beta tester is not checking for firmware")
                    return
            version_info = self.device_version_info()
            if version_info:
                # Check affiliate.conf for the affiliate unless the early update check box is selected
                affiliate = "whsmith"
                affiliate_file = os.path.join(self.device_path, ".kobo", "affiliate.conf")
                if not self.current_firmware_check_config[cfg.KEY_DO_EARLY_FIRMWARE_CHECK] and os.path.isfile(affiliate_file):
                    affiliate_config = ConfigParser.SafeConfigParser(allow_no_value=True)
                    affiliate_config.optionxform = str
                    affiliate_config.read(affiliate_file)
                    if affiliate_config.has_section("General") and affiliate_config.has_option("General", "affiliate"):
                        affiliate = affiliate_config.get("General", "affiliate")
                serial_no = self.device_serial_no()
                update_url = KOBO_FIRMWARE_UPDATE_CHECK_URL.format(version_info[5], affiliate, version_info[2], serial_no)
                debug_print("auto_firmware_update_check - update_url:%s" % update_url)
                update_data = None
                import urllib2
                resp = urllib2.urlopen(update_url)
                if resp.getcode() == 200:
                    import json
                    update_data = json.loads(resp.read())
                    debug_print("do_check_firmware_update - update_data:\n%s" % unicode(update_data))
                    if update_data["UpgradeURL"] is not None:
                        import re
                        m = re.search(r'\/kobo\-update\-(?P<version>\d\.\d\.\d)\.zip$', update_data["UpgradeURL"])
                        if m:
                            upgrade_version = m.group("version")
                        else:
                            upgrade_version = '(Unknown)'
                        upgrade_continue = question_dialog(self.gui, _("Kobo Firmware Update"), _("A Kobo firmware update to version {0} is available. Do you want to update? You have version {1}.".format(upgrade_version, version_info[2])), show_copy_button=False)
                        if upgrade_continue:
                            self._firmware_update(update_data)
                    else:
                        debug_print('auto_firmware_update_check - No firmware upgrade available')
                        if from_menu:
                            info_dialog(self.gui, _("Kobo Firmware Update"), _("Kobo firmware update check complete - no updates available"), show=True, show_copy_button=False)
                else:
                    raise ValueError("Couldn't check for firmware update: got HTTP%s" % resp.getcode())
        else:
            debug_print("auto_firmware_update_check - Not checking for firmware, only checking once per day")

    def device_version_info(self):
        if not self.version_info:
            version_file = os.path.join(self.device_path, ".kobo", "version")
            if os.path.isfile(version_file):
                vf = open(version_file, "r")
                self.version_info = vf.read().strip().split(",")
                vf.close()
        return self.version_info

    def device_serial_no(self):
        return self.device_version_info()[0]


    def menu_backup_device_database(self):
        self.auto_backup_device_database(from_menu=True)


    def auto_backup_device_database(self, from_menu=False):
        debug_print('auto_backup_device_database - start')
        if not self.current_backup_config:
            debug_print('auto_backup_device_database - no backup configuration')
            return
        self.device_path = self.get_device_path()

        dest_dir = self.current_backup_config[cfg.KEY_BACKUP_DEST_DIRECTORY]
        debug_print('auto_backup_device_database - destination directory=', dest_dir)
        if not dest_dir or len(dest_dir) == 0:
            debug_print('auto_backup_device_database - destination directory not set, not doing backup')
            return

        # Backup file names will be KoboReader-devicename-serialnumber-timestamp.sqlite
        backup_file_template = 'KoboReader-{0}-{1}-{2}'
        version_info = self.device_version_info()
        debug_print('auto_backup_device_database - version_info=', version_info)
        serial_number = version_info[0].translate(None, ':')
        device_name = "".join(self.device.gui_name.split())
        debug_print('auto_backup_device_database - device_information=', self.device.get_device_information())
        debug_print('auto_backup_device_database - device_name=', device_name)
        debug_print('auto_backup_device_database - backup_file_template=', backup_file_template.format(device_name, serial_number, ''))

        backup_options = {}
        backup_options[cfg.KEY_BACKUP_DEST_DIRECTORY]  = dest_dir
        backup_options[cfg.KEY_BACKUP_COPIES_TO_KEEP]  = self.current_backup_config[cfg.KEY_BACKUP_COPIES_TO_KEEP]
        backup_options[cfg.KEY_DO_DAILY_BACKUP]        = self.current_backup_config[cfg.KEY_DO_DAILY_BACKUP]
        backup_options[cfg.KEY_BACKUP_EACH_CONNECTION] = self.current_backup_config[cfg.KEY_BACKUP_EACH_CONNECTION]
        backup_options['device_name']                 = device_name
        backup_options['serial_number']               = serial_number
        backup_options['backup_file_template']        = backup_file_template
        backup_options['database_file']               = self.device_database_path()
        backup_options["device_path"]                   = self.device._main_prefix
        debug_print('auto_backup_device_database - backup_options=', backup_options)

        self._device_database_backup(backup_options)
        debug_print('auto_backup_device_database - end')


    def store_current_bookmark(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot update metadata in device library."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        self.options["device_database_path"]       = self.device_database_path()
        self.options["job_function"]               = 'store_current_bookmark'
        self.options["supports_ratings"]           = self.supports_ratings
        self.options['allOnDevice']                = False
        self.options[cfg.KEY_PROMPT_TO_STORE]      = True
        debug_print("store_current_bookmark - self.options:", self.options)

        if self.options[cfg.KEY_BACKGROUND_JOB]:
            QueueProgressDialog(self.gui, [], None, self.options, self._store_queue_job, self.gui.current_view().model().db, plugin_action=self)
        else:
            selectedIDs = self._get_selected_ids()
        
            if len(selectedIDs) == 0:
                return
            debug_print("store_current_bookmark - selectedIDs:", selectedIDs)
            books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
            for book in books:
                device_book_paths = self.get_device_paths_from_id(book.calibre_id)
    #            debug_print("store_current_bookmark - device_book_paths:", device_book_paths)
                book.paths = device_book_paths
                book.contentIDs = [self.contentid_from_path(path, self.CONTENTTYPE) for path in device_book_paths]

            books_with_bookmark, books_without_bookmark, count_books = self._store_current_bookmark(books)
            result_message = _("Update summary:") + "\n\t" + _("Bookmarks retrieved={0}\n\tBooks with no bookmarks={1}\n\tTotal books={2}").format(books_with_bookmark, books_without_bookmark, count_books)
            info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Library updated"),
                        result_message,
                        show=True)

    def restore_current_bookmark(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot set bookmark in device library."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return
        debug_print("restore_current_bookmark - selectedIDs:", selectedIDs)
        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
        for book in books:
            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            debug_print("store_current_bookmark - device_book_paths:", device_book_paths)
            book.paths = device_book_paths
            book.contentIDs = [self.contentid_from_path(path, self.CONTENTTYPE) for path in device_book_paths]
        
        updated_books, not_on_device_books, count_books = self._restore_current_bookmark(books)
        result_message = _("Update summary:") + "\n\t" + _("Books updated={0}\n\tBooks not on device={1}\n\tTotal books={2}").format(updated_books, not_on_device_books, count_books)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Device library updated"),
                    result_message,
                    show=True)

    def backup_device_database(self):
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot backup the device database."),
                                 _("No device connected."),
                                show=True)
        self.device_path = self.get_device_path()

        debug_print("backup_device_database")

        fd = FileDialog(parent=self.gui, name='Kobo Utilities plugin:choose backup destination', 
                        title= _("Choose Backup Destination"),
                        filters=[( _("SQLite database"), ['sqlite'])], 
                        add_all_files_filter=False,
                        mode=QFileDialog.AnyFile
                        )
        if not fd.accepted:
                return
        backup_file = fd.get_files()[0]

        if not backup_file:
            return

        debug_print("backup_device_database - backup file selected=", backup_file)
        source_file = self.device.normalize_path(self.device_path +'.kobo/KoboReader.sqlite')
        shutil.copyfile(source_file, backup_file)

    def backup_annotation_files(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return

        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot backup annotation files from device."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return

        dlg = BackupAnnotationsOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return

        dest_path = dlg.dest_path()
        debug_print("backup_annotation_files - selectedIDs:", selectedIDs)
        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
        for book in books:
            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            debug_print("backup_annotation_files - device_book_paths:", device_book_paths)
            book.paths = device_book_paths
            book.contentIDs = [self.contentid_from_path(path, self.CONTENTTYPE) for path in device_book_paths]

        debug_print("backup_annotation_files - dest_path=", dest_path)
        annotations_found, no_annotations, kepubs, count_books = self._backup_annotation_files(books, dest_path)
        result_message = _("Annotations backup summary:\n\tBooks with annotations={0}\n\tBooks without annotations={1}\n\tKobo epubs={2}\n\tTotal books={3}").format(annotations_found, no_annotations, kepubs, count_books)
        info_dialog(self.gui,  _("Kobo Utilities") + _(" - Annotations backup"),
                    result_message,
                    show=True)

    def refresh_device_books(self):
        self.gui.device_detected(True, KOBOTOUCH)

    def change_reading_status(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot change reading status in device library."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        books = self._get_books_for_selected()
        
        if len(books) == 0:
            return
        for book in books:
#            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            debug_print("change_reading_status - book:", book)
            book.contentIDs = [book.contentID]
        debug_print("change_reading_status - books:", books)

        dlg = ChangeReadingStatusOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options
        self.options[cfg.KEY_USE_PLUGBOARD]        = False
        self.options[cfg.KEY_USE_TITLE_SORT]       = False
        self.options[cfg.KEY_USE_AUTHOR_SORT]      = False
        self.options[cfg.KEY_SET_TAGS_IN_SUBTITLE] = False

        updated_books, unchanged_books, not_on_device_books, count_books = self._update_metadata(books)
        result_message = _("Update summary:") + "\n\t" + _("Books updated={0}\n\tUnchanged books={1}\n\tBooks not on device={2}\n\tTotal books={3}").format(updated_books, unchanged_books, not_on_device_books, count_books)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Device library updated"),
                    result_message,
                    show=True)


    def mark_not_interested(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot change reading status in device library."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        books = self._get_books_for_selected()
        
        if len(books) == 0:
            return
        recommendations = []
        for book in books:
#            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            if 'Recommendation' in book.device_collections:
                debug_print("mark_not_interested - book:", book)
                book.contentIDs = [book.contentID]
                recommendations.append(book)
                debug_print("mark_not_interested - book.device_collections:", book.device_collections)
        debug_print("mark_not_interested - recommendations:", recommendations)
        self.options = self.default_options()
        self.options['mark_not_interested'] = True

        updated_books, unchanged_books, not_on_device_books, count_books = self._update_metadata(recommendations)
        result_message = _("Books marked as Not Interested:\n\tBooks updated={0}\n\tUnchanged books={1}\n\tTotal books={2}").format(updated_books, unchanged_books, count_books)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Device library updated"),
                    result_message,
                    show=True)


    def show_books_not_in_database(self):

        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot list books not in device library."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        books = self._get_books_for_selected()
        
        if len(books) == 0:
            books = self.gui.current_view().model().db

        books_not_in_database = self._check_book_in_database(books)
#        for book in books:
#            debug_print("show_books_not_in_database - book.title='%s'" % book.title)
#            if not book.contentID:
#                books_not_in_database.append(book)
#            else:
#                debug_print("show_books_not_in_database - book.contentID='%s'" % book.contentID)

        dlg = ShowBooksNotInDeviceDatabaseDialog(self.gui, books_not_in_database)
        dlg.show()


    def fix_duplicate_shelves(self):

        #debug_print("fix_duplicate_shelves - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot fix the duplicate shelves in the device library."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        shelves = self._get_shelf_count()
        dlg = FixDuplicateShelvesDialog(self.gui, self, shelves)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            debug_print("fix_duplicate_shelves - dialog cancelled")
            return
        self.options = dlg.options
        debug_print("fix_duplicate_shelves - about to fix shelves - options=%s" % self.options)

        starting_shelves, shelves_removed, finished_shelves = self._remove_duplicate_shelves(shelves, self.options)
        result_message = _("Update summary:") + "\n\t" + _("Starting number of shelves={0}\n\tShelves removed={1}\n\tTotal shelves={2}").format(starting_shelves, shelves_removed, finished_shelves)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Duplicate Shelves Fixed"),
                    result_message,
                    show=True)


    def order_series_shelves(self):

        #debug_print("order_series_shelves - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot order the series shelves in the device library."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        shelves = []
        dlg = OrderSeriesShelvesDialog(self.gui, self, shelves)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            debug_print("order_series_shelves - dialog cancelled")
            return
        self.options = dlg.options
        shelves      = dlg.get_shelves()
        debug_print("order_series_shelves - about to order shelves - options=%s" % self.options)
        debug_print("order_series_shelves - shelves=", shelves)

        starting_shelves, shelves_ordered = self._order_series_shelves(shelves, self.options)
        result_message = _("Update summary:") + "\n\t" + _("Starting number of shelves={0}\n\tShelves reordered={1}").format(starting_shelves, shelves_ordered)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Order Series Shelves"),
                    result_message,
                    show=True)


    def get_shelves_from_device(self):
        debug_print("get_shelves_from_device - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot get the shelves from device."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        dlg = GetShelvesFromDeviceDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            debug_print("get_shelves_from_device - dialog cancelled")
            return
        self.options = dlg.options

        # Check if driver is configured to manage shelves. If so, warn if selected column is one of
        # the configured columns.
        driver_shelves = self.device.get_collections_attributes()
        debug_print("get_shelves_from_device - driver_shelves=", driver_shelves)
        debug_print("get_shelves_from_device - selected column=", self.options[cfg.KEY_SHELVES_CUSTOM_COLUMN])
        if self.options[cfg.KEY_SHELVES_CUSTOM_COLUMN] in driver_shelves:
            debug_print("get_shelves_from_device - selected column is one of the columns used in the driver configuration!")
            details_msg = _(
                           "The selected column is {0}."
                           "\n"
                           "The driver shelf management columns are: {1}"
                           ).format(self.options[cfg.KEY_SHELVES_CUSTOM_COLUMN], ', '.join(driver_shelves))
            mb = question_dialog(self.gui,  
                                 _("Getting shelves from device"),  
                                 _("The column selected is one of the columns used in the driver configuration for shelf management. "
                                   "Updating this column might affect the shelf management the next time you connect the device. "
                                   "\n\nAre you sure you want to do this?"),
                                 override_icon=QIcon(I('dialog_warning.png')),
                                 show_copy_button=False, det_msg=details_msg)
            if not mb:
                debug_print("get_shelves_from_device - User cancelled because of column used.")
                return

#        self.pb = ProgressBar(parent=self.gui, window_title=_("Getting shelves from device"), on_top=True)
        self.pb = ProgressBar(parent=self.gui, window_title=_("Getting shelves from device"), on_top=False)
        self.pb.show()
        self.pb.set_label(_("Getting list of books"))

        library_db = self.gui.current_view().model().db
        if self.options[cfg.KEY_ALL_BOOKS]:
            selectedIDs = set(library_db.search_getting_ids('ondevice:True', None, sort_results=False, use_virtual_library=False))
        else:
            selectedIDs = self._get_selected_ids()

        if len(selectedIDs) == 0:
            return
        debug_print("get_shelves_from_device - selectedIDs:", selectedIDs)
        books = self._convert_calibre_ids_to_books(library_db, selectedIDs)
        self.pb.set_label(_("Number of books to get shelves for {0}").format(len(books)))
        for book in books:
            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            debug_print("get_shelves_from_device - device_book_paths:", device_book_paths)
            book.paths = device_book_paths
            book.contentIDs = [self.contentid_from_path(path, self.CONTENTTYPE) for path in device_book_paths]

        debug_print("get_shelves_from_device - about get shelves - options=%s" % self.options)

        books_with_shelves, books_without_shelves, count_books = self._get_shelves_from_device(books, self.options)
        result_message = _("Update summary:") + "\n\t" + _("Books processed={0}\n\tBooks with Shelves={1}\n\tBooks without Shelves={2}").format(count_books, books_with_shelves, books_without_shelves)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Get Shelves from Device"),
                    result_message,
                    show=True)


    def check_device_database(self):
        #debug_print("check_device_database - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot check Kobo device database."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        check_result = self._check_device_database()

        check_result = _("Result of running 'PRAGMA integrity_check' on database on the Kobo device:\n\n") + check_result

        d = ViewLog("Kobo Utilities - Device Database Check", check_result, parent=self.gui)
        d.setWindowIcon(self.qaction.icon())
        d.exec_()


    def block_analytics(self):
        debug_print("block_analytics - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot block analytics events."), 
                 _("No device connected."), show=True)
        self.device_path = self.get_device_path()

        debug_print("block_analytics")

        dlg = BlockAnalyticsOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options

        block_analytics_result = self._block_analytics()
        if block_analytics_result:
            info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Block Analytics Events"),
                    block_analytics_result, show=True)
        else:
            result_message = _("Failed to block analytics events.")
            d = ViewLog( _("Kobo Utilities") + " - " + _("Block Analytics Events"),
                    result_message, parent=self.gui)
            d.setWindowIcon(self.qaction.icon())
            d.exec_()

    
    def vacuum_device_database(self):
        debug_print("vacuum_device_database - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot compress Kobo device database."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()

        uncompressed_db_size = os.path.getsize(self.device_database_path())
        vacuum_result = self._vacuum_device_database()

        if vacuum_result == '':
            compressed_db_size = os.path.getsize(self.device_database_path())
            result_message = _("The database on the device has been compressed.\n\tOriginal size = {0}MB\n\tCompressed size = {1}MB").format("%.3f"%(uncompressed_db_size / 1024 / 1024), "%.3f"%(compressed_db_size / 1024 / 1024))
            info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Compress Device Database"),
                    result_message,
                    show=True)

        else:
            vacuum_result = _("Result of running 'vacuum' on database on the Kobo device:\n\n") + vacuum_result

            d = ViewLog("Kobo Utilities - Compress Device Database", vacuum_result, parent=self.gui)
            d.setWindowIcon(self.qaction.icon())
            d.exec_()


    def default_options(self):
        options = cfg.METADATA_OPTIONS_DEFAULTS
        return options

    def manage_series_on_device(self):
        def digits(f):
            return len(str(f).split('.')[1].rstrip('0'))

        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return

        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot manage series in device library."),
                     _("No device connected."),
                    show=True)
        series_columns = self.get_series_columns()
        self.device_path = self.get_device_path()

        books = self._get_books_for_selected()
        debug_print("manage_series_on_device - books[0].__class__=", books[0].__class__)

        
        if len(books) == 0:
            return
        seriesBooks = [SeriesBook(book, series_columns) for book in books]
        seriesBooks = sorted(seriesBooks, key=lambda k: k.sort_key(sort_by_name=True))
        debug_print("manage_series_on_device - seriesBooks[0]._mi.__class__=", seriesBooks[0]._mi.__class__)
        debug_print("manage_series_on_device - seriesBooks[0]._mi.kobo_series=", seriesBooks[0]._mi.kobo_series)
        debug_print("manage_series_on_device - seriesBooks[0]._mi.kobo_series_number=", seriesBooks[0]._mi.kobo_series_number)
        debug_print("manage_series_on_device - books:", seriesBooks)

        library_db = self.gui.library_view.model().db
        all_series = library_db.all_series()
        all_series.sort(key=lambda x : sort_key(x[1]))

        d = ManageSeriesDeviceDialog(self.gui, self, seriesBooks, all_series, series_columns)
        d.exec_()
        if d.result() != d.Accepted:
            return
        
        debug_print("manage_series_on_device - done series management - books:", seriesBooks)

        self.options = self.default_options()
        books = []
        for seriesBook in seriesBooks:
            debug_print("manage_series_on_device - seriesBook._mi.contentID=", seriesBook._mi.contentID)
            if seriesBook.is_title_changed() or seriesBook.is_pubdate_changed() or seriesBook.is_series_changed():
                book = seriesBook._mi
                book.series_index_string = seriesBook.series_index_string()
                book.kobo_series_number  = seriesBook.series_index_string()
                book.kobo_series         = seriesBook.series_name()
                book._new_book           = True
                book.contentIDs          = [book.contentID]
                books.append(book)
                self.options['title']          = self.options['title'] or seriesBook.is_title_changed()
                self.options['series']         = self.options['series'] or seriesBook.is_series_changed()
                self.options['published_date'] = self.options['published_date'] or seriesBook.is_pubdate_changed()
                debug_print("manage_series_on_device - seriesBook._mi.__class__=", seriesBook._mi.__class__)
                debug_print("manage_series_on_device - seriesBook.is_pubdate_changed()=%s"%seriesBook.is_pubdate_changed())
                debug_print("manage_series_on_device - book.kobo_series=", book.kobo_series)
                debug_print("manage_series_on_device - book.kobo_series_number=", book.kobo_series_number)
                debug_print("manage_series_on_device - book.series=", book.series)
                debug_print("manage_series_on_device - book.series_index=%s"%unicode(book.series_index))


        if self.options['title'] or self.options['series'] or self.options['published_date']:
            updated_books, unchanged_books, not_on_device_books, count_books = self._update_metadata(books)
            
            debug_print("manage_series_on_device - about to call sync_booklists")
    #        self.device.sync_booklists((self.gui.current_view().model().db, None, None))
            USBMS.sync_booklists(self.device, (self.gui.current_view().model().db, None, None))
            result_message = _("Update summary:") + "\n\t" + _("Books updated={0}\n\tUnchanged books={1}\n\tBooks not on device={2}\n\tTotal books={3}").format(updated_books, unchanged_books, not_on_device_books, count_books)
        else:
            result_message = _("No changes made to series information.")
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Manage Series On Device"),
                    result_message,
                    show=True)


    def get_series_columns(self):
        custom_columns = self.gui.library_view.model().custom_columns
        series_columns = OrderedDict()
        for key, column in custom_columns.iteritems():
            typ = column['datatype']
            if typ == 'series':
                series_columns[key] = column
        return series_columns

    def get_selected_books(self, rows, series_columns):
        def digits(f):
            return len(str(f).split('.')[1].rstrip('0'))

        db = self.gui.library_view.model().db
        idxs = [row.row() for row in rows]
        books = []
        for idx in idxs:
            mi = db.get_metadata(idx)
            book = SeriesBook(mi, series_columns)
            books.append(book)
        # Sort books by the current series
        books = sorted(books, key=lambda k: k.sort_key())
        return books


    def upload_covers(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,
                                 _("Cannot upload covers."),
                                 _("No device connected."),
                                show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return
        debug_print("upload_covers - selectedIDs:", selectedIDs)
        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
        
        dlg = CoverUploadOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options
        
        total_books, uploaded_covers, not_on_device_books = self._upload_covers(books)
        result_message = _("Change summary:") + "\n\t" + _("Covers uploaded={0}\n\tBooks not on device={1}\n\tTotal books={2}").format(uploaded_covers, not_on_device_books, total_books)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Covers uploaded"),
                    result_message,
                    show=True)

    def remove_covers(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("remove_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot remove covers."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()
        debug_print("remove_covers - self.device_path", self.device_path)

#        contentIDs = self._get_contentIDs_for_selected()
        if self.gui.stack.currentIndex() == 0:
            selectedIDs = self._get_selected_ids()
            books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)

        else:
            books = self._get_books_for_selected()

        
        if len(books) == 0:
            return

        dlg = RemoveCoverOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options
        
        removed_covers, not_on_device_books, total_books = self._remove_covers(books)
        result_message = _("Change summary:") + "\n\t" + _("Covers removed={0}\n\tBooks not on device={1}\n\tTotal books={2}").format(removed_covers, not_on_device_books, total_books)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Covers removed"),
                    result_message,
                    show=True)


    def test_covers(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("remove_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot remove covers."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()
        debug_print("test_covers - self.device_path", self.device_path)

#        contentIDs = self._get_contentIDs_for_selected()
        if self.gui.stack.currentIndex() == 0:
            selectedIDs = self._get_selected_ids()
            books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)

        else:
            books = self._get_books_for_selected()

        
        if len(books) == 0:
            return

        dlg = RemoveCoverOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options
        
        removed_covers, not_on_device_books, total_books = self._test_covers(books)
        result_message = _("Change summary:") + "\n\t" + _("Covers removed={0}\n\tBooks not on device={1}\n\tTotal books={2}").format(removed_covers, not_on_device_books, total_books)
        info_dialog(self.gui,  _("Kobo Utilities") + " - " + _("Covers removed"),
                    result_message,
                    show=True)


    def clean_images_dir(self):
        debug_print("clean_images_dir - start")

        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,  _("Cannot clean covers directory."),
                     _("No device connected."),
                    show=True)
        self.device_path = self.get_device_path()
        debug_print("clean_images_dir - self.device_path", self.device_path)

        dlg = CleanImagesDirOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options
        if self.device.fwversion >= self.device.min_fwversion_images_tree:
            self.main_image_path = os.path.join(self.device._main_prefix,   '.kobo-images')
            self.sd_image_path = os.path.join(self.device._card_a_prefix, 'koboExtStorage/images-cache/') if self.device._card_a_prefix else None
            self.options['images_tree'] = True
        else:
            self.main_image_path = os.path.join(self.device._main_prefix,   '.kobo/images')
            self.sd_image_path = os.path.join(self.device._card_a_prefix, 'koboExtStorage/images') if self.device._card_a_prefix else None
            self.options['images_tree'] = False
        self.options['main_image_path']      = self.device.normalize_path(self.main_image_path)
        self.options['sd_image_path']        = self.device.normalize_path(self.sd_image_path)
        self.options["device_database_path"] = self.device_database_path()
        self.options['job_function']         = 'clean_images_dir'
        debug_print("clean_images_dir - self.options=", self.options)
        QueueProgressDialog(self.gui, [], None, self.options, self._clean_images_dir_job, None)


    def getAnnotationForSelected(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,
                                 _("Cannot upload covers."),
                                 _("No device connected."),
                                show=True)

        self._getAnnotationForSelected()


    def _get_selected_ids(self):
        rows = self.gui.current_view().selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return []
        return map(self.gui.current_view().model().id, rows)

    def contentid_from_path(self, path, ContentType):
#        debug_print("KoboUtilities.action:contentid_from_path - path='%s'"%path, "ContentType='%s'"%ContentType)
#        debug_print("KoboUtilities.action:contentid_from_path - self.device._main_prefix='%s'"%self.device._main_prefix, "self.device.device._card_a_prefix='%s'"%self.device._card_a_prefix)
        if ContentType == 6:
            extension =  os.path.splitext(path)[1]
            if extension == '.kobo':
#                debug_print("KoboUtilities.action:contentid_from_path - have kepub")
                ContentID = os.path.splitext(path)[0]
                # Remove the prefix on the file.  it could be either
                ContentID = ContentID.replace(self.device._main_prefix, '')
            elif extension == '':
#                debug_print("KoboUtilities.action:contentid_from_path - no extension:", os.path.join(self.device._main_prefix, '.kobo/kepub/'))
                ContentID = path
                ContentID = ContentID.replace(self.device._main_prefix + self.device.normalize_path('.kobo/kepub/'), '')
            else:
#                debug_print("KoboUtilities.action:contentid_from_path - All others")
                ContentID = path
                ContentID = ContentID.replace(self.device._main_prefix, "file:///mnt/onboard/")

#            debug_print("KoboUtilities.action:contentid_from_path - 1 ContentID='%s'"%ContentID)

            if self.device._card_a_prefix is not None:
                ContentID = ContentID.replace(self.device._card_a_prefix,  "file:///mnt/sd/")
        else:  # ContentType = 16
#            debug_print("KoboUtilities.action:contentid_from_path ContentType other than 6 - ContentType='%d'"%ContentType, "path='%s'"%path)
            ContentID = path
            ContentID = ContentID.replace(self.device._main_prefix, "file:///mnt/onboard/")
            if self.device._card_a_prefix is not None:
                ContentID = ContentID.replace(self.device._card_a_prefix, "file:///mnt/sd/")
        ContentID = ContentID.replace("\\", '/')
#        debug_print("KoboUtilities.action:contentid_from_path - end - ContentID='%s'"%ContentID)
        return ContentID

    def get_contentIDs_for_books(self, ids):
        contentIDs= []
        for book_id in ids:
            device_book_path = self.get_device_path_from_id(book_id)
            debug_print('get_contentIDs_for_books - device_book_path', device_book_path)
            if device_book_path is None:
                continue
            contentID = self.contentid_from_path(device_book_path, self.CONTENTTYPE)
            debug_print('get_contentIDs_for_books - contentID', contentID)
            contentIDs.append(contentID)
        return contentIDs

    def _get_books_for_selected(self):
        view = self.gui.current_view()
        if self.isDeviceView():
            rows  = view.selectionModel().selectedRows()
            books = []
            for r in rows:
#                debug_print('_get_books_for_selected - r.row()', r.row())
                book = view.model().db[view.model().map[r.row()]]
                book.calibre_id = r.row()
                books.append(book)
            #books = [view.model().db[view.model().map[r.row()]] for r in rows]
        else:
            books = []
            
        return books

    def _convert_calibre_ids_to_books(self, db, ids):
        books = []
        for book_id in ids:
            book = self._convert_calibre_id_to_book(db, book_id)
#            debug_print('_convert_calibre_ids_to_books - book', book)
            books.append(book)
        return books

    def _convert_calibre_id_to_book(self, db, book_id):
        mi = db.get_metadata(book_id, index_is_id=True, get_cover=True)
#        debug_print('_convert_calibre_id_to_book - mi', mi)
#        debug_print('_convert_calibre_id_to_book - mi.application_id', mi.application_id)
#        debug_print('_convert_calibre_id_to_book - mi.in_library', mi.in_library)
        book = Book('', 'lpath', title=mi.title, other=mi)
#        book = {}
        book.calibre_id  = mi.id
#        book['title']       = mi.title
#        book['metadata']    = mi

        return book


    def get_device_path(self):
        debug_print('BEGIN Get Device Path')
    
        device_path = ''
        try:
            # If we're in test mode TEST_DEVICE is defined, use the predefined test directory
            #TEST_DEVICE = 'fakeKindleDir2'
            device_path = TEST_DEVICE
            debug_print('RUNNING IN TEST MODE')
        except:
            # Not in test mode, so confirm a device is connected
            device_connected = self.gui.library_view.model().device_connected
            try:
                device_connected = self.gui.library_view.model().device_connected
            except:
                debug_print('No device connected')
                device_connected = None
    
            # If there is a device connected, test if we can retrieve the mount point from Calibre
            if device_connected is not None:
                try:
                    # _main_prefix is not reset when device is ejected so must be sure device_connected above
                    device_path = self.gui.device_manager.connected_device._main_prefix
                    debug_print('Root path of device: %s' % device_path)
                except:
                    debug_print('A device appears to be connected, but device path not defined')
            else:
                debug_print('No device appears to be connected')
    
        debug_print('END Get Device Path')
        return device_path

    def get_device(self):
#        debug_print('BEGIN Get Device Path')

        self.device = None
        try:
            self.device = self.gui.device_manager.connected_device
        except:
            debug_print('No device connected')
            self.device = None

        self.current_device_profile = None
        self.current_device_config = None
        self.current_backup_config = None
        self.current_firmware_check_config = None
        self.device_uuid  = None
        self.version_info = None
    
        # If there is a device connected, test if we can retrieve the mount point from Calibre
        if self.device is None or not isinstance(self.device, KOBO):
            debug_print('No Kobo Touch, Glo or Mini appears to be connected')
        else:
            debug_print('Have a Kobo device connected connected')

        if self.device:
            self.device_path = self.get_device_path()
            self.connected_device_info = self.gui.device_manager.get_current_device_information().get('info', None)
            drive_info = self.connected_device_info[4]
            debug_print('KoboUtilities:get_device - drive_info:', drive_info)
            library_db = self.gui.library_view.model().db
            self.device_uuid = drive_info['main']['device_store_uuid']
            self.current_device_profile = cfg.get_book_profile_for_device(library_db, self.device_uuid, use_any_device=True)
            self.current_device_config = cfg.get_device_config(self.device_uuid)
            self.individual_device_options = cfg.get_plugin_pref(cfg.COMMON_OPTIONS_STORE_NAME, cfg.KEY_INDIVIDUAL_DEVICE_OPTIONS)
            if self.individual_device_options:
                self.current_backup_config = self.current_device_config[cfg.BACKUP_OPTIONS_STORE_NAME]
                self.current_firmware_check_config = self.current_device_config[cfg.UPDATE_OPTIONS_STORE_NAME]
            else:
                self.current_backup_config = cfg.get_plugin_prefs(cfg.BACKUP_OPTIONS_STORE_NAME)
                self.current_firmware_check_config =  cfg.get_plugin_prefs(cfg.UPDATE_OPTIONS_STORE_NAME)

        self.supports_series  = self.haveKoboTouch() and "supports_series" in dir(self.device) and self.device.supports_series()
        self.supports_ratings = self.haveKoboTouch() and self.device.dbversion > 36

#        debug_print('END Get Device Path')
        return self.device
    
    def device_fwversion(self):
        return self.device.fwversion

    def get_device_path_from_id(self, book_id):
        paths = []
        for x in ('memory', 'card_a', 'card_b'):
            x = getattr(self.gui, x+'_view').model()
            paths += x.paths_for_db_ids(set([book_id]), as_map=True)[book_id]
        return paths[0].path if paths else None


    def get_device_paths_from_id(self, book_id):
        paths = []
        for x in ('memory', 'card_a', 'card_b'):
            x = getattr(self.gui, x+'_view').model()
            paths += x.paths_for_db_ids(set([book_id]), as_map=True)[book_id]
#        debug_print("get_device_paths_from_id - paths=", paths)
        return [r.path for r in paths]

    def get_contentIDs_from_id(self, book_id):
        debug_print("get_contentIDs_from_id - book_id=", book_id)
        paths = []
        for x in ('memory', 'card_a', 'card_b'):
#            debug_print("get_contentIDs_from_id - x=", x)
            x = getattr(self.gui, x+'_view').model()
#            debug_print("get_contentIDs_from_id - x=", x)
            paths += x.paths_for_db_ids(set([book_id]), as_map=True)[book_id]
        debug_print("get_contentIDs_from_id - paths=", paths)
#        return [r.contentID if r.contentID else self.contentid_from_path(r.path, self.CONTENTTYPE) for r in paths]
        return [r.contentID for r in paths]

    def get_contentIDs_from_book(self, book):
        paths = []
        for x in ('memory', 'card_a', 'card_b'):
            x = getattr(self.gui, x+'_view').model()
            paths += x.paths_for_db_ids(set([book_id]), as_map=True)[book_id]
        debug_print("get_contentIDs_from_book - paths=", paths)
        return [r.contentID for r in paths]


#    def _store_queue_job(self, tdir, options, books_to_modify):
#        debug_print("KoboUtilitiesAction::_store_queue_job")
#        if not books_to_modify:
#            # All failed so cleanup our temp directory
#            remove_dir(tdir)
#            return
#
#        func = 'arbitrary_n'
#        cpus = self.gui.job_manager.server.pool_size
#        args = ['calibre_plugins.koboutilities.jobs', 'do_store_locations',
#                (books_to_modify, options, cpus)]
#        desc = _('Storing reading positions for {0} books').format(len(books_to_modify))
#        job = self.gui.job_manager.run_job(
#                self.Dispatcher(self._store_completed), func, args=args,
#                    description=desc)
#        job._tdir = tdir
#        self.gui.status_bar.show_message(_('Kobo Utilities') + ' - ' + desc, 3000)

    def _store_queue_job(self, tdir, options, books_to_modify):
        debug_print("KoboUtilitiesAction::_store_queue_job")
        if not books_to_modify:
            # All failed so cleanup our temp directory
            remove_dir(tdir)
            return

        cpus = 1# self.gui.device_manager.server.pool_size
        from calibre_plugins.koboutilities.jobs import do_store_locations
        args = [books_to_modify, options, cpus]
        desc = _('Storing reading positions for {0} books').format(len(books_to_modify))
        job = self.gui.device_manager.create_job(do_store_locations, self.Dispatcher(self._store_completed), description=desc, args=args)
        job._tdir = tdir
        self.gui.status_bar.show_message(_('Kobo Utilities') + ' - ' + desc, 3000)


    def _store_completed(self, job):
        if job.failed:
            self.gui.job_exception(job, dialog_title=_('Failed to get reading positions'))
            return
        modified_epubs_map, options = job.result
        debug_print("KoboUtilitiesAction::_store_completed - options", options)

        update_count = len(modified_epubs_map) if modified_epubs_map else 0
        if update_count == 0:
            msg = _('No reading positions were found that need to be updated')
            if options[cfg.KEY_PROMPT_TO_STORE]:
                return info_dialog(self.gui, _('Kobo Utilities'), msg,
                                    show_copy_button=True, show=True,
                                    det_msg=job.details)
            else:
                self.gui.status_bar.show_message(_('Kobo Utilities') + ' - ' + _('Storing reading positions completed - No changes found'), 3000)
        else:
            msg = _('Kobo Utilities stored reading locations for <b>{0} book(s)</b>').format(update_count)

            if options[cfg.KEY_PROMPT_TO_STORE]:
                db = self.gui.current_db
                dlg = ShowReadingPositionChangesDialog(self.gui, self, job.result, db)
                dlg.exec_()
                if dlg.result() != dlg.Accepted:
                    debug_print("_store_completed - dialog cancelled")
                    return
                modified_epubs_map = dlg.reading_locations
            self._update_database_columns(modified_epubs_map)


#    def _firmware_update(self, update_data):
#        debug_print("KoboUtilitiesAction::_firmware_update")
#
#        func = 'arbitrary_n'
#        cpus = self.gui.job_manager.server.pool_size
#        args = ['calibre_plugins.koboutilities.jobs', 'do_check_firmware_update',
#                (update_data, os.path.join(self.get_device_path(), KOBO_FIRMWARE_DL_LOCATION), cpus)]
#        desc = _("Downloading Kobo firmware update")
#        job = self.gui.job_manager.run_job(self.Dispatcher(self._firmware_completed), func, args=args, description=desc)
#        job._tdir = None
#        self.gui.status_bar.show_message(_("Kobo Utilities") + " - " + desc, 3000)

    def _firmware_update(self, update_data):
        debug_print("KoboUtilitiesAction::_firmware_update")

        cpus = 1# self.gui.device_manager.server.pool_size
        from calibre_plugins.koboutilities.jobs import do_check_firmware_update
        args = [update_data, os.path.join(self.get_device_path(), KOBO_FIRMWARE_DL_LOCATION), cpus]
        desc = _("Downloading Kobo firmware update")
        job = self.gui.device_manager.create_job(do_check_firmware_update, self.Dispatcher(self._firmware_completed), description=desc, args=args)
        job._tdir = None
        self.gui.status_bar.show_message(_("Kobo Utilities") + " - " + desc, 3000)


    def _firmware_completed(self, job):
        if job.failed:
            self.gui.job_exception(job, dialog_title=_("Failed to update Kobo firmware"))
            return

        update_results = job.result
        if update_results is True:
            debug_print("KoboUtilitiesAction::_firmware_completed - Done updating Kobo firmware")
            ret = question_dialog(self.gui, _("Kobo Firmware Update"), _("Keep Kobo firmware update files?"), show_copy_button=False)
            if not ret:
                device_path = self.get_device_path()
                update_dir = os.path.join(device_path, KOBO_FIRMWARE_DL_LOCATION)
                os.unlink(os.path.join(update_dir, "KoboRoot.tgz"))
                try:
                    shutil.rmtree(os.path.join(update_dir, "upgrade"))
                except Exception as e:
                    debug_print("WARNING: _firmware_completed - Couldn't remove 'upgrade' directory: %s" % (str(e),))
                try:
                    os.unlink(os.path.join(update_dir, "manifest.md5sum"))
                except Exception as e:
                    debug_print("WARNING: _firmware_completed - Couldn't remove 'manifest.md5sum' file: %s" % (str(e),))
            else:
                info_dialog(self.gui, _("Kobo Firmware Update"), _("Eject and unplug your Kobo device to complete the update"), show=True, show_copy_button=False)
        elif isinstance(update_results, basestring):
            info_dialog(self.gui, _("Kobo Firmware Update"), update_results, show=True)
        elif isinstance(update_results, BaseException):
            error_dialog(self.gui, _("Kobo Firmware Update"), _("Exception encountered updating Kobo firmware"), det_msg=update_results.message, show=True)


#    def _device_database_backup(self, backup_options):
#        debug_print("KoboUtilitiesAction::_firmware_update")
#
#        func = 'arbitrary_n'
#        cpus = self.gui.job_manager.server.pool_size
#        args = ['calibre_plugins.koboutilities.jobs', 'do_device_database_backup',
#                (backup_options,  cpus)]
#        desc = _("Backing up Kobo device database")
#        job = self.gui.job_manager.run_job(self.Dispatcher(self._device_database_backup_completed), func, args=args, description=desc)
#        job._tdir = None
#        self.gui.status_bar.show_message(_("Kobo Utilities") + " - " + desc, 3000)

    def _device_database_backup(self, backup_options):
        debug_print("KoboUtilitiesAction::_firmware_update")

#        func = 'arbitrary_n'
        cpus = 1# self.gui.device_manager.server.pool_size
        from calibre_plugins.koboutilities.jobs import do_device_database_backup
        args = [backup_options,  cpus]
        desc = _("Backing up Kobo device database")
        job = self.gui.device_manager.create_job(do_device_database_backup, self.Dispatcher(self._device_database_backup_completed), description=desc, args=args)
        job._tdir = None
        self.gui.status_bar.show_message(_("Kobo Utilities") + " - " + desc, 3000)


    def _device_database_backup_completed(self, job):
        if job.failed:
            self.gui.job_exception(job, dialog_title=_("Failed to backup device database"))
            return


    def _clean_images_dir_job(self, tdir, options):
        debug_print("KoboUtilitiesAction::_clean_images_dir_job")

        func = 'arbitrary_n'
        cpus = self.gui.job_manager.server.pool_size
        args = ['calibre_plugins.koboutilities.jobs', 'do_clean_images_dir',
                (options, cpus)]
        desc = _("Cleaning images directory")
        job = self.gui.job_manager.run_job(
                self.Dispatcher(self._clean_images_dir_completed), func, args=args,
                    description=desc)
        job._tdir = tdir
        self.gui.status_bar.show_message(_("Cleaning images directory") + '...')


    def _clean_images_dir_completed(self, job):
        if job.failed:
            self.gui.job_exception(job, dialog_title=_('Failed to check cover directory on device'))
            return
        extra_image_files  = job.result
        extra_covers_count = len(extra_image_files['main_memory']) + len(extra_image_files['sd_card'])
        self.gui.status_bar.show_message(_('Checking cover directory completed'), 3000)

        details = ''
        if extra_covers_count == 0:
            msg = _('No extra files found')
        else:
            msg = _("Kobo Utilities found <b>{0} extra cover(s)</b> in the cover directory.").format(extra_covers_count)
            if self.options['delete_extra_covers']:
                msg += "\n" +_("All files have been deleted.")
            if len(extra_image_files['main_memory']):
                details += "\n" +_("Extra files found in main memory images directory:") + "\n"
                for filename in extra_image_files['main_memory']:
                    details += "\t%s\n" % filename

            if len(extra_image_files['sd_card']):
                details += "\n" +_("Extra files found in SD card images directory:") + "\n"
                for filename in extra_image_files['sd_card']:
                    details += "\t%s\n" % filename

        return info_dialog(self.gui, _("Kobo Utilities") + " - " + _("Finished"), msg,
                                show_copy_button=True, show=True,
                                det_msg=details)


    def get_column_names(self, profile_name=None):
        if profile_name:
            profile = cfg.get_profile_info(self.gui.current_db, profile_name)
            columns_config = profile[cfg.CUSTOM_COLUMNS_STORE_NAME]
        else:
            columns_config = self.current_device_profile[cfg.CUSTOM_COLUMNS_STORE_NAME]

        debug_print("get_column_names - columns_config:", columns_config)
        kobo_chapteridbookmarked_column = columns_config.get(cfg.KEY_CURRENT_LOCATION_CUSTOM_COLUMN, cfg.CUSTOM_COLUMNS_OPTIONS_DEFAULTS[cfg.KEY_CURRENT_LOCATION_CUSTOM_COLUMN])
        kobo_percentRead_column = columns_config.get(cfg.KEY_PERCENT_READ_CUSTOM_COLUMN, cfg.CUSTOM_COLUMNS_OPTIONS_DEFAULTS[cfg.KEY_PERCENT_READ_CUSTOM_COLUMN])
        rating_column = columns_config.get(cfg.KEY_RATING_CUSTOM_COLUMN, cfg.CUSTOM_COLUMNS_OPTIONS_DEFAULTS[cfg.KEY_RATING_CUSTOM_COLUMN])
        last_read_column = columns_config.get(cfg.KEY_LAST_READ_CUSTOM_COLUMN, cfg.CUSTOM_COLUMNS_OPTIONS_DEFAULTS[cfg.KEY_LAST_READ_CUSTOM_COLUMN])

        return kobo_chapteridbookmarked_column, kobo_percentRead_column, rating_column, last_read_column

    def _update_database_columns(self, reading_locations):
#        reading_locations, options = payload
#        debug_print("_store_current_bookmark - reading_locations=", reading_locations)
        debug_print("_update_database_columns - start number of reading_locations= %d" % (len(reading_locations)))
        pb = ProgressBar(parent=self.gui, window_title=_("Storing reading positions"), on_top=True)
        total_books = len(reading_locations)
        pb.set_maximum(total_books)
        pb.set_value(0)
        pb.show()

        library_db   = self.gui.current_db
        custom_cols = library_db.field_metadata.custom_field_metadata()

        kobo_chapteridbookmarked_column, kobo_percentRead_column, rating_column, last_read_column = self.get_column_names()

        if kobo_chapteridbookmarked_column:
            debug_print("_update_database_columns - kobo_chapteridbookmarked_column=", kobo_chapteridbookmarked_column)
            kobo_chapteridbookmarked_col = custom_cols[kobo_chapteridbookmarked_column]
#                debug_print("_update_database_columns - kobo_chapteridbookmarked_col=", kobo_chapteridbookmarked_col)
            kobo_chapteridbookmarked_col_label = library_db.field_metadata.key_to_label(kobo_chapteridbookmarked_column)
            debug_print("_update_database_columns - kobo_chapteridbookmarked_col_label=", kobo_chapteridbookmarked_col_label)
        if kobo_percentRead_column:
            kobo_percentRead_col = custom_cols[kobo_percentRead_column]
#            kobo_percentRead_col_label = library_db.field_metadata.key_to_label(kobo_percentRead_column)

        if rating_column:
            if not rating_column == 'rating':
                rating_col = custom_cols[rating_column]
#                rating_col_label = library_db.field_metadata.key_to_label(rating_column) if rating_column else ''

        if last_read_column:
            last_read_col = custom_cols[last_read_column]
#            last_read_col_label = library_db.field_metadata.key_to_label(last_read_column)

        debug_print("_update_database_columns - kobo_chapteridbookmarked_column=", kobo_chapteridbookmarked_column)
        debug_print("_update_database_columns - kobo_percentRead_column=", kobo_percentRead_column) 
        debug_print("_update_database_columns - rating_column=", rating_column) 
        debug_print("_update_database_columns - last_read_column=", last_read_column) 
        # At this point we want to re-use code in edit_metadata to go ahead and
        # apply the changes. So we will create empty Metadata objects so only
        # the custom column field gets updated
        id_map = {}
        id_map_percentRead         = {}
        id_map_chapteridbookmarked = {}
        id_map_rating              = {}
        id_map_last_read           = {}
        for book_id, reading_location in reading_locations.iteritems():
            mi      = Metadata(_('Unknown'))
            book_mi = library_db.get_metadata(book_id, index_is_id=True, get_cover=True)
            book    = Book('', 'lpath', title=book_mi.title, other=book_mi)
            pb.set_label(_("Updating ") + book_mi.title)
            pb.increment()

            kobo_chapteridbookmarked = None
            kobo_adobe_location      = None
            kobo_percentRead         = None
            last_read                = None
            if reading_location is not None: # and not result[7] == MIMETYPE_KOBO:
                debug_print("_update_database_columns - result=", reading_location)
#                books_with_bookmark      += 1
                if reading_location[7] == MIMETYPE_KOBO:
                    kobo_chapteridbookmarked = reading_location[0]
                    kobo_adobe_location      = None
                else:
                    kobo_chapteridbookmarked = reading_location[0][len(reading_location[9]) + 1:] if reading_location[0] else None
                    kobo_adobe_location      = reading_location[1]

                if reading_location[2] == 1:
                    kobo_percentRead     = reading_location[3]
                elif reading_location[2] == 2:
                    kobo_percentRead     = 100

                if reading_location[8]:
                    kobo_rating = reading_location[8] * 2
                else:
                    kobo_rating = 0

                if reading_location[5]:
#                    debug_print("_update_database_columns - reading_location[5]=", reading_location[5])
                    last_read = convert_kobo_date(reading_location[5])
#                    debug_print("_update_database_columns - last_read=", last_read)

            elif self.options[cfg.KEY_CLEAR_IF_UNREAD]:
#                books_with_bookmark      += 1
                kobo_chapteridbookmarked = None
                kobo_adobe_location      = None
                kobo_percentRead         = None
                last_read                = None
                kobo_rating              = 0
            else:
#                books_without_bookmark += 1
                continue
            
            debug_print("_update_database_columns - kobo_chapteridbookmarked='%s'" % (kobo_chapteridbookmarked))
            debug_print("_update_database_columns - kobo_adobe_location='%s'" % (kobo_adobe_location))
            debug_print("_update_database_columns - kobo_percentRead=", kobo_percentRead)
            if kobo_chapteridbookmarked_column:
                if kobo_chapteridbookmarked and kobo_adobe_location:
#                    kobo_chapteridbookmarked_col['#value#'] = kobo_chapteridbookmarked + BOOKMARK_SEPARATOR + kobo_adobe_location
                    new_value = kobo_chapteridbookmarked + BOOKMARK_SEPARATOR + kobo_adobe_location
                elif kobo_chapteridbookmarked:
#                    kobo_chapteridbookmarked_col['#value#'] = kobo_chapteridbookmarked + BOOKMARK_SEPARATOR + kobo_adobe_location
                    new_value = kobo_chapteridbookmarked
                else:
                    kobo_chapteridbookmarked_col['#value#'] = None
                    new_value        = None
#                    debug_print("_update_database_columns - setting bookmark column to None")
                kobo_chapteridbookmarked_col['#value#'] = new_value
                if not hasattr(library_db, 'new_api'):
                    mi.set_user_metadata(kobo_chapteridbookmarked_column, kobo_chapteridbookmarked_col)
                else:
                    old_value = book.get_user_metadata(kobo_chapteridbookmarked_column, True)['#value#']
                    if not old_value == new_value: 
                        id_map_chapteridbookmarked[book_id] = new_value
#                library_db.set_custom(book.calibre_id, new_value, label=kobo_chapteridbookmarked_col_label, commit=False)

            if kobo_percentRead_column:
                kobo_percentRead_col['#value#'] = kobo_percentRead
                debug_print("_update_database_columns - setting mi.kobo_percentRead=", kobo_percentRead)
#                library_db.set_custom(book.calibre_id, kobo_percentRead, label=kobo_percentRead_col_label, commit=False)
                if not hasattr(library_db, 'new_api'):
                    mi.set_user_metadata(kobo_percentRead_column, kobo_percentRead_col)
                current_percentRead = book.get_user_metadata(kobo_percentRead_column, True)['#value#']
                debug_print("_update_database_columns - percent read - in book=", current_percentRead)
                if not current_percentRead == kobo_percentRead:
                    id_map_percentRead[book_id] = kobo_percentRead

            if rating_column and kobo_rating > 0:
                debug_print("_update_database_columns - rating_column=", rating_column)
                if rating_column == 'rating':
                    current_rating = book.rating
#                    library_db.set_rating(book.calibre_id, kobo_rating, commit=False)
                    if not hasattr(library_db, 'new_api'):
                        mi.rating = kobo_rating
                else:
                    current_rating = book.get_user_metadata(rating_column, True)['#value#']
                    rating_col['#value#'] = kobo_rating
#                    library_db.set_custom(book.calibre_id, kobo_rating, label=rating_col_label, commit=False)
                    if not hasattr(library_db, 'new_api'):
                        mi.set_user_metadata(rating_column, rating_col)
                if not current_rating == kobo_rating:
                    id_map_rating[book_id] = kobo_rating

            if last_read_column:
                current_last_read = book.get_user_metadata(last_read_column, True)['#value#']
                last_read_col['#value#'] = last_read
                debug_print("_update_database_columns - last_read=", last_read)
                if not hasattr(library_db, 'new_api'):
                    mi.set_user_metadata(last_read_column, last_read_col)
                if not current_last_read == last_read:
                    id_map_last_read[book_id] = last_read

#            debug_print("_update_database_columns - mi=", mi)
            id_map[book_id] = mi

        if hasattr(library_db, 'new_api'):
            if kobo_chapteridbookmarked_column:
                debug_print("_update_database_columns - Updating metadata - for column: %s number of changes=%d" % (kobo_chapteridbookmarked_column, len(id_map_chapteridbookmarked)))
                library_db.new_api.set_field(kobo_chapteridbookmarked_column, id_map_chapteridbookmarked)
            if kobo_percentRead_column:
                debug_print("_update_database_columns - Updating metadata - for column: %s number of changes=%d" % (kobo_percentRead_column, len(id_map_percentRead)))
                library_db.new_api.set_field(kobo_percentRead_column, id_map_percentRead)
            if rating_column:
                debug_print("_update_database_columns - Updating metadata - for column: %s number of changes=%d" % (rating_column, len(id_map_rating)))
                library_db.new_api.set_field(rating_column, id_map_rating)
            if last_read_column:
                debug_print("_update_database_columns - Updating metadata - for column: %s number of changes=%d" % (last_read_column, len(id_map_last_read)))
                library_db.new_api.set_field(last_read_column, id_map_last_read)

        
        if hasattr(library_db, 'new_api'):
            debug_print("_update_database_columns - Updating GUI - new DB engine")
            self.gui.iactions['Edit Metadata'].refresh_gui(list(reading_locations))
        else:
            edit_metadata_action = self.gui.iactions['Edit Metadata']
            debug_print("_update_database_columns - Updating GUI - old DB engine")
            edit_metadata_action.apply_metadata_changes(id_map)
        debug_print("_update_database_columns - finished")

        pb.hide()
        self.gui.status_bar.show_message(_('Kobo Utilities') + ' - ' + _('Storing reading positions completed - {0} changed.').format(len(reading_locations)), 3000)


    def _getAnnotationForSelected(self, *args):
        # Generate a path_map from selected ids
        def get_ids_from_selected_rows():
            rows = self.gui.library_view.selectionModel().selectedRows()
            if not rows or len(rows) < 1:
                rows = xrange(self.gui.library_view.model().rowCount(QModelIndex()))
            ids = map(self.gui.library_view.model().id, rows)
            return ids

        def get_formats(_id):
            formats = db.formats(_id, index_is_id=True)
            fmts = []
            if formats:
                for format in formats.split(','):
                    fmts.append(format.lower())
            return fmts

        def get_device_path_from_id(id_):
            paths = []
            for x in ('memory', 'card_a', 'card_b'):
                x = getattr(self.gui, x+'_view').model()
                paths += x.paths_for_db_ids(set([id_]), as_map=True)[id_]
            return paths[0].path if paths else None

        def generate_annotation_paths(ids, db, device):
            # Generate path templates
            # Individual storage mount points scanned/resolved in driver.get_annotations()
            path_map = {}
            for _id in ids:
                paths = self.get_device_paths_from_id(_id)
                debug_print("generate_annotation_paths - paths=", paths)
#                mi = db.get_metadata(_id, index_is_id=True)
#                a_path = device.create_annotations_path(mi, device_path=paths)
                if len(paths) > 0:
                    the_path = paths[0]
                    if len(paths) > 1:
                        if os.path.splitext(paths[0]) > 1: # No extension - is kepub
                            the_path = paths[1]
                    path_map[_id] = dict(path=the_path, fmts=get_formats(_id))
            return path_map

        annotationText = []

        if self.gui.current_view() is not self.gui.library_view:
            return error_dialog(self.gui,  _("Use library only"),
                     _("User annotations generated from main library only"),
                    show=True)
        db = self.gui.library_view.model().db

        # Get the list of ids
        ids = get_ids_from_selected_rows()
        if not ids:
            return error_dialog(self.gui,  _("No books selected"),
                     _("No books selected to fetch annotations from"),
                    show=True)

        debug_print("_getAnnotationForSelected - ids=", ids)
        # Map ids to paths
        path_map = generate_annotation_paths(ids, db, self.device)
        debug_print("_getAnnotationForSelected - path_map=", path_map)
        if len(path_map) == 0:
            return error_dialog(self.gui,  _("No books on device selected"),
                     _("None of the books selected were on the device. Annotations can only be copied for books on the device."),
                    show=True)

        from calibre.ebooks.BeautifulSoup import BeautifulSoup, Tag, NavigableString
        from calibre.ebooks.metadata import authors_to_string

        # Dispatch to the device get_annotations()
        debug_print("_getAnnotationForSelected - path_map=", path_map)
        bookmarked_books = self.device.get_annotations(path_map)
        debug_print("_getAnnotationForSelected - bookmarked_books=", bookmarked_books)

        for i, id_ in enumerate(bookmarked_books):
            bm = self.device.UserAnnotation(bookmarked_books[id_][0], bookmarked_books[id_][1])

            mi = db.get_metadata(id_, index_is_id=True)

            user_notes_soup = self.device.generate_annotation_html(bm.value)
            spanTag = Tag(user_notes_soup, 'span')
            spanTag['style'] = 'font-weight:normal'
            spanTag.insert(0,NavigableString(
                "<hr /><b>%(title)s</b> by <b>%(author)s</b>" % \
                            dict(title=mi.title,
                            #loc=last_read_location,
                            author=authors_to_string(mi.authors))))
            user_notes_soup.insert(0, spanTag)
            bookmark_html = unicode(user_notes_soup.prettify())
            debug_print(bookmark_html)
            annotationText.append(bookmark_html)

        d = ViewLog("Kobo Touch Annotation", "\n".join(annotationText), parent=self.gui)
#        d = ViewLog("Kobo Touch Annotation", unicode(j), parent=self.gui)
        d.setWindowIcon(self.qaction.icon())
#       d.setWindowIcon(get_icon('bookmarks.png'))
        d.exec_()


    def _upload_covers(self, books):

        uploaded_covers     = 0
        total_books         = 0
        not_on_device_books = len(books)

        kobo_kepub_dir = self.device.normalize_path('.kobo/kepub/')
        sd_kepub_dir   = self.device.normalize_path('koboExtStorage/kepub/')
        debug_print("_upload_covers - kobo_kepub_dir=", kobo_kepub_dir)

        for book in books:
            total_books += 1
#            debug_print("_upload_covers - book=", book)
#            debug_print("_upload_covers - thumbnail=", book.thumbnail)
            paths = self.get_device_paths_from_id(book.calibre_id)
            not_on_device_books -= 1 if len(paths) > 0 else 0
            for path in paths:
                debug_print("_upload_covers - path=", path)
                if (kobo_kepub_dir not in path and sd_kepub_dir not in path) or self.options['kepub_covers']:
                    if self.haveKoboTouch():
                        self.device._upload_cover(path, '', book, path, self.options['blackandwhite'], keep_cover_aspect=self.options['keep_cover_aspect'])
                    else:
                        self.device._upload_cover(path, '', book, path, self.options['blackandwhite'])
                    uploaded_covers += 1

        return total_books, uploaded_covers, not_on_device_books


    def _remove_covers(self, books):
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            total_books         = 0
            removed_covers      = 0
            not_on_device_books = 0

            imageId_query = 'SELECT ImageId '       \
                            'FROM content '         \
                            'WHERE ContentType = ? '\
                            'AND ContentId = ?'
            cursor = connection.cursor()

            for book in books:
                debug_print("_remove_covers - book=", book)
                debug_print("_remove_covers - book.__class__=", book.__class__)
                debug_print("_remove_covers - book.contentID=", book.contentID)
                debug_print("_remove_covers - book.lpath=", book.lpath)
                debug_print("_remove_covers - book.path=", book.path)
                contentIDs = [book.contentID] if book.contentID is not None else self.get_contentIDs_from_id(book.calibre_id)
                debug_print("_remove_covers - contentIDs=", contentIDs)
                for contentID in contentIDs:
                    debug_print("_remove_covers - contentID=", contentID)
                    if 'file:///' not in contentID and not self.options['kepub_covers']:
                        continue
                    
                    if contentID.startswith("file:///mnt/sd/"):
                        path = self.device._card_a_prefix
                    else:
                        path = self.device._main_prefix

                    query_values = (self.CONTENTTYPE, contentID,)
                    cursor.execute(imageId_query, query_values)
                    result = cursor.fetchone()
                    if result is not None:
                        debug_print("_remove_covers - contentId='%s', imageId='%s'" % (contentID, result[0]))
                        self.device.delete_images(result[0], path)
                        removed_covers +=1
                    else:
                        debug_print("_remove_covers - no match for contentId='%s'" % (contentID,))
                        not_on_device_books += 1
                    connection.commit()
                    total_books += 1

            cursor.close()

        return removed_covers, not_on_device_books, total_books


    def _test_covers(self, books):


        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            total_books         = 0
            removed_covers      = 0
            not_on_device_books = 0

            imageId_query = 'SELECT ImageId '       \
                            'FROM content '         \
                            'WHERE ContentType = ? '\
                            'AND ContentId = ?'
            cursor = connection.cursor()

            for book in books:
                debug_print("_test_covers - book=", book)
                debug_print("_test_covers - book.__class__=", book.__class__)
                debug_print("_test_covers - book.contentID=", book.contentID)
                debug_print("_test_covers - book.lpath=", book.lpath)
                debug_print("_test_covers - book.path=", book.path)
                contentIDs = [book.contentID] if book.contentID is not None else self.get_contentIDs_from_id(book.calibre_id)
                debug_print("_test_covers - contentIDs=", contentIDs)
                for contentID in contentIDs:
                    debug_print("_test_covers - contentID=", contentID)
                    if 'file:///' not in contentID and not self.options['kepub_covers']:
                        continue
                    
                    if contentID.startswith("file:///mnt/sd/"):
                        path = self.device._card_a_prefix
                    else:
                        path = self.device._main_prefix

                    query_values = (self.CONTENTTYPE, contentID,)
                    cursor.execute(imageId_query, query_values)
                    result = cursor.fetchone()
                    if result is not None:
                        debug_print("_test_covers - contentId='%s', imageId='%s'" % (contentID, result[0]))
                        hash1 = qhash(result[0])
#                        debug_print("_test_covers - hash1='%s'" % (hash1))
                        xff   = 0xff
                        dir1  = hash1 & xff
                        dir1  &= 0xff
#                        debug_print("_test_covers - dir1='%s', xff='%s'" % (dir1, xff))
                        xff00 = 0xff00
                        dir2  = (hash1 & xff00) >> 8
#                        debug_print("_test_covers - hash1='%s', dir1='%s', dir2='%s'" % (hash1, dir1, dir2))
                        cover_dir = os.path.join(path, ".kobo-images", "%s" % dir1, "%s" % dir2)
                        debug_print("_test_covers - cover_dir='%s'" % (cover_dir))
#                        self.device.delete_images(result[0], path)
                        removed_covers +=1
                    else:
                        debug_print("_test_covers - no match for contentId='%s'" % (contentID,))
                        not_on_device_books += 1
                    connection.commit()
                    total_books += 1

            cursor.close()

        return removed_covers, not_on_device_books, total_books


    def _get_imageid_set(self):
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            imageId_query = 'SELECT DISTINCT ImageId '       \
                            'FROM content '         \
                            'WHERE BookID IS NULL'
            cursor = connection.cursor()

            imageIDs = []
            cursor.execute(imageId_query)
            for i, row in enumerate(cursor):
                imageIDs.append(row[0])
#                debug_print("_get_imageid_set - row[0]='%s'" % (row[0]))
            connection.commit()

            cursor.close()

        return set(imageIDs)


    def _check_book_in_database(self, books):
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device.normalize_path(
            self.device_path +'.kobo/KoboReader.sqlite'))) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            not_on_device_books = []

            imageId_query = 'SELECT 1 '            \
                            'FROM content '        \
                            'WHERE BookID is NULL '\
                            'AND ContentId = ?'
            cursor = connection.cursor()

            for book in books:
                if not book.contentID:
                    book.contentID = self.contentid_from_path(book.path, self.CONTENTTYPE)
#                    not_on_device_books.append(book)
#                    continue

                query_values = (book.contentID,)
                cursor.execute(imageId_query, query_values)
                result = cursor.fetchone()
                if result is None:
                    debug_print("_check_book_in_database - no match for contentId='%s'" % (book.contentID,))
                    not_on_device_books.append(book)
#                else:
#                    debug_print("_check_book_in_database - match for contentId='%s', imageId='%s'" % (book.contentID, result[0]))
                connection.commit()

            cursor.close()

        return not_on_device_books


    def _get_shelf_count(self):
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            shelves = []

            shelves_query = ("SELECT Name, MIN(CreationDate), MAX(CreationDate), COUNT(*), MAX(Id) "
                            "FROM Shelf "
                            "WHERE _IsDeleted = 'false' "
                            "GROUP BY Name")

            cursor = connection.cursor()
            cursor.execute(shelves_query)
    #        count_bookshelves = 0
            for i, row in enumerate(cursor):
                debug_print("_get_shelf_count - row:", i, row[0], row[1], row[2], row[3], row[4])
                shelves.append([row[0], convert_kobo_date(row[1]), convert_kobo_date(row[2]), int(row[3]), row[4] ])
    
            cursor.close()
        return shelves


    def _get_series_shelf_count(self, order_shelf_type):
        debug_print("_get_series_shelf_count - order_shelf_type:", order_shelf_type)
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            shelves = []

            series_query = ("SELECT Name, count(*) "
                            "FROM Shelf s JOIN ShelfContent sc on name = ShelfName "
                            "WHERE s._IsDeleted = 'false' "
                            "AND sc._IsDeleted = 'false' "
                            "AND EXISTS (SELECT 1 FROM content c WHERE s.Name = c.Series ) "
                            "GROUP BY Name"
                            )
            authors_query = ("SELECT Name, count(*) "
                            "FROM Shelf s JOIN ShelfContent sc on name = ShelfName "
                            "WHERE s._IsDeleted = 'false' "
                            "AND sc._IsDeleted = 'false' "
                            "AND EXISTS (SELECT 1 FROM content c WHERE s.Name = c.Attribution ) "
                            "GROUP BY Name"
                            )
            other_query = ("SELECT Name, count(*) "
                            "FROM Shelf s JOIN ShelfContent sc on name = ShelfName "
                            "WHERE s._IsDeleted = 'false' "
                            "AND sc._IsDeleted = 'false' "
                            "AND NOT EXISTS (SELECT 1 FROM content c WHERE s.Name = c.Attribution ) "
                            "AND NOT EXISTS (SELECT 1 FROM content c WHERE s.Name = c.Series ) "
                            "GROUP BY Name"
                            )
            all_query = ("SELECT Name, count(*) "
                            "FROM Shelf s JOIN ShelfContent sc on name = ShelfName "
                            "WHERE s._IsDeleted = 'false' "
                            "AND sc._IsDeleted = 'false' "
                            "GROUP BY Name"
                            )

            shelves_queries= [series_query, authors_query, other_query, all_query]
            shelves_query = shelves_queries[order_shelf_type]
            debug_print("_get_series_shelf_count - shelves_query:", shelves_query)

            cursor = connection.cursor()
            cursor.execute(shelves_query)
    #        count_bookshelves = 0
            for i, row in enumerate(cursor):
                debug_print("_get_series_shelf_count - row:", i, row[0], row[1])
                shelf = {}
                shelf['name']  = row[0]
                shelf['count'] = int(row[1])
                shelves.append(shelf)

            cursor.close()
        debug_print("_get_series_shelf_count - shelves:", shelves)
        return shelves


    def _order_series_shelves(self, shelves, options):
        
        def urlquote(shelf_name):
            """ Quote URL-unsafe characters, For unsafe characters, need "%xx" rather than the 
            other encoding used for urls.  
            Pulled from calibre.ebooks.oeb.base.py:urlquote"""
            ASCII_CHARS   = set(chr(x) for x in xrange(128))
            UNIBYTE_CHARS = set(chr(x) for x in xrange(256))
            URL_SAFE      = set('ABCDEFGHIJKLMNOPQRSTUVWXYZ'
                                'abcdefghijklmnopqrstuvwxyz'
                                '0123456789' '_.-/~')
            URL_UNSAFE = [ASCII_CHARS - URL_SAFE, UNIBYTE_CHARS - URL_SAFE]
            result = []
            unsafe = 1 if isinstance(shelf_name, unicode) else 0
            unsafe = URL_UNSAFE[unsafe]
            for char in shelf_name:
                try:
                    if not char in URL_SAFE:
                        char = ("%%%02x" % ord(char)).upper()
                        debug_print("urlquote - unsafe after ord char=", char)
                except:
                    char = "%%%02x" % ord(char).upper()
                result.append(char)
            return ''.join(result)


        debug_print("_order_series_shelves - shelves:", shelves, " options:", options)
        import re
        from urllib import quote
#        from calibre.ebooks.oeb.base import urlquote
        
        starting_shelves = 0
        shelves_ordered  = 0
        timeDiff         = timedelta(0, 1)
        sort_descending  = not options[cfg.KEY_SORT_DESCENDING]
        order_by         = options[cfg.KEY_ORDER_SHELVES_BY]
        update_config = options[cfg.KEY_SORT_UPDATE_CONFIG]
        if update_config:
            koboConfig, config_file_path = self.get_config_file()

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")
            connection.row_factory  = sqlite.Row

            shelves_query = ("SELECT ShelfName, c.ContentId, c.Title, c.DateCreated, DateModified, Series, SeriesNumber "
                             "FROM ShelfContent sc JOIN content c on sc.ContentId= c.ContentId "
                             "WHERE sc._IsDeleted = 'false' "
                             "AND ShelfName = ? "
                             "ORDER BY ShelfName, SeriesNumber"
                            )
            update_query = ("UPDATE ShelfContent "
                            "SET DateModified = ? "
                            "WHERE ShelfName = ? "
                            "AND ContentID = ? "
                            )

            cursor = connection.cursor()
            for shelf in shelves:
                starting_shelves += 1
                debug_print("_order_series_shelves - shelf=%s, count=%d" % (shelf['name'], shelf['count']))
                if shelf['count'] <= 1:
                    continue
                shelves_ordered += 1
                shelf_data = (shelf['name'],)
                debug_print("_order_series_shelves - shelf_data:", shelf_data)
                cursor.execute(shelves_query, shelf_data)
                shelf_dict = {}
                for i, row in enumerate(cursor):
                    debug_print("_order_series_shelves - row:", i, row["ShelfName"], row["ContentId"], row['Series'], row["SeriesNumber"])
                    series_name = row['Series'] if row['Series'] else ''
                    try:
                        series_index = float(row["SeriesNumber"]) if row["SeriesNumber"] is not None else None
                    except:
                        debug_print("_order_series_shelves - non numeric number")
                        numbers = re.findall(r"\d*\.?\d+", row["SeriesNumber"])
                        if len(numbers) > 0:
                            series_index = float(numbers[0])
                    debug_print("_order_series_shelves - series_index=", series_index)
#                    series_index_str = "%10.4f"%series_index if series_index else ''
#                    sort_str = series_name + series_index_str + row['Title']
                    if order_by == cfg.KEY_ORDER_SHELVES_PUBLISHED:
                        sort_key = (row['DateCreated'], row['Title'])
                    else:
                        sort_key = (series_name, series_index, row['Title']) if not series_name == '' else (row['Title'])
                    series_entry = shelf_dict.get(sort_key, None)
                    if series_entry:
                        shelf_dict[sort_key].append(row['ContentId'])
                    else:
                        shelf_dict[sort_key] = [row['ContentId']]
                debug_print("_order_series_shelves - shelf_dict:", shelf_dict)
                
                debug_print("_order_series_shelves - sorted shelf_dict:", sorted(shelf_dict))
                
                lastModifiedTime = datetime.fromtimestamp(time.mktime(time.gmtime()))
                
                debug_print("_order_series_shelves - lastModifiedTime=", lastModifiedTime, " timeDiff:", timeDiff)
                for sort_key in sorted(shelf_dict, reverse=sort_descending):
                    for contentId in shelf_dict[sort_key]:
                        update_data = (strftime(self.device_timestamp_string(), lastModifiedTime.timetuple()), shelf['name'], contentId)
                        debug_print("_order_series_shelves - sort_key: ", sort_key,  " update_data:", update_data)
                        cursor.execute(update_query, update_data)
                        lastModifiedTime += timeDiff
                if update_config:
                    try:
                        shelf_key = quote("LastLibrarySorter_shelf_filterByBookshelf(" + shelf['name'] + ")")
                    except:
                        debug_print("_order_series_shelves - cannot encode shelf name=", shelf['name'])
                        if isinstance(shelf['name'], unicode):
                            debug_print("_order_series_shelves - is unicode")
                            shelf_key = urlquote(shelf['name'])
                            shelf_key = quote("LastLibrarySorter_shelf_filterByBookshelf(") + shelf_key + quote(")")
                        else:
                            debug_print("_order_series_shelves - not unicode")
                            shelf_key = "LastLibrarySorter_shelf_filterByBookshelf(" + shelf['name'] + ")"
                    koboConfig.set('ApplicationPreferences', shelf_key , "sortByDateAddedToShelf()")
#                    debug_print("_order_series_shelves - set shelf_key=", shelf_key)

            cursor.close()
            connection.commit()
            if update_config:
                with open(config_file_path, 'wb') as config_file:
                    debug_print("_order_series_shelves - writing config file")
                    koboConfig.write(config_file)
        debug_print("_order_series_shelves - end")
        return starting_shelves, shelves_ordered


    def _remove_duplicate_shelves(self, shelves, options):
        debug_print("_remove_duplicate_shelves - total shelves=%d: options=%s" % (len(shelves), options))
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")
            connection.row_factory = sqlite.Row

            starting_shelves    = 0
            shelves_removed     = 0
            finished_shelves    = 0
            self.pb = ProgressBar(parent=self.gui, window_title=_("Duplicate Shelves in Device Database"), on_top=False)
            total_shelves = len(shelves)
            self.pb.set_maximum(total_shelves)
            self.pb.set_value(0)
            self.pb.show()
            self.pb.left_align_label()

            shelves_update_timestamp = ("UPDATE Shelf "
                              "SET _IsDeleted = 'true', "
                              "LastModified = ? "
                              "WHERE _IsSynced = 'true' "
                              "AND Name = ? "
                              "AND CreationDate <> ?"
                              )
            shelves_update_id = ("UPDATE Shelf "
                              "SET _IsDeleted = 'true', "
                              "LastModified = ? "
                              "WHERE _IsSynced = 'true' "
                              "AND Name = ? "
                              "AND id <> ?"
                              )
            shelves_query = ("SELECT * FROM Shelf "
                              "WHERE _IsSynced = 'true' "
                              "AND Name = ? "
                              "AND CreationDate = ?"
                              )

            shelves_delete_timestamp = ("DELETE FROM Shelf "
                              "WHERE _IsSynced = 'false' "
                              "AND Name = ? "
                              "AND CreationDate <> ? "
                              "AND _IsDeleted = 'true'"
                              )
            shelves_delete_id = ("DELETE FROM Shelf "
                              "WHERE _IsSynced = 'false' "
                              "AND Name = ? "
                              "AND id <> ?"
                              "AND _IsDeleted = 'true'"
                              )

            shelves_purge = ("DELETE FROM Shelf "
                             "WHERE _IsDeleted = 'true'"
                            )

            purge_shelves = options[cfg.KEY_PURGE_SHELVES]
            keep_newest   = options[cfg.KEY_KEEP_NEWEST_SHELF]

            cursor = connection.cursor()
    #        count_bookshelves = 0
            for shelf in shelves:
                starting_shelves += shelf[3]
                finished_shelves += 1
                self.pb.set_label(_("Removing duplicates of shelf ") + shelf[0])
                self.pb.increment()

                if shelf[3] > 1:
                    debug_print("_remove_duplicate_shelves - shelf: %s, '%s', '%s', '%s', '%s'" % (shelf[0], shelf[1], shelf[2], shelf[3], shelf[4]))
                    timestamp = shelf[2] if keep_newest else shelf[1]
                    shelf_id = shelf[4] if shelf[1] == shelf[2] else None
                    shelves_values = (shelf[0], timestamp.strftime(self.device_timestamp_string()))

                    # Following lines are for debug purposes
#                    cursor.execute(shelves_query, shelves_values)
#                    for i, row in enumerate(cursor):
#                        debug_print("_remove_duplicate_shelves - row: ", row['Name'], row['CreationDate'], row['_IsDeleted'], row['_IsSynced'], row['Id'])

                    if shelf_id:
                        shelves_update_query  = shelves_update_id
                        shelves_delete_query  = shelves_delete_id
                        shelves_update_values = (strftime(self.device_timestamp_string(), time.gmtime()), shelf[0], shelf_id)
                        shelves_delete_values = (shelf[0], shelf_id)
                    else:
                        shelves_update_query  = shelves_update_timestamp
                        shelves_delete_query  = shelves_delete_timestamp
                        shelves_update_values = (strftime(self.device_timestamp_string(), time.gmtime()), shelf[0], timestamp.strftime(self.device_timestamp_string()))
                        shelves_delete_values = shelves_values
                    debug_print("_remove_duplicate_shelves - marking as deleted:", shelves_update_values)
                    debug_print("_remove_duplicate_shelves - shelves_update_query:", shelves_update_query)
                    debug_print("_remove_duplicate_shelves - shelves_delete_query:", shelves_delete_query)
                    debug_print("_remove_duplicate_shelves - shelves_delete_values:", shelves_delete_values)
                    cursor.execute(shelves_update_query, shelves_update_values)
                    cursor.execute(shelves_delete_query, shelves_delete_values)
                    shelves_removed += shelf[3] - 1

            if purge_shelves:
                debug_print("_remove_duplicate_shelves - purging all shelves marked as deleted")
                cursor.execute(shelves_purge)

            cursor.close()
            connection.commit()

            self.pb.hide()
        return starting_shelves, shelves_removed, finished_shelves


    def _check_device_database(self):
        return check_device_database(self.device_database_path())
#        import sqlite3 as sqlite
##        with closing(sqlite.connect(self.device.normalize_path(
##            'D:\Download\Books\Kobo\Test Books\prospero/KoboReader-paola.sqlite'))) as connection:
#        with closing(sqlite.connect(self.device_database_path())) as connection:
#            # return bytestrings if the content cannot the decoded as unicode
#            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")
#
#            check_query = 'PRAGMA integrity_check'
#            cursor = connection.cursor()
#
#            check_result = ''
#            cursor.execute(check_query)
#            result = cursor.fetchall()
#            if not result is None:
#                for line in result:
#                    check_result += '\n' + line[0]
#                    debug_print("_check_device_database - result line=", line[0])
#            else:
#                check_result = _("Execution of '%s' failed") % check_query
#
#            connection.commit()
#
#            cursor.close()
#
#        return check_result


    def _block_analytics(self):
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")
            cursor = connection.cursor()

            block_result = "The trigger on the AnalyticsEvents table has been removed."

            cursor.execute("DROP TRIGGER IF EXISTS BlockAnalyticsEvents")
            # Delete the Extended drvier version if it is there.
            cursor.execute("DROP TRIGGER IF EXISTS KTE_BlockAnalyticsEvents")

            if self.options[cfg.KEY_CREATE_ANALYTICSEVENTS_TRIGGER]:
                cursor.execute('DELETE FROM AnalyticsEvents')
                debug_print("KoboUtilities:_block_analytics - creating trigger.")
                trigger_query = ('CREATE TRIGGER IF NOT EXISTS BlockAnalyticsEvents '
                                'AFTER INSERT ON AnalyticsEvents '
                                'BEGIN '
                                'DELETE FROM AnalyticsEvents; '
                                'END'
                                )
                cursor.execute(trigger_query)
                result = cursor.fetchall()

                if result is None:
                    block_result = None
                else:
                    debug_print("_block_analytics - result=", result)
                    block_result = "AnalyticsEvents have been blocked in the database."

            connection.commit()
            cursor.close()
        return block_result


    def _vacuum_device_database(self):
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            compress_query = 'VACUUM'
            cursor = connection.cursor()

            compress_result = ''
            cursor.execute(compress_query)
            result = cursor.fetchall()
            if not result is None:
                debug_print("_vacuum_device_database - result=", result)
                for line in result:
                    compress_result += '\n' + line[0]
                    debug_print("_vacuum_device_database - result line=", line[0])
            else:
                compress_result = _("Execution of '%s' failed") % compress_query

            connection.commit()

            cursor.close()

        return compress_result


    def generate_metadata_query(self):
        debug_print("generate_metadata_query - self.supports_series=", self.supports_series)
        test_query = 'SELECT Title,   '\
                    '    Attribution, '\
                    '    Description, '\
                    '    Publisher,   '
        if self.supports_series:
            debug_print("generate_metadata_query - supports series is true")
            test_query += ' Series,       '\
                          ' SeriesNumber, '\
                          ' Subtitle, '
        else:
            test_query += ' null as Series, '      \
                          ' null as SeriesNumber,'
        test_query += ' ReadStatus, '        \
                      ' DateCreated, '       \
                      ' Language, '
        if self.supports_ratings:
            test_query += ' ISBN, '              \
                          ' FeedbackType, '      \
                          ' FeedbackTypeSynced, '\
                          ' r.Rating, '          \
                          ' r.DateModified '
        else:
            test_query += ' NULL as ISBN, '              \
                          ' NULL as FeedbackType, '      \
                          ' NULL as FeedbackTypeSynced, '\
                          ' NULL as Rating, '            \
                          ' NULL as DateModified '

        test_query += 'FROM content c1 '
        if self.supports_ratings:
            test_query += ' left outer join ratings r on c1.ContentID = r.ContentID '

        test_query += 'WHERE c1.BookId IS NULL '  \
                      'AND c1.ContentId = ?'
        debug_print("generate_metadata_query - test_query=%s" % test_query)
        return test_query


    def get_rating_column(self):
        columns_config =  self.current_device_profile[cfg.CUSTOM_COLUMNS_STORE_NAME]
        rating_column  = columns_config.get(cfg.KEY_RATING_CUSTOM_COLUMN, cfg.CUSTOM_COLUMNS_OPTIONS_DEFAULTS[cfg.KEY_RATING_CUSTOM_COLUMN])
        return rating_column

    def _update_metadata(self, books):
        from calibre.ebooks.metadata import authors_to_string
        from calibre.utils.localization import canonicalize_lang, lang_as_iso639_1

        debug_print("_update_metadata: number books=", len(books), "options=", self.options)

        updated_books       = 0
        not_on_device_books = 0
        unchanged_books     = 0
        count_books         = 0

        from calibre.library.save_to_disk import find_plugboard
        plugboards = self.gui.library_view.model().db.prefs.get('plugboards', {})
        debug_print("_update_metadata: plugboards=", plugboards)
        debug_print("_update_metadata: self.device.__class__.__name__=", self.device.__class__.__name__)

        rating_column = self.get_rating_column()

        rating_update = 'UPDATE ratings '           \
                            'SET Rating = ?, '      \
                                'DateModified = ? ' \
                            'WHERE ContentID  = ?'
        rating_insert = 'INSERT INTO ratings (' \
                            'Rating, '          \
                            'DateModified, '    \
                            'ContentID '        \
                            ')'                 \
                            'VALUES (?, ?, ?)'
        rating_delete = 'DELETE FROM ratings '     \
                            'WHERE ContentID = ?'

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")
            connection.row_factory = sqlite.Row

            test_query = self.generate_metadata_query()
            cursor = connection.cursor()

            for book in books:
#                device_book_paths = self.get_device_paths_from_id(book.id)
                for contentID in book.contentIDs:
                    debug_print("_update_metadata: searching for contentId='%s'" % (contentID))
                    if not contentID:
                        contentID = self.contentid_from_path(book.path, self.CONTENTTYPE)
                    count_books += 1
                    query_values = (contentID,)
                    cursor.execute(test_query, query_values)
                    result = cursor.fetchone()
                    if result is not None:
                        debug_print("_update_metadata: found contentId='%s'" % (contentID))
#                        debug_print("    result=", result)
#                        debug_print("    result.keys()=", result.keys())
#                        debug_print("    result[0]=", result[0])
                        debug_print("    result['Title']='%s'" % (result['Title']))
                        debug_print("    result['Attribution']='%s'" % (result['Attribution']))
#                        debug_print("    result['Title']=", result[result.keys()[0]])
#                        debug_print("    result.keys()[0]=", result.keys()[0])
#                        debug_print("    type(result.keys()[0])=", type(result.keys()[0]))
#                        debug_print("    type('title')=", type('title'))
#                        debug_print("    type('title)=", type("title"))
                        #self.device.delete_images(result[0])

                        title_string = None
                        authors_string = None
                        if self.options[cfg.KEY_USE_PLUGBOARD] and plugboards is not None:
                            book_format = os.path.splitext(contentID)[1][1:]
                            debug_print("_update_metadata: format='%s'" % (book_format))
                            plugboard = find_plugboard(self.device.__class__.__name__,
                                                       book_format, plugboards)
                            debug_print("_update_metadata: plugboard=", plugboard)
                            newmi = book.deepcopy_metadata()
                            if plugboard is not None:
                                debug_print("_update_metadata: applying plugboard")
                                newmi.template_to_attribute(book, plugboard)
                            newmi.series_index_string = book.series_index_string
                            debug_print("_update_metadata: newmi.title=", newmi.title)
                            debug_print("_update_metadata: newmi.authors=", newmi.authors)
                        else:
                            newmi = book
                            if self.options[cfg.KEY_USE_TITLE_SORT]:
                                title_string = newmi.title_sort
                            if self.options[cfg.KEY_USE_AUTHOR_SORT]:
                                debug_print("_update_metadata: author=", newmi.authors)
                                debug_print("_update_metadata: using author_sort=", newmi.author_sort)
#                                newmi.authors = newmi.author_sort
                                debug_print("_update_metadata: using author_sort - author=", newmi.authors)
                                authors_string = newmi.author_sort
                        debug_print("_update_metadata: title_string=", title_string)
                        title_string   = newmi.title if title_string is None else title_string
                        debug_print("_update_metadata: title_string=", title_string)
                        debug_print("_update_metadata: authors_string=", authors_string)
                        authors_string = authors_to_string(newmi.authors) if authors_string is None else authors_string
                        debug_print("_update_metadata: authors_string=", authors_string)

                        update_query  = 'UPDATE content SET '
                        update_values = []
                        set_clause    = ''
                        changes_found = False
                        rating_values = []
                        rating_change_query = None
    
                        if self.options[cfg.KEY_SET_TITLE] and not result["Title"] == title_string:
                            set_clause += ', Title  = ? '
                            debug_print("_update_metadata: set_clause=", set_clause)
                            update_values.append(title_string)
                        if self.options[cfg.KEY_SET_AUTHOR] and not result["Attribution"] == authors_string:
                            set_clause += ', Attribution  = ? '
                            debug_print("_update_metadata: set_clause=", set_clause)
                            update_values.append(authors_string)
                        if self.options[cfg.KEY_SET_DESCRIPTION]  and not result["Description"] == newmi.comments:
                            set_clause += ', Description = ? '
                            update_values.append(newmi.comments)
                        if self.options[cfg.KEY_SET_PUBLISHER]  and not result["Publisher"] == newmi.publisher:
                            set_clause += ', Publisher = ? '
                            update_values.append(newmi.publisher)
                        if self.options[cfg.KEY_SET_PUBLISHED_DATE]:
                            pubdate_string = strftime(self.device_timestamp_string(), newmi.pubdate)
                            if not (result["DateCreated"] == pubdate_string):
                                set_clause += ', DateCreated = ? '
                                debug_print("_update_metadata: convert_kobo_date(result['DateCreated'])=", convert_kobo_date(result["DateCreated"]))
                                debug_print("_update_metadata: convert_kobo_date(result['DateCreated']).__class__=", convert_kobo_date(result["DateCreated"]).__class__)
                                debug_print("_update_metadata: newmi.pubdate  =", newmi.pubdate)
                                debug_print("_update_metadata: result['DateCreated']     =", result["DateCreated"])
                                debug_print("_update_metadata: pubdate_string=", pubdate_string)
                                debug_print("_update_metadata: newmi.pubdate.__class__=", newmi.pubdate.__class__)
                                update_values.append(pubdate_string)

                        if self.options[cfg.KEY_SET_ISBN]  and not result["ISBN"] == newmi.isbn:
                            set_clause += ', ISBN = ? '
                            update_values.append(newmi.isbn)

                        if self.options[cfg.KEY_SET_LANGUAGE] and not result["Language"] == lang_as_iso639_1(newmi.language):
                            debug_print("_update_metadata: newmi.language =", newmi.language)
                            debug_print("_update_metadata: lang_as_iso639_1(newmi.language)=", lang_as_iso639_1(newmi.language))
                            debug_print("_update_metadata: canonicalize_lang(newmi.language)=", canonicalize_lang(newmi.language))
#                            set_clause += ', ISBN = ? '
#                            update_values.append(newmi.isbn)

                        if self.options[cfg.KEY_SET_NOT_INTERESTED] and not (result["FeedbackType"] == 2 or result["FeedbackTypeSynced"] == 1):
                            set_clause += ', FeedbackType = ? '
                            update_values.append(2)
                            set_clause += ', FeedbackTypeSynced = ? '
                            update_values.append(1)

                        debug_print("_update_metadata: self.options[cfg.KEY_SET_RATING]= ", self.options[cfg.KEY_SET_RATING])
                        if rating_column and self.options[cfg.KEY_SET_RATING]:
                            if rating_column == 'rating':
                                rating = newmi.rating
                            else:
                                rating = newmi.get_user_metadata(rating_column, True)['#value#']
                            debug_print("_update_metadata: rating=", rating, "result[Rating]=", result["Rating"])
                            rating = None if not rating or rating == 0 else rating / 2
                            debug_print("_update_metadata: rating=", rating, "result[Rating]=", result["Rating"])
                            rating_values.append(rating)
                            rating_values.append(strftime(self.device_timestamp_string(), time.gmtime()))
                            rating_values.append(contentID)
                            if not rating == result["Rating"]:
                                if not rating:
                                    rating_change_query = rating_delete
                                    rating_values = (contentID, )
                                elif result["DateModified"] is None: # If the date modified column does not have a value, there is no rating column
                                    rating_change_query = rating_insert
                                else:
                                    rating_change_query = rating_update

                        if self.supports_series:
                            debug_print("_update_metadata: self.options['series']", self.options['series'])
                            debug_print("_update_metadata: newmi.series=", newmi.series, "newmi.series_index=", newmi.series_index)
                            debug_print("_update_metadata: result['Series'] ='%s' result['SeriesNumber'] =%s" % (result["Series"], result["SeriesNumber"]))
                            debug_print("_update_metadata: result['Series'] == newmi.series =", (result["Series"] == newmi.series))
                            series_index_str = ("%g" % newmi.series_index) if newmi.series_index is not None else None
                            debug_print('_update_metadata: result["SeriesNumber"] == series_index_str =', (result["SeriesNumber"] == series_index_str))
                            debug_print('_update_metadata: not (result["Series"] == newmi.series or result["SeriesNumber"] == series_index_str) =', not (result["Series"] == newmi.series or result["SeriesNumber"] == series_index_str))
                            if self.options['series'] and not (result["Series"] == newmi.series and (result["SeriesNumber"] == book.series_index_string or result["SeriesNumber"] == series_index_str)):
                                debug_print("_update_metadata: setting series")
                                set_clause += ', Series  = ? '
                                set_clause += ', SeriesNumber   = ? '
                                if newmi.series is None or newmi.series == '':
                                    update_values.append(None)
                                    update_values.append(None)
                                else:
                                    update_values.append(newmi.series)
    #                                update_values.append("%g" % newmi.series_index)
                                    if newmi.series_index_string is not None:
                                        update_values.append(newmi.series_index_string)
                                    elif newmi.series_index is None:
                                        update_values.append(None)
                                    else:
                                        update_values.append("%g" % newmi.series_index)
    
                        if self.options[cfg.KEY_SET_TAGS_IN_SUBTITLE] and (
                                result["Subtitle"] is None or result["Subtitle"] == '' or result["Subtitle"][:3] == "t::" or result["Subtitle"][1] == "@"):
                            debug_print("_update_metadata: newmi.tags =", newmi.tags)
                            tag_str = None
                            if len(newmi.tags):
                                tag_str = " @".join(newmi.tags)
                                tag_str = "@" + " @".join(newmi.tags)
                            debug_print("_update_metadata: tag_str =", tag_str)
                            set_clause += ', Subtitle = ? '
                            update_values.append(tag_str)

    #                    debug_print("_update_metadata: self.options['setRreadingStatus']", self.options['setRreadingStatus'])
    #                    debug_print("_update_metadata: self.options['readingStatus']", self.options['readingStatus'])
    #                    debug_print("_update_metadata: not (result[6] == self.options['readingStatus'])", not (result[6] == self.options['readingStatus']))
                        if self.options['setRreadingStatus'] and (not (result["ReadStatus"] == self.options['readingStatus']) or self.options['resetPosition']):
                            set_clause += ', ReadStatus  = ? '
                            update_values.append(self.options['readingStatus'])
                            if self.options['resetPosition']:
                                set_clause += ', DateLastRead = ?'
                                update_values.append(None)
                                set_clause += ', ChapterIDBookmarked = ?'
                                update_values.append(None)
                                set_clause += ', ___PercentRead = ?'
                                update_values.append(0)
                                set_clause += ', FirstTimeReading = ? '
                                update_values.append(self.options['readingStatus'] < 2)
    
                        if len(set_clause) > 0:
                            update_query += set_clause[1:]
                            changes_found = True

                        if not (changes_found or rating_change_query):
                            debug_print("_update_metadata: no changes found to selected metadata. No changes being made.")
                            unchanged_books += 1
                            continue
    
                        update_query += 'WHERE ContentID = ? AND BookID IS NULL'
                        update_values.append(contentID)
                        debug_print("_update_metadata: update_query=%s" % update_query)
                        debug_print("_update_metadata: update_values= ", update_values)
                        try:
                            if changes_found:
                                cursor.execute(update_query, update_values)

                            if rating_change_query:
                                debug_print("_update_metadata: rating_change_query=%s" % rating_change_query)
                                debug_print("_update_metadata: rating_values= ", rating_values)
                                cursor.execute(rating_change_query, rating_values)

                            updated_books += 1
                        except:
                            debug_print('    Database Exception:  Unable to set series info')
                            raise
                    else:
                        debug_print("_update_metadata: no match for title='%s' contentId='%s'" % (book.title, contentID))
                        not_on_device_books += 1
                    connection.commit()
            debug_print("Update summary: Books updated=%d, unchanged books=%d, not on device=%d, Total=%d" % (updated_books, unchanged_books, not_on_device_books, count_books))

            cursor.close()
        
        return (updated_books, unchanged_books, not_on_device_books, count_books)


    def _store_current_bookmark(self, books, options=None):
        
        if options:
            self.options = options

        books_with_bookmark    = 0
        books_without_bookmark = 0
        count_books            = 0

        profileName              = self.options.get('profileName', None)
        debug_print("_store_current_bookmark - profileName=", profileName)
        clear_if_unread          = self.options[cfg.KEY_CLEAR_IF_UNREAD]
        store_if_more_recent     = self.options[cfg.KEY_STORE_IF_MORE_RECENT]
        do_not_store_if_reopened = self.options[cfg.KEY_DO_NOT_STORE_IF_REOPENED]
        
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            library_db = self.gui.current_db
            kobo_chapteridbookmarked_column, kobo_percentRead_column, rating_column, last_read_column = self.get_column_names(profileName)
            if kobo_chapteridbookmarked_column:
                debug_print("_store_current_bookmark - kobo_chapteridbookmarked_column=", kobo_chapteridbookmarked_column)
#                kobo_chapteridbookmarked_col = custom_cols[kobo_chapteridbookmarked_column]
#                debug_print("_store_current_bookmark - kobo_chapteridbookmarked_col=", kobo_chapteridbookmarked_col)
                kobo_chapteridbookmarked_col_label = library_db.field_metadata.key_to_label(kobo_chapteridbookmarked_column)
                debug_print("_store_current_bookmark - kobo_chapteridbookmarked_col_label=", kobo_chapteridbookmarked_col_label)
            if kobo_percentRead_column:
#                kobo_percentRead_col = custom_cols[kobo_percentRead_column]
                kobo_percentRead_col_label = library_db.field_metadata.key_to_label(kobo_percentRead_column)

            rating_col_label = library_db.field_metadata.key_to_label(rating_column) if rating_column else ''
            if last_read_column:
#                kobo_percentRead_col = custom_cols[kobo_percentRead_column]
                last_read_col_label = library_db.field_metadata.key_to_label(last_read_column)

            debug_print("_store_current_bookmark - kobo_chapteridbookmarked_column=", kobo_chapteridbookmarked_column)
            debug_print("_store_current_bookmark - kobo_percentRead_column=", kobo_percentRead_column) 
            debug_print("_store_current_bookmark - rating_column=", rating_column) 
            debug_print("_store_current_bookmark - rating_col_label=", rating_col_label) 
            debug_print("_store_current_bookmark - last_read_column=", last_read_column) 

#            id_map = {}
            cursor = connection.cursor()
            for book in books:
                count_books += 1
#                mi = Metadata('Unknown')
                for contentID in book.contentIDs:
                    debug_print("_store_current_bookmark - contentId='%s'" % (contentID))
                    fetch_values = (contentID,)
                    if contentID.endswith(".kepub.epub"):
                        if self.supports_ratings:
                            fetch_query = KEPUB_FETCH_QUERY
                        else:
                            fetch_query = KEPUB_FETCH_QUERY_NORATING
                    else:
                        if self.supports_ratings:
                            fetch_query = EPUB_FETCH_QUERY
                        else:
                            fetch_query = EPUB_FETCH_QUERY_NORATING
                    debug_print("_store_current_bookmark - tetch_query='%s'" % (fetch_query))
                    cursor.execute(fetch_query, fetch_values)
                    result = cursor.fetchone()
                    
                    kobo_chapteridbookmarked = None
                    kobo_adobe_location      = None
                    kobo_percentRead         = None
                    last_read                = None
                    update_library           = False
                    if result is not None: # and not result[7] == MIMETYPE_KOBO:
                        debug_print("_store_current_bookmark - result=", result)
                        books_with_bookmark += 1
                        if result[2] == 0 and clear_if_unread:
                            kobo_chapteridbookmarked = None
                            kobo_adobe_location      = None
                            kobo_percentRead         = None
                            last_read                = None
                            kobo_rating              = 0
                            update_library           = True
                        else:
                            update_library = True
                            if result[7] == MIMETYPE_KOBO:
                                kobo_chapteridbookmarked = result[0]
                                kobo_adobe_location      = None
                            else:
                                kobo_chapteridbookmarked = result[0][len(contentID) + 1:] if result[0] else None
                                kobo_adobe_location      = result[1]
    
                            if result[2] == 1: # or (result[2] == 0 and result[3] > 0):
                                kobo_percentRead = result[3]
                            elif result[2] == 2:
                                kobo_percentRead = 100

                            if result[8]:
                                kobo_rating = result[8] * 2
                            else:
                                kobo_rating = 0
                                
                            if result[5]:
                                debug_print("_store_current_bookmark - result[5]=", result[5])
                                last_read = convert_kobo_date(result[5])
                                debug_print("_store_current_bookmark - last_read=", last_read)

                            if last_read_column and store_if_more_recent:
            #                    last_read_col_label['#value#'] = last_read
                                current_last_read = book.get_user_metadata(last_read_column, True)['#value#']
                                debug_print("_store_current_bookmark - book.get_user_metadata(last_read_column, True)['#value#']=", current_last_read)
                                debug_print("_store_current_bookmark - setting mi.last_read=", last_read)
                                debug_print("_store_current_bookmark - store_if_more_recent - current_last_read < last_read=", current_last_read < last_read)
                                if current_last_read and last_read:
                                    update_library &= current_last_read < last_read
                                elif last_read:
                                    update_library &= True

                            if kobo_percentRead_column and do_not_store_if_reopened:
                                current_percentRead = book.get_user_metadata(kobo_percentRead_column, True)['#value#']
                                debug_print("_store_current_bookmark - do_not_store_if_reopened - current_percentRead=", current_percentRead)
                                update_library &= current_percentRead < 100

                    elif self.options[cfg.KEY_CLEAR_IF_UNREAD]:
                        books_with_bookmark      += 1
                        kobo_chapteridbookmarked = None
                        kobo_adobe_location      = None
                        kobo_percentRead         = None
                        last_read                = None
                        kobo_rating              = 0
                        update_library           = True
                    else:
                        books_without_bookmark += 1
                        continue
                    
                    if update_library:
                        debug_print("_store_current_bookmark - kobo_chapteridbookmarked='%s'" % (kobo_chapteridbookmarked))
                        debug_print("_store_current_bookmark - kobo_adobe_location='%s'" % (kobo_adobe_location))
                        debug_print("_store_current_bookmark - kobo_percentRead=", kobo_percentRead)
                        if kobo_chapteridbookmarked_column:
                            if kobo_chapteridbookmarked and kobo_adobe_location:
        #                        kobo_chapteridbookmarked_col['#value#'] = kobo_chapteridbookmarked + '|||' + kobo_adobe_location
                                new_value = kobo_chapteridbookmarked + BOOKMARK_SEPARATOR + kobo_adobe_location
                            elif kobo_chapteridbookmarked:
        #                        kobo_chapteridbookmarked_col['#value#'] = kobo_chapteridbookmarked + '|||' + kobo_adobe_location
                                new_value = kobo_chapteridbookmarked
                            else:
        #                        kobo_chapteridbookmarked_col['#value#'] = None
                                new_value = None
                                debug_print("_store_current_bookmark - setting bookmark column to None")
        #                    mi.set_user_metadata(kobo_chapteridbookmarked_column, kobo_chapteridbookmarked_col)
                            debug_print("_store_current_bookmark - chapterIdBookmark - on kobo=", new_value)
                            debug_print("_store_current_bookmark - chapterIdBookmark - in library=", book.get_user_metadata(kobo_chapteridbookmarked_column, True)['#value#'])
                            debug_print("_store_current_bookmark - chapterIdBookmark - on kobo==in library=", new_value == book.get_user_metadata(kobo_chapteridbookmarked_column, True)['#value#'])
                            old_value = book.get_user_metadata(kobo_chapteridbookmarked_column, True)['#value#']
                            if not old_value == new_value: 
                                library_db.set_custom(book.calibre_id, new_value, label=kobo_chapteridbookmarked_col_label, commit=False)
    
                        if kobo_percentRead_column:
        #                    kobo_percentRead_col['#value#'] = kobo_percentRead
                            debug_print("_store_current_bookmark - setting mi.kobo_percentRead=", kobo_percentRead)
                            current_percentRead = book.get_user_metadata(kobo_percentRead_column, True)['#value#']
                            debug_print("_store_current_bookmark - percent read - in book=", current_percentRead)
                            if not current_percentRead == kobo_percentRead:
                                library_db.set_custom(book.calibre_id, kobo_percentRead, label=kobo_percentRead_col_label, commit=False)
        #                    mi.set_user_metadata(kobo_percentRead_column, kobo_percentRead_col)
    
                        if rating_column and kobo_rating > 0:
        #                    kobo_percentRead_col['#value#'] = kobo_percentRead
                            debug_print("_store_current_bookmark - setting mi.rating_column=", rating_column)
                            if rating_column == 'rating':
                                current_rating = book.rating
                                debug_print("_store_current_bookmark - rating - in book=", current_rating)
                                if not current_rating == kobo_rating:
                                    library_db.set_rating(book.calibre_id, kobo_rating, commit=False)
                            else:
                                current_rating = book.get_user_metadata(rating_column, True)['#value#']
                                if not current_rating == kobo_rating:
                                    library_db.set_custom(book.calibre_id, kobo_rating, label=rating_col_label, commit=False)
    
                        if last_read_column:
        #                    last_read_col_label['#value#'] = last_read
                            current_last_read = book.get_user_metadata(last_read_column, True)['#value#']
                            debug_print("_store_current_bookmark - book.get_user_metadata(last_read_column, True)['#value#']=", current_last_read)
                            debug_print("_store_current_bookmark - setting mi.last_read=", last_read)
                            debug_print("_store_current_bookmark - current_last_read == last_read=", current_last_read == last_read)
                            if not current_last_read == last_read:
                                library_db.set_custom(book.calibre_id, last_read, label=last_read_col_label, commit=False)
        #                    mi.set_user_metadata(last_read_column, last_read_col_label)
    
        #                debug_print("_store_current_bookmark - mi=", mi)
        #                id_map[book.calibre_id] = mi
                    else:
                        books_with_bookmark    -= 1
                        books_without_bookmark += 1

            connection.commit()
            cursor.close()
            
#            edit_metadata_action = self.gui.iactions['Edit Metadata']
#            edit_metadata_action.apply_metadata_changes(id_map)
            library_db.commit()
        
        return (books_with_bookmark, books_without_bookmark, count_books)


    def _restore_current_bookmark(self, books):
        from calibre.ebooks.metadata import authors_to_string

        updated_books       = 0
        not_on_device_books = 0
        count_books         = 0

        profileName = self.options.get('profileName', None)
        kobo_chapteridbookmarked_column, kobo_percentRead_column, rating_column, last_read_column = self.get_column_names(profileName)
        chapter_query = 'SELECT c1.ChapterIDBookmarked, ' \
                               'c1.ReadStatus, '          \
                               'c1.___PercentRead, '      \
                               'c1.Attribution, '         \
                               'c1.DateLastRead, '        \
                               'c1.Title, '               \
                               'c1.MimeType, '
        if self.supports_ratings:
            chapter_query += ' r.Rating, ' \
                             ' r.DateModified '
        else:
            chapter_query += ' NULL as Rating, ' \
                             ' NULL as DateModified '
        chapter_query += 'FROM content c1 '
        if self.supports_ratings:
            chapter_query += ' left outer join ratings r on c1.ContentID = r.ContentID '
        chapter_query += 'WHERE c1.BookId IS NULL '  \
                      'AND c1.ContentId = ?'

        chapter_update  = 'UPDATE content '                \
                            'SET ChapterIDBookmarked = ? ' \
                            '  , FirstTimeReading = ? '    \
                            '  , ReadStatus = ? '          \
                            '  , ___PercentRead = ? '      \
                            '  , DateLastRead = ? '        \
                            'WHERE BookID IS NULL '        \
                            'AND ContentID = ?'
        location_update  = 'UPDATE content '           \
                             'SET adobe_location = ? ' \
                             'WHERE ContentType = 9 '  \
                             'AND ContentID = ?'
        rating_update = 'UPDATE ratings '           \
                            'SET Rating = ?, '      \
                                'DateModified = ? ' \
                            'WHERE ContentID  = ?'
        rating_insert = 'INSERT INTO ratings (' \
                            'Rating, '          \
                            'DateModified, '    \
                            'ContentID '        \
                            ')'                 \
                            'VALUES (?, ?, ?)'
        rating_delete = 'DELETE FROM ratings '  \
                            'WHERE ContentID = ?'

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path()
#            'D:/Download/Books/CalibrePlugins/KoboTouchTOCUpdatePlugin/KoboReader.sqlite'))) as connection:
            )) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")
            connection.row_factory = sqlite.Row

            cursor = connection.cursor()

            for book in books:
                count_books += 1
                for contentID in book.contentIDs:
                    chapter_values = (contentID,)
                    cursor.execute(chapter_query, chapter_values)
                    result = cursor.fetchone()
                    
                    if result is not None: # and not result[6] == MIMETYPE_KOBO:
                        debug_print("_restore_current_bookmark - result= ", result)
                        chapter_update          = 'UPDATE content SET '
                        chapter_set_clause      = ''
                        chapter_values          = []
                        location_update         = 'UPDATE content SET '
                        location_set_clause     = ''
                        location_values         = []
                        rating_change_query     = None
                        rating_values           = []
    
                        kobo_chapteridbookmarked = None
                        kobo_adobe_location      = None
                        kobo_percentRead         = None
    
                        if kobo_chapteridbookmarked_column:
                            reading_location_string  = book.get_user_metadata(kobo_chapteridbookmarked_column, True)['#value#']
                            debug_print("_restore_current_bookmark - reading_location_string=", reading_location_string)
                            if reading_location_string:
                                if result['MimeType'] == MIMETYPE_KOBO:
                                    kobo_chapteridbookmarked = reading_location_string
                                    kobo_adobe_location      = None
                                else:
                                    reading_location_parts   = reading_location_string.split(BOOKMARK_SEPARATOR)
                                    kobo_chapteridbookmarked = (contentID + "#" + reading_location_parts[0]) if len(reading_location_parts) > 0 else None
                                    kobo_adobe_location      = reading_location_parts[1] if len(reading_location_parts) > 1 else None
                            else:
                                kobo_chapteridbookmarked = None
                                kobo_adobe_location      = None
    
                            if reading_location_string:
                                chapter_values.append(kobo_chapteridbookmarked)
                                chapter_set_clause += ', ChapterIDBookmarked  = ? '
                                location_values.append(kobo_adobe_location)
                                location_set_clause += ', adobe_location  = ? '
                            else:
                                debug_print("_restore_current_bookmark - reading_location_string=", reading_location_string)

                        if kobo_percentRead_column:
                            kobo_percentRead = book.get_user_metadata(kobo_percentRead_column, True)['#value#']
                            kobo_percentRead = kobo_percentRead if kobo_percentRead else result['___PercentRead']
                            chapter_values.append(kobo_percentRead)
                            chapter_set_clause += ', ___PercentRead  = ? '

                        if self.options[cfg.KEY_READING_STATUS]:
                            if kobo_percentRead:
                                debug_print("_restore_current_bookmark - chapter_values= ", chapter_values)
                                if kobo_percentRead == 100:
                                    chapter_values.append(2)
                                    debug_print("_restore_current_bookmark - chapter_values= ", chapter_values)
                                else:
                                    chapter_values.append(1)
                                    debug_print("_restore_current_bookmark - chapter_values= ", chapter_values)
                                chapter_set_clause += ', ReadStatus  = ? '
                                chapter_values.append('false')
                                chapter_set_clause += ', FirstTimeReading = ? '

                        last_read = None
                        if self.options[cfg.KEY_DATE_TO_NOW]:
                            chapter_values.append(strftime(self.device_timestamp_string(), time.gmtime()))
                            chapter_set_clause += ', DateLastRead  = ? '
                        elif last_read_column:
                            last_read = book.get_user_metadata(last_read_column, True)['#value#']
                            if last_read is not None:
                                chapter_values.append(last_read.strftime(self.device_timestamp_string()))
                                chapter_set_clause += ', DateLastRead  = ? '

                        debug_print("_restore_current_bookmark - self.options[cfg.KEY_SET_RATING]= ", self.options[cfg.KEY_SET_RATING])
                        rating = None
                        if rating_column and self.options[cfg.KEY_SET_RATING]:
                            if rating_column == 'rating':
                                rating = book.rating
                            else:
                                rating = book.get_user_metadata(rating_column, True)['#value#']
                            rating = None if not rating or rating == 0 else rating / 2
                            debug_print("_restore_current_bookmark - rating=", rating, " result['Rating']=", result['Rating'])
                            rating_values.append(rating)
                            if last_read:
                                rating_values.append(last_read)
                            else:
                                rating_values.append(strftime(self.device_timestamp_string(), time.gmtime()))
                            rating_values.append(contentID)
                            if not rating:
                                rating_change_query = rating_delete
                                rating_values = (contentID, )
                            elif result['DateModified'] is None: # If the date modified column does not have a value, there is no rating column
                                rating_change_query = rating_insert
                            else:
                                rating_change_query = rating_update

                        debug_print("_restore_current_bookmark - found contentId='%s'" % (contentID))
                        debug_print("_restore_current_bookmark - kobo_chapteridbookmarked=", kobo_chapteridbookmarked)
                        debug_print("_restore_current_bookmark - kobo_adobe_location=", kobo_adobe_location)
                        debug_print("_restore_current_bookmark - kobo_percentRead=", kobo_percentRead)
                        debug_print("_restore_current_bookmark - rating=", rating)
                        debug_print("_restore_current_bookmark - last_read=", last_read)
#                        debug_print("    result=", result)
    
                        if len(chapter_set_clause) > 0:
                            chapter_update += chapter_set_clause[1:]
                            chapter_update += 'WHERE ContentID = ? AND BookID IS NULL'
                            chapter_values.append(contentID)
                        else:
                            debug_print("_restore_current_bookmark - no changes found to selected metadata. No changes being made.")
                            not_on_device_books += 1
                            continue
    
                        debug_print("_restore_current_bookmark - chapter_update=%s" % chapter_update)
                        debug_print("_restore_current_bookmark - chapter_values= ", chapter_values)
                        try:
                            cursor.execute(chapter_update, chapter_values)
                            if len(location_set_clause) > 0 and not result['MimeType'] == MIMETYPE_KOBO:
                                location_update += location_set_clause[1:]
                                location_update += ' WHERE ContentID = ? AND BookID IS NOT NULL'
                                location_values.append(kobo_chapteridbookmarked)
                                debug_print("_restore_current_bookmark - location_update=%s" % location_update)
                                debug_print("_restore_current_bookmark - location_values= ", location_values)
                                cursor.execute(location_update, location_values)
                            if rating_change_query:
                                debug_print("_restore_current_bookmark - rating_change_query=%s" % rating_change_query)
                                debug_print("_restore_current_bookmark - rating_values= ", rating_values)
                                cursor.execute(rating_change_query, rating_values)
                                
                            updated_books += 1
                        except:
                            debug_print('    Database Exception:  Unable to set bookmark info.')
                            raise
                    else:
                        debug_print("_restore_current_bookmark - no match for title='%s' contentId='%s'" % (book.title, book.contentID))
                        not_on_device_books += 1
                    connection.commit()
            debug_print("_restore_current_bookmark - Update summary: Books updated=%d, not on device=%d, Total=%d" % (updated_books, not_on_device_books, count_books))

            cursor.close()
        
        return (updated_books, not_on_device_books, count_books)


    def _get_shelves_from_device(self, books, options=None):
        
        if options:
            self.options = options

        count_books        = 0
        books_with_shelves = 0
        books_without_shelves = 0
        shelves_retrieved  = 0
        all_books          = self.options[cfg.KEY_ALL_BOOKS]
        replace_shelves    = self.options[cfg.KEY_REPLACE_SHELVES]

#        self.pb = ProgressBar(parent=self.gui, window_title=_("Getting shelves from device"), on_top=True)
        total_books = len(books)
        self.pb.set_maximum(total_books)
        self.pb.set_value(0)
        self.pb.show()

        fetch_query = ("SELECT c.ContentId, sc.ShelfName "
                       "FROM content c LEFT OUTER JOIN ShelfContent sc "
                            "ON c.ContentId = sc.ContentId AND c.ContentType = 6  AND sc._IsDeleted = 'false' "
                            "JOIN Shelf s ON s.Name = sc.ShelfName AND s._IsDeleted = 'false' "
                        "WHERE c.ContentId = ? "
                        "ORDER BY c.ContentId, sc.ShelfName"
                        )
        
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            library_db = self.gui.current_db
            library_config = cfg.get_library_config(library_db)
            bookshelf_column_name = library_config.get(cfg.KEY_SHELVES_CUSTOM_COLUMN, cfg.GET_SHELVES_OPTIONS_DEFAULTS[cfg.KEY_SHELVES_CUSTOM_COLUMN])
            debug_print("_get_shelves_from_device - bookshelf_column_name=", bookshelf_column_name)
            bookshelf_column = library_db.field_metadata[bookshelf_column_name]
            bookshelf_column_label = library_db.field_metadata.key_to_label(bookshelf_column_name)
            bookshelf_column_is_multiple = (bookshelf_column['is_multiple'] is not None and len(bookshelf_column['is_multiple']) > 0)
            debug_print("_get_shelves_from_device - bookshelf_column_label=", bookshelf_column_label)
            debug_print("_get_shelves_from_device - bookshelf_column_is_multiple=", bookshelf_column_is_multiple)

            cursor = connection.cursor()
            for book in books:
                self.pb.set_label(_("Getting shelves for ") + book.title)
                self.pb.increment()
                count_books += 1
                shelf_names     = []
                update_library  = False
#                mi = Metadata('Unknown')
                for contentID in book.contentIDs:
                    debug_print("_get_shelves_from_device - title='%s' contentId='%s'" % (book.title, contentID))
                    fetch_values = (contentID,)
                    debug_print("_get_shelves_from_device - tetch_query='%s'" % (fetch_query))
                    cursor.execute(fetch_query, fetch_values)

                    for i, row in enumerate(cursor):
                        debug_print("_get_shelves_from_device - result=", row)
                        shelf_names.append(row[1])
                        update_library = True

                if len(shelf_names) > 0:
                    books_with_shelves += 1
                else:
                    books_without_shelves += 1
                    continue

                if update_library and len(shelf_names) > 0:
                    debug_print("_get_shelves_from_device - device shelf_names='%s'" % (shelf_names))
                    debug_print("_get_shelves_from_device - device set(shelf_names)='%s'" % (set(shelf_names)))
                    old_value = book.get_user_metadata(bookshelf_column_name, True)['#value#']
                    debug_print("_get_shelves_from_device - library shelf names='%s'" % (old_value))
                    if old_value is None or not set(old_value) == set(shelf_names): 
                        debug_print("_get_shelves_from_device - shelves are not the same")
                        shelf_names = list(set(shelf_names)) if bookshelf_column_is_multiple else ', '.join(shelf_names)
                        debug_print("_get_shelves_from_device - device shelf_names='%s'" % (shelf_names))
                        if replace_shelves or old_value is None:
                            new_value = shelf_names
                        elif bookshelf_column_is_multiple:
                            new_value = old_value + shelf_names
                        else:
                            new_value = old_value + ', ' + shelf_names
                        debug_print("_get_shelves_from_device - new shelf names='%s'" % (new_value))
                        library_db.set_custom(book.calibre_id, new_value, label=bookshelf_column_label, commit=False)

                else:
                    books_with_shelves -= 1
                    books_without_shelves+= 1

            connection.commit()
            cursor.close()

            library_db.commit()
        self.pb.hide()

        return (books_with_shelves, books_without_shelves, count_books)


    def fetch_book_fonts(self):
        debug_print("fetch_book_fonts - start")
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device.normalize_path(
            self.device_path +'.kobo/KoboReader.sqlite'))) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            book_options = {}
            
            fetch_query = 'SELECT  '                   \
                            '"ReadingFontFamily", '    \
                            '"ReadingFontSize", '      \
                            '"ReadingAlignment", '     \
                            '"ReadingLineHeight", '    \
                            '"ReadingLeftMargin", '    \
                            '"ReadingRightMargin"  '   \
                            'FROM content_settings '   \
                            'WHERE ContentType = ? '   \
                            'AND ContentId = ?'
            fetch_values = (self.CONTENTTYPE, self.single_contentID,)

            cursor = connection.cursor()
            cursor.execute(fetch_query, fetch_values)
            result = cursor.fetchone()
            if result is not None:
                book_options['readingFontFamily']   = result[0]
                book_options['readingFontSize']     = result[1]
                book_options['readingAlignment']    = result[2].title()
                book_options['readingLineHeight']   = result[3]
                book_options['readingLeftMargin']   = result[4]
                book_options['readingRightMargin']  = result[5]
            connection.commit()

            cursor.close()
        
        return book_options


    def device_timestamp_string(self):
        if not self.timestamp_string:
            if "TIMESTAMP_STRING" in dir(self.device):
                self.timestamp_string = self.device.TIMESTAMP_STRING
            else:
                self.timestamp_string = "%Y-%m-%dT%H:%M:%SZ"
        return self.timestamp_string


    def _set_reader_fonts(self, contentIDs, delete=False):
        debug_print("_set_reader_fonts - start")
        updated_fonts  = 0
        added_fonts    = 0
        deleted_fonts  = 0
        count_books    = 0

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device.normalize_path(
            self.device_path +'.kobo/KoboReader.sqlite'))) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            test_query = 'SELECT 1 '                    \
                            'FROM content_settings '    \
                            'WHERE ContentType = ? '    \
                            'AND ContentId = ?'
            delete_query = 'DELETE '                    \
                            'FROM content_settings '    \
                            'WHERE ContentType = ? '    \
                            'AND ContentId = ?'

            if not delete:
                font_face       = self.options[cfg.KEY_READING_FONT_FAMILY]
                justification   = self.options[cfg.KEY_READING_ALIGNMENT].lower()
                justification   = '' if justification == 'off' else justification
                font_size       = self.options[cfg.KEY_READING_FONT_SIZE]
                line_spacing    = self.options[cfg.KEY_READING_LINE_HEIGHT]
                left_margins    = self.options[cfg.KEY_READING_LEFT_MARGIN]
                right_margins   = self.options[cfg.KEY_READING_RIGHT_MARGIN]
               
                add_query = 'INSERT INTO content_settings ( '   \
                                '"ContentType", '               \
                                '"DateModified", '              \
                                '"ReadingFontFamily", '         \
                                '"ReadingFontSize", '           \
                                '"ReadingAlignment", '          \
                                '"ReadingLineHeight", '         \
                                '"ReadingLeftMargin", '         \
                                '"ReadingRightMargin", '        \
                                '"ContentID" '                  \
                                ') '                            \
                            'VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)'
                add_values = (
                              self.CONTENTTYPE, 
                              time.strftime(self.device_timestamp_string(), time.gmtime()), 
                              font_face, 
                              font_size, 
                              justification, 
                              line_spacing, 
                              left_margins, 
                              right_margins, 
                              )
                update_query = 'UPDATE content_settings '    \
                                'SET "DateModified" = ?, '   \
                                '"ReadingFontFamily" = ?, '  \
                                '"ReadingFontSize" = ?, '    \
                                '"ReadingAlignment" = ?, '   \
                                '"ReadingLineHeight" = ?, '  \
                                '"ReadingLeftMargin" = ?, '  \
                                '"ReadingRightMargin" = ? '  \
                                'WHERE ContentType = ?  '    \
                                'AND ContentId = ?'
                update_values = (
                                 time.strftime(self.device_timestamp_string(), time.gmtime()), 
                                 font_face, 
                                 font_size, 
                                 justification, 
                                 line_spacing, 
                                 left_margins, 
                                 right_margins, 
                                 self.CONTENTTYPE, 
                                 )

            cursor = connection.cursor()

            for contentID in contentIDs:
                test_values = (self.CONTENTTYPE, contentID,)
                if delete:
                    cursor.execute(delete_query, test_values)
                    deleted_fonts += 1
                else:
                    cursor.execute(test_query, test_values)
                    result = cursor.fetchone()
                    if result is None:
                        cursor.execute(add_query, add_values + (contentID,))
                        added_fonts += 1
                    else:
                        cursor.execute(update_query, update_values + (contentID,))
                        updated_fonts += 1
                connection.commit()
                count_books += 1

            cursor.close()
        
        return updated_fonts, added_fonts, deleted_fonts, count_books


    def get_config_file(self):
        config_file_path = self.device.normalize_path(self.device._main_prefix + '.kobo/Kobo/Kobo eReader.conf')
        koboConfig = ConfigParser.SafeConfigParser(allow_no_value=True)
        koboConfig.optionxform = str
        debug_print("_update_config_reader_settings - config_file_path=", config_file_path)
        koboConfig.read(config_file_path)
        
        return koboConfig, config_file_path

    def _update_config_reader_settings(self, options):
        koboConfig, config_file_path = self.get_config_file()

        koboConfig.set('Reading', cfg.KEY_READING_FONT_FAMILY,  options[cfg.KEY_READING_FONT_FAMILY])
        koboConfig.set('Reading', cfg.KEY_READING_ALIGNMENT,    options[cfg.KEY_READING_ALIGNMENT])
        koboConfig.set('Reading', cfg.KEY_READING_FONT_SIZE,    "%g" % options[cfg.KEY_READING_FONT_SIZE])
        koboConfig.set('Reading', cfg.KEY_READING_LINE_HEIGHT,  "%g" % options[cfg.KEY_READING_LINE_HEIGHT])
        koboConfig.set('Reading', cfg.KEY_READING_LEFT_MARGIN,  "%g" % options[cfg.KEY_READING_LEFT_MARGIN])
        koboConfig.set('Reading', cfg.KEY_READING_RIGHT_MARGIN, "%g" % options[cfg.KEY_READING_RIGHT_MARGIN])
        
        with open(config_file_path, 'wb') as config_file:
            koboConfig.write(config_file)


    def _dismiss_tiles(self):
        
        debug_print("_dismiss_tiles - self.options[cfg.KEY_TILE_OPTIONS]", self.options[cfg.KEY_TILE_OPTIONS])
        where_clause = ''
        for option in self.options[cfg.KEY_TILE_OPTIONS]:
            where_clause += ", '" + option + "'" if self.options[cfg.KEY_TILE_OPTIONS][option] else ''

        recent_books_where = ''
        recent_books_when  = ''
        if self.options[cfg.KEY_TILE_RECENT_NEW] or self.options[cfg.KEY_TILE_RECENT_FINISHED] or self.options[cfg.KEY_TILE_RECENT_IN_THE_CLOUD]:
            recent_books_status = ', 0' if self.options[cfg.KEY_TILE_RECENT_NEW] else ''
            recent_books_status += ', 2' if self.options[cfg.KEY_TILE_RECENT_FINISHED] else ''
            recent_books_status = recent_books_status[1:] if len(recent_books_status) > 0 else ''
            recent_books_status_clause = "c.ReadStatus in (%s)" % (recent_books_status) if len(recent_books_status) > 0 else ''
            recent_books_in_cloud_clause = "c.IsDownloaded = 'false'" if self.options[cfg.KEY_TILE_RECENT_IN_THE_CLOUD] else ''
            if len(recent_books_status) > 0 and len(recent_books_in_cloud_clause) > 0:
                recent_books_clause = recent_books_status_clause + " OR " + recent_books_in_cloud_clause
            elif len(recent_books_status) > 0:
                recent_books_clause = recent_books_status_clause
            elif len(recent_books_in_cloud_clause) > 0:
                recent_books_clause = recent_books_in_cloud_clause
            else: # Should never reach here, but just in case...
                recent_books_clause = "'' <> ''"
            recent_books_where = "Type IN ('RecentBook') AND EXISTS (SELECT 1 FROM content c where c.contentId = Id and (%s))" % recent_books_clause
            recent_books_when  = "NEW.Type IN ('RecentBook') AND EXISTS (SELECT 1 FROM content c where c.contentId = NEW.Id and (%s))" % recent_books_clause
            

        if len(where_clause) > 0 or len(recent_books_where) > 0:
            if self.options[cfg.KEY_CHANGE_DISMISS_TRIGGER] and self.options[cfg.KEY_CREATE_DISMISS_TRIGGER]:
                trigger_when_clause = "new.Type IN (" + where_clause[1:] + ")" if len(where_clause) > 0 else ''
                trigger_when_clause += " OR " if len(trigger_when_clause) > 0 and len(recent_books_when) > 0 else ''
                trigger_when_clause += recent_books_when if len(recent_books_when) > 0 else ''
            where_clause = "type in (" + where_clause[1:] + ")" if len(where_clause) > 0 else ''
            where_clause += " OR " if len(where_clause) > 0 and len(recent_books_where) > 0 else ''
            where_clause += recent_books_where if len(recent_books_where) > 0 else ''
            where_clause = 'WHERE ' + where_clause
        else:
            return 0
        
        if self.options[cfg.KEY_CHANGE_DISMISS_TRIGGER]:
            if self.options[cfg.KEY_CREATE_DISMISS_TRIGGER]:
                trigger_change_statements = (
                                            "CREATE TRIGGER Activity_DismissTiles_INSERT\n"
                                            "AFTER INSERT ON Activity\n"
                                            "FOR EACH ROW\n"
                                            "WHEN ( " + trigger_when_clause + ")\n"
                                            "BEGIN\n"
                                                "UPDATE Activity\n"
                                                "SET Enabled    = 'false'\n"
                                                "WHERE rowid = new.rowid;\n"
                                            "END",
                                            "CREATE TRIGGER Activity_DismissTiles_UPDATE\n"
                                            "AFTER UPDATE ON Activity\n"
                                            "FOR EACH ROW\n"
                                            "WHEN ( " + trigger_when_clause + ")\n"
                                            "BEGIN\n"
                                                "UPDATE Activity\n"
                                                "SET Enabled    = 'false'\n"
                                                "WHERE rowid = new.rowid;\n"
                                            "END",
                                            )
                
                trigger_delete_statements= (
                                            "DROP TRIGGER IF EXISTS Activity_DismissTiles",
                                            "DROP TRIGGER IF EXISTS Activity_DismissTiles_INSERT",
                                            "DROP TRIGGER IF EXISTS Activity_DismissTiles_UPDATE",
                                            "DROP TRIGGER IF EXISTS KTE_Activity_DismissNewBookTiles",
                                            )
#            trigger_kte_delete_statement = "DROP TRIGGER IF EXISTS KTE_Activity_DismissNewBookTiles"

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            update_query = ("UPDATE Activity "
                            "SET Enabled = 'false' "
                            + where_clause
                            )

            cursor = connection.cursor()

            debug_print("KoboUtilities:_dismiss_tiles - executing update_query=", update_query)
            cursor.execute(update_query)
            
            if self.options[cfg.KEY_CHANGE_DISMISS_TRIGGER]:
                if self.options[cfg.KEY_DELETE_DISMISS_TRIGGER] or self.options[cfg.KEY_CREATE_DISMISS_TRIGGER]:
                    for trigger_statement in trigger_delete_statements:
                        debug_print("KoboUtilities:_dismiss_tiles - executing trigger_statement=", trigger_statement)
                        cursor.execute(trigger_statement)
#                    cursor.execute(trigger_kte_delete_statement)
                if self.options[cfg.KEY_CREATE_DISMISS_TRIGGER]:
                    for trigger_statement in trigger_change_statements:
                        debug_print("KoboUtilities:_dismiss_tiles - executing trigger_statement=", trigger_statement)
                        cursor.execute(trigger_statement)

            connection.commit()

            cursor.close()
        
        return 1

    def _backup_annotation_files(self, books, dest_path):

        annotations_found = 0
        kepubs            = 0
        no_annotations    = 0
        count_books       = 0
#        source_file = self.device.normalize_path(self.device_path +'.kobo/KoboReader.sqlite')
#        shutil.copyfile(source_file, backup_file)
        debug_print("_backup_annotation_files - self.device_path='%s'" % (self.device_path))
        kepub_dir = self.device.normalize_path('.kobo/kepub/')
        annotations_dir = self.device.normalize_path(self.device_path + 'Digital Editions/Annotations/')
        annotations_ext = '.annot'

        for book in books:
            count_books += 1

            for book_path in book.paths:
                relative_path = book_path.replace(self.device_path, '')
                annotation_file = self.device.normalize_path(annotations_dir + relative_path + annotations_ext)
                debug_print("_backup_annotation_files - kepub title='%s' annotation_file='%s'" % (book.title, annotation_file))
                if relative_path.startswith(kepub_dir):
                    debug_print("_backup_annotation_files - kepub title='%s' book_path='%s'" % (book.title, book_path))
                    kepubs += 1
                elif os.path.exists(annotation_file):
                    debug_print("_backup_annotation_files - book_path='%s'" % (book_path))
                    backup_file = self.device.normalize_path(dest_path + '/'+ relative_path + annotations_ext)
                    debug_print("_backup_annotation_files - backup_file='%s'" % (backup_file))
                    d, p = os.path.splitdrive(backup_file)
                    debug_print("_backup_annotation_files - d='%s' p='%s'" % (d, p))
                    backup_path = os.path.dirname(unicode(backup_file))
                    try:
                        os.makedirs(backup_path)
                    except OSError:
                        debug_print("_backup_annotation_files - path exists: backup_path='%s'" % (backup_path))
                        pass
                    shutil.copyfile(annotation_file, backup_file)
                    annotations_found += 1
                else:
                    debug_print("_backup_annotation_files - book_path='%s'" % (book_path))
                    no_annotations += 1

        debug_print("Backup summary: annotations_found=%d, no_annotations=%d, kepubs=%d Total=%d" % (annotations_found, no_annotations, kepubs, count_books))

        return (annotations_found, no_annotations, kepubs, count_books)


    def device_database_path(self):
        return self.device.normalize_path(self.device._main_prefix + '.kobo/KoboReader.sqlite')
#        return "D:\Development\CalibrePlugins\KoboUtilities\KoboReader-Glo.sqlite"


    def show_help1(self):
        self.show_help()

    def show_help(self, anchor=''):
        debug_print("show_help - anchor=", anchor)
        # Extract on demand the help file resource
        def get_help_file_resource():
            # We will write the help file out every time, in case the user upgrades the plugin zip
            # and there is a later help file contained within it.
            HELP_FILE = 'KoboUtilities_Help.html'
            file_path = os.path.join(config_dir, 'plugins', HELP_FILE)
            file_data = self.load_resources(HELP_FILE)[HELP_FILE]
            with open(file_path,'w') as f:
                f.write(file_data)
            return file_path
        debug_print("show_help - anchor=", anchor)
        url = 'file:///' + get_help_file_resource()
        url = QUrl(url)
        if not (anchor or anchor == ''):
            url.setFragment(anchor)
        open_url(url)

    def convert_kobo_date(self, kobo_date):
        return convert_kobo_date(kobo_date)


def check_device_database(database_path):
    import sqlite3 as sqlite
#        with closing(sqlite.connect(self.device.normalize_path(
#            'D:\Download\Books\Kobo\Test Books\prospero/KoboReader-paola.sqlite'))) as connection:
    with closing(sqlite.connect(database_path)) as connection:
        # return bytestrings if the content cannot the decoded as unicode
        connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

        check_query = 'PRAGMA integrity_check'
        cursor = connection.cursor()

        check_result = ''
        cursor.execute(check_query)
        result = cursor.fetchall()
        if not result is None:
            for line in result:
                check_result += '\n' + line[0]
#                debug_print("_check_device_database - result line=", line[0])
        else:
            check_result = _("Execution of '%s' failed") % check_query

        connection.commit()

        cursor.close()

    return check_result


def convert_kobo_date(kobo_date):
    from calibre.utils.date import utc_tz

    try:
        converted_date = datetime.strptime(kobo_date, "%Y-%m-%dT%H:%M:%S.%f")
        converted_date = datetime.strptime(kobo_date[0:19], "%Y-%m-%dT%H:%M:%S")
        converted_date = converted_date.replace(tzinfo=utc_tz)
#            debug_print("convert_kobo_date - '%Y-%m-%dT%H:%M:%S.%f' - kobo_date={0}'".format(kobo_date))
    except:
        try:
            converted_date = datetime.strptime(kobo_date, "%Y-%m-%dT%H:%M:%S%+00:00")
#                debug_print("convert_kobo_date - '%Y-%m-%dT%H:%M:%S+00:00' - kobo_date=%s' - kobo_date={0}'".format(kobo_date))
        except:
            try:
                converted_date = datetime.strptime(kobo_date.split('+')[0], "%Y-%m-%dT%H:%M:%S")
                converted_date = converted_date.replace(tzinfo=utc_tz)
#                    debug_print("convert_kobo_date - '%Y-%m-%dT%H:%M:%S' - kobo_date={0}'".format(kobo_date))
            except:
                try:
                    converted_date = datetime.strptime(kobo_date.split('+')[0], "%Y-%m-%d")
                    converted_date = converted_date.replace(tzinfo=utc_tz)
#                        debug_print("convert_kobo_date - '%Y-%m-%d' - kobo_date={0}'".format(kobo_date))
                except:
                    try:
                        from calibre.utils.date import parse_date
                        converted_date = parse_date(kobo_date, assume_utc=True)
#                            debug_print("convert_kobo_date - parse_date - kobo_date=%s' - kobo_date={0}'".format(kobo_date))
                    except:
#                        try:
#                            converted_date = time.gmtime(os.path.getctime(self.path))
#                            debug_print("convert_kobo_date - time.gmtime(os.path.getctime(self.path)) - kobo_date={0}'".format(kobo_date))
#                        except:
                        converted_date = time.gmtime()
                        debug_print("convert_kobo_date - time.gmtime() - kobo_date={0}'".format(kobo_date))
    return converted_date
