#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy, traceback

try:
    from PyQt5 import QtCore
    from PyQt5 import QtWidgets as QtGui
    from PyQt5.Qt import (Qt, QWidget, QGridLayout, QLabel, QPushButton, QVBoxLayout, QSpinBox,
                          QGroupBox, QCheckBox, QLineEdit, QTabWidget, QTableWidget, QAbstractItemView,
                          QHBoxLayout, QIcon, QInputDialog, QComboBox)
except ImportError:
    from PyQt4 import QtGui, QtCore
    from PyQt4.Qt import (Qt, QWidget, QGridLayout, QLabel, QPushButton, QVBoxLayout, QSpinBox,
                          QGroupBox, QCheckBox, QLineEdit, QTabWidget, QTableWidget, QAbstractItemView,
                          QHBoxLayout, QIcon, QInputDialog, QComboBox)

from calibre.gui2 import open_url, choose_dir, error_dialog, question_dialog
from calibre.gui2.dialogs.confirm_delete import confirm
from calibre.utils.config import JSONConfig
from calibre.constants import DEBUG as _DEBUG

from calibre_plugins.koboutilities.common_utils import (get_library_uuid, debug_print, get_icon,
                                     CustomColumnComboBox, KeyboardConfigDialog, KeyComboBox, ImageTitleLayout,
                                     PrefsViewerDialog, ReadOnlyTextIconWidgetItem, CheckableTableWidgetItem,
                                     ReadOnlyTableWidgetItem,
                                     convert_qvariant)

# Redefine the debug here so the jobs can see it.
DEBUG = _DEBUG

PREFS_NAMESPACE = 'KoboUtilitiesPlugin'
PREFS_KEY_SETTINGS = 'settings'

KEY_SCHEMA_VERSION = 'SchemaVersion'
DEFAULT_SCHEMA_VERSION = 0.1

STORE_LIBRARIES = 'libraries'
KEY_PROFILES                       = 'profiles'
KEY_CURRENT_LOCATION_CUSTOM_COLUMN = 'currentReadingLocationColumn'
KEY_PERCENT_READ_CUSTOM_COLUMN     = 'percentReadColumn'
KEY_RATING_CUSTOM_COLUMN           = 'ratingColumn'
KEY_LAST_READ_CUSTOM_COLUMN        = 'lastReadColumn'
KEY_STORE_ON_CONNECT               = 'storeOnConnect'
KEY_PROMPT_TO_STORE                = 'promptToStore'
KEY_STORE_IF_MORE_RECENT           = 'storeIfMoreRecent'
KEY_DO_NOT_STORE_IF_REOPENED       = 'doNotStoreIfReopened'
KEY_DO_UPDATE_CHECK                = 'doFirmwareUpdateCheck'
KEY_LAST_FIRMWARE_CHECK_TIME       = 'firmwareUpdateCheckLastTime'
KEY_DO_EARLY_FIRMWARE_CHECK        = 'doEarlyFirmwareUpdate'
KEY_FOR_DEVICE                     = 'forDevice'
BACKUP_OPTIONS_STORE_NAME               = 'backupOptionsStore'
BOOKMARK_OPTIONS_STORE_NAME             = 'BookmarkOptions'
COMMON_OPTIONS_STORE_NAME               = 'commonOptionsStore'
CUSTOM_COLUMNS_STORE_NAME               = 'customColumnOptions'
METADATA_OPTIONS_STORE_NAME             = 'MetadataOptions'
READING_OPTIONS_STORE_NAME              = 'ReadingOptions'
STORE_OPTIONS_STORE_NAME                = 'storeOptionsStore'
DISMISSTILES_OPTIONS_STORE_NAME         = 'dismissTilesOptionsStore'
FIXDUPLICATESHELVES_OPTIONS_STORE_NAME  = 'fixDuplicatesOptionsStore'
ORDERSERIESSHELVES_OPTIONS_STORE_NAME   = 'orderSeriesShelvesOptionsStore'
UPDATE_OPTIONS_STORE_NAME               = 'updateOptionsStore'
GET_SHELVES_OPTIONS_STORE_NAME          = 'getShelvesOptionStore'

KEY_STORE_BOOKMARK          = 'storeBookmarks'
KEY_DATE_TO_NOW             = 'setDateToNow'
KEY_SET_RATING              = 'setRating'
KEY_CLEAR_IF_UNREAD         = 'clearIfUnread'
KEY_BACKGROUND_JOB          = 'backgroundJob'
KEY_SET_TITLE               = 'title'
KEY_USE_TITLE_SORT          = 'titleSort'
KEY_SET_AUTHOR              = 'author'
KEY_USE_AUTHOR_SORT         = 'authourSort'
KEY_SET_DESCRIPTION         = 'description'
KEY_SET_PUBLISHER           = 'publisher'
KEY_SET_RATING              = 'rating'
KEY_SET_SERIES              = 'series'
KEY_SET_TAGS_IN_SUBTITLE    = 'tagsInSubtitle'
KEY_USE_PLUGBOARD           = 'usePlugboard'
KEY_SET_READING_STATUS      = 'setRreadingStatus'
KEY_READING_STATUS          = 'readingStatus'
KEY_SET_PUBLISHED_DATE      = 'published_date'
KEY_SET_ISBN                = 'isbn'
KEY_SET_NOT_INTERESTED      = 'mark_not_interested'
KEY_SET_LANGUAGE            = 'language'
KEY_RESET_POSITION          = 'resetPosition'
KEY_TILE_OPTIONS            = 'tileOptions'
KEY_CHANGE_DISMISS_TRIGGER  = 'changeDismissTrigger'
KEY_CREATE_DISMISS_TRIGGER  = 'createDismissTrigger'
KEY_DELETE_DISMISS_TRIGGER  = 'deleteDismissTrigger'
KEY_CREATE_ANALYTICSEVENTS_TRIGGER  = 'createAnalyticsEventsTrigger'
KEY_DELETE_ANALYTICSEVENTS_TRIGGER  = 'deleteAnalyticsEventsTrigger'
KEY_TILE_RECENT_NEW             = 'tileRecentBooksNew'
KEY_TILE_RECENT_FINISHED        = 'tileRecentBooksFinished'
KEY_TILE_RECENT_IN_THE_CLOUD    = 'tileRecentBooksInTheCLoud'

KEY_READING_FONT_FAMILY     = 'readingFontFamily'
KEY_READING_ALIGNMENT       = 'readingAlignment'
KEY_READING_FONT_SIZE       = 'readingFontSize'
KEY_READING_LINE_HEIGHT     = 'readingLineHeight'
KEY_READING_LEFT_MARGIN     = 'readingLeftMargin'
KEY_READING_RIGHT_MARGIN    = 'readingRightMargin'
KEY_READING_LOCK_MARGINS    = 'lockMargins'
KEY_UPDATE_CONFIG_FILE      = 'updateConfigFile'

KEY_BUTTON_ACTION_DEVICE    = 'buttonActionDevice'
KEY_BUTTON_ACTION_LIBRARY   = 'buttonActionLibrary'

KEY_KEEP_NEWEST_SHELF       = 'keepNewestShelf'
KEY_PURGE_SHELVES           = 'purgeShelves'

KEY_SORT_DESCENDING         = 'sortDescending'
KEY_SORT_UPDATE_CONFIG      = 'updateConfig'

KEY_ORDER_SHELVES_SERIES    = 0
KEY_ORDER_SHELVES_AUTHORS   = 1
KEY_ORDER_SHELVES_OTHER     = 2
KEY_ORDER_SHELVES_ALL       = 3
KEY_ORDER_SHELVES_TYPE      = 'orderShelvesType'

KEY_ORDER_SHELVES_BY_SERIES = 0
KEY_ORDER_SHELVES_PUBLISHED = 1
KEY_ORDER_SHELVES_BY        = 'orderShelvesBy'

KEY_DO_DAILY_BACKUP         = 'doDailyBackp'
KEY_BACKUP_EACH_CONNECTION  = 'backupEachCOnnection'
KEY_BACKUP_COPIES_TO_KEEP   = 'backupCopiesToKeepSpin'
KEY_BACKUP_DEST_DIRECTORY   = 'backupDestDirectory'

KEY_SHELVES_CUSTOM_COLUMN   = 'shelvesColumn'
KEY_ALL_BOOKS               = 'allBooks'
KEY_REPLACE_SHELVES         = 'replaceShelves'

TOKEN_ANY_DEVICE = '*Any Device'

STORE_DEVICES = 'Devices'
# Devices store consists of:
# 'Devices': { 'dev_uuid': {'type':'xxx', 'uuid':'xxx', 'name:'xxx', 'location_code':'main',
#                           'active':True, 'collections':False} ,
# For iTunes
#              'iTunes':   {'type':'iTunes', 'uuid':iTunes', 'name':'iTunes', 'location_code':'',
#                           'active':True, 'collections':False}, ...}
DEFAULT_DEVICES_VALUES = {}

BOOKMARK_OPTIONS_DEFAULTS = {
                KEY_STORE_BOOKMARK:             True,
                KEY_READING_STATUS:             True,
                KEY_DATE_TO_NOW:                True, 
                KEY_SET_RATING:                 True, 
                KEY_CLEAR_IF_UNREAD:            False, 
                KEY_BACKGROUND_JOB:             False, 
                KEY_STORE_IF_MORE_RECENT:       False,
                KEY_DO_NOT_STORE_IF_REOPENED:   False
                }
METADATA_OPTIONS_DEFAULTS = {
                KEY_SET_TITLE:          False,
                KEY_SET_AUTHOR:         False,
                KEY_SET_DESCRIPTION:    False,
                KEY_SET_PUBLISHER:      False,
                KEY_SET_RATING:         False,
                KEY_SET_SERIES:         False,
                KEY_SET_READING_STATUS: False,
                KEY_READING_STATUS:     -1,
                KEY_SET_PUBLISHED_DATE: False,
                KEY_SET_ISBN:           False,
                KEY_SET_NOT_INTERESTED: False,
                KEY_SET_LANGUAGE:       False,
                KEY_RESET_POSITION:     False,
                KEY_USE_PLUGBOARD:      False,
                KEY_USE_TITLE_SORT:     False,
                KEY_USE_AUTHOR_SORT:    False,
                KEY_SET_TAGS_IN_SUBTITLE: False
                }
READING_OPTIONS_DEFAULTS = {
                KEY_READING_FONT_FAMILY:  'Georgia',
                KEY_READING_ALIGNMENT:    'Off',
                KEY_READING_FONT_SIZE:    22,
                KEY_READING_LINE_HEIGHT:  1.3,
                KEY_READING_LEFT_MARGIN:  3,
                KEY_READING_RIGHT_MARGIN: 3,
                KEY_READING_LOCK_MARGINS: False,
                KEY_UPDATE_CONFIG_FILE:   False,
                }
STORE_OPTIONS_DEFAULTS = {
                KEY_STORE_ON_CONNECT:           False,
                KEY_PROMPT_TO_STORE:            True,
                KEY_STORE_IF_MORE_RECENT:       False,
                KEY_DO_NOT_STORE_IF_REOPENED:   False,
                }
COMMON_OPTIONS_DEFAULTS = {
                KEY_BUTTON_ACTION_DEVICE:       '',
                KEY_BUTTON_ACTION_LIBRARY:      '',
                }
DISMISSTILES_OPTIONS_DEFAULTS = {
                KEY_TILE_OPTIONS:               {},
                KEY_TILE_RECENT_NEW:            False,
                KEY_TILE_RECENT_FINISHED:       False,
                KEY_TILE_RECENT_IN_THE_CLOUD:   False
                }

FIXDUPLICATESHELVES_OPTIONS_DEFAULTS = {
                KEY_KEEP_NEWEST_SHELF:  True,
                KEY_PURGE_SHELVES:      False
                }

ORDERSERIESSHELVES_OPTIONS_DEFAULTS = {
                KEY_SORT_DESCENDING:    False,
                KEY_SORT_UPDATE_CONFIG: True,
                KEY_ORDER_SHELVES_TYPE: KEY_ORDER_SHELVES_SERIES,
                KEY_ORDER_SHELVES_BY:   KEY_ORDER_SHELVES_BY_SERIES
                }

UPDATE_OPTIONS_DEFAULTS = {
                KEY_DO_UPDATE_CHECK: False,
                KEY_LAST_FIRMWARE_CHECK_TIME: 0,
                KEY_DO_EARLY_FIRMWARE_CHECK: False
                }

BACKUP_OPTIONS_DEFAULTS = {
                KEY_DO_DAILY_BACKUP:        False,
                KEY_BACKUP_EACH_CONNECTION: False,
                KEY_BACKUP_COPIES_TO_KEEP:  5,
                KEY_BACKUP_DEST_DIRECTORY:  ''
                }

GET_SHELVES_OPTIONS_DEFAULTS = {
                KEY_SHELVES_CUSTOM_COLUMN: None,
                KEY_ALL_BOOKS:             True,
                KEY_REPLACE_SHELVES:       True
                }

CUSTOM_COLUMNS_OPTIONS_DEFAULTS = {
                          KEY_CURRENT_LOCATION_CUSTOM_COLUMN: '',
                          KEY_PERCENT_READ_CUSTOM_COLUMN:     '',
                          KEY_RATING_CUSTOM_COLUMN:           None,
                          KEY_LAST_READ_CUSTOM_COLUMN:        None,
                         }

DEFAULT_PROFILE_VALUES = {
                          KEY_FOR_DEVICE:                     None,
                          UPDATE_OPTIONS_STORE_NAME:    UPDATE_OPTIONS_DEFAULTS,
                          STORE_OPTIONS_STORE_NAME:     STORE_OPTIONS_DEFAULTS,
                          BACKUP_OPTIONS_STORE_NAME:    BACKUP_OPTIONS_DEFAULTS,
                          CUSTOM_COLUMNS_STORE_NAME:    CUSTOM_COLUMNS_OPTIONS_DEFAULTS,
                         }
DEFAULT_LIBRARY_VALUES = {
#                          KEY_CURRENT_LOCATION_CUSTOM_COLUMN: '',
#                          KEY_PERCENT_READ_CUSTOM_COLUMN:     '',
#                          KEY_RATING_CUSTOM_COLUMN:           None,
#                          KEY_LAST_READ_CUSTOM_COLUMN:        None,
                          KEY_PROFILES: { 'Default': DEFAULT_PROFILE_VALUES },
                         }

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Kobo Utilities')

# Set defaults
plugin_prefs.defaults[BOOKMARK_OPTIONS_STORE_NAME]      = BOOKMARK_OPTIONS_DEFAULTS
plugin_prefs.defaults[METADATA_OPTIONS_STORE_NAME]      = METADATA_OPTIONS_DEFAULTS
plugin_prefs.defaults[READING_OPTIONS_STORE_NAME]       = READING_OPTIONS_DEFAULTS
plugin_prefs.defaults[COMMON_OPTIONS_STORE_NAME]        = COMMON_OPTIONS_DEFAULTS
plugin_prefs.defaults[DISMISSTILES_OPTIONS_STORE_NAME]  = DISMISSTILES_OPTIONS_DEFAULTS
plugin_prefs.defaults[FIXDUPLICATESHELVES_OPTIONS_STORE_NAME]  = FIXDUPLICATESHELVES_OPTIONS_DEFAULTS
plugin_prefs.defaults[ORDERSERIESSHELVES_OPTIONS_STORE_NAME]   = ORDERSERIESSHELVES_OPTIONS_DEFAULTS
plugin_prefs.defaults[STORE_LIBRARIES]                  = {}
plugin_prefs.defaults[UPDATE_OPTIONS_STORE_NAME]        = UPDATE_OPTIONS_DEFAULTS
plugin_prefs.defaults[BACKUP_OPTIONS_STORE_NAME]        = BACKUP_OPTIONS_DEFAULTS
plugin_prefs.defaults[GET_SHELVES_OPTIONS_STORE_NAME]   = GET_SHELVES_OPTIONS_DEFAULTS
plugin_prefs.defaults[STORE_DEVICES]                    = DEFAULT_DEVICES_VALUES


try:
    debug_print("KoboUtilites::action.py - loading translations")
    load_translations()
except NameError:
    debug_print("KoboUtilites::action.py - exception when loading translations")
    pass # load_translations() added in calibre 1.9


def get_plugin_pref(store_name, option):
    c = plugin_prefs[store_name]
    default_value = plugin_prefs.defaults[store_name][option]
    return c.get(option, default_value)

def get_plugin_prefs(store_name):
    c = plugin_prefs[store_name]
    return c

def migrate_library_config_if_required(db, library_config):
    schema_version = library_config.get(KEY_SCHEMA_VERSION, 0)
    if schema_version == DEFAULT_SCHEMA_VERSION:
        return
    # We have changes to be made - mark schema as updated
    library_config[KEY_SCHEMA_VERSION] = DEFAULT_SCHEMA_VERSION

    # Any migration code in future will exist in here.
    if schema_version < 0.1:
        if 'customColumn' in library_config:
            print('Migrating Count Pages plugin custom column for pages to new schema')
            library_config[KEY_PAGES_CUSTOM_COLUMN] = library_config['customColumn']
            del library_config['customColumn']
        store_prefs = plugin_prefs[STORE_NAME]
        if KEY_PAGES_ALGORITHM not in library_config:
            print('Migrating Count Pages plugin algorithm for pages to new schema')
            library_config[KEY_PAGES_ALGORITHM] = store_prefs.get('algorithm', 0)
            # Unfortunately cannot delete since user may have other libraries
        if 'algorithmWords' in store_prefs:
            print('Deleting Count Pages plugin word algorithm')
            del store_prefs['algorithmWords']
            plugin_prefs[STORE_NAME] = store_prefs

    set_library_config(db, library_config)


def get_library_config(db):
    library_id = get_library_uuid(db)
    library_config = None
    # Check whether this is a configuration needing to be migrated from json into database
    if 'libraries' in plugin_prefs:
        libraries = plugin_prefs['libraries']
        if library_id in libraries:
            # We will migrate this below
            library_config = libraries[library_id]
            # Cleanup from json file so we don't ever do this again
            del libraries[library_id]
            if len(libraries) == 0:
                # We have migrated the last library for this user
                del plugin_prefs['libraries']
            else:
                plugin_prefs['libraries'] = libraries

    if library_config is None:
        library_config = db.prefs.get_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS,
                                                 copy.deepcopy(DEFAULT_LIBRARY_VALUES))
#    migrate_library_config_if_required(db, library_config)
    return library_config

def get_profile_info(db, profile_name):
    library_config = get_library_config(db)
    profiles = library_config.get(KEY_PROFILES, {})
    profile_map = profiles.get(profile_name, DEFAULT_PROFILE_VALUES)
    return profile_map


def set_default_profile(db, profile_name):
    library_config = get_library_config(db)
#    library_config[KEY_DEFAULT_LIST] = profile_name
    set_library_config(db, library_config)

def get_book_profiles_for_device(db, device_uuid, exclude_auto=True):
    library_config = get_library_config(db)
    profiles_map = library_config[KEY_PROFILES]
    device_profiles = {}
    for profile_name, profile_info in profiles_map.iteritems():
        if profile_info[KEY_FOR_DEVICE] in [device_uuid, TOKEN_ANY_DEVICE]:
            if not exclude_auto:
                device_profiles[profile_name] = profile_info
    return device_profiles

def get_book_profile_for_device(db, device_uuid, use_any_device=False):
    library_config = get_library_config(db)
    profiles_map = library_config.get(KEY_PROFILES, None)
    selected_profile = None
    if profiles_map:
        for profile_name, profile_info in profiles_map.iteritems():
            if profile_info[KEY_FOR_DEVICE] == device_uuid:
                profile_info['profileName'] = profile_name
                selected_profile = profile_info
                break
            elif use_any_device and profile_info[KEY_FOR_DEVICE] == TOKEN_ANY_DEVICE:
                profile_info['profileName'] = profile_name
                selected_profile = profile_info

    return selected_profile

def get_profile_names(db, exclude_auto=True):
    library_config = get_library_config(db)
    profiles = library_config[KEY_PROFILES]
    if not exclude_auto:
        return sorted(list(profiles.keys()))

    profile_names = []
    for profile_name, profile_info in profiles.iteritems():
        if profile_info.get(KEY_FOR_DEVICE, DEFAULT_PROFILE_VALUES[KEY_FOR_DEVICE]) == 'POPMANUAL':
            profile_names.append(profile_name)
    return sorted(profile_names)

def get_device_name(device_uuid):
    device = plugin_prefs[STORE_DEVICES].get(device_uuid, None)
    device_name = device['name'] if device else _("(Unknown device)")
    return device_name


def set_library_config(db, library_config):
    debug_print("set_library_config - library_config:", library_config)
    db.prefs.set_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS, library_config)


class ProfilesTab(QWidget):

    def __init__(self, parent_dialog, plugin_action):
        self.parent_dialog = parent_dialog
        QWidget.__init__(self)
        
        self.plugin_action = plugin_action
        self.help_anchor = "configuration"
        self.library_config = get_library_config(self.plugin_action.gui.current_db)
        debug_print("ProfilesTab.__init__ - self.library_config", self.library_config)
        self.profiles = self.library_config.get(KEY_PROFILES, {})
        self.current_device_profile = self.plugin_action.current_device_profile
        self.profile_name = self.current_device_profile['profileName'] if self.current_device_profile else None

        layout = QVBoxLayout(self)
        self.setLayout(layout)
        
        # -------- Lists configuration ---------
        select_profile_layout = QHBoxLayout()
        layout.addLayout(select_profile_layout)
        profiles_label = QLabel(_('Profiles:'), self)
        select_profile_layout.addWidget(profiles_label)
        self.select_profile_combo = ListComboBox(self, self.profiles, self.profile_name)
        self.select_profile_combo.setMinimumSize(150, 20)
        self.select_profile_combo.currentIndexChanged.connect(self._select_profile_combo_changed)
        select_profile_layout.addWidget(self.select_profile_combo)
        self.add_profile_button = QtGui.QToolButton(self)
        self.add_profile_button.setToolTip(_('Add profile'))
        self.add_profile_button.setIcon(QIcon(I('plus.png')))
        self.add_profile_button.clicked.connect(self.add_profile)
        select_profile_layout.addWidget(self.add_profile_button)
        self.delete_profile_button = QtGui.QToolButton(self)
        self.delete_profile_button.setToolTip(_('Delete profile'))
        self.delete_profile_button.setIcon(QIcon(I('minus.png')))
        self.delete_profile_button.clicked.connect(self.delete_profile)
        select_profile_layout.addWidget(self.delete_profile_button)
        self.rename_profile_button = QtGui.QToolButton(self)
        self.rename_profile_button.setToolTip(_('Rename profile'))
        self.rename_profile_button.setIcon(QIcon(I('edit-undo.png')))
        self.rename_profile_button.clicked.connect(self.rename_profile)
        select_profile_layout.addWidget(self.rename_profile_button)
        select_profile_layout.insertStretch(-1)

        device_layout = QHBoxLayout()
        layout.addLayout(device_layout)
        device_label = QLabel(_('&Device this profile is for:'), self)
        device_label.setToolTip(_('Select the device this profile is for.'))
        self.device_combo = DeviceColumnComboBox(self)
        device_label.setBuddy(self.device_combo)
        device_layout.addWidget(device_label)
        device_layout.addWidget(self.device_combo)

        custom_column_group = QGroupBox(_('Custom Columns'), self)
        layout.addWidget(custom_column_group )
        options_layout = QGridLayout()
        custom_column_group.setLayout(options_layout)

        self.avail_text_columns   = self.get_text_custom_columns()
        self.avail_number_columns = self.get_number_custom_columns()
        self.avail_rating_columns = self.get_rating_custom_columns()
        self.avail_date_columns   = self.get_date_custom_columns()

        current_Location_label = QLabel(_('Current Reading Location Column:'), self)
        current_Location_label.setToolTip(_("Select a custom column to store the current reading location. The column type must be 'text'. Leave this blank if you do not want to store or restore the current reading location."))
        self.current_Location_combo = CustomColumnComboBox(self, self.avail_text_columns)
        current_Location_label.setBuddy(self.current_Location_combo)
        options_layout.addWidget(current_Location_label, 0, 0, 1, 1)
        options_layout.addWidget(self.current_Location_combo, 0, 1, 1, 1)
        
        percent_read_label = QLabel(_('Percent Read Column:'), self)
        percent_read_label.setToolTip(_("Column used to store the current percent read. The column type must be a 'integer'. Leave this blank if you do not want to store or restore the percentage read."))
        self.percent_read_combo = CustomColumnComboBox(self, self.avail_number_columns)
        percent_read_label.setBuddy(self.percent_read_combo)
        options_layout.addWidget(percent_read_label, 2, 0, 1, 1)
        options_layout.addWidget(self.percent_read_combo, 2, 1, 1, 1)

        rating_label = QLabel(_('Rating Column:'), self)
        rating_label.setToolTip(_("Column used to store the rating. The column type must be a 'integer'. Leave this blank if you do not want to store or restore the rating."))
        self.rating_combo = CustomColumnComboBox(self, self.avail_rating_columns)
        rating_label.setBuddy(self.rating_combo)
        options_layout.addWidget(rating_label, 3, 0, 1, 1)
        options_layout.addWidget(self.rating_combo, 3, 1, 1, 1)

        last_read_label = QLabel(_('Last Read Column:'), self)
        last_read_label.setToolTip(_("Column used to store when the book was last read. The column type must be a 'Date'. Leave this blank if you do not want to store the last read timestamp."))
        self.last_read_combo = CustomColumnComboBox(self, self.avail_date_columns)
        last_read_label.setBuddy(self.last_read_combo)
        options_layout.addWidget(last_read_label, 4, 0, 1, 1)
        options_layout.addWidget(self.last_read_combo, 4, 1, 1, 1)

        auto_store_group = QGroupBox(_('Store on connect'), self)
        layout.addWidget(auto_store_group )
        options_layout = QGridLayout()
        auto_store_group.setLayout(options_layout)

        self.store_on_connect_checkbox = QCheckBox(_("Store current bookmarks on connect"), self)
        self.store_on_connect_checkbox.setToolTip(_("When this is checked, the library will be updated with the current bookmark for all books on the device."))
        self.store_on_connect_checkbox.clicked.connect(self.store_on_connect_checkbox_clicked)
        options_layout.addWidget(self.store_on_connect_checkbox, 0, 0, 1, 3)

        self.prompt_to_store_checkbox = QCheckBox(_("Prompt to store any changes"), self)
        self.prompt_to_store_checkbox.setToolTip(_("Enable this to be prompted to save the changed bookmarks after an automatic store is done."))
        options_layout.addWidget(self.prompt_to_store_checkbox, 1, 0, 1, 1)

        self.store_if_more_recent_checkbox = QCheckBox(_("Only if more recent"), self)
        self.store_if_more_recent_checkbox.setToolTip(_("Only store the reading position if the last read timestamp on the device is more recent than in the library."))
        options_layout.addWidget(self.store_if_more_recent_checkbox, 1, 1, 1, 1)

        self.do_not_store_if_reopened_checkbox = QCheckBox(_("Not if finished in library"), self)
        self.do_not_store_if_reopened_checkbox.setToolTip(_("Do not store the reading position if the library has the book as finished. This is if the percent read is 100%."))
        options_layout.addWidget(self.do_not_store_if_reopened_checkbox, 1, 2, 1, 1)

        update_options_group = QGroupBox(_('Firmware Update Options'), self)
        layout.addWidget(update_options_group)
        options_layout = QGridLayout()
        update_options_group.setLayout(options_layout)

        self.do_update_check = QCheckBox(_('Check for Kobo firmware updates daily?'), self)
        self.do_update_check.setToolTip(_('If this is selected the plugin will check for Kobo firmware updates when your Kobo device is plugged in, once per 24-hour period.'))
        options_layout.addWidget(self.do_update_check, 0, 0, 1, 1)

        self.do_early_firmware_check = QCheckBox(_('Use early firmware adopter affiliate?'), self)
        self.do_early_firmware_check.setToolTip(_('WARNING: THIS OPTION RISKS DOWNLOADING THE WRONG FIRMWARE FOR YOUR DEVICE! YOUR DEVICE MAY NOT FUNCTION PROPERLY IF THIS HAPPENS! Choose this option to attempt to download Kobo firmware updates before they are officially available for your device.'))
        options_layout.addWidget(self.do_early_firmware_check, 0, 1, 1, 1)

        backup_options_group = QGroupBox(_('Device Database Backup'), self)
        layout.addWidget(backup_options_group)
        options_layout = QGridLayout()
        backup_options_group.setLayout(options_layout)

        self.do_daily_backp_checkbox = QCheckBox(_('Backup the device database daily'), self)
        self.do_daily_backp_checkbox.setToolTip(_('If this is selected the plugin will backup the device database the first time it is connected each day.'))
        self.do_daily_backp_checkbox.clicked.connect(self.do_daily_backp_checkbox_clicked)
        options_layout.addWidget(self.do_daily_backp_checkbox, 0, 0, 1, 2)

        self.backup_each_connection_checkbox = QCheckBox(_('Backup the device database on each connection'), self)
        self.backup_each_connection_checkbox.setToolTip(_('If this is selected the plugin will backup the device database each time the device is connected.'))
        self.backup_each_connection_checkbox.clicked.connect(self.backup_each_connection_checkbox_clicked)
        options_layout.addWidget(self.backup_each_connection_checkbox, 0, 2, 1, 3)

        self.dest_directory_label = QLabel(_("Destination:"), self)
        self.dest_directory_label.setToolTip(_("Select the destination to backup the device database to."))
        self.dest_directory_edit = QLineEdit(self)
        self.dest_directory_edit.setMinimumSize(150, 0)
        self.dest_directory_label.setBuddy(self.dest_directory_edit)
        self.dest_pick_button = QPushButton(_("..."), self)
        self.dest_pick_button.setMaximumSize(24, 20)
        self.dest_pick_button.clicked.connect(self._get_dest_directory_name)
        options_layout.addWidget(self.dest_directory_label, 1, 0, 1, 1)
        options_layout.addWidget(self.dest_directory_edit, 1, 1, 1, 1)
        options_layout.addWidget(self.dest_pick_button, 1, 2, 1, 1)

        self.copies_to_keep_checkbox = QCheckBox(_('Copies to keep'), self)
        self.copies_to_keep_checkbox.setToolTip(_("Select this to limit the number of backup kept. If not set, the backup files must be manually deleted."))
        self.copies_to_keep_spin = QSpinBox(self)
        self.copies_to_keep_spin.setMinimum(2)
        self.copies_to_keep_spin.setToolTip(_("The number of backup copies of the database to keep. The minimum is 2."))
        options_layout.addWidget(self.copies_to_keep_checkbox, 1, 3, 1, 1)
        options_layout.addWidget(self.copies_to_keep_spin, 1, 4, 1, 1)
        self.copies_to_keep_checkbox.clicked.connect(self.copies_to_keep_checkbox_clicked)

        self.toggle_backup_options_state(False)

        layout.addStretch(1)

    def _select_profile_combo_changed(self):
        self.persist_profile_config()
        self.refresh_current_profile_info()

    def store_on_connect_checkbox_clicked(self, checked):
        self.prompt_to_store_checkbox.setEnabled(checked)
        self.store_if_more_recent_checkbox.setEnabled(checked)
        self.do_not_store_if_reopened_checkbox.setEnabled(checked)

    def toggle_backup_options_state(self, enabled):
        self.dest_directory_edit.setEnabled(enabled)
        self.dest_pick_button.setEnabled(enabled)
        self.dest_directory_label.setEnabled(enabled)
        self.copies_to_keep_checkbox.setEnabled(enabled)
        self.copies_to_keep_checkbox_clicked(enabled and self.copies_to_keep_checkbox.checkState() == Qt.Checked)

    def do_daily_backp_checkbox_clicked(self, checked):
        enable_backup_options = checked or self.backup_each_connection_checkbox.checkState() ==  Qt.Checked
        self.toggle_backup_options_state(enable_backup_options)
        if self.backup_each_connection_checkbox.checkState() ==  Qt.Checked:
            self.backup_each_connection_checkbox.setCheckState(Qt.Unchecked)

    def backup_each_connection_checkbox_clicked(self, checked):
        enable_backup_options = checked or self.do_daily_backp_checkbox.checkState() ==  Qt.Checked
        self.toggle_backup_options_state(enable_backup_options)
        if self.do_daily_backp_checkbox.checkState() ==  Qt.Checked:
            self.do_daily_backp_checkbox.setCheckState(Qt.Unchecked)

    def copies_to_keep_checkbox_clicked(self, checked):
        self.copies_to_keep_spin.setEnabled(checked)

    # Called by Calibre before save_settings 
    def validate(self):
#        import traceback
#        traceback.print_stack()
        
        debug_print('BEGIN Validate')
        valid = True
        # Only save if we were able to get data to avoid corrupting stored data
#        if self.do_daily_backp_checkbox.checkState() == Qt.Checked and not len(self.dest_directory_edit.text()):
#            error_dialog(self, 'No destination directory',
#                            'If the automatic device backup is set, there must be a destination directory.',
#                            show=True, show_copy_button=False)
#            valid = False

        debug_print('END Validate, status = %s' % valid)
        return valid

    def add_profile(self):
        debug_print("ProfilesTab:add_profile - Start")
        # Display a prompt allowing user to specify a new profile
        new_profile_name, ok = QInputDialog.getText(self, _('Add new profile'),
                    _('Enter a unique display name for this profile:'), text='Default')
        if not ok:
            # Operation cancelled
            return
        new_profile_name = unicode(new_profile_name).strip()
        # Verify it does not clash with any other profiles in the profile
        for profile_name in self.profiles.keys():
            debug_print("ProfilesTab:add_profile - existing profile: ", profile_name)
            if profile_name.lower() == new_profile_name.lower():
                return error_dialog(self,
                                    _('Add failed'),
                                    _('A profile with the same name already exists'),
                                    show=True)

        # As we are about to switch profile, persist the current profiles details if any
        self.persist_profile_config()
        self.profile_name = new_profile_name
        self.profiles[new_profile_name] = copy.deepcopy(DEFAULT_PROFILE_VALUES)
        debug_print("ProfilesTab:add_profile - new profile: ", self.profiles[new_profile_name])
        # Now update the profiles combobox
        self.select_profile_combo.populate_combo(self.profiles, new_profile_name)
        self.refresh_current_profile_info()
        debug_print("ProfilesTab:add_profile - End")

    def rename_profile(self):
        if not self.profile_name:
            return
        # Display a prompt allowing user to specify a rename profile
        old_profile_name = self.profile_name
        new_profile_name, ok = QInputDialog.getText(self, _('Rename profile'),
                    _('Enter a new display name for this profile:'), text=old_profile_name)
        if not ok:
            # Operation cancelled
            return
        new_profile_name = unicode(new_profile_name).strip()
        if new_profile_name == old_profile_name:
            return
        # Verify it does not clash with any other profiles in the profile
        for profile_name in self.profiles.keys():
            if profile_name == old_profile_name:
                continue
            if profile_name.lower() == new_profile_name.lower():
                return error_dialog(self, _('Add failed'), _('A profile with the same name already exists'),
                                    show=True, show_copy_button=False)

        # As we are about to rename profile, persist the current profiles details if any
        self.persist_profile_config()
        self.profiles[new_profile_name] = self.profiles[old_profile_name]
#        if self.default_profile == old_profile_name:
#            self.default_profile = new_profile_name
        del self.profiles[old_profile_name]
        self.profile_name = new_profile_name
        # Now update the profiles combobox
        self.select_profile_combo.populate_combo(self.profiles, new_profile_name)
        self.refresh_current_profile_info()

    def delete_profile(self):
        if not self.profile_name:
            return
        if len(self.profiles) == 1:
            return error_dialog(self, _('Cannot delete'), _('You must have at least one profile'),
                                    show=True, show_copy_button=False)
        if not confirm(_("Do you want to delete the profile named '{0}'".format(self.profile_name)),
                        'reading_profile_delete_profile', self):
            return
        del self.profiles[self.profile_name]
#        if self.default_profile == self.profile_name:
#            self.default_profile = self.profiles.keys()[0]
        # Now update the profiles combobox
        self.select_profile_combo.populate_combo(self.profiles)
        self.refresh_current_profile_info()

    def refresh_current_profile_info(self):
        debug_print("ConfigWidget:refresh_current_profile_info - Start")
        # Get configuration for the selected profile
        self.profile_name = unicode(self.select_profile_combo.currentText()).strip()
        profile_map = get_profile_info(self.plugin_action.gui.current_db, self.profile_name)

        device_uuid = profile_map.get(KEY_FOR_DEVICE, None)

        column_prefs = profile_map.get(CUSTOM_COLUMNS_STORE_NAME, CUSTOM_COLUMNS_OPTIONS_DEFAULTS)
        current_Location_column  = column_prefs.get(KEY_CURRENT_LOCATION_CUSTOM_COLUMN)
        percent_read_column      = column_prefs.get(KEY_PERCENT_READ_CUSTOM_COLUMN)
        rating_column            = column_prefs.get(KEY_RATING_CUSTOM_COLUMN)
        last_read_column         = column_prefs.get(KEY_LAST_READ_CUSTOM_COLUMN)
#        debug_print("ConfigWidget:__init__ - current_Location_column=%s, percent_read_column=%s, rating_column=%s" % (current_Location_column, percent_read_column, rating_column))

        store_prefs = profile_map.get(STORE_OPTIONS_STORE_NAME, STORE_OPTIONS_DEFAULTS)
        store_on_connect         = store_prefs.get(KEY_STORE_ON_CONNECT)
        prompt_to_store          = store_prefs.get(KEY_PROMPT_TO_STORE)
        store_if_more_recent     = store_prefs.get(KEY_STORE_IF_MORE_RECENT)
        do_not_store_if_reopened = store_prefs.get(KEY_DO_NOT_STORE_IF_REOPENED)

        update_prefs = profile_map.get(UPDATE_OPTIONS_STORE_NAME, UPDATE_OPTIONS_DEFAULTS)
        do_check_for_firmware_updates = update_prefs.get(KEY_DO_UPDATE_CHECK)
        do_early_firmware_updates     = update_prefs.get(KEY_DO_EARLY_FIRMWARE_CHECK)
        self.update_check_last_time   = update_prefs.get(KEY_LAST_FIRMWARE_CHECK_TIME)

        backup_prefs = profile_map.get(BACKUP_OPTIONS_STORE_NAME, BACKUP_OPTIONS_DEFAULTS)
        do_daily_backup          = backup_prefs.get(KEY_DO_DAILY_BACKUP)
        backup_each_connection   = backup_prefs.get(KEY_BACKUP_EACH_CONNECTION)
        dest_directory           = backup_prefs.get(KEY_BACKUP_DEST_DIRECTORY)
        copies_to_keep           = backup_prefs.get(KEY_BACKUP_COPIES_TO_KEEP,)

        # Display profile configuration in the controls
        self.current_Location_combo.populate_combo(self.avail_text_columns, current_Location_column)
        self.percent_read_combo.populate_combo(self.avail_number_columns, percent_read_column)
        self.rating_combo.populate_combo(self.avail_rating_columns, rating_column)
        self.last_read_combo.populate_combo(self.avail_date_columns, last_read_column)

        self.device_combo.populate_combo(self.parent_dialog.get_devices_list(), device_uuid)
        self.store_on_connect_checkbox.setCheckState(Qt.Checked if store_on_connect else Qt.Unchecked)
        self.prompt_to_store_checkbox.setCheckState(Qt.Checked if prompt_to_store else Qt.Unchecked)
        self.prompt_to_store_checkbox.setEnabled(store_on_connect)
        self.store_if_more_recent_checkbox.setCheckState(Qt.Checked if store_if_more_recent else Qt.Unchecked)
        self.store_if_more_recent_checkbox.setEnabled(store_on_connect)
        self.do_not_store_if_reopened_checkbox.setCheckState(Qt.Checked if do_not_store_if_reopened else Qt.Unchecked)
        self.do_not_store_if_reopened_checkbox.setEnabled(store_on_connect)
        self.do_update_check.setCheckState(Qt.Checked if do_check_for_firmware_updates else Qt.Unchecked)
        self.do_early_firmware_check.setCheckState(Qt.Checked if do_early_firmware_updates else Qt.Unchecked)
        self.do_daily_backp_checkbox.setCheckState(Qt.Checked if do_daily_backup else Qt.Unchecked)
        self.backup_each_connection_checkbox.setCheckState(Qt.Checked if backup_each_connection else Qt.Unchecked)
        self.dest_directory_edit.setText(dest_directory)
        if copies_to_keep == -1:
            self.copies_to_keep_checkbox.setCheckState(not Qt.Checked)
        else:
            self.copies_to_keep_checkbox.setCheckState(Qt.Checked)
            self.copies_to_keep_spin.setProperty('value', copies_to_keep)
        debug_print("ConfigWidget:refresh_current_profile_info - do_daily_backup=%s, backup_each_connection=%s" % (do_daily_backup, backup_each_connection))
        if do_daily_backup:
            self.do_daily_backp_checkbox_clicked(do_daily_backup)
        if backup_each_connection:
            self.backup_each_connection_checkbox_clicked(backup_each_connection)
        debug_print("ConfigWidget:refresh_current_profile_info - end")

    def persist_profile_config(self):
        debug_print("ConfigWidget:persist_profile_config - Start")
        if not self.profile_name:
            return

        profile_config = self.profiles[self.profile_name]
        debug_print("ConfigWidget:persist_profile_config - profile_config:", profile_config)

        profile_config[KEY_FOR_DEVICE] = self.device_combo.get_selected_device()

        store_prefs = {}
        store_prefs[KEY_STORE_ON_CONNECT]         = self.store_on_connect_checkbox.checkState() == Qt.Checked
        store_prefs[KEY_PROMPT_TO_STORE]          = self.prompt_to_store_checkbox.checkState() == Qt.Checked
        store_prefs[KEY_STORE_IF_MORE_RECENT]     = self.store_if_more_recent_checkbox.checkState() == Qt.Checked
        store_prefs[KEY_DO_NOT_STORE_IF_REOPENED] = self.do_not_store_if_reopened_checkbox.checkState() == Qt.Checked
        profile_config[STORE_OPTIONS_STORE_NAME]  = store_prefs
        debug_print("ConfigWidget:persist_profile_config - store_prefs:", store_prefs)

        new_update_prefs = {}
        new_update_prefs[KEY_DO_UPDATE_CHECK]          = self.do_update_check.checkState() == Qt.Checked
        new_update_prefs[KEY_DO_EARLY_FIRMWARE_CHECK]  = self.do_early_firmware_check.checkState() == Qt.Checked
        new_update_prefs[KEY_LAST_FIRMWARE_CHECK_TIME] = self.update_check_last_time
        profile_config[UPDATE_OPTIONS_STORE_NAME]        = new_update_prefs
        debug_print("ConfigWidget:persist_profile_config - new_update_prefs:", new_update_prefs)

        backup_prefs = {}
        backup_prefs[KEY_DO_DAILY_BACKUP]       = self.do_daily_backp_checkbox.checkState() == Qt.Checked
        backup_prefs[KEY_BACKUP_EACH_CONNECTION]= self.backup_each_connection_checkbox.checkState() == Qt.Checked
        backup_prefs[KEY_BACKUP_DEST_DIRECTORY] = unicode(self.dest_directory_edit.text())
        backup_prefs[KEY_BACKUP_COPIES_TO_KEEP] = int(unicode(self.copies_to_keep_spin.value())) if self.copies_to_keep_checkbox.checkState() == Qt.Checked else -1 
        profile_config[BACKUP_OPTIONS_STORE_NAME] = backup_prefs
        debug_print("ConfigWidget:persist_profile_config - backup_prefs:", backup_prefs)

        column_prefs = {}
        column_prefs[KEY_CURRENT_LOCATION_CUSTOM_COLUMN] = self.current_Location_combo.get_selected_column()
        debug_print("ConfigWidget:persist_profile_config - column_prefs[KEY_CURRENT_LOCATION_CUSTOM_COLUMN]:", column_prefs[KEY_CURRENT_LOCATION_CUSTOM_COLUMN])
        column_prefs[KEY_PERCENT_READ_CUSTOM_COLUMN]     = self.percent_read_combo.get_selected_column()
        column_prefs[KEY_RATING_CUSTOM_COLUMN]           = self.rating_combo.get_selected_column()
        column_prefs[KEY_LAST_READ_CUSTOM_COLUMN]        = self.last_read_combo.get_selected_column()
        profile_config[CUSTOM_COLUMNS_STORE_NAME]        = column_prefs

        self.profiles[self.profile_name] = profile_config

        debug_print("ConfigWidget:persist_profile_config - end")

    def get_number_custom_columns(self):
        column_types = ['float','int']
        return self.get_custom_columns(column_types)

    def get_rating_custom_columns(self):
        column_types = ['rating','int']
        custom_columns = self.get_custom_columns(column_types)
        ratings_column_name = self.plugin_action.gui.library_view.model().orig_headers['rating']
        custom_columns['rating'] = {'name': ratings_column_name}
        return custom_columns

    def get_text_custom_columns(self):
        column_types = ['text']
        return self.get_custom_columns(column_types)

    def get_date_custom_columns(self):
        column_types = ['datetime']
        return self.get_custom_columns(column_types)

    def get_custom_columns(self, column_types):
        custom_columns = self.plugin_action.gui.library_view.model().custom_columns
        available_columns = {}
        for key, column in custom_columns.iteritems():
            typ = column['datatype']
            if typ in column_types and not column['is_multiple']:
                available_columns[key] = column
        return available_columns

    def _get_dest_directory_name(self):
        path = choose_dir(self, 'backup annotations destination dialog','Choose destination directory')
        if path:
            self.dest_directory_edit.setText(path)


class DevicesTab(QWidget):

    def __init__(self, parent_dialog, plugin_action):
        self.parent_dialog = parent_dialog
        QWidget.__init__(self)

        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self._connected_device_info = plugin_action.connected_device_info
        self.library_config = get_library_config(self.gui.current_db)

        layout = QVBoxLayout()
        self.setLayout(layout)

        # -------- Device configuration ---------
        devices_group_box = QGroupBox(_('Devices:'), self)
        layout.addWidget(devices_group_box)
        devices_group_box_layout = QVBoxLayout()
        devices_group_box.setLayout(devices_group_box_layout)

        self.devices_table = DevicesTableWidget(self)
        self.devices_table.itemSelectionChanged.connect(self._devices_table_item_selection_changed)
        devices_group_box_layout.addWidget(self.devices_table)

        buttons_layout = QHBoxLayout()
        devices_group_box_layout.addLayout(buttons_layout)

        self.add_device_btn = QPushButton(_('Add connected device'), self)
        self.add_device_btn.setToolTip(
                _('If you do not have a device connected currently, either plug one\n'
                'in now or exit the dialog and connect to folder/iTunes first'))
        self.add_device_btn.setIcon(QIcon(I('plus.png')))
        self.add_device_btn.clicked.connect(self._add_device_clicked)
        buttons_layout.addWidget(self.add_device_btn, 1)
        
        self.rename_device_btn = QtGui.QToolButton(self)
        self.rename_device_btn.setIcon(get_icon('edit-undo.png'))
        self.rename_device_btn.setToolTip(_('Rename the currently connected device'))
        self.rename_device_btn.clicked.connect(self._rename_device_clicked)
        self.rename_device_btn.setEnabled(False)
        buttons_layout.addWidget(self.rename_device_btn)
        
        self.delete_device_btn = QtGui.QToolButton(self)
        self.delete_device_btn.setIcon(QIcon(I('trash.png')))
        self.delete_device_btn.setToolTip(_('Delete this device from the device list'))
        self.delete_device_btn.clicked.connect(self._delete_device_clicked)
        self.delete_device_btn.setEnabled(False)
        buttons_layout.addWidget(self.delete_device_btn)

        layout.insertStretch(-1)

    def on_device_connection_changed(self, is_connected):
        if not is_connected:
            self._connected_device_info = None
            self.update_from_connection_status()

    def on_device_metadata_available(self):
        if self.plugin_action.have_kobo():
            self._connected_device_info = self.gui.device_manager.get_current_device_information().get('info', None)
            self.update_from_connection_status()

    def _devices_table_item_selection_changed(self):
        if len(self.devices_table.selectedIndexes()) > 0:
            self.rename_device_btn.setEnabled(True)
            self.delete_device_btn.setEnabled(True)
        else:
            self.rename_device_btn.setEnabled(False)
            self.delete_device_btn.setEnabled(False)

    def _add_device_clicked(self):
        devices = self.devices_table.get_data()
        drive_info = self._connected_device_info[4]
        for location_info in drive_info.values():
            if location_info['location_code'] == 'main':
                new_device = {}
                new_device['type'] = self._connected_device_info[0]
                new_device['active'] = True
                new_device['kindle_col'] = False
                new_device['uuid'] = location_info['device_store_uuid']
                new_device['name'] = location_info['device_name']
                new_device['location_code'] = location_info['location_code']
                new_device['serial_no'] = self.plugin_action.device_serial_no()
                devices[new_device['uuid']] = new_device

        self.devices_table.populate_table(devices, self._connected_device_info)
        self.update_from_connection_status(update_table=False)
        # Ensure the devices combo is refreshed for the current list
        self.parent_dialog.profiles_tab.refresh_current_profile_info()

    def _rename_device_clicked(self):
        (device_info, is_connected) = self.devices_table.get_selected_device_info()
        if not device_info:
            return error_dialog(self, _('Rename failed'), _('You must select a device first'),
                                show=True, show_copy_button=False)
        if not is_connected:
            return error_dialog(self, _('Rename failed'),
                                _('You can only rename a device that is currently connected'),
                                show=True, show_copy_button=False)

        old_name = device_info['name']
        new_device_name, ok = QInputDialog.getText(self, _('Rename device'),
                    _('Enter a new display name for this device:'), text=old_name)
        if not ok:
            # Operation cancelled
            return
        new_device_name = unicode(new_device_name).strip()
        if new_device_name == old_name:
            return
        try:
            self.gui.device_manager.set_driveinfo_name(device_info['location_code'], new_device_name)
            self.devices_table.set_current_row_device_name(new_device_name)
            # Ensure the devices combo is refreshed for the current list
            self.parent_dialog.profiles_tab.refresh_current_profile_info()
        except:
            return error_dialog(self, _('Rename failed'), _('An error occured while renaming.'),
                                det_msg=traceback.format_exc(), show=True)

    def _delete_device_clicked(self):
        (device_info, _is_connected) = self.devices_table.get_selected_device_info()
        if not device_info:
            return error_dialog(self, _('Delete failed'), _('You must select a device first'),
                                show=True, show_copy_button=False)
        name = device_info['name']
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('You are about to remove the <b>{0}</b> device from this list. '.format(name)) +
                _('Are you sure you want to continue?')):
            return
        self.parent_dialog.profiles_tab.persist_profile_config()
        self.devices_table.delete_selected_row()
        self.update_from_connection_status(update_table=False)

        # Ensure any lists are no longer associated with this device
        # NOTE: As of version 1.5 we can no longer do this since we only know the lists
        #       for the current library, not all libraries. So just reset this library
        #       and put some "self-healing" logic elsewhere to ensure a user loading a
        #       list for a deleted device in another library gets it reset at that point.
        self.parent_dialog.delete_device_from_lists(self.library_config, device_info['uuid'])
        # Ensure the devices combo is refreshed for the current list
        self.parent_dialog.profiles_tab.refresh_current_profile_info()

    def update_from_connection_status(self, first_time=False, update_table=True):
        if first_time:
            devices = plugin_prefs[STORE_DEVICES]
        else:
            devices = self.devices_table.get_data()

        if self._connected_device_info is None or not self.plugin_action.haveKobo():
            self.add_device_btn.setEnabled(False)
#            self.rename_device_btn.setEnabled(False)
        else:
            # Check to see whether we are connected to a device we already know about
            is_new_device = True
            can_rename = False
            drive_info = self._connected_device_info[4]
            if drive_info:
                # This is a non iTunes device that we can check to see if we have the UUID for
                device_uuid = drive_info['main']['device_store_uuid']
                if device_uuid in devices:
                    is_new_device = False
                    can_rename = True
            else:
                # This is a device without drive info like iTunes
                device_type = self._connected_device_info[0]
                if device_type in devices:
                    is_new_device = False
            self.add_device_btn.setEnabled(is_new_device)
#            self.rename_device_btn.setEnabled(can_rename)
        if update_table:
            self.devices_table.populate_table(devices, self._connected_device_info)


class DeviceColumnComboBox(QComboBox):

    def __init__(self, parent):
        QComboBox.__init__(self, parent)

    def populate_combo(self, devices, selected_device_uuid):
        self.clear()
        self.device_ids = [None, TOKEN_ANY_DEVICE]
        self.addItem('')
        self.addItem(TOKEN_ANY_DEVICE)
        selected_idx = 0
        if selected_device_uuid == TOKEN_ANY_DEVICE:
            selected_idx = 1
        for idx, key in enumerate(devices.keys()):
            self.addItem('%s'%(devices[key]['name']))
            self.device_ids.append(key)
            if key == selected_device_uuid:
                selected_idx = idx + 2
        self.setCurrentIndex(selected_idx)

    def get_selected_device(self):
        return self.device_ids[self.currentIndex()]


class ListComboBox(QComboBox):

    def __init__(self, parent, lists, selected_text=None):
        QComboBox.__init__(self, parent)
        self.populate_combo(lists, selected_text)

    def populate_combo(self, lists, selected_text=None):
        self.blockSignals(True)
        self.clear()
        for list_name in sorted(lists.keys()):
            self.addItem(list_name)
        self.select_view(selected_text)

    def select_view(self, selected_text):
        self.blockSignals(True)
        if selected_text:
            idx = self.findText(selected_text)
            self.setCurrentIndex(idx)
        elif self.count() > 0:
            self.setCurrentIndex(0)
        self.blockSignals(False)


class NoWheelComboBox(QComboBox):

    def wheelEvent (self, event):
        # Disable the mouse wheel on top of the combo box changing selection as plays havoc in a grid
        event.ignore()


class BoolColumnComboBox(NoWheelComboBox):

    def __init__(self, parent, selected=True):
        NoWheelComboBox.__init__(self, parent)
        self.populate_combo(selected)

    def populate_combo(self, selected):
        self.clear()
        self.addItem(QIcon(I('ok.png')), 'Y')
        self.addItem(QIcon(I('list_remove.png')), 'N')
        if selected:
            self.setCurrentIndex(0)
        else:
            self.setCurrentIndex(1)


class DevicesTableWidget(QTableWidget):

    def __init__(self, parent):
        QTableWidget.__init__(self, parent)
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        self.setMinimumSize(380, 0)

    def populate_table(self, devices, connected_device_info):
        self.clear()
        self.setRowCount(len(devices))
        header_labels = [ _('Menu'), _('Name'), _('Model'), _('Serial Number'),_('Status')]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(32)
        self.horizontalHeader().setStretchLastSection(False)
        self.setIconSize(QtCore.QSize(32, 32))

        for row, uuid in enumerate(devices.keys()):
            self.populate_table_row(row, uuid, devices[uuid], connected_device_info)

        self.resizeColumnsToContents()
        self.setMinimumColumnWidth(1, 100)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, uuid, device_config, connected_device_info):
        debug_print("DevicesTableWidget:populate_table_row - device_config:", device_config)
        device_type = device_config['type']
        device_uuid = device_config['uuid']
        device_icon = 'reader.png'
        is_connected = False
        if connected_device_info is not None:
            debug_print("DevicesTableWidget:populate_table_row - connected_device_info:", connected_device_info)
            if device_type == connected_device_info[0]:
                drive_info = connected_device_info[4]
                if not drive_info:
                    is_connected = False
                else:
                    for connected_info in drive_info.values():
                        if connected_info['device_store_uuid'] == device_uuid:
                            is_connected = True
                            break
        connected_icon = 'images/device_connected.png' if is_connected else None
        debug_print("DevicesTableWidget:populate_table_row - connected_icon=%s" % connected_icon)

        name_widget = ReadOnlyTextIconWidgetItem(device_config['name'], get_icon(device_icon))
        name_widget.setData(Qt.UserRole, (device_config, is_connected))
        type_widget = ReadOnlyTableWidgetItem(device_config['type'])
        serial_no = device_config.get('serial_no', '')
        serial_no_widget = ReadOnlyTableWidgetItem(serial_no)
        self.setItem(row, 0, CheckableTableWidgetItem(device_config['active']))
        self.setItem(row, 1, name_widget)
        self.setItem(row, 2, type_widget)
        self.setItem(row, 3, serial_no_widget)
        self.setItem(row, 4, ReadOnlyTextIconWidgetItem('', get_icon(connected_icon)))

    def get_data(self):
        debug_print("DevicesTableWidget::get_data - start")
        devices = {}
        for row in range(self.rowCount()):
            (device_config, _is_connected) = convert_qvariant(self.item(row, 1).data(Qt.UserRole))
#            debug_print("DevicesTableWidget::get_data - device_config", device_config)
#            debug_print("DevicesTableWidget::get_data - _is_connected", _is_connected)
            device_config['active'] = self.item(row, 0).get_boolean_value()
            w = self.cellWidget(row, 4)
            if w:
                device_config['collections'] = unicode(w.currentText()).strip() == 'Y'
            else:
                device_config['collections'] = False
            devices[device_config['uuid']] = device_config
#        debug_print("DevicesTableWidget::get_data - devices:", devices)
        return devices

    def get_selected_device_info(self):
        if self.currentRow() >= 0:
            (device_config, is_connected) = convert_qvariant(self.item(self.currentRow(), 1).data(Qt.UserRole))
            return (device_config, is_connected)
        return None, None

    def set_current_row_device_name(self, device_name):
        if self.currentRow() >= 0:
            widget = self.item(self.currentRow(), 1)
            (device_config, is_connected) = convert_qvariant(widget.data(Qt.UserRole))
            device_config['name'] = device_name
            widget.setData(Qt.UserRole, (device_config, is_connected))
            widget.setText(device_name)

    def delete_selected_row(self):
        if self.currentRow() >= 0:
            self.removeRow(self.currentRow())


class OtherTab(QWidget):

    def __init__(self, parent_dialog):
        self.parent_dialog = parent_dialog
        QWidget.__init__(self)
        layout = QVBoxLayout()
        self.setLayout(layout)

        other_options_group = QGroupBox(_('Other Options'), self)
        layout.addWidget(other_options_group )
        options_layout = QGridLayout()
        other_options_group.setLayout(options_layout)

        library_default_label = QLabel(_('&Library Button default:'), self)
        library_default_label.setToolTip(_('If plugin is placed as a toolbar button, choose a default action when clicked on'))
        self.library_default_combo = KeyComboBox(self, self.parent_dialog.plugin_action.library_actions_map, unicode(get_plugin_pref(COMMON_OPTIONS_STORE_NAME, KEY_BUTTON_ACTION_LIBRARY)))
        library_default_label.setBuddy(self.library_default_combo)
        options_layout.addWidget(library_default_label, 0, 0, 1, 1)
        options_layout.addWidget(self.library_default_combo, 0, 1, 1, 2)

        device_default_label = QLabel(_('&Device Button default:'), self)
        device_default_label.setToolTip(_('If plugin is placed as a toolbar button, choose a default action when clicked on'))
        self.device_default_combo = KeyComboBox(self, self.parent_dialog.plugin_action.device_actions_map, unicode(get_plugin_pref(COMMON_OPTIONS_STORE_NAME, KEY_BUTTON_ACTION_DEVICE)))
        device_default_label.setBuddy(self.device_default_combo)
        options_layout.addWidget(device_default_label, 1, 0, 1, 1)
        options_layout.addWidget(self.device_default_combo, 1, 1, 1, 2)

        keyboard_shortcuts_button = QPushButton(_('Keyboard shortcuts...'), self)
        keyboard_shortcuts_button.setToolTip(
                    _('Edit the keyboard shortcuts associated with this plugin'))
        keyboard_shortcuts_button.clicked.connect(parent_dialog.edit_shortcuts)
        layout.addWidget(keyboard_shortcuts_button)

        view_prefs_button = QPushButton(_('&View library preferences...'), self)
        view_prefs_button.setToolTip(_('View data stored in the library database for this plugin'))
        view_prefs_button.clicked.connect(parent_dialog.view_prefs)
        layout.addWidget(view_prefs_button)

        layout.insertStretch(-1)

    def persist_other_config(self):

        new_prefs = {}
        new_prefs[KEY_BUTTON_ACTION_DEVICE]     = unicode(self.device_default_combo.currentText())
        new_prefs[KEY_BUTTON_ACTION_LIBRARY]    = unicode(self.library_default_combo.currentText())
        plugin_prefs[COMMON_OPTIONS_STORE_NAME] = new_prefs


class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        layout = QVBoxLayout(self)
        self.setLayout(layout)
        self.help_anchor = "configuration"
        
        title_layout = ImageTitleLayout(self, 'images/icon.png', _('Kobo Utilities Options'))
        layout.addLayout(title_layout)

        tab_widget = QTabWidget(self)
        layout.addWidget(tab_widget)

        self.profiles_tab = ProfilesTab(self, plugin_action)
        self.devices_tab = DevicesTab(self, plugin_action)
        self.other_tab = OtherTab(self)
        tab_widget.addTab(self.profiles_tab, _('Profiles'))
        tab_widget.addTab(self.devices_tab, _('Devices'))
        tab_widget.addTab(self.other_tab, _('Other'))

        # Force an initial display of list information
        self.devices_tab.update_from_connection_status(first_time=True)
        self.profiles_tab.refresh_current_profile_info()

    def connect_signals(self):
        self.plugin_action.plugin_device_connection_changed.connect(self.devices_tab.on_device_connection_changed)
        self.plugin_action.plugin_device_metadata_available.connect(self.devices_tab.on_device_metadata_available)

    def disconnect_signals(self):
        self.plugin_action.plugin_device_connection_changed.disconnect()
        self.plugin_action.plugin_device_metadata_available.disconnect()

    def refresh_devices_dropdown(self):
        self.profiles_tab.refresh_current_profile_info()

    def get_devices_list(self):
        return self.devices_tab.devices_table.get_data()

    def delete_device_from_lists(self, library_config, device_uuid):
#        for list_info in library_config[KEY_PROFILES].itervalues():
#            if list_info[KEY_FOR_DEVICE] == device_uuid:
#                list_info[KEY_FOR_DEVICE] = DEFAULT_PROFILE_VALUES[KEY_FOR_DEVICE]
#                list_info[KEY_SYNC_AUTO] = DEFAULT_PROFILE_VALUES[KEY_SYNC_AUTO]
#                list_info[KEY_SYNC_CLEAR] = DEFAULT_PROFILE_VALUES[KEY_SYNC_CLEAR]
        set_library_config(self.plugin_action.gui.current_db, library_config)

    def save_settings(self):
        device_prefs = self.get_devices_list()
        plugin_prefs[STORE_DEVICES] = device_prefs

        # We only need to update the store for the current list, as switching lists
        # will have updated the other lists
        self.profiles_tab.persist_profile_config()
        self.other_tab.persist_other_config()

        library_config = self.profiles_tab.library_config
        library_config[KEY_PROFILES] = self.profiles_tab.profiles
#        library_config[KEY_DEFAULT_LIST] = self.profiles_tab.default_list
        set_library_config(self.plugin_action.gui.current_db, library_config)

#        options = {}
#        plugin_prefs[STORE_OPTIONS] = options

    def edit_shortcuts(self):
        self.save_settings()
        # Force the menus to be rebuilt immediately, so we have all our actions registered
        self.plugin_action.rebuild_menus()
        d = KeyboardConfigDialog(self.plugin_action.gui, self.plugin_action.action_spec[0])
        if d.exec_() == d.Accepted:
            self.plugin_action.gui.keyboard.finalize()

    def view_prefs(self):
        d = PrefsViewerDialog(self.plugin_action.gui, PREFS_NAMESPACE)
        d.exec_()

    def help_link_activated(self, url):
        self.plugin_action.show_help(anchor="configuration")

