﻿using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.Runtime.InteropServices.ComTypes;
using System.Threading.Tasks;
using EnvDTE;
using EnvDTE80;

namespace StartDebug
{
	class Program
	{
		static void Main(string[] args)
		{
			if (args.Length > 0)
			{
				string fileName = args[0];
				string processName = Path.GetFileNameWithoutExtension(fileName);
				string calibreDevelopFrom = null;
				string arguments = null;
				if (args.Length > 1)
				{
					arguments = "";
					for (int index = 1; index < args.Length; index++)
					{
						string arg = args[index];
						if (arg.StartsWith("CALIBRE_DEVELOP_FROM="))
						{
							// Setting the Calibre source directory.
							calibreDevelopFrom = arg.Substring(21);
						}
						else
						{
							if (arguments.Length > 0)
							{
								arguments += " ";
							}
							arguments += arg;
						}
					}
				}
				var process = new System.Diagnostics.Process();
				process.StartInfo.FileName = fileName;
				process.StartInfo.WorkingDirectory = Path.GetDirectoryName(fileName);
				if (arguments != null) process.StartInfo.Arguments = arguments;
				process.StartInfo.UseShellExecute = false;
				process.StartInfo.RedirectStandardError = true;
				process.StartInfo.RedirectStandardOutput = true;
				process.StartInfo.CreateNoWindow = true;
				process.StartInfo.WindowStyle = System.Diagnostics.ProcessWindowStyle.Hidden;
				if (calibreDevelopFrom != null) process.StartInfo.EnvironmentVariables.Add("CALIBRE_DEVELOP_FROM", calibreDevelopFrom);
				bool success = process.Start();
				string standardOutput = process.StandardOutput.ReadToEnd();
				string standardError = process.StandardError.ReadToEnd();
				EnvDTE80.DTE2 dte = (EnvDTE80.DTE2)FindTopDte();
				EnvDTE80.Debugger2 debugger = (EnvDTE80.Debugger2)dte.Debugger;
				EnvDTE80.Transport trans = debugger.Transports.Item("Default");
				//EnvDTE80.Engine nativeEngine = trans.Engines.Item("Native");
				EnvDTE80.Engine pythonEngine = trans.Engines.Item("Python");
				EnvDTE80.Engine[] engines = new[] { pythonEngine };
				foreach (EnvDTE80.Process2 proc in debugger.GetProcesses(trans, ""))
				{
					bool match = false;
					try
					{
						match = string.Compare(Path.GetFileNameWithoutExtension(proc.Name), processName, true) == 0;
					}
					catch (COMException ex)
					{
						Console.WriteLine(ex.Message);
					}
					if (match)
					{
						proc.Attach2(engines);
						break;
					}
				}
			}
			else
			{
				Console.WriteLine("Usage: calibreDebug \"<calibreExePath>\" [arguments] [\"CALIBRE_DEVELOP_FROM=<calibreSrcDirectory>\"]");
			}
		}

		[System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Naming", "CA1704:IdentifiersShouldBeSpelledCorrectly", MessageId = "Dte")]
		[CLSCompliant(false)]
		private static EnvDTE.DTE FindTopDte()
		{
			EnvDTE.DTE topDte = null;

			try
			{
				Dictionary<string, object> runningObjects = GetRunningObjects();

				// Find the DTE process whose main window is the topmost in the z-order. 
				// This will the currently "active" Visual Studio IDE.
				int topz = int.MaxValue;
				foreach (KeyValuePair<string, object> runningObject in runningObjects)
				{
					if (runningObject.Key.StartsWith("!VisualStudio.DTE", StringComparison.OrdinalIgnoreCase))
					{
						EnvDTE.DTE dte = runningObject.Value as EnvDTE.DTE;
						if (dte != null)
						{
							IntPtr handle = new IntPtr(dte.MainWindow.HWnd);
							int z = 0;
							do
							{
								z++;
								handle = GetWindow(handle, GetWindowCommand.GW_HWNDPREV);
							} while (handle != IntPtr.Zero);

							if (z < topz)
							{
								topDte = dte;
								topz = z;
							}
						}
					}
				}
			}
			catch (Exception ex)
			{
				System.Diagnostics.Debug.WriteLine(ex.ToString());
			}

			return topDte;
		}

		/// <summary>
		/// Get a snapshot of the running object table (ROT).
		/// </summary>
		/// <returns>
		/// A dictionary mapping the name of the object in the ROT to the corresponding object.
		/// </returns>
		private static Dictionary<string, object> GetRunningObjects()
		{
			Dictionary<string, object> runningObjects = new Dictionary<string, object>();

			IRunningObjectTable runningObjectTable;
			if (GetRunningObjectTable(0, out runningObjectTable) == 0)
			{
				IEnumMoniker monikerEnumerator;
				runningObjectTable.EnumRunning(out monikerEnumerator);
				monikerEnumerator.Reset();

				IMoniker[] monikers = new IMoniker[1];
				IntPtr numFetched = new IntPtr(0);

				while (monikerEnumerator.Next(1, monikers, numFetched) == 0)
				{
					IBindCtx ctx;
					if (CreateBindCtx(0, out ctx) == 0)
					{
						string runningObjectName;
						monikers[0].GetDisplayName(ctx, null, out runningObjectName);

						object runningObjectVal;
						if (runningObjectTable.GetObject(monikers[0], out runningObjectVal) == 0)
						{
							runningObjects[runningObjectName] = runningObjectVal;
						}
					}
				}
			}

			return runningObjects;
		}

		[DllImport("user32.dll", SetLastError = true)]
		public static extern IntPtr GetWindow(IntPtr hWnd, GetWindowCommand getWindowCommand);

		public enum GetWindowCommand : uint
		{
			GW_HWNDFIRST = 0,
			GW_HWNDLAST = 1,
			GW_HWNDNEXT = 2,
			GW_HWNDPREV = 3,
			GW_OWNER = 4,
			GW_CHILD = 5,
			GW_ENABLEDPOPUP = 6
		}

		[DllImport("ole32.dll")]
		public static extern int GetRunningObjectTable(int reserved, out IRunningObjectTable runningObjectTable);

		[DllImport("ole32.dll")]
		public static extern int CreateBindCtx(int reserved, out IBindCtx bindContext);
	}
}
