#!/usr/bin/env  calibre-debug

from calibre.ebooks.BeautifulSoup import BeautifulSoup
from datetime import datetime
import time

import subprocess
import logging
import sqlite3
import sys
import os

# Update Calibre Database
updateCalibre = True

# if annotations_column_name = '': search for custom_column with name annotations
# if cannot find custom column named annotations, search for annotations in Comments
annotations_column_name = ''

# Set TargetDir to '' or a directory present on your system or this script will fail
# When TargetDir is '' the output file is written to the directory where script is invoked
# If updateCalibre is True then the output file is deleted after being added to Calibre database
TargetDir = ''

# Extention for output file: using .clip for adding to Calibre Database  
# If you want to be able to open this file from within Calibre, 
# tell your OS to associate filetype with the text editor of your choice
# I have chosen not to use .txt to avoid conflict with other .txt files already in database
AnnotationsExtension = '.clip'

# used for logging and for prepending to output file
program_name = "Annotations"

# Set logfile to '' or something present on your system or this script will fail
# logfile = program_name + '.log'
logfile = ''

##### Methods #####

def write_file(data, filename):
   with open(filename, "wb") as file: file.write(data)
   return

def find_annotations_table(dbloc, annotations_column_name):
   conn = sqlite3.connect(dbloc)
   c = conn.cursor()

   # comments is standard Calibre column name, so return
   if annotations_column_name.lower() == 'comments': return annotations_column_name.lower()
   # if '' search for 'annotations' in custom columns
   custom_column_number = None
   query = "SELECT * FROM custom_columns"
   c.execute(query)
   results = c.fetchall()

   if annotations_column_name == '': annotations_column_name = 'annotations' 
   for i, row in enumerate(results):
      if row[2].lower() == annotations_column_name: 
         custom_column_number = row[0]
         logger.info('Found Custom Column %s with name %s', custom_column_number, annotations_column_name.upper())

   c.close()
   if custom_column_number == None: 
      logger.info('Will search for Annotations in Comments column')
      return 'comments'
   annotations_table = 'custom_column_' + str(custom_column_number)
   return annotations_table

def extract_annotations(dbloc, search_id, annotations_table):
   conn = sqlite3.connect(dbloc)
   c = conn.cursor()

   if search_id != None:
      query = 'select * from ' + annotations_table + ' where book = ' + str(dbid)
   else: query = 'select * from ' + annotations_table

   logger.debug('Query: %s', query)

   c.execute(query)
   booksWithAnnotations = c.fetchall()
   logger.info('Returned with %s Records', len(booksWithAnnotations))

   data = ''; a_counter = 0; ASR = {}
   # ASR is a dictionary to be reverse sorted by key so that latest is first
   for i, bookWithAnnotations in enumerate(booksWithAnnotations):
      write_annotation = False
      book_id = bookWithAnnotations[1]
      query = 'select title, author_sort from books where id = ' + str(book_id)
      logger.debug('Query: %s', query)
      c.execute(query)
      book = c.fetchone()
      title = book[0]
      author = book[1]

      logger.debug('Search id: %s Book_id %s', search_id, book_id)
      if search_id == book_id: 
         write_annotation = True
         logger.info('DIRECT HIT: ID')

      asoup = BeautifulSoup(bookWithAnnotations[2])
      annotations = asoup.findAll('div', attrs={'class':'annotation'})
      for a, annotation in enumerate(annotations):
         location = annotation.find('td', attrs={'class':'location'}).string
         timestamp = annotation.find('td', attrs={'class':'timestamp'})
         dt = datetime.fromtimestamp(float(timestamp['uts']))
         timedisplay = dt.strftime('%A, %B %d, %Y %H:%M:%S %p')
         if annotation.p: highlight = annotation.p.string
         else: highlight = ''

         if search_id == None or write_annotation == True:
            a_counter += 1
            new_annotation  = title + " (" + author + ")" + "\n"
            new_annotation += "- Your Highlight on Page | Location " + location + "| Added on " + timedisplay + "\n"
            new_annotation += "\n" + highlight + "\n"
            new_annotation += "==========" + "\n"
            ASR[timestamp['uts']] = new_annotation

   for key in sorted(ASR, reverse=True):
      data += ASR[key]

   c.close()

   logger.info('Writing %s Annotations', a_counter)
   return data


###### Main Body of Program #########

logstat = False
if '/' in logfile:
   logdir = logfile.rsplit('/',1)[0]
   logstat = os.path.isdir(logdir)

# if writing to TargetDir prepend to output file name
annotations_file = program_name.upper() 

# set up logging
logger = logging.getLogger(program_name)
logger.setLevel(logging.DEBUG)
formatter = logging.Formatter('%(name)-6s - %(levelname)-5s - %(message)s')
ch = logging.StreamHandler()
ch.setLevel(logging.INFO)
ch.setFormatter(formatter)
logger.addHandler(ch)

if logfile != '' and logstat:
   logger.info('Writing DEBUG log to %s', logfile)
   fh = logging.FileHandler(logfile) 
   fh.setLevel(logging.DEBUG) 
   fh.setFormatter(formatter)
   logger.addHandler(fh)

logger.info('##### Started Annotation Extraction #####')

# Process Command Line Arguments
# If there is no os.path.dirname then assume we are running locally
local = True; fullExport = True
if len(sys.argv) > 1:
   book = sys.argv[1]
   logger.info('Found Book Name: %s', book)
   try:
      target_dir = os.path.dirname(book)
      logger.info('path basename is: %s', target_dir)
      os.chdir(target_dir)
      local = False; fullExport = False
   except:
      e = sys.exc_info()
      logger.debug('Error: %s', e)
      progname = os.path.basename(__file__)
      logger.info('%s is running locally', progname)

cwdir = os.getcwd()
logger.info('Current working directory is: %s', cwdir)
dirparts = cwdir.split('/')
testdir = '/'.join(dirparts[:(len(dirparts) - 2)])

# determine where to find database and how to name output file
dbloc = cwdir + '/metadata.db'
if local and os.path.isfile(dbloc):
   annotations_file += '-' + dirparts[len(dirparts)-1] + AnnotationsExtension
   logger.info('Database Exists!')
elif len(dirparts) > 2: 
   local = False; fullExport = False
   logger.info('Splitting PATH to find metadata.db')
   dbdir = '/'.join(dirparts[:(len(dirparts) - 2)])
   dbloc = dbdir + '/metadata.db'
   annotations_file  += '-' + dirparts[len(dirparts)-1][:30].rsplit(' ',1)[0] + '-' + dirparts[len(dirparts)-2][:20] + AnnotationsExtension
   if os.path.isfile(dbloc): 
      logger.info('Database Exists!')
   else:
      logger.info('Calibre Database: %s NOT FOUND', dbloc)
      logger.info('##### Terminating Annotation Extraction #####')
      sys.exit()
else:
   logger.info('Calibre Database: %s NOT FOUND', dbloc)
   logger.info('##### Terminating Annotation Extraction #####')
   sys.exit()

# Check if database is where we think it is:
logger.info('Checking for Calibre Database: %s', dbloc)

# if dbid is None then export all annotations found in annotations_table
dbid = None
# attempt to determine book id from Calibre directory structure
if not fullExport:
   idstr = cwdir.rsplit(' ',1)[1]
   dbid = int(''.join(ele for ele in idstr if ele.isdigit()))

   # Check if file matching variable book is present
   logger.info('Checking Book File: %s', book)
   if os.path.isfile(book):
      logger.info('Book Exists!')
   else:
      logger.info('Book File: %s NOT FOUND', book)

   # necessary because when run from Automator book contains fully qualified path names
   book_basename = os.path.basename(book)
   logger.info('Book Basename: %s', book_basename)

annotations_table = find_annotations_table(dbloc, annotations_column_name)
logger.info('Looking for Annotations in table: %s', annotations_table)
data = extract_annotations(dbloc, dbid, annotations_table.lower())

if data == '': 
   logger.info('##### No Annotations Found')
   logger.info('##### Terminating Annotation Extraction #####')
   sys.exit()

# if a TargetDir is set, then write to that directory, don't update Calibre Database
if TargetDir != '' and os.path.isdir(TargetDir):
   os.chdir(TargetDir)
   updateCalibre = False

write_file(data, annotations_file)

# we have successfully exported annotations; proceed with updating calibre database  
if updateCalibre and dbid:
   command_args = ["calibredb", "add_format", "--dont-replace", str(dbid), annotations_file]
   logger.debug('Command Args: %s', command_args)
   logger.info('Upating Calibre Book ID: %s', dbid)
   output = subprocess.check_output(command_args)
   logger.debug(output)
   os.remove(annotations_file)

logger.info('##### Completed Annotation Extraction #####')

###########
