# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (division, absolute_import, print_function)

__license__   = 'GPL v3'
__docformat__ = 'restructuredtext en'

import os, zlib, zipfile, codecs, re
from mobiml2xhtml import MobiMLConverter

def sanitizeFilename(filename):
    import string, unicodedata
    validFilenameChars = "-_.() %s%s" % (string.ascii_letters, string.digits)
    cleanedFilename = unicodedata.normalize('NFKD', filename).encode('ASCII', 'ignore')
    return ''.join(c for c in cleanedFilename if c in validFilenameChars)


class ZipInfo(zipfile.ZipInfo):
    def __init__(self, *args, **kwargs):
        if 'compress_type' in kwargs:
            compress_type = kwargs.pop('compress_type')
        super(ZipInfo, self).__init__(*args, **kwargs)
        self.compress_type = compress_type

class QuickEpub(object):
    def __init__(self, outdir, htmlfile, opffile, basename ):
        self.outdir, self.htmlfile, self.opffile, self.basename = outdir, htmlfile, opffile, basename
        #self.epubname = os.path.join(outdir, basename + '.epub')
        self.epubname = ''
        self.metainf = os.path.join(outdir,'META-INF')
        if not os.path.exists(self.metainf):
            os.mkdir(self.metainf)
        
        container = '<?xml version="1.0" encoding="UTF-8"?>\n'
        container += '<container version="1.0" xmlns="urn:oasis:names:tc:opendocument:xmlns:container">\n'
        container += '    <rootfiles>\n'
        container += '<rootfile full-path="{0}" media-type="application/oebps-package+xml"/>'.format(os.path.basename(self.opffile))
        container += '    </rootfiles>\n</container>\n'
        fileout = os.path.join(self.metainf,'container.xml')
        open(fileout,'wb').write(container)
        
    # recursive zip creation support routine
    def zipUpDir(self, myzip, tdir, localname):
        currentdir = tdir
        if localname != "":
            currentdir = os.path.join(currentdir,localname)
        list = os.listdir(currentdir)
        for file in list:
            afilename = file
            localfilePath = os.path.join(localname, afilename)
            realfilePath = os.path.join(currentdir,file)
            if os.path.isfile(realfilePath):
                myzip.write(realfilePath, localfilePath, zipfile.ZIP_DEFLATED)
            elif os.path.isdir(realfilePath):
                self.zipUpDir(myzip, tdir, localfilePath)
        
    def makeEPUB(self):
        ml2html = MobiMLConverter(self.htmlfile)
        xhtmlstr, css, cssname = ml2html.processml()
        title = ''
        open(self.htmlfile,'wb').write(xhtmlstr)
        open(cssname,'wb').write(css)
        
        with codecs.open(self.opffile, 'r', 'utf-8') as fp:
            newopf = ''
            for line in fp:
                #if line.startswith('<item') and line.find('text/x-oeb1-document'):
                #    line = line.replace('text/x-oeb1-document', 'application/xhtml+xml' )
                newopf += line
                if line.startswith('<manifest>'):
                    newopf += '<item id="css_file" media-type="text/css" href="styles.css" />\n'
                if line.startswith('<dc:title>'):
                    m = re.search('<dc:title>([^<]+)</dc:title>', line)
                    if m.groups() > 0:
                        title = m.group(1)
                if line.startswith('<meta name="Updated_Title"') and not len(title):
                    m = re.search('content="(.*?)"', line)
                    if m.groups() > 0:
                        title = m.group(1)

        codecs.open(self.opffile,'w', 'utf-8').write(newopf)
        
        if len(title):
            candidate = sanitizeFilename(title)
            if len(candidate):
                self.epubname = os.path.join(self.outdir, candidate + '.epub')
                
        if not len(self.epubname):
            candidate = sanitizeFilename(self.basename)
            if len(candidate):
                self.epubname = os.path.join(self.outdir, candidate + '.epub')
            else:
                self.epubname = os.path.join(self.outdir, 'new.epub')
            
        
        outzip = zipfile.ZipFile(self.epubname, 'w')
        
        # add the mimetype file uncompressed
        mimetype = 'application/epub+zip'
        fileout = os.path.join(self.outdir,'mimetype')
        open(fileout,'wb').write(mimetype)
        nzinfo = ZipInfo('mimetype', compress_type=zipfile.ZIP_STORED)
        outzip.writestr(nzinfo, mimetype)
        
        
        self.zipUpDir(outzip,self.outdir,'META-INF')
        if os.path.exists(os.path.join(self.outdir,'Images')):
            self.zipUpDir(outzip,self.outdir,'Images')
            
        outzip.write(self.htmlfile, os.path.basename(self.htmlfile), zipfile.ZIP_DEFLATED)
        outzip.write(self.opffile, os.path.basename(self.opffile), zipfile.ZIP_DEFLATED)
        outzip.write(cssname, 'styles.css', zipfile.ZIP_DEFLATED)
        print (cssname)
        if os.path.exists(os.path.join(self.outdir, 'toc.ncx')):
            outzip.write(os.path.join(self.outdir, 'toc.ncx'), 'toc.ncx', zipfile.ZIP_DEFLATED)
        outzip.close()
        return self.epubname
            

