#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2014, Stan King <stanking2@gmail.com>'
__docformat__ = 'restructuredtext en'

from calibre.ebooks.metadata import check_isbn10
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import error_dialog, info_dialog
from calibre.gui2.actions import InterfaceAction

import calibre_plugins.polish_isbn.config as cfg
from calibre_plugins.polish_isbn.common_utils import set_plugin_icon_resources, get_icon

PLUGIN_ICONS = ['images/polish_isbn.png']

class PolishISBNAction(InterfaceAction):

    name = 'Polish ISBN'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Polish ISBN', None, 'Validate ISBN and update to ISBN-13', ())
    action_type = 'current'

    def genesis(self):
        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        # Assign our menu to this action and an icon
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.pull_current_isbn) 

    def pull_current_isbn(self):
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, 'No rows selected',
                                'You must select one or more books to perform this action.', show=True)
        book_ids = self.gui.library_view.get_selected_ids()
        db = self.gui.current_db.new_api
        for i in book_ids:
            mi = db.get_metadata(i, False, True, False)
            nuisbn = self.convert_10to13(mi.isbn)
            if nuisbn is None:
                nuisbn = ""
            else:
                mi.set_identifier('isbn', nuisbn)
                id_map = {}
                id_map[i] = mi
                edit_metadata_action = self.gui.iactions['Edit Metadata']
                edit_metadata_action.apply_metadata_changes(id_map,
                                                        callback=self._mark_and_display_results)
        return
    
    def convert_10to13(self, isbn10): #takes an ISBN-10 as input string, returns an ISBN-13
        isbn13 = None
        if check_isbn10(isbn10) is None:
            isbn13 = None
            # insert code to suggest correct isbn?
            # add record to the log
        else:
            fourth = int(isbn10[0])*3
            fifth = int(isbn10[1])
            sixth = int(isbn10[2])*3
            seventh = int(isbn10[3])
            eighth = int(isbn10[4])*3
            ninth = int(isbn10[5])
            tenth = int(isbn10[6])*3
            eleventh = int(isbn10[7])
            twelfth = int(isbn10[8])*3
            thirteenth = 38 + fourth + fifth + sixth + seventh + eighth + ninth + tenth + eleventh + twelfth
            thirteenth %= 10
            last = str(10 - thirteenth)
            isbn13 = '978' + isbn10[0:9] + last[-1]
        return isbn13

    def _mark_and_display_results(self, applied_ids):
        #Stan: remove code to mark updates until all updates can be marked simultaneously
        marked_ids = {}
        for book_id in applied_ids:
            marked_ids[book_id] = 'isbn_updated'
        self.gui.current_db.set_marked_ids(marked_ids)
        action = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.KEY_POST_TASK, 'none')
        if action == 'updated' and len(applied_ids) > 0:
            self.gui.search.set_search_string('marked:isbn_updated')
