#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

# target script

import os, sys
import ConfigParser, inspect, ntpath
from contextlib import contextmanager
from Tkinter import *
import Tkconstants, tkFileDialog


def path_leaf(path):
    head, tail = ntpath.split(path)
    return tail or ntpath.basename(head)

SCRIPT_NAME = path_leaf(os.path.normpath(inspect.getfile(inspect.currentframe())))
SCRIPT_DIR = os.path.normpath(os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe()))))
PROGNAME = os.path.splitext(SCRIPT_NAME)[0]
CONFIGFILE = os.path.join(SCRIPT_DIR, '{0}.ini'.format(PROGNAME))

@contextmanager
def make_temp_directory():
    import tempfile, shutil
    temp_dir = tempfile.mkdtemp()
    yield temp_dir
    shutil.rmtree(temp_dir)

class guiMain(Frame):
    def __init__(self, parent, bk):
        Frame.__init__(self, parent, border=5)
        self.parent = parent
        self.bk = bk
        self.inpath = ''
        self.prefs = {}
       
        try:
            config = ConfigParser.RawConfigParser()
            config.read(CONFIGFILE)
            self.prefs['useFilePath'] = config.get('Defaults', 'use_file_path')
            self.prefs['windowGeometry'] = config.get('Geometry', 'window_geometry')
        except:
            self.prefs['useFilePath'] = os.path.expanduser('~')
            self.prefs['windowGeometry'] = None
            
        self.initUI()
        parent.protocol('WM_DELETE_WINDOW', self.quitApp)
        
    def initUI(self):
        self.parent.title("Choose Kindlebook")
        body = Frame(self)
        body.pack(fill=Tkconstants.BOTH, expand=True)
        
        entry_frame1 = Frame(body, bd=1)
        self.cust_file_path = Entry(entry_frame1)
        self.choose_button = Button(entry_frame1, text='Select Kindlebook', command=self.fileChooser)
        self.choose_button.pack(side=Tkconstants.TOP, fill=Tkconstants.BOTH)
        entry_frame1.pack(side=Tkconstants.TOP, fill=Tkconstants.BOTH)
        
        if self.prefs['windowGeometry'] is None:
            self.parent.update()
            w = self.parent.winfo_screenwidth()
            h = self.parent.winfo_screenheight()
            rootsize = tuple(int(_) for _ in self.parent.geometry().split('+')[0].split('x'))
            x = w/2 - rootsize[0]/2
            y = h/2 - rootsize[1]/2
            self.parent.geometry('%dx%d+%d+%d' % (rootsize + (x, y)))
        else:
            self.parent.geometry(self.prefs['windowGeometry'])

    def fileChooser(self):
        file_opt = {}
        file_opt['parent'] = None
        file_opt['title']= 'Select Kindlebook file'
        file_opt['defaultextension'] = '.azw3'
        file_opt['initialdir'] = self.prefs['useFilePath']
        file_opt['multiple'] = False
        file_opt['filetypes'] = [('Kindlebooks', ('.azw3', '.prc', '.mobi'))]
        self.inpath = tkFileDialog.askopenfilename(**file_opt)
        self.doit()
        
    def doit(self):
        from mobi_stuff import mobiProcessor, topaz
        if self.inpath == '' or not os.path.exists(self.inpath):
            raise Exception('No input file selected!')
        print ('Path to Kindlebook {0}'.format(self.inpath))
        if topaz(self.inpath):
            raise Exception('Kindlebook is in Topaz format: can\'t open!')
        mp = mobiProcessor(self.inpath)
        if mp.isEncrypted:
            raise Exception('Kindlebook is encrypted: can\'t open!')
        if mp.isPrintReplica:
            raise Exception('Kindlebook is a Print Replica: can\'t open!')
        if not mp.hasKF8:
            print ('Kindlebook has no KF8 section: maybe this will work later.')
            self.quitApp()
        with make_temp_directory() as temp_dir:
            epub = mp.unpackEPUB(temp_dir)
            print ('Path to new epub {0}'.format(epub))
            with open(epub,'rb')as fp:
                data = fp.read()
            self.bk.addotherfile('new.epub', data)
        self.quitApp()

    def quitApp(self):
        config = ConfigParser.RawConfigParser()
        config.add_section('Defaults')
        config.add_section('Geometry')
        config.set('Geometry', 'window_geometry', self.parent.geometry())
        if self.inpath != '':
            config.set('Defaults', 'use_file_path', os.path.dirname(self.inpath))
            INFILE = self.inpath
        try:
            with open(CONFIGFILE, 'wb') as configfile:
                config.write(configfile)
        except:
            pass
        self.parent.destroy()
        self.quit()

def run(bk):
    
    print ('Entering KindleUnpack input plugin\'s run() routine')
    
    root = Tk()
    root.title('')
    root.resizable(True, True)
    guiMain(root, bk).pack(fill=Tkconstants.BOTH, expand=False)
    root.mainloop()
 
    return 0
 
def main():
    print ('I reached main when I should not have\n')
    return -1
    
if __name__ == "__main__":
    sys.exit(main())

