#!/usr/bin/python3

import sys
import mmap
import os
import kobo_common
kc = kobo_common

def main():
    log_path = os.path.join(kc.installer_log_path, 'kobo_patch.log')
    logger = kc.KoboDebugger(log_path, stream=sys.stdout).logger
    
    actions_add = {
        'add-after': 'add-after', 
        'add-before': 'add-before'
    }
    
    actions = {
        'append': 'append'
    }
    
    actions.update(actions_add)
    add_values = actions_add.values()
    
    nargs = len(sys.argv) - 1
    debugf = None
    scriptname = os.path.basename(sys.argv[0])
    
    usage = '''
********************************************************************************
Usage: 
 {0} {append} PROGNAME FILEPATH CODE
 {0} {add-after} PROGNAME FILEPATH SEARCH_STRING CODE
 {0} {add-before} PROGNAME FILEPATH SEARCH_STRING CODE

"{append}" action appends CODE at the end of FILEPATH.

With "{add-after}" and "{add-before}", CODE is added after or before 
SEARCH_STRING in FILEPATH.

FILEPATH will be backed up to FILEPATH_before_PROGNAME before any change will 
be applied. No patch will be performed if FILEPATH is already patched.
If FILEPATH will be successfully patched and {0} is invoked by Kobo Installer,
system will be rebooted.
********************************************************************************
'''.format(scriptname, **actions)
    
    nargs_err = False
    nargs_expected = 0
    
    if nargs < 1:
        nargs_err = True
    else:
        action = sys.argv[1]
        
        if action not in actions.values():
            logger.error('Unknown action "{0}"'.format(action))
            print(usage)
            sys.exit(2)
        
        if action == actions['append']:
            nargs_expected = 4
        elif action in add_values:
            nargs_expected = 5

        if nargs != nargs_expected:
            nargs_err = True
    
    if nargs_err:
        errstr = 'Bad number of arguments'
        if nargs_expected:
            errstr += ', expected: {0}, found: {1}'.format(nargs_expected, 
                                                           nargs)
        logger.error(errstr)
        print(usage)
        sys.exit(2)
    
    progname = sys.argv[2]
    
    filepath = sys.argv[3]
    
    if action == actions['append']:
        newstr = sys.argv[4]
    elif action in add_values:
        searchb = bytes(sys.argv[4], 'utf8')
        newb = bytes(sys.argv[5], 'utf8')
    
    cp_err_str = 'Failed to backup "{0}"'.format(filepath)
    cpstr = 'cp "{0}" "{0}_before_{1}" 2>/dev/null'
    
    try:
        # python doesn't handle all metadata
        cp_err = os.system(cpstr.format(filepath, progname))
    except OSError:
        logger.warning(cp_err_str, exc_info=True)
        cp_err = 0
    
    if cp_err:
        logger.warning(cp_err_str)
    
    if action == actions['append']:
        with open(filepath, 'a') as f:
            if not newstr.startswith(os.linesep):
                newstr = os.linesep + newstr
            
            f.write(newstr)
    elif action in add_values:
        with open(filepath, 'r+') as f, mmap.mmap(f.fileno(), 0) as mm:
            pos = mm.find(searchb)
            
            if pos < 0:
                errstr = 'String "{0}" not found in file {1}'
                logger.error(errstr.format(searchb.decode(), filepath))
                sys.exit(1)
            
            if action == actions['add-after']:
                allb = searchb + newb
            elif action == actions['add-before']:
                allb = newb + searchb
            
            if mm.find(allb) >= 0:
                errstr = 'String "{0}" already present in file "{1}"'
                logger.warning(errstr.format(allb.decode(), filepath))
                sys.exit(0)
            
            newlen = len(newb)
            searchlen = len(searchb)
            addpos = pos
            mvpos = pos
            mvnewpos = pos + newlen
            oldsize = mm.size()
            mvlen = oldsize - pos
            
            if action == actions['add-after']:
                addpos += searchlen
                mvpos += searchlen
                mvnewpos += searchlen
                mvlen -= searchlen
            
            mm.resize(oldsize + newlen)
            mm.move(mvnewpos, mvpos, mvlen)
            mm.seek(addpos)
            mm.write(newb)
    
    with open(kc.reboot_path, mode='w') as dbgf:
        dbgf.write('1')

if __name__ == '__main__':
    main()

