#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2013, Pr.BarnArt' 
__docformat__ = 'restructuredtext en'

import time
import sys
from urllib import quote
from Queue import Queue, Empty

from lxml.html import fromstring, tostring

from calibre import as_unicode
from calibre.ebooks.metadata import check_isbn
from calibre.ebooks.metadata.sources.base import Source
from calibre.utils.icu import lower
from calibre.utils.cleantext import clean_ascii_chars
from calibre.utils.localization import get_udc
#from calibre import ipython



class INMONDADORI(Source):

    name                    = 'INMONDADORI'
    description             = _('Get book details from inmondadori.com book page in a separate thread')
    author                  = 'Pr. BarnArt'
    version                 = (0, 2, 1)
    minimum_calibre_version = (0, 8, 5)

    capabilities = frozenset(['identify', 'cover'])
    touched_fields = frozenset(['title', 'authors', 'identifier:isbn', 'rating', 'comments', 'publisher', 'pubdate'])
	
    has_html_comments = True
	supports_gzip_transfer_encoding = True
	cached_cover_url_is_reliable = True
     

	BASE_URL = 'http://www.inmondadori.it/search/?tpr=10&g='
	BASE_URL0= 'http://www.inmondadori.it'
	#BASE_URL0 ='http://www.nmondadori.com'
	#BASE_URL = 'http://www.nmondadori.com/search?keyword='
	#BASE_ISBN= '/Nty/1/search/true/searchType/adv/section/books/N/8293/Ntk/isbncode/index.html?_requestid=116710'
	#BASE_URL_1 = '/Ntk/nl_books_all/Nty/1/N/8299+8293/Ne/8299+8293'
	BASE_URL_LAST = '&swe=N&search-input=active' 
	
    def config_widget(self):
        '''
        Overriding the default configuration screen for our own custom configuration
        '''
        from calibre_plugins.INMONDADORI.config import ConfigWidget
        return ConfigWidget(self)

    def get_book_url(self, identifiers):
        inmondadori_id = identifiers.get('inmondadori', None)
		if inmondadori_id:
		    return ('inmondadori', inmondadori_id,
                  '%s%s%s'%(INMONDADORI.BASE_URL, inmondadori_id))
				  

    def create_query(self, log, title=None, authors=None, identifiers={}):

        isbn = check_isbn(identifiers.get('isbn', None))		
        q = ''
        if isbn is not None:		    
            return '%s%s'%(INMONDADORI.BASE_URL,isbn,INMONDADORI.BASE_URL_LAST)
        if title:		   
            # inmondadori  maybe doesn't cope very well with non ascii names so convert
            title = get_udc().decode(title)
            title_tokens = list(self.get_title_tokens(title,
                                strip_joiners=False, strip_subtitle=True))
            if title_tokens:
                tokens = [quote(t.encode('utf-8') if isinstance(t, unicode) else t) for t in title_tokens]               
				q='+'.join(tokens)
				
			
        if authors:
            # anobi  maybe doesn't cope very well with non ascii names so convert
            authors = [get_udc().decode(a) for a in authors]
            author_tokens = self.get_author_tokens(authors,
                    only_first_author=True)
            if author_tokens:
                tokens = [quote(t.encode('utf-8') if isinstance(t, unicode) else t) for t in author_tokens]
				q+='+'+'+'.join(tokens)
			
        if not q:
            return None
		#print("query:",'%s%s%s'%(INMONDADORI.BASE_URL, q,INMONDADORI.BASE_URL_LAST ))
        return '%s%s%s'%(INMONDADORI.BASE_URL, q,INMONDADORI.BASE_URL_LAST ) 

    def get_cached_cover_url(self, identifiers):
        url = None
        inmondadori_id = identifiers.get('inmondadori', None)
        if inmondadori_id is None:
            isbn = identifiers.get('isbn', None)
            if isbn is not None:
               inmondadori_id = self.cached_isbn_to_identifier(isbn)
        if inmondadori_id is not None:
            url = self.cached_identifier_to_cover_url(inmondadori_id)
        return url

    def cached_identifier_to_cover_url(self, id_):
        with self.cache_lock:
            url = self._get_cached_identifier_to_cover_url(id_)
            if not url:
                # Try for a "small" image in the cache
                url = self._get_cached_identifier_to_cover_url('small/'+id_)
            return url

    def _get_cached_identifier_to_cover_url(self, id_):
        # This must only be called once we have the cache lock
        url = self._identifier_to_cover_url_cache.get(id_, None)
        if not url:
            # We could not get a url for this particular B&N id
            # However we might have one for a different isbn for this book
            # Barnes & Noble are not very consistent with their covers and
            # it could be that the particular ISBN we chose does not have
            # a large image but another ISBN we retrieved does.
            key_prefix = id_.rpartition('/')[0]
            for key in self._identifier_to_cover_url_cache.keys():
                if key.startswith('key_prefix'):
                    return self._identifier_to_cover_url_cache[key]
        return url

    def identify(self, log, result_queue, abort, title=None, authors=None,
            identifiers={}, timeout=30):
        '''
        Note this method will retry without identifiers automatically if no
        match is found with identifiers.
        '''
        matches = []
		#need to read the ratings of an indexpage, i cannot trace them on the productpage
		#use ratings for the selected matches, so the order of rating is equal to matches 
		ratings = []
	    inmondadori_id = identifiers.get('inmondadori', None)
	    isbn = check_isbn(identifiers.get('isbn', None))		
        br = self.browser

        if inmondadori_id:		    
            matches.append('LAST')
        else:	
            query = self.create_query(log, title=title, authors=authors,
                    identifiers=identifiers)	
            if query is None:
                log.error('Insufficient metadata to construct query')
                return
            isbn_match_failed = False
            log.info('Querying: %s'%query)
            response = br.open_novisit(query, timeout=timeout)
            if isbn:
			    try:				    
                    raw = response.read().strip()
                    raw = raw.decode('utf-8', errors='replace')
                    if not raw:
                        log.error('Failed to get raw result for query: %r'%query)
                        return
                    root = fromstring(clean_ascii_chars(raw))
                except:
                    msg = 'Failed to parse inmondadori.it page for query: %r'%query
                    log.exception(msg)
                    return msg
                # Now grab the matches from the search results, provided the
                # title and authors appear to be for the same book
			    self._parse_search_results(log, title, authors, root, matches,ratings,timeout)			
			
            # For successful ISBN based searches we have already done everything we need to
            # So anything from this point below is for title/author based searches.
			if not isbn:
                try:				    
                    raw = response.read().strip()
                    raw = raw.decode('utf-8', errors='replace')
                    if not raw:
                        log.error('Failed to get raw result for query: %r'%query)
                        return
                    root = fromstring(clean_ascii_chars(raw))
                except:
                    msg = 'Failed to parse inmondadori.it page for query: %r'%query
                    log.exception(msg)
                    return msg
                # Now grab the matches from the search results, provided the
                # title and authors appear to be for the same book
                self._parse_search_results(log, title, authors, root, matches,ratings,timeout)

        if abort.is_set():
            return
		
        if not matches:		   
            if identifiers and title and authors:
                log.info('No matches found with identifiers, retrying using only'
                        ' title and authors')
                return self.identify(log, result_queue, abort, title=title,
                        authors=authors, timeout=timeout)
            log.error('No matches found with query: %r'%query)
            return

		
				
        from calibre_plugins.INMONDADORI.worker import Worker
		#need to mix url and ratings for the worker class
		combos= zip(matches,ratings)
        workers = [Worker(combo, result_queue, br, log, i, self) for  i,combo in
                enumerate(combos)]
		

        for w in workers:
            w.start()
            # Don't send all requests at the same time
            time.sleep(0.1)

        while not abort.is_set():
            a_worker_is_alive = False
            for w in workers:
                w.join(0.2)
                if abort.is_set():
                    break
                if w.is_alive():
                    a_worker_is_alive = True
            if not a_worker_is_alive:
                break

        return None

    def _parse_search_results(self, log, orig_title, orig_authors, root, matches, ratings, timeout):
        results = root.xpath('//div [@class="single-box"]/div [@class="product-info"]/h3  [@class="title"]/a')

		for result in results:
           result_url=result.xpath('@href')		
		if not results:
           return		

        def ismatch(title):
		     #only if the title exact matches
			match = False
			if lower(title)==lower(orig_title):
				match= True
			if match==False:
				if lower(orig_title[:3])=='de ':
					if lower(title)==lower(orig_title[3:]):
					    match=True
				if lower(orig_title[:4])=='het ':
					if lower(title)==lower(orig_title[4:]):
					    match=True
				if lower(orig_title[:4])=='een ':
					if lower(title)==lower(orig_title[4:]):
					    match=True		
			return match 
           
        import calibre_plugins.INMONDADORI.config as cfg
        max_results = cfg.plugin_prefs[cfg.STORE_NAME][cfg.KEY_MAX_DOWNLOADS]
        for result in results:
			title=result.text_content().strip()
			url=result.xpath('@href')
            while '  ' in title:
                title = title.replace('  ',' ')
            # Strip off any series information from the title
            if '(' in title:
                title = title.rpartition('(')[0].strip()
                title_tokens = list(self.get_title_tokens(orig_title))
			valid_cat = True
			rating = 0
           
			if not ismatch(title):
                log.error('Rejecting as not close enough match: %s '%(title))
			else: 
			    valid_cat = True
				if valid_cat:
					#result_url = INMONDADORI.BASE_URL0 + url[0]
					result_url=url[0]
					matches.append( result_url)	
					ratings.append(rating)
					if len(matches) >= max_results:
						break
        
		    
    def download_cover(self, log, result_queue, abort,title=None, authors=None, identifiers={}, timeout=30):
        cached_url = self.get_cached_cover_url(identifiers)
        if cached_url is None:
            log.info('No cached cover found, running identify')
            rq = Queue()
            self.identify(log, rq, abort, title=title, authors=authors,
                    identifiers=identifiers)
            if abort.is_set():
                return
            results = []
            while True:
                try:
                    results.append(rq.get_nowait())
                except Empty:
                    break
            results.sort(key=self.identify_results_keygen(
                title=title, authors=authors, identifiers=identifiers))
            for mi in results:
                cached_url = self.get_cached_cover_url(mi.identifiers)
                if cached_url is not None:
                    break
        if cached_url is None:
            log.info('No cover found')
            return

        if abort.is_set():
            return
        br = self.browser
        log('Downloading cover from:', cached_url)
        try:
            cdata = br.open_novisit(cached_url, timeout=timeout).read()
			if sys.getsizeof(cdata) <> 9361 :
				result_queue.put((self, cdata))	
        except:
            log.exception('Failed to download cover from:', cached_url)


if __name__ == '__main__': # tests
    # To run these test use:
    # calibre-debug -e __init__.py
    from calibre.ebooks.metadata.sources.test import (test_identify_plugin,
            title_test, authors_test, series_test)
    test_identify_plugin(INMONDADORI.name,
        [

            ( # A book with no ISBN specified
                {'title':"Le Luci Di Atlantide", 'authors':['Marion Zimmer Bradley']},
                [title_test("Le Luci Di Atlantide",
                    exact=True), authors_test(['Marion Zimmer Bradley']),
                    series_test('Harry Potter', 1.0)]

            ),

            ( # A book with an ISBN
                {'identifiers':{'isbn': '9780439064866'},
                    'title':'Chamber of Secrets', 'authors':['J.K. Rowling']},
                [title_test('Harry Potter and the Chamber of Secrets',
                    exact=True), authors_test(['J. K. Rowling']),
                    series_test('Harry Potter', 2.0)]

            ),

            ( # A book with a Barnes & Noble id
                {'identifiers':{'INMONDADORI': '61-Hours/Lee-Child/e/9780440243694'},
                    'title':'61 Hours', 'authors':['Lee Child']},
                [title_test('61 Hours',
                    exact=True), authors_test(['Lee Child']),
                    series_test('Jack Reacher', 14.0)]

            ),

        ])


