#!/usr/bin/env python
# vim:fileencoding=utf-8
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__ = 'GPL v3'
__copyright__ = '2014, David Forrester <davidfor@internode.on.net>'

import re
try:
    from PyQt5.Qt import QAction, QInputDialog, QMenu, QIcon
except:
    from PyQt4.Qt import QAction, QInputDialog, QMenu, QIcon

from cssutils.css import CSSRule

# The base class that all tools must inherit from
from calibre.gui2.tweak_book.plugin import Tool

from calibre import force_unicode
from calibre.gui2 import error_dialog, info_dialog
from calibre.ebooks.oeb.polish.container import OEB_DOCS, OEB_STYLES
from calibre.utils.config import JSONConfig


# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Sort CSS')

class SortCSSTool(Tool):

    #: Set this to a unique name it will be used as a key
    name = 'SortCSS'

    #: If True the user can choose to place this tool in the plugins toolbar
    allowed_in_toolbar = True

    #: If True the user can choose to place this tool in the plugins menu
    allowed_in_menu = True
    
    sort_current = True
    DEFAULT_STATE = {
        'sort_current': True,
    }

    def create_action(self, for_toolbar=True):
        # Create an action, this will be added to the plugins toolbar and
        # the plugins menu
        ac = QAction(get_icons('images/icon.png'), 'SortCSS', self.gui)  # noqa
        self.restore_state()
        if not for_toolbar:
            # Register a keyboard shortcut for this toolbar action. We only
            # register it for the action created for the menu, not the toolbar,
            # to avoid a double trigger
            self.register_shortcut(ac, 'sort-css-tool', default_keys=('Ctrl+Shift+Alt+D',))
        else:
            menu = QMenu()
            ac.setMenu(menu)
            checked_menu_item = menu.addAction('Sort current file', self.toggle_sort_current)
            checked_menu_item.setCheckable(True)
            checked_menu_item.setChecked(self.sort_current)

        ac.triggered.connect(self.sort_stylesheets)
        return ac

    def toggle_sort_current(self):
        self.sort_current = not self.sort_current
        self.save_state()

    def sort_stylesheets(self):
        container = self.current_container

        if self.sort_current:
            from calibre.gui2.tweak_book import editor_name
            current_name = editor_name(self.gui.central.current_editor)
            if not current_name or container.mime_map[current_name] not in OEB_STYLES:
                return info_dialog(self.gui, _('Cannot Sort'),
                        _('The current file is not a stylesheet.'), show=True)

        # Ensure any in progress editing the user is doing is present in the container
        self.boss.commit_all_editors_to_container()
        try:
            if self.sort_current:
                self.sort_stylesheet(current_name)
            else:
                for name, media_type in container.mime_map.iteritems():
                    if media_type in OEB_STYLES:
                        # A stylesheet. Parsed stylesheets are cssutils CSSStylesheet
                        # objects.
                        self.sort_stylesheet(name)
        except Exception:
            # Something bad happened report the error to the user
            import traceback
            error_dialog(self.gui, _('Failed to sort the CSS'), _(
                'Failed to sort the CSS, click "Show details" for more info'),
                det_msg=traceback.format_exc(), show=True)
            # Revert to the saved restore point
            self.boss.revert_requested(self.boss.global_undo.previous_container)
        else:
            # Show the user what changes we have made, allowing her to
            # revert them if necessary
            self.boss.show_current_diff()
            # Update the editor UI to take into account all the changes we
            # have made
            self.boss.apply_container_update_to_gui()

    def sort_stylesheet(self, stylesheet_name):
        # Magnify all font sizes defined in the book by the specified factor
        # First we create a restore point so that the user can undo all changes
        # we make.
        self.boss.add_savepoint('Before: sort_css')

        container = self.current_container  # The book being edited as a container object

        sheet = container.parsed(stylesheet_name)
        rules = sheet.cssRules
#        rules.sort(cmp=lambda x,y:cmp(x.cssText.lower(), y.cssText.lower()))
        rules.sort(cmp=lambda x,y:cmp(x.selectorText.lower(), y.selectorText.lower()))
        container.dirty(stylesheet_name)  # Tell the container that we have changed the stylesheet

    def restore_state(self):
        self.sort_current = plugin_prefs.get('sort_current', self.DEFAULT_STATE)

    def save_state(self):
        plugin_prefs['sort_current'] = self.sort_current

