#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

import sys, os, re
from uuid import uuid4

try:
    import xml.etree.cElementTree as ET
except ImportError:
    import xml.etree.ElementTree as ET

NSMAP = {'opf': '{http://www.idpf.org/2007/opf}',
         'dc':  '{http://purl.org/dc/elements/1.1/}'}

def fixup_element_prefixes(elem, uri_map, memo):
   def fixup(name):
      try:
         return memo[name]
      except KeyError:
         if name[0] != "{":
            return
         uri, tag = name[1:].split("}")
         if uri in uri_map:
            new_name = uri_map[uri] + ":" + tag
            memo[name] = new_name
            return new_name
   # fix element name
   name = fixup(elem.tag)
   if name:
      elem.tag = name
   # fix attribute names
   for key, value in elem.items():
      name = fixup(key)
      if name:
         elem.set(name, value)
         del elem.attrib[key]

def parse_xmlns(file):
   events = "start", "start-ns"
   root = None
   ns_map = []

   for event, elem in ET.iterparse(file, events):
      if event == "start-ns":
         ns_map.append(elem)
      elif event == "start":
         if root is None:
            root = elem
         for prefix, uri in ns_map:
            elem.set("xmlns:" + prefix, uri)
         ns_map = []
   return ET.ElementTree(root)
   
def fixup_xmlns(elem, maps=None):
   if maps is None:
      maps = [{}]
      
   # check for local overrides
   xmlns = {}
   for key, value in elem.items():
      if key[:6] == "xmlns:":
         xmlns[value] = key[6:]
   if xmlns:
      uri_map = maps[-1].copy()
      uri_map.update(xmlns)
   else:
      uri_map = maps[-1]

   # fixup this element
   fixup_element_prefixes(elem, uri_map, {})

   # process elements
   maps.append(uri_map)
   for elem in elem:
      fixup_xmlns(elem, maps)
   maps.pop()


def write_xmlns(elem, file=''):
   if not ET.iselement(elem):
      elem = elem.getroot()
   fixup_xmlns(elem)
   return elem
   #ET.ElementTree(elem).write(file)

def main(argv=sys.argv):
   if len(argv) != 2:
      print "Usage:"
      print "  new_uuid.py infile"
      return 1
   else:  
      infile = argv[1]
      
      opftree = parse_xmlns(infile)
      root = opftree.getroot()
      if root.tag == '{0}package'.format(NSMAP['opf']):
         print root.get('unique-identifier')
         identstring = root.get('unique-identifier')
      else:
         return 1
      
      dirty = False
      for item in opftree.findall('.//{0}metadata/{1}identifier'.format(NSMAP['opf'], NSMAP['dc'])):
         if item.get('id')== identstring:
            scheme = item.get('{0}scheme'.format(NSMAP['opf']), None)
            if scheme is None:
               item.set('{0}scheme'.format(NSMAP['opf']), 'UUID')
            else:
               if scheme != 'UUID' and scheme != 'uuid':
                  item.set('{0}scheme'.format(NSMAP['opf']), 'UUID')
            print item.text
            
            item.text = 'urn:uuid:' + str(uuid4())
            print item.text
            
            # Regex to polish Python's xml.etree namespace mess
            output = write_xmlns(opftree)
            output = ET.tostring(output)
            pattern = re.compile(r'''<(/?)(opf)?:''', re.IGNORECASE)
            output = pattern.sub(r'''<\1''', output)
            pattern = re.compile(r''':=''')
            output = pattern.sub(r'''=''', output)
            output = '<?xml version="1.0" encoding="utf-8" standalone="yes"?>\n' + output
            dirty = True
            break
      if dirty:
         with open(infile, 'wb') as fout:
            fout.write(output)

if __name__ == "__main__":
   sys.exit(main())
