#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
try:
    from PyQt5.Qt import QMenu, QToolButton
except ImportError:
    from PyQt4.Qt import QMenu, QToolButton

from calibre.gui2 import question_dialog, config
from calibre.gui2.actions import InterfaceAction

import calibre_plugins.walk_search_history.config as cfg
from calibre_plugins.walk_search_history.common_utils import (set_plugin_icon_resources, get_icon,
                                                              create_menu_action_unique)

PLUGIN_ICONS = ['images/goto_previous.png', 'images/goto_next.png']

class SearchHistoryState:
    # Keep a history of searches.
    # This class is our "visited" history, and is used for the dropdown menu
    # It is also the base class for our ActiveSearchHistoryState class which is
    # used for the fwd/bwd navigation.

    MAX_HISTORY = 25

    def __init__(self, search):
        self.clear()
        # Seed our history list with the contents of the search combobox history as a starting point
        self.snapshot = list()
        for i in range(search.count()):
            self.snapshot.append(unicode(search.itemText(i)))

    def clear(self):
        self.snapshot = []

    def items(self):
        return self.snapshot

    def insert(self, text):
        # If the item exists in our history, move it to the top
        # Otherwise add it at the top
        if text in self.snapshot:
            self.snapshot.remove(text)
        self.snapshot.insert(0, text)
        # Make sure we don't keep too much history...
        while len(self.snapshot) > self.MAX_HISTORY:
            self.snapshot.pop()


class NavigationSearchHistoryState(SearchHistoryState):
    # Keep track of the recent searches allowing fwd/bwd navigation.
    # When the user issues a new search (breaking the sequence) all
    # "forward" entries if any will be removed.

    def __init__(self, search):
        SearchHistoryState.__init__(self, search)
        # Ensure we have a blank search (as this is system startup) to move forward to if go back
        self.snapshot.insert(0, '')
        self.position = 0

    def clear(self):
        SearchHistoryState.clear(self)
        self.position = -1

    def insert(self, text):
        # Override the base behaviour slightly
        # If we have navigated backwards, remove searches that are considered in future
        if self.position > 0:
            for i in range(self.position):
                self.snapshot.pop(0)
        # It is ok to have duplicates in our list as this reflects order of searches
        self.snapshot.insert(0, text)
        # Make sure we don't keep too much history...
        while len(self.snapshot) > self.MAX_HISTORY:
            self.snapshot.pop()
        # Reset our search position
        self.position = 0

    def get_current(self):
        if self.position != -1:
            return self.snapshot[self.position]
        return ''

    def get_current_position(self):
        return self.position

    def goto_previous(self):
        if self.position < len(self.snapshot) - 1:
            self.position = self.position + 1
            return True
        return False

    def goto_next(self):
        if self.position > 0:
            self.position = self.position - 1
            return True
        return False


class WalkSearchHistoryAction(InterfaceAction):

    name = 'Walk Search History'
    action_spec = ('History', None, None, None)
    action_type = 'current'

    def genesis(self):
        self.default_action_is_previous = False
        self.is_navigating_history = False
        self.last_search_text = ''
        self.menu = QMenu(self.gui)

        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))

    def initialization_complete(self):
        # We must delay constructing until the gui view is available to modify the toolbar button
        # This is not available from the genesis() method.
        self.visited_history_state = SearchHistoryState(self.gui.search)
        self.navigation_history_state = NavigationSearchHistoryState(self.gui.search)

        self.rebuild_menus()
        # Setup hooks so that we rebuild the dropdown menus each time to represent latest history
        self.menu.aboutToShow.connect(self.rebuild_menus)
        # Hook into the search signal so that we can keep track of the searches being performed
        self.gui.search.search.connect(self.search_performed)

    def rebuild_menus(self):
        c = cfg.plugin_prefs[cfg.STORE_NAME]
        m = self.menu
        m.clear()
        create_menu_action_unique(self, m, 'Goto previous', PLUGIN_ICONS[0],
                                  shortcut='Alt+Left',  triggered=self.find_previous_search)
        create_menu_action_unique(self, m, 'Goto next', PLUGIN_ICONS[1],
                                  shortcut='Alt+Right', triggered=self.find_next_search)

        # Now create menu items for each of the searches currently in the history
        if self.gui.search.count() > 0:
            m.addSeparator()
            current_search = self.current_search_text()
            limit = c[cfg.KEY_LIMIT]
            for i, history_item in enumerate(self.visited_history_state.items()):
                if limit > 0 and i == limit:
                    break
                ac = create_menu_action_unique(self, m, history_item, shortcut=False,
                                      triggered=partial(self.goto_search_item, history_item))
                if history_item == current_search:
                    ac.setCheckable(True)
                    ac.setChecked(True)

        # Finally add a menu item for configuring the plugin
        m.addSeparator()
        create_menu_action_unique(self, m, 'Clear search history',
                                  triggered=self.clear_search_history)
        create_menu_action_unique(self, m, _('&Customize plugin')+'...', 'config.png',
                                  shortcut=False, triggered=self.show_configuration)
        self.gui.keyboard.finalize()

        # Set the desired default action for the toolbar button when clicked on
        if self.default_action_is_previous:
            self.qaction.triggered.disconnect()
        self.default_action_is_previous = c[cfg.KEY_DEFAULT_ACTION] == cfg.KEY_PREVIOUS
        popup_type = QToolButton.InstantPopup
        if self.default_action_is_previous:
            popup_type = QToolButton.MenuButtonPopup
            self.qaction.triggered.connect(self.find_previous_search)
        # Apply the popup type for this action in the toolbar
        # Only update on the toolbar if it is actually visible there
        self.change_toolbar_popup_type(popup_type)

    def change_toolbar_popup_type(self, popup_type):
        self.popup_type = popup_type
        for bar in self.gui.bars_manager.bars:
            if hasattr(bar, 'setup_tool_button'):
                if self.qaction in bar.added_actions:
                    bar.setup_tool_button(bar, self.qaction, self.popup_type)

    def current_search_text(self):
        if self.gui.search.count() == 0:
            return ''
        return self.last_search_text

    def search_performed(self, text):
        # A search has been performed - either triggered by this plugin or by user using the GUI
        # If search is the same as the last search executed, nothing to do
        if text == self.last_search_text:
            return
        self.last_search_text = text
        if self.is_navigating_history:
            # Search has been triggered by this plugin - nothing to do:
            return
        # Only add it to our visited history if it is not an empty search
        if text:
            self.visited_history_state.insert(text)
        # Ensure our navigation history of searches is updated
        self.navigation_history_state.insert(text)

    def find_previous_search(self):
        if self.gui.search.count() == 0:
            # First-time user, has never done a search
            return
        # Only if there is a previous search to display do we perform a search
        if self.navigation_history_state.goto_previous():
            previous_search = self.navigation_history_state.get_current()
            try:
                self.is_navigating_history = True
                self.gui.search.set_search_string(previous_search, store_in_history=True)
            finally:
                self.is_navigating_history = False

    def find_next_search(self):
        if self.gui.search.count() == 0:
            # First-time user
            return
        # Only if there is a next search to display do we perform a search
        if self.navigation_history_state.goto_next():
            next_search = self.navigation_history_state.get_current()
            # We want to store the selection in the combo history so as to "re-add" it
            try:
                self.is_navigating_history = True
                self.gui.search.set_search_string(next_search, store_in_history=True)
            finally:
                self.is_navigating_history = False

    def goto_search_item(self, selected_text):
        # User has selected an item in the history to jump to from the menu
        # Update for our history menu to reflect the new position
        self.visited_history_state.insert(selected_text)
        self.gui.search.set_search_string(selected_text, store_in_history=True)

    def clear_search_history(self):
        if not question_dialog(self.gui, _('Are you sure?'), '<p>'+
                           'Are you sure you want to clear the search history?', show_copy_button=False):
            return
        # Turn off event signals for combobox to prevent a search being triggered
        self.gui.search.block_signals(True)
        while self.gui.search.count() > 0:
            self.gui.search.removeItem(0)
        self.gui.search.block_signals(False)
        # Clear the config history where the history is persisted
        config[self.gui.search.opt_name] = []
        # Ensure our own state histories are cleared too
        self.navigation_history_state.clear()
        self.visited_history_state.clear()

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
