#!/usr/bin/env python
# -*- coding: utf-8 -*-

# In EPUB3, NCX and <guide> MAY exist in OPF, although the NCX is superseded
# by the Navigation Document and the <guide> is deprecated. Currently, EPUB3_WITH_NCX
# and EPUB3_WITH_GUIDE are set to True due to compatibility with epub2 reading systems.
# They might be change to set to False in the future.
EPUB3_WITH_NCX = True
""" Set to True to create a toc.ncx when converting to epub3. """

EPUB3_WITH_GUIDE = True
""" Set to True to create a guide element in an opf when converting to epub3. """

EPUB_OPF = 'content.opf'
""" The name for the OPF of EPUB. """

TOC_NCX = 'toc.ncx'
""" The name for the TOC of EPUB2. """

NAVIGATION_DOCUMENT = 'nav.xhtml'
""" The name for the navigation document of EPUB3. """

BEGIN_INFO_ONLY = '<!-- BEGIN INFORMATION ONLY '
""" The comment to indicate the beginning of metadata which will be ignored by kindlegen. """

END_INFO_ONLY = 'END INFORMATION ONLY -->'
""" The comment to indicate the end of metadata which will be ignored by kindlegen. """

PPD_KEY = 'page-progression-direction'
""" The meta name for page-progression-direction. """

PWM_KEY = 'primary-writing-mode'
""" The meta name for primary-writing-mode. """


import sys, os, re, uuid
from path import pathof
from xml.sax.saxutils import escape as xmlescape
from HTMLParser import HTMLParser
EXTRA_ENTITIES = {'"':'&quot;', "'":"&apos;"}

class OPFProcessor(object):
    def __init__(self, files, metadata, fileinfo, imgnames, isNCX, mh, usedmap, pagemapxml='', guidetext='', k8resc=None, epubver='2'):
        self.files = files
        self.metadata = metadata
        self.fileinfo = fileinfo
        self.imgnames = imgnames
        self.isNCX = isNCX
        self.codec = mh.codec
        self.isK8 = mh.isK8()
        self.printReplica = mh.isPrintReplica()
        self.guidetext = guidetext
        self.used = usedmap
        self.k8resc = k8resc
        self.covername = None
        self.cover_id = 'cover_img'
        self.resc_metadata = []
        self.h = HTMLParser()
        # Create a unique urn uuid
        self.BookId = str(uuid.uuid4())
        #self.starting_offset = None # FIXME: regarding guidetext for mobi7 is gatherd to getMobi7GuideText().
        self.pagemap = pagemapxml
        self.ncxname = None
        self.navname = None
        self.epubver = epubver
        if self.epubver == 'A':
            self.epubver = self.autodetectEPUBVersion()

        # get page-progression-direction
        self.page_progression_direction = None
        self.page_progression_direction = metadata.get(PPD_KEY, [None])[0]
        if 'rl' in metadata.get(PWM_KEY, [''])[0]:
            self.page_progression_direction = 'rtl'

        if not self.isK8 and not self.printReplica:
            self.getMobi7GuideText()


    def escapeit(self, sval, EXTRAS=None):
        # note, xmlescape and unescape do not work with utf-8 bytestrings
        # so pre-convert to full unicode and then convert back since opf is utf-8 encoded
        uval = sval.decode('utf-8')
        if EXTRAS:
            ures = xmlescape(self.h.unescape(uval), EXTRAS)
        else:
            ures = xmlescape(self.h.unescape(uval))
        return ures.encode('utf-8')


    # FIXME: regarding guidetext for mobi7 is gatherd to getMobi7GuideText().
    def getMobi7GuideText(self):
        if 'StartOffset' in self.metadata:
            value = self.metadata['StartOffset'][0]
            if int(value) == 0xffffffff:
                value = '0'
            so = value
            metaguidetext = '<reference type="text" href="'+self.fileinfo[0][1]+'#filepos'+so+'" />\n'
            self.guidetext = metaguidetext + self.guidetext


    # will be invoked inside buildOPFMetadata().
    def getRESCExtraMetaData(self):
        if self.k8resc is not None:
            cover_id = self.k8resc.cover_name
            if cover_id is not None:
                self.cover_id = cover_id
            new_metadata = ''
            for taginfo in self.k8resc.extrameta:
                new_metadata += self.k8resc.taginfo_toxml(taginfo)
            self.resc_metadata = new_metadata


    def buildOPFMetadata(self, start_tag, has_obfuscated_fonts=False):
        # convert from EXTH metadata format to target epub version metadata
        # epub 3 will ignore <meta name="xxxx" content="yyyy" /> style metatags
        #    but allows them to be present for backwards compatibility
        #    instead the new format is
        #    <meta property="xxxx" id="iiii" ... > property_value</meta>
        #       and DCMES elements such as:
        #    <dc:blah id="iiii">value</dc:blah>

        metadata = self.metadata

        META_TAGS = ['Drm Server Id', 'Drm Commerce Id', 'Drm Ebookbase Book Id', 'ASIN', 'ThumbOffset', 'Fake Cover',
                                                'Creator Software', 'Creator Major Version', 'Creator Minor Version', 'Creator Build Number',
                                                'Watermark', 'Clipping Limit', 'Publisher Limit', 'Text to Speech Disabled', 'CDE Type',
                                                'Updated Title', 'Font Signature (hex)', 'Tamper Proof Keys (hex)',  ]


        def createMetaTag(data, property, content):
            data.append('<meta property="%s">%s</meta>\n' % (property, content))

        def handleTag(data, metadata, key, tag):
            '''
            Format metadata values.

            @param data: List of formatted metadata entries.
            @param metadata: The metadata dictionary.
            @param key: The key of the metadata value to handle.
            @param tag: The opf tag the the metadata value.
            '''
            if key in metadata.keys():
                for value in metadata[key]:
                    closingTag = tag.split(" ")[0]
                    res = '<%s>%s</%s>\n' % (tag, self.escapeit(value), closingTag)
                    data.append(res)
                del metadata[key]

        # these are allowed but ignored by epub3
        def handleMetaPairs(data, metadata, key, name):
            if key in metadata.keys():
                for value in metadata[key]:
                    res = '<meta name="%s" content="%s" />\n' % (name, self.escapeit(value, EXTRA_ENTITIES))
                    data.append(res)
                del metadata[key]


        data = []
        data.append(start_tag + '\n')
        # Handle standard metadata
        if 'Title' in metadata.keys():
            handleTag(data, metadata, 'Title', 'dc:title')
        else:
            data.append('<dc:title>Untitled</dc:title>\n')
        handleTag(data, metadata, 'Language', 'dc:language')
        if 'UniqueID' in metadata.keys():
            handleTag(data, metadata, 'UniqueID', 'dc:identifier id="uid"')
        else:
            # No unique ID in original, give it a generic one.
            data.append('<dc:identifier id="uid">0</dc:identifier>\n')
        if self.isK8 and has_obfuscated_fonts:
            # Use the random generated urn:uuid so obuscated fonts work.
            # It doesn't need to be _THE_ unique identifier to work as a key
            # for obfuscated fonts in Sigil, ADE and calibre. Its just has
            # to use the opf:scheme="UUID" and have the urn:uuid: prefix.
            data.append('<dc:identifier opf:scheme="UUID">urn:uuid:'+self.BookId+'</dc:identifier>\n')

        handleTag(data, metadata, 'Creator', 'dc:creator')
        handleTag(data, metadata, 'Contributor', 'dc:contributor')
        handleTag(data, metadata, 'Publisher', 'dc:publisher')
        handleTag(data, metadata, 'Source', 'dc:source')
        handleTag(data, metadata, 'Type', 'dc:type')
        handleTag(data, metadata, 'ISBN', 'dc:identifier opf:scheme="ISBN"')
        if 'Subject' in metadata.keys():
            if 'SubjectCode' in metadata.keys():
                codeList = metadata['SubjectCode']
                del metadata['SubjectCode']
            else:
                codeList = None
            for i in range(len(metadata['Subject'])):
                if codeList and i < len(codeList):
                    data.append('<dc:subject BASICCode="'+codeList[i]+'">')
                else:
                    data.append('<dc:subject>')
                data.append(self.escapeit(metadata['Subject'][i])+'</dc:subject>\n')
            del metadata['Subject']
        handleTag(data, metadata, 'Description', 'dc:description')
        handleTag(data, metadata, 'Published', 'dc:date opf:event="publication"')
        handleTag(data, metadata, 'Rights', 'dc:rights')

        # Append metadata in RESC section.
        self.getRESCExtraMetaData()
        if len(self.resc_metadata) > 0:
            data.append('<!-- Extra MetaData from RESC\n')
            data += self.resc_metadata
            data.append('-->\n')

        if 'CoverOffset' in metadata.keys():
            imageNumber = int(metadata['CoverOffset'][0])
            self.covername = self.imgnames[imageNumber]
            if self.covername is None:
                print "Error: Cover image %s was not recognized as a valid image" % imageNumber
            else:
                #XXX: since whole metadata in RESC is commented out.
                data.append('<meta name="cover" content="' + self.cover_id + '" />\n')
                self.used[self.covername] = 'used'
            del metadata['CoverOffset']

        # XXX: It seems kindlegen does not recognaize <meta name="output encoding" content="XXX"/> since it is always stored in RESC.
        handleMetaPairs(data, metadata, 'Codec', 'output encoding')
        # handle kindlegen specifc tags
        handleTag(data, metadata, 'DictInLanguage', 'DictionaryInLanguage')
        handleTag(data, metadata, 'DictOutLanguage', 'DictionaryOutLanguage')
        handleMetaPairs(data, metadata, 'RegionMagnification', 'RegionMagnification')
        handleMetaPairs(data, metadata, 'book-type', 'book-type')
        handleMetaPairs(data, metadata, 'zero-gutter', 'zero-gutter')
        handleMetaPairs(data, metadata, 'zero-margin', 'zero-margin')
        handleMetaPairs(data, metadata, PWM_KEY, PWM_KEY)
        metadata.pop(PPD_KEY, None) # 'page-progression-direction' is placed in <spine>.

        # now handle fixed layout and convert to epub3 format if needed
        if self.epubver == "3":
            #epub3metadata = self.getEPUB3MetaData(metadata)
            #FIXME: I would like to offload bellows into getEPUB3MetaData()
            if 'fixed-layout' in metadata and  metadata['fixed-layout'][0].lower() == 'true':
                data.append('<meta property="rendition:layout">pre-paginated</meta>\n')
            if 'orientation-lock' in metadata:
                orientation = metadata['orientation-lock'][0].lower()
                if orientation != 'none':
                    createMetaTag(data, 'rendition:orientation', orientation)
            # XXX: You are right.
            # according to epub3 spec about correspondence with Amazon
            # if 'original-resolution' is provided it needs to be converted to
            # meta viewport property tag stored in the <head></head> of **each**
            # xhtml page - so this tag would need to be handled by editing each part
            # before reaching this routine
            # we need to add support for this to the k8html routine
            if False: # if 'original-resolution' in metadata.keys():
                resolution = metadata['original-resolution'][0]
                [width, height] = resolution.split('x')
                if width.isdigit() and int(width) > 0 and height.isdigit() and int(height) > 0:
                    viewport = 'width=%s, height=%s' % (width, height)
                    createMetaTag(data, 'rendition:viewport', viewport)
        handleMetaPairs(data, metadata, 'fixed-layout', 'fixed-layout')
        handleMetaPairs(data, metadata, 'orientation-lock', 'orientation-lock')
        handleMetaPairs(data, metadata, 'original-resolution', 'original-resolution')

        # these are not allowed in epub2 or 3 so convert them to meta name content pairs
        # perhaps these could better be mapped into the dcterms namespace instead
        handleMetaPairs(data, metadata, 'Review', 'review')
        handleMetaPairs(data, metadata, 'Imprint', 'imprint')
        handleMetaPairs(data, metadata, 'Adult', 'adult')
        handleMetaPairs(data, metadata, 'DictShortName', 'DictionaryVeryShortName')

        # these are needed by kobo books upon submission but not sure if legal metadata in epub2 or epub3
        if 'Price' in metadata.keys() and 'Currency' in metadata.keys():
            priceList = metadata['Price']
            currencyList = metadata['Currency']
            if len(priceList) != len(currencyList):
                print "Error: found %s price entries, but %s currency entries."
            else:
                for i in range(len(priceList)):
                    data.append('<SRP Currency="'+currencyList[i]+'">'+priceList[i]+'</SRP>\n')
            del metadata['Price']
            del metadata['Currency']

        # all that remains is extra EXTH info we will store inside a comment inside meta name/content pairs
        # so it can not impact anything and will be automatically stripped out if found again in a RESC section
        data.append(BEGIN_INFO_ONLY + '\n')
        if 'ThumbOffset' in metadata.keys():
            imageNumber = int(metadata['ThumbOffset'][0])
            imageName = self.imgnames[imageNumber]
            if imageName is None:
                print "Error: Cover Thumbnail image %s was not recognized as a valid image" % imageNumber
            else:
                data.append('<meta name="Cover ThumbNail Image" content="'+ 'Images/'+imageName+'" />\n')
                # self.used[imageName] = 'used' # thumbnail image is always generated by Kindlegen, so don't include in manifest
                self.used[imageName] = 'not used'
            del metadata['ThumbOffset']
        for metaName in META_TAGS:
            if metaName in metadata.keys():
                for value in metadata[metaName]:
                    data.append('<meta name="'+metaName+'" content="'+self.escapeit(value, EXTRA_ENTITIES)+'" />\n')
                del metadata[metaName]
        for key in metadata.keys():
            for value in metadata[key]:
                #FIXME: regarding metaguidetext is offloaded to getMobi7GuideText()
                #if key == 'StartOffset':
                #    if int(value) == 0xffffffff:
                #        value = '0'
                #    self.starting_offset = value
                data.append('<meta name="'+key+'" content="'+self.escapeit(value, EXTRA_ENTITIES)+'" />\n')
            del metadata[key]
        data.append(END_INFO_ONLY + '\n')
        data.append('</metadata>\n')
        return data


    def buildOPFManifest(self, ncxname, navname=None):
        # Build manifest.
        files = self.files
        k8resc = self.k8resc
        cover_id = self.cover_id
        hasK8RescSpine = k8resc is not None and k8resc.hasSpine()
        isEpub3 = False
        if navname is not None:
            isEpub3 = True
        self.ncxname = ncxname
        self.navname = navname

        data = []
        # build manifest
        data.append('<manifest>\n')
        media_map = {
                '.jpg'  : 'image/jpeg',
                '.jpeg' : 'image/jpeg',
                '.png'  : 'image/png',
                '.gif'  : 'image/gif',
                '.svg'  : 'image/svg+xml',
                '.xhtml': 'application/xhtml+xml',
                '.html' : 'text/x-oeb1-document', # for mobi7
                '.pdf'  : 'application/pdf', # for azw4(print replica textbook)
                '.ttf'  : 'application/x-font-ttf',
                '.otf'  : 'application/x-font-opentype', # replaced?
                #'.otf' : 'application/vnd.ms-opentype', # [OpenType] OpenType fonts
                #'.woff' : 'application/font-woff', # [WOFF] WOFF fonts
                #'.smil' : 'application/smil+xml', # [MediaOverlays301] EPUB Media Overlay documents
                #'.pls' : 'application/pls+xml', # [PLS] Text-to-Speech (TTS) Pronunciation lexicons
                #'.mp3'  : 'audio/mpeg',
                #'.mp4'  : 'audio/mp4',
                #'.js'   : 'text/javascript', # not supported in K8
                '.css'  : 'text/css'
                }
        spinerefs = []

        idcnt = 0
        for [key,dir,fname] in self.fileinfo:
            name, ext = os.path.splitext(fname)
            ext = ext.lower()
            media = media_map.get(ext)
            ref = "item%d" % idcnt
            if hasK8RescSpine:
                if key is not None and key in k8resc.spine_idrefs.keys():
                    ref = k8resc.spine_idrefs[key]
            #properties = ''
            if dir != '':
                fpath = dir + '/' + fname
            else:
                fpath = fname
            data.append('<item id="{:s}" media-type="{:s}" href="{:s}" />\n'.format(ref, media, fpath))

            if ext in ['.xhtml', '.html']:
                spinerefs.append(ref)
            idcnt += 1

        for fname in self.imgnames:
            if fname is not None:
                if self.used.get(fname,'not used') == 'not used':
                    continue
                name, ext = os.path.splitext(fname)
                ext = ext.lower()
                media = media_map.get(ext,ext[1:])
                properties = ''
                if fname == self.covername:
                    ref = cover_id
                    if isEpub3:
                        properties = 'properties="cover-image"'
                else:
                    ref = "item%d" % idcnt
                if ext == '.ttf' or ext == '.otf':
                    fpath = 'Fonts/' + fname # fonts only exist in mobi8
                else:
                    fpath = 'Images/' + fname
                data.append('<item id="{:s}" media-type="{:s}" href="{:s}" {:s}/>\n'.format(ref, media, fpath, properties))
                idcnt += 1

        if navname is not None:
            data.append('<item id="nav" media-type="application/xhtml+xml" href="Text/' + navname + '" properties="nav"/>\n')
        if ncxname is not None:
            data.append('<item id="ncx" media-type="application/x-dtbncx+xml" href="' + ncxname +'"></item>\n')
        if self.pagemap != '':
            data.append('<item id="map" media-type="application/oebs-page-map+xml" href="page-map.xml"></item>\n')
        data.append('</manifest>\n')
        return [data, spinerefs]


    def buildOPFSpine(self, spinerefs, isNCX):
        # build spine
        k8resc = self.k8resc
        hasK8RescSpine = k8resc is not None and k8resc.hasSpine()
        data = []
        ppd = ''
        if self.isK8 and self.page_progression_direction is not None:
            ppd = ' page-progression-direction="{:s}"'.format(self.page_progression_direction)
        ncx = ''
        if isNCX:
            ncx = ' toc="ncx"'
        map=''
        if self.pagemap != '':
            map = ' page-map="map"'
        spine_start_tag = '<spine{:s}{:s}{:s}>\n'.format(ppd, map, ncx)
        data.append(spine_start_tag)
        if hasK8RescSpine:
            for key in k8resc.spine_order:
                idref = k8resc.spine_idrefs[key]
                attribs = k8resc.spine_pageattributes[key]
                tag = '<itemref idref="%s"' % idref
                for aname, val in attribs.items():
                    if val is not None:
                        tag += ' %s="%s"' % (aname, val)
                tag += '/>\n'
                data.append(tag)
        else:
            start = 0
            # special case the created coverpage if need be
            [key, dir, fname] = self.fileinfo[0]
            if key is not None and key == "coverpage":
                entry = spinerefs[start]
                data.append('<itemref idref="%s" linear="no"/>\n' % entry)
                start += 1
            for entry in spinerefs[start:]:
                data.append('<itemref idref="' + entry + '"/>\n')
        data.append('</spine>\n')
        return data


    def buildOPF(self, has_obfuscated_fonts=False):
        # Build an OPF for mobi7,azw4 and epub2.
        print "Building an opf for mobi7,azw4 and epub2."
        data = []
        data.append('<?xml version="1.0" encoding="utf-8"?>\n')
        data.append('<package version="2.0" xmlns="http://www.idpf.org/2007/opf" unique-identifier="uid">\n')
        metadata_tag = '<metadata xmlns:dc="http://purl.org/dc/elements/1.1/" xmlns:opf="http://www.idpf.org/2007/opf">'
        opf_metadata = self.buildOPFMetadata(metadata_tag, has_obfuscated_fonts)
        data += opf_metadata
        if self.isK8:
            ncxname = TOC_NCX
        elif self.isNCX:
            ncxname = self.files.getInputFileBasename() + '.ncx'
        else:
            ncxname = None
        [opf_manifest, spinerefs] = self.buildOPFManifest(ncxname)
        data += opf_manifest
        opf_spine = self.buildOPFSpine(spinerefs, self.isNCX)
        data += opf_spine
        data.append('<tours>\n</tours>\n')
        if not self.printReplica:
            #FIXME: regarding metaguidetext is offloaded to getMobi7GuideText()
            #metaguidetext = ''
            #if not self.isK8:
            #    # get guide items from metadata (note starting offset previsouly processed)
            #    if self.starting_offset is not None:
            #        so = self.starting_offset
            #        metaguidetext += '<reference type="text" href="'+self.fileinfo[0][1]+'#filepos'+so+'" />\n'
            #guide ='<guide>\n' + metaguidetext + self.guidetext + '</guide>\n'
            guide ='<guide>\n' + self.guidetext + '</guide>\n'
            data.append(guide)
        data.append('</package>\n')
        return ''.join(data)


    def buildEPUB3OPF(self, has_obfuscated_fonts=False):
        # Build an OPF for EPUB3.
        print "Building an opf for epub3"
        has_ncx = EPUB3_WITH_NCX
        has_guide = EPUB3_WITH_GUIDE
        #self.sovleRefineID()
        data = []
        data.append('<?xml version="1.0" encoding="utf-8"?>\n')
        data.append('<package version="3.0" xmlns="http://www.idpf.org/2007/opf" prefix="rendition: http://www.idpf.org/vocab/rendition/#" unique-identifier="uid">\n')
        #metadata_tag = '<metadata xmlns:dc="http://purl.org/dc/elements/1.1/">') # XXX: I will check this again.
        metadata_tag = '<metadata xmlns:dc="http://purl.org/dc/elements/1.1/" xmlns:opf="http://www.idpf.org/2007/opf">'
        opf_metadata = self.buildOPFMetadata(metadata_tag, has_obfuscated_fonts)
        data += opf_metadata

        if has_ncx:
            ncxname = TOC_NCX
        else:
            ncxname = None
        navname = NAVIGATION_DOCUMENT
        [opf_manifest, spinerefs] = self.buildOPFManifest(ncxname, navname)
        data += opf_manifest
        opf_spine = self.buildOPFSpine(spinerefs, has_ncx)
        data += opf_spine
        # tours are deprecated in epub3: data.append('<tours>\n</tours>\n')
        if has_guide:
            guide ='<guide>\n' + self.guidetext + '</guide>\n'
            data.append(guide)
        data.append('</package>\n')
        return ''.join(data)


    def buildK8OPF(self, has_obfuscated_fonts=False):
        if self.epubver == '2':
            return self.buildOPF(has_obfuscated_fonts)
        else:
            return self.buildEPUB3OPF(has_obfuscated_fonts)


    def writeOPF(self, has_obfuscated_fonts=False):
        # write out the metadata as an OEB 1.0 OPF file
        #print "Write opf"
        if not self.isK8:
            data = self.buildOPF(has_obfuscated_fonts)
            outopf = os.path.join(self.files.mobi7dir, self.files.getInputFileBasename() + '.opf')
            open(pathof(outopf), 'wb').write(data)
            return
        else:
            # Write opf for mobi8
            data = self.buildK8OPF(has_obfuscated_fonts)
            outopf = os.path.join(self.files.k8oebps, EPUB_OPF)
            open(pathof(outopf), 'wb').write(data)
            return self.BookId

    # XXX: Will be removed.
    def writeK8OPF(self, has_obfuscated_fonts=False):
        return self.writeOPF(has_obfuscated_fonts)

    def getNCXName(self):
        return self.ncxname

    def getNAVName(self):
        return self.navname

    def getEPUBVersion(self):
        return self.epubver

    def hasNCX(self):
        return self.ncxname is not None

    def hasNAV(self):
        return self.navname is not None


    def autodetectEPUBVersion(self):
        # Determine EPUB version from metadata and RESC.
        epubver = '2'
        if 'true' == self.metadata.get('fixed-layout', [''])[0].lower():
            epubver = '3'
        elif self.page_progression_direction == 'rtl':
            epubver = '3'
        elif self.k8resc is not None and self.k8resc.needEPUB3():
            epubver = '3'
        return epubver



    # XXX: Under construction. Need more investigation of kindlegen.
    # def sovleRefineID(self):
    #     metadata = self.metadata
    #     dc_title = metadata.get('Title', [])
    #     dc_creator = metadata.get('Creator', [])
    #     dc_publisher = metadata.get('Publisher', [])
    #     fileas_title = metadata.get('Title file-as', [])
    #     fileas_creator = metadata.get('Creator file-as', [])
    #     fileas_publisher = metadata.get('Publisher file-as', [])
    #
    #     title_ids = []
    #     creator_ids = []
    #     publisher_ids = []
    #
    #     k8resc = self.k8resc
    #     if k8resc is not None:
    #         for [tname, tattr, tcontent] in k8resc.extrameta:
    #             if 'refines' in tattr:
    #                 id_ = tattr['refines'][1:]
    #                 if 'title' in id_.lower():
    #                     title_ids.append(id_)
    #                 elif 'creator' in id_.lower():
    #                     creator_ids.append(id_)
    #                 elif 'publisher' in id_.lower():
    #                     publisher_ids.append(id_)
    #
    #     if len(dc_title) == 1:
    #         if len(title_ids) == 1 and len(fileas_title) <= 1:
    #             pass
    #         elif len(title_ids) == 0 and len(fileas_title) == 1:
    #             title_ids.append('title')
    #         else:
    #             title_ids = []
    #     else:
    #         title_ids = []
    #
    #     if len(dc_publisher) == 1:
    #         if len(publisher_ids) == 1and len(fileas_publisher) <= 1:
    #             pass
    #         elif len(publisher_ids) == 0 and len(fileas_publisher) == 1:
    #             publisher_ids.append('publisher')
    #         else:
    #             publisher_ids = []
    #     else:
    #         publisher_ids = []
    #
    #     if len(dc_creator) == 1:
    #         if len(creator_ids) == 1 and len(fileas_creator) <= 1:
    #             pass
    #         elif len(creator_ids) == 0 and len(fileas_creator) == 1:
    #             creator_ids.append('creator')
    #         else:
    #             creator_ids = []
    #     else:
    #         creator_ids = []
    #
    #     self.title_ids = title_ids
    #     self.creator_ids = creator_ids
    #     self.publisher_ids = publisher_ids
    #     return
