
#include <libgen.h>

#include <assert.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <pgm.h>

#include "list.h"

#define error_exit(ec, fmt...)			\
	do {					\
		fprintf(stderr, fmt);		\
		fprintf(stderr, "\n");		\
		assert(0);			\
		exit(ec);			\
	} while (0)

#define error_exit_on(cond, ec, fmt...)		\
	do {					\
		if (cond)			\
			error_exit(ec, fmt);	\
	} while (0)

#define error_exit_errno(ec, fmt...)				\
	do {							\
		fprintf(stderr, fmt);				\
		fprintf(stderr, ": %s.\n", strerror(errno));	\
		assert(0);					\
		exit(ec);					\
	} while (0)

#define error_exit_errno_on(cond, ec, fmt...)		\
	do {						\
		if (cond)				\
			error_exit_errno(ec, fmt);	\
	} while (0)

struct pi_rect {
	int left;
	int top;
	int width;
	int height;
};

struct pi_image {
	gray maxval;
	int width;
	int height;
	gray **pxls;
};

struct pi_page;
struct pi_doc;
struct pi_line {
	int bl;
	int al;
	struct pi_page *page;
	struct pi_rect bbox;
	struct list_head list;
};

struct pi_page {
	struct pi_image *img;
	struct pi_doc *doc;
	struct pi_rect bbox;
	struct list_head lines;
};

struct pi_doc {
	gray empty;
};

struct pi_image *pi_image_load(const char *file_name)
{
	FILE *fin;
	struct pi_image *img;

	img = malloc(sizeof(struct pi_image));
	error_exit_on(!img, -1, "Error alloc memory for pi_image.");
	fin = fopen(file_name, "r");
	error_exit_errno_on(!fin, -1, "Error open pgm image file %s",
			    file_name);
	img->pxls = pgm_readpgm(fin, &img->width, &img->height, &img->maxval);
	error_exit_on(!img->pxls, -1, "Error read pgm imge file %s", file_name);
	fclose(fin);

	return img;
}

void pi_image_free(struct pi_image *img)
{
	pgm_freearray(img->pxls, img->height);
	free(img);
}

static inline gray pi_image_get_pxl(struct pi_image *img,
				    int x, int y)
{
	return img->pxls[y][x];
}

static inline void pi_image_set_pxl(struct pi_image *img,
				    int x, int y,
				    gray val)
{
	img->pxls[y][x] = val;
}

static inline int pi_image_row_find_nonempty(struct pi_image *img,
					     int y, int sx, int ex,
					     gray empty)
{
	int x;
	gray *r = img->pxls[y];

	for (x = sx; x < ex; x++) {
		if (r[x] < empty)
			return x;
	}
	return ex;
}

static inline int pi_image_row_rfind_nonempty(struct pi_image *img,
					     int y, int sx, int ex,
					     gray empty)
{
	int x;
	gray *r = img->pxls[y];

	for (x = sx; x > ex; x--) {
		if (r[x] < empty)
			return x;
	}
	return ex;
}

static inline int pi_image_col_find_nonempty(struct pi_image *img,
					     int x, int sy, int ey,
					     gray empty)
{
	int y;
	gray **pxls = img->pxls;

	for (y = sy; y < ey; y++) {
		if (pxls[y][x] < empty)
			return y;
	}
	return ey;
}

void pi_image_save(struct pi_image *img, const char *file_name)
{
	FILE *fout;

	fout = fopen(file_name, "w");
	error_exit_errno_on(!fout, -1, "Error open pgm image file %s",
			    file_name);
	pgm_writepgm(fout, img->pxls, img->width, img->height, img->maxval, 1);
	fclose(fout);
}

struct pi_image *pi_image_new_from(struct pi_image *img,
				   int width, int height)
{
	struct pi_image *nimg;

	assert(width > 0 && height > 0);

	nimg = malloc(sizeof(struct pi_image));
	error_exit_on(!nimg, -1, "Error alloc memory.");
	nimg->width = width;
	nimg->height = height;
	nimg->maxval = img->maxval;
	nimg->pxls = pgm_allocarray(width, height);

	return nimg;
}

void pi_image_fill(struct pi_image *img, gray v)
{
	int x, y;
	int iw = img->width, ih = img->height;
	gray **pxls = img->pxls;
	gray *row;

	for (y = 0; y < ih; y++) {
		row = pxls[y];
		for (x = 0; x < iw; x++)
			row[x] = v;
	}
}

void pi_image_copy(struct pi_image *dst, struct pi_image *src,
		   int dst_x, int dst_y, int src_x, int src_y,
		   int width, int height)
{
	gray **dst_pxls = dst->pxls;
	gray **src_pxls = src->pxls;
	gray *dst_row, *src_row;
	int x, y;

	for (y = 0; y < height; y++) {
		dst_row = dst_pxls[dst_y+y];
		src_row = src_pxls[src_y+y];
		for (x = 0; x < width; x++)
			dst_row[dst_x+x] = src_row[src_x+x];
	}
}

struct pi_image *pi_image_crop(struct pi_image *img,
			       int left, int top,
			       int width, int height)
{
	struct pi_image *crop;
	int x, y;

	assert(left >= 0 && top >= 0 && width > 0 && height > 0);
	assert(left + width <= img->width);
	assert(top + height <= img->height);

	crop = pi_image_new_from(img, width, height);
	pi_image_copy(crop, img, 0, 0, left, top, width, height);

	return crop;
}

struct pi_line *pi_line_new(struct pi_page *page,
			    int left, int top,
			    int width, int height,
			    int bl, int al)
{
	struct pi_line *line;

	line = malloc(sizeof(struct pi_line));
	error_exit_on(!line, -1, "Error alloc memory for pi_line.");
	line->page = page;
	line->bbox.left = left;
	line->bbox.top = top;
	line->bbox.width = width;
	line->bbox.height = height;
	line->bl = bl;
	line->al = al;

	return line;
}

void pi_line_free(struct pi_line *line)
{
	free(line);
}

void pi_line_save(struct pi_line *line, const char *file_name)
{
	struct pi_image *img;
	struct pi_rect *bbox = &line->bbox;

	img = pi_image_crop(line->page->img,
			    bbox->left, bbox->top,
			    bbox->width, bbox->height);
	pi_image_save(img, file_name);
	pi_image_free(img);
}

void pi_line_divide(struct pi_line *line, int dpts[], int ndpt,
		    int flex, struct pi_line *segs[])
{
	int i, n;
	int x, dx, dw, cw, sy, ey, sx, ex;
	int ln_lft, al, bl, il, ln_rt;
	struct pi_image *img = line->page->img;
	gray empty = line->page->doc->empty;

	sy = line->bbox.top;
	ey = line->bbox.top + line->bbox.height;
	ln_lft = line->bbox.left;
	ln_rt = line->bbox.left + line->bbox.width;
	bl = line->bl;
	al = line->al;
	il = al > bl ? bl : al;
	il = il <= 0 ? 5 : il; /* fix me */
	n = 0;
	for (i = 0; i < ndpt; i++) {
		dx = dpts[i];
		if (dx - flex <= ln_lft)
			continue;
		if (dx + flex >= ln_rt)
			break;
		dw = 0;
		cw = 0;
		for (x = dpts[i] - flex; x < dpts[i] + flex; x++) {
			if (pi_image_col_find_nonempty(img, x, sy, ey, empty)
			    == ey)
				cw++;
			else if (cw) {
				if (cw > dw) {
					dw = cw;
					dx = x;
				}
				cw = 0;
			}
		}
		segs[n++] = pi_line_new(line->page,
					ln_lft, sy,
					dx - dw - ln_lft, ey - sy,
					il, il);
		ln_lft = dx;
	}
	if (ln_lft < ln_rt) {
		segs[n++] = pi_line_new(line->page,
					ln_lft, sy,
					ln_rt - ln_lft, ey - sy,
					il, il);
	}
	if (n <= ndpt)
		segs[n] = NULL;
	segs[0]->bl = bl;
	segs[n-1]->al = al;
}

int pi_line_divide2(struct pi_line *line, int flex, struct pi_line *segs[])
{
	struct pi_page *page = line->page;
	struct pi_image *img = page->img;
	struct pi_rect *lbox = &line->bbox;
	int x, m, dx, dw, cw, sy, ey, il;
	int nrln_lft = (lbox->left - page->bbox.left) / 2;
	int npw = page->bbox.width / 2 + flex;
	gray empty = page->doc->empty;

	if (nrln_lft + line->bbox.width < npw) {
		segs[0] = pi_line_new(page, lbox->left, lbox->top,
				      lbox->width, lbox->height,
				      line->bl, line->al);
		return 1;
	}
	sy = lbox->top;
	ey = lbox->top + lbox->height;
	if (nrln_lft < page->bbox.width / 16)
		m = page->bbox.left + nrln_lft + page->bbox.width / 2;
	else
		m = lbox->left + lbox->width / 2;
	dx = m;
	dw = 0;
	cw = 0;
	for (x = m - flex; x < m + flex; x++) {
		if (pi_image_col_find_nonempty(img, x, sy, ey, empty) == ey)
			cw++;
		else if (cw) {
			if (cw > dw || (cw == dw && abs(x - m) < abs(dx - m))) {
				dw = cw;
				dx = x;
			}
			cw = 0;
		}
	}
	il = line->al > line->bl ? line->bl : line->al;
	il = il <= 0 ? 5 : il; /* fix me */
	segs[0] = pi_line_new(page, lbox->left, sy,
			      dx - dw - lbox->left, ey - sy,
			      line->bl, il);
	segs[1] = pi_line_new(page, dx, sy,
			      lbox->left + lbox->width - dx, ey - sy,
			      il, line->al);
	return 2;
}

#define pi_page_for_each_line(line, page)	\
	list_for_each_entry(line, &page->lines, list)

void pi_page_append_line(struct pi_page *page,
			 struct pi_line *line)
{
	list_add_tail(&line->list, &page->lines);
}

#define PS_TOP_MARGIN		0
#define PS_IN_LINE		1
#define PS_BETWEEN_LINE		2

void pi_page_parse(struct pi_page *page)
{
	struct pi_line *line;
	struct pi_image *img = page->img;
	int iw = img->width, ih = img->height;
	int x, rx, y;
	gray empty = page->doc->empty;
	int pg_lft = 0, pg_rt = 0, pg_top, pg_btm;
	int ln_lft, ln_rt, ln_top, ln_btm, bl, al;
	int state = PS_TOP_MARGIN;

	for (y = 0; y < ih; y++) {
		x = pi_image_row_find_nonempty(img, y, 0, iw, empty);
		// empty line
		if (x == iw) {
			if (state == PS_TOP_MARGIN) {
			} else if (state == PS_IN_LINE) {
				ln_btm = y;
				al++;
				state = PS_BETWEEN_LINE;
			} else {
				al++;
			}
		} else {
			rx = 1 + pi_image_row_rfind_nonempty(img, y, iw - 1,
							     0, empty);
			if (state == PS_TOP_MARGIN) {
				/* setup for first line */
				bl = ln_top = pg_top = y;
				al = 0;
				pg_lft = ln_lft = x;
				pg_rt = ln_rt = rx;
				state = PS_IN_LINE;
			} else if (state == PS_IN_LINE) {
				if (x < ln_lft)
					ln_lft = x;
				if (rx > ln_rt)
					ln_rt = rx;
			} else {
				if (ln_lft < pg_lft)
					pg_lft = ln_lft;
				if (ln_rt > pg_rt)
					pg_rt = ln_rt;
				line = pi_line_new(page,
						   ln_lft, ln_top,
						   ln_rt - ln_lft,
						   ln_btm - ln_top,
						   bl, al);
				pi_page_append_line(page, line);
				/* setup for next line */
				ln_top = y;
				ln_lft = x;
				ln_rt = rx;
				bl = al;
				al = 0;
				state = PS_IN_LINE;
			}
		}
	}

	if (state == PS_BETWEEN_LINE) {
		if (ln_lft < pg_lft)
			pg_lft = ln_lft;
		if (ln_rt > pg_rt)
			pg_rt = ln_rt;
		line = pi_line_new(page,
				   ln_lft, ln_top,
				   ln_rt - ln_lft,
				   ln_btm - ln_top,
				   bl, al);
		pi_page_append_line(page, line);
		pg_btm = ln_btm;
	} else if (state == PS_IN_LINE) {
		if (ln_lft < pg_lft)
			pg_lft = ln_lft;
		if (ln_rt > pg_rt)
			pg_rt = ln_rt;
		line = pi_line_new(page,
				   ln_lft, ln_top,
				   ln_rt - ln_lft,
				   y - ln_top,
				   bl, al);
		pi_page_append_line(page, line);
		pg_btm = y;
	} else {
		pg_lft = pg_rt = pg_top = pg_btm = 0;
	}

	page->bbox.left = pg_lft;
	page->bbox.top = pg_top;
	page->bbox.width = pg_rt - pg_lft;
	page->bbox.height = pg_btm - pg_top;
}

struct pi_page *pi_page_load(struct pi_doc *doc, const char *file_name)
{
	struct pi_page *page;

	page = malloc(sizeof(struct pi_page));
	error_exit_on(!page, -1, "Error alloc memory for pi_page.");
	page->doc = doc;
	INIT_LIST_HEAD(&page->lines);
	page->img = pi_image_load(file_name);
	pi_page_parse(page);
	return page;
}

void pi_page_free(struct pi_page *page)
{
	struct pi_line *l, *n;

	list_for_each_entry_safe(l, n, &page->lines, list) {
		list_del(&l->list);
		pi_line_free(l);
	}
	pi_image_free(page->img);
	free(page);
}

struct pi_doc *pi_doc_new(gray empty)
{
	struct pi_doc *doc;

	doc = malloc(sizeof(struct pi_doc));
	error_exit_on(!doc, -1, "Error alloc memory for pi_doc.");
	doc->empty = empty;
	return doc;
}

void pi_doc_free(struct pi_doc *doc)
{
	free(doc);
}

void pi_init(int *argc, char *argv[])
{
	pgm_init(argc, argv);
}

void usage(char *cmd)
{
	printf("usage: %s <in_pgm> <out_pgm>\n", basename(cmd));
}

int main1(int argc, char *argv[])
{
	struct pi_doc *doc;
	struct pi_page *page;
	struct pi_line *line, *segs[2], **pl, *l;
	struct pi_image *img, *cimg;
	struct pi_rect *bbox;
	int n = 0;
	char buf[32];
	int dpts[1];
	int y = 0, pg_lft, npg_rt = 0, ln_lft, npw;
	int il, ph;

	pi_init(&argc, argv);
	if (argc != 3) {
		usage(argv[0]);
		exit(-1);
	}
	doc = pi_doc_new(255);
	page = pi_page_load(doc, argv[1]);
	img = pi_image_new_from(page->img, page->bbox.width,
				page->bbox.height * 2);
	pi_image_fill(img, 255);
	pg_lft = page->bbox.left;
	ph = page->img->height;
	npw = page->bbox.width / 2;
	dpts[0] = page->bbox.left + npw;
	pi_page_for_each_line(line, page) {
		pi_line_divide(line, dpts, 1, npw / 16, segs);
		l = segs[0];
		bbox = &l->bbox;
		ln_lft = bbox->left - pg_lft;
		ln_lft %= npw;
		if (ln_lft + bbox->width - npw > npw / 4)
			ln_lft = 0;
		pi_image_copy(img, l->page->img, ln_lft, y,
			      bbox->left, bbox->top,
			      bbox->width, bbox->height);
		if (ln_lft + bbox->width > npg_rt)
			npg_rt = ln_lft + bbox->width;
		if (ln_lft > npw / 2)
			ln_lft = 0;
		y += bbox->height;
		il = l->al > bbox->height ? bbox->height : l->al;
		for (pl = segs + 1; pl < segs + 2 && *pl; pl++) {
			l = *pl;
			bbox = &l->bbox;
			y += il;
			pi_image_copy(img, l->page->img, ln_lft, y,
				      bbox->left, bbox->top,
				      bbox->width, bbox->height);
			y += bbox->height;
			if (ln_lft + bbox->width > npg_rt)
				npg_rt = ln_lft + bbox->width;
		}
		il = l->al > ph / 16 ? ph / 16 : l->al;
		y += il;
	}
	y -= il;
	cimg = pi_image_crop(img, 0, 0, npg_rt, y);
	pi_image_save(cimg, argv[2]);
	pi_image_free(cimg);
	pi_image_free(img);
	pi_page_free(page);
	pi_doc_free(doc);

	return 0;
}

int main(int argc, char *argv[])
{
	struct pi_doc *doc;
	struct pi_page *page;
	struct pi_line *line, *segs[2], *l;
	struct pi_image *oimg, *img, *cimg;
	struct pi_rect *lbox;
	int n = 0;
	int y = 0, pg_lft, npg_rt = 0, npw;
	int il, ph;
	int flex, nln_lft;

	pi_init(&argc, argv);
	if (argc != 3) {
		usage(argv[0]);
		exit(-1);
	}
	doc = pi_doc_new(255);
	page = pi_page_load(doc, argv[1]);
	oimg = page->img;
	img = pi_image_new_from(oimg, page->bbox.width,
				page->bbox.height * 2);
	pi_image_fill(img, 255);
	pg_lft = page->bbox.left;
	ph = oimg->height;
	npw = page->bbox.width / 2;
	flex = npw / 6;

	pi_page_for_each_line(line, page) {
		n = pi_line_divide2(line, flex, segs);
		l = segs[0];
		lbox = &l->bbox;
		nln_lft = (lbox->left - pg_lft) / 2;
		pi_image_copy(img, oimg, nln_lft, y,
			      lbox->left, lbox->top,
			      lbox->width, lbox->height);
		if (nln_lft + lbox->width > npg_rt)
			npg_rt = nln_lft + lbox->width;
		y += lbox->height;
		if (n == 2) {
			il = l->al > lbox->height ? lbox->height : l->al;
			y += il;
			l = segs[1];
			lbox = &l->bbox;
			pi_image_copy(img, oimg, nln_lft, y,
				      lbox->left, lbox->top,
				      lbox->width, lbox->height);
			y += lbox->height;
			if (nln_lft + lbox->width > npg_rt)
				npg_rt = nln_lft + lbox->width;
		}
		il = l->al > ph / 16 ? ph / 16 : l->al;
		y += il;
		pi_line_free(segs[0]);
		if (n == 2)
			pi_line_free(segs[1]);
	}
	y -= il;
	cimg = pi_image_crop(img, 0, 0, npg_rt, y);
	pi_image_save(cimg, argv[2]);
	pi_image_free(cimg);
	pi_image_free(img);
	pi_page_free(page);
	pi_doc_free(doc);

	return 0;
}
