#!/usr/bin/env python
# -*- coding: utf-8 -*-

EPUB3_WITH_NCX = True
""" Set to True to create a toc.ncx when converting to epub3. """

EPUB3_WITH_GUIDE = True
""" Set to True to create a guide element in an opf when converting to epub3. """

DEBUG = False
if DEBUG:
    EPUB3_WITH_NCX = False
    EPUB3_WITH_GUIDE = False


import sys, os, re, uuid
from path import pathof
from xml.sax.saxutils import escape as xmlescape
from HTMLParser import HTMLParser
EXTRA_ENTITIES = {'"':'&quot;', "'":"&apos;"}
import mobi_taglist as taglist

class OPFProcessor(object):
    def __init__(self, files, metadata, filenames, imgnames, isNCX, mh, usedmap, guidetext=False, k8resc=None, navname=None):
        self.files = files
        self.metadata = metadata
        self.filenames = filenames
        self.imgnames = imgnames
        self.isNCX = isNCX
        self.codec = mh.codec
        self.isK8 = mh.isK8()
        self.printReplica = mh.isPrintReplica()
        self.guidetext = guidetext
        self.used = usedmap
        self.k8resc = k8resc
        self.navname = navname
        self.covername = None
        self.cover_id = None
        self.h = HTMLParser()
        # Create a unique urn uuid
        self.BookId = str(uuid.uuid4())
        self.starting_offset = None
        self.page_progression_direction = None

    def escapeit(self, sval, EXTRAS=None):
        # note, xmlescape and unescape do not work with utf-8 bytestrings
        # so pre-convert to full unicode and then convert back since opf is utf-8 encoded
        uval = sval.decode('utf-8')
        if EXTRAS:
            ures = xmlescape(self.h.unescape(uval), EXTRAS)
        else:
            ures = xmlescape(self.h.unescape(uval))
        return ures.encode('utf-8')

    def createItemidSet(self):
        # Create an id set to prevent id confliction.
        k8resc = self.k8resc
        cover_id = self.cover_id
        if k8resc != None and k8resc.hasSpine():
            #itemidset = set(k8resc.getSpineItemidList() + (cover_id,))
            itemidset = set(k8resc.getSpineItemidList())
        else:
            #itemidset = set((cover_id,))
            itemidset = set()
        if cover_id != None:
            itemidset.add(cover_id)

        return itemidset

    def createItemid(self, base, itemidset):
        # Create an itemid.
            idsuf = 0
            itemid = base
            while itemid in itemidset:
                itemid = '{0:s}-{1:d}'.format(base, idsuf)
                idsuf += 1
            itemidset.add(itemid)
            return itemid


    def buildOPFMetadata(self, has_obfuscated_fonts=False):
        metadata = self.metadata
        k8resc = self.k8resc

        META_TAGS = ['Drm Server Id', 'Drm Commerce Id', 'Drm Ebookbase Book Id', 'ASIN', 'ThumbOffset', 'Fake Cover',
                                                'Creator Software', 'Creator Major Version', 'Creator Minor Version', 'Creator Build Number',
                                                'Watermark', 'Clipping Limit', 'Publisher Limit', 'Text to Speech Disabled', 'CDE Type',
                                                'Updated Title', 'Font Signature (hex)', 'Tamper Proof Keys (hex)',  ]
        def handleTag(data, metadata, key, tag):
            '''
            Format metadata values.

            @param data: List of formatted metadata entries.
            @param metadata: The metadata dictionary.
            @param key: The key of the metadata value to handle.
            @param tag: The opf tag the the metadata value.
            '''
            if key in metadata.keys():
                for value in metadata[key]:
                    closingTag = tag.split(" ")[0]
                    res = '<%s>%s</%s>\n' % (tag, self.escapeit(value), closingTag)
                    data.append(res)
                del metadata[key]

        def handleMetaPairs(data, metadata, key, name):
            if key in metadata.keys():
                for value in metadata[key]:
                    res = '<meta name="%s" content="%s" />\n' % (name, self.escapeit(value, EXTRA_ENTITIES))
                    data.append(res)
                del metadata[key]

        data = []
        data.append('<metadata xmlns:dc="http://purl.org/dc/elements/1.1/" xmlns:opf="http://www.idpf.org/2007/opf">\n')
        # Handle standard metadata
        if 'Title' in metadata.keys():
            handleTag(data, metadata, 'Title', 'dc:title')
        else:
            data.append('<dc:title>Untitled</dc:title>\n')
        handleTag(data, metadata, 'Language', 'dc:language')
        if 'UniqueID' in metadata.keys():
            handleTag(data, metadata, 'UniqueID', 'dc:identifier id="uid"')
        else:
            # No unique ID in original, give it a generic one.
            data.append('<dc:identifier id="uid">0</dc:identifier>\n')
        if self.isK8 and has_obfuscated_fonts:
            # Use the random generated urn:uuid so obuscated fonts work.
            # It doesn't need to be _THE_ unique identifier to work as a key
            # for obfuscated fonts in Sigil, ADE and calibre. Its just has
            # to use the opf:scheme="UUID" and have the urn:uuid: prefix.
            data.append('<dc:identifier opf:scheme="UUID">urn:uuid:'+self.BookId+'</dc:identifier>\n')

        handleTag(data, metadata, 'Creator', 'dc:creator')
        handleTag(data, metadata, 'Contributor', 'dc:contributor')
        handleTag(data, metadata, 'Publisher', 'dc:publisher')
        handleTag(data, metadata, 'Source', 'dc:source')
        handleTag(data, metadata, 'Type', 'dc:type')
        handleTag(data, metadata, 'ISBN', 'dc:identifier opf:scheme="ISBN"')
        if 'Subject' in metadata.keys():
            if 'SubjectCode' in metadata.keys():
                codeList = metadata['SubjectCode']
                del metadata['SubjectCode']
            else:
                codeList = None
            for i in range(len(metadata['Subject'])):
                if codeList and i < len(codeList):
                    data.append('<dc:subject BASICCode="'+codeList[i]+'">')
                else:
                    data.append('<dc:subject>')
                data.append(self.escapeit(metadata['Subject'][i])+'</dc:subject>\n')
            del metadata['Subject']
        handleTag(data, metadata, 'Description', 'dc:description')
        handleTag(data, metadata, 'Published', 'dc:date opf:event="publication"')
        handleTag(data, metadata, 'Rights', 'dc:rights')
        handleTag(data, metadata, 'DictInLanguage', 'DictionaryInLanguage')
        handleTag(data, metadata, 'DictOutLanguage', 'DictionaryOutLanguage')

       # page-progression-direction
        ppd_key = 'page-progression-direction'
        self.page_progression_direction = metadata.pop(ppd_key, [None])[0]
        pwm_key = 'primary-writing-mode'
        pwm_value = metadata.pop(pwm_key, [None])[0]
        if pwm_value != None:
            data.append('<meta name="'+pwm_key+'" content="'+self.escapeit(pwm_value, EXTRA_ENTITIES)+'" />\n')
            if 'rl' in pwm_value:
                self.page_progression_direction = 'rtl'

        # Append metadata in RESC section.
        cover_id = None
        if k8resc != None:
            cover_id = k8resc.cover_id
            resc_metadata_ = k8resc.metadata_toxml()
            #index = taglist.find(resc_metadata_, 'meta', 'name', 'cover')
            #if index != None:
            #    cover_id = taglist.get_attrib(resc_metadata_, index, 'content')
            indices = taglist.findall(resc_metadata_, '<!--')
            #indices += taglist.findall(resc_metadata_, 'meta', 'refines')
            indices_start = taglist.findall(resc_metadata_, '<!-- Begin: Tags will be ignored by kindlegen')
            indices_end = taglist.findall(resc_metadata_, '<!-- End: Tags will be ignored by kindlegen')
            if len(indices_start) > 0 and len(indices_end) > 0 and len(indices_start) == len(indices_end):
                for istart, iend in zip(indices_start, indices_end):
                    indices += range(istart, iend + 1)
            new_metadata = taglist.remove_tags(resc_metadata_, indices)
            if len(new_metadata) > 0:
                data.append('<!-- Begin imported from RESC section -->\n')
                data += new_metadata
                data.append('<!-- End imported from RESC section -->\n')

        if 'CoverOffset' in metadata.keys():
            imageNumber = int(metadata['CoverOffset'][0])
            self.covername = self.imgnames[imageNumber]
            if self.covername is None:
                print "Error: Cover image %s was not recognized as a valid image" % imageNumber
            else:
                if cover_id == None:
                    cover_id = 'cover_img'
                    data.append('<meta name="cover" content="' + cover_id + '" />\n')
                self.used[self.covername] = 'used'
            del metadata['CoverOffset']
        self.cover_id = cover_id
        i = taglist.find(data, 'meta', 'name', 'output encoding')
        if i != None:
            data[i] = ''
        handleMetaPairs(data, metadata, 'Codec', 'output encoding')
        # handle kindlegen specifc tags
        handleMetaPairs(data, metadata, 'RegionMagnification', 'RegionMagnification')
        handleMetaPairs(data, metadata, 'fixed-layout', 'fixed-layout')
        handleMetaPairs(data, metadata, 'original-resolution', 'original-resolution')
        handleMetaPairs(data, metadata, 'orientation-lock', 'orientation-lock')
        handleMetaPairs(data, metadata, 'book-type', 'book-type')
        handleMetaPairs(data, metadata, 'zero-gutter', 'zero-gutter')
        handleMetaPairs(data, metadata, 'zero-margin', 'zero-margin')

        handleTag(data, metadata, 'Review', 'Review')
        handleTag(data, metadata, 'Imprint', 'Imprint')
        handleTag(data, metadata, 'Adult', 'Adult')
        handleTag(data, metadata, 'DictShortName', 'DictionaryVeryShortName')
        if 'Price' in metadata.keys() and 'Currency' in metadata.keys():
            priceList = metadata['Price']
            currencyList = metadata['Currency']
            if len(priceList) != len(currencyList):
                print "Error: found %s price entries, but %s currency entries."
            else:
                for i in range(len(priceList)):
                    data.append('<SRP Currency="'+currencyList[i]+'">'+priceList[i]+'</SRP>\n')
            del metadata['Price']
            del metadata['Currency']
        data.append('<!-- Begin: Tags will be ignored by kindlegen -->\n')
        data.append("<!-- The following meta tags are just for information and will be ignored by mobigen/kindlegen. -->\n")
        if 'ThumbOffset' in metadata.keys():
            imageNumber = int(metadata['ThumbOffset'][0])
            imageName = self.imgnames[imageNumber]
            if imageName is None:
                print "Error: Cover Thumbnail image %s was not recognized as a valid image" % imageNumber
            else:
                data.append('<meta name="Cover ThumbNail Image" content="'+ 'Images/'+imageName+'" />\n')
                # self.used[imageName] = 'used' # thumbnail image is always generated by Kindlegen, so don't include in manifest
                self.used[imageName] = 'not used'
            del metadata['ThumbOffset']
        for metaName in META_TAGS:
            if metaName in metadata.keys():
                for value in metadata[metaName]:
                    data.append('<meta name="'+metaName+'" content="'+self.escapeit(value, EXTRA_ENTITIES)+'" />\n')
                    del metadata[metaName]
        for key in metadata.keys():
            for value in metadata[key]:
                if key == 'StartOffset':
                    if int(value) == 0xffffffff:
                        value = '0'
                    self.starting_offset = value
                data.append('<meta name="'+key+'" content="'+self.escapeit(value, EXTRA_ENTITIES)+'" />\n')
            del metadata[key]
        data.append('<!-- End: Tags will be ignored by kindlegen -->\n')
        data.append('</metadata>\n')
        return data


    def buildOPFManifest(self):
        # for EPUB2 manifest.
        k8resc = self.k8resc
        cover_id = self.cover_id
        hasK8RescSpine = k8resc != None and k8resc.hasSpine()

        data = []
        # build manifest
        data.append('<manifest>\n')
        media_map = {
                '.jpg'  : 'image/jpeg',
                '.jpeg' : 'image/jpeg',
                '.png'  : 'image/png',
                '.gif'  : 'image/gif',
                '.svg'  : 'image/svg+xml',
                '.xhtml': 'application/xhtml+xml',
                '.html' : 'text/x-oeb1-document',
                '.pdf'  : 'application/pdf',
                '.ttf'  : 'application/x-font-ttf',
                '.otf'  : 'application/x-font-opentype',
                '.css'  : 'text/css'
                }
        spinerefs = []

        # Create an id set to prevent id confliction.
        itemidset = self.createItemidSet()

        idcnt = 0
        for [dir,fname] in self.filenames:
            name, ext = os.path.splitext(fname)
            ext = ext.lower()
            media = media_map.get(ext)
            ref = self.createItemid('item{:d}'.format(idcnt), itemidset)
            #if k8resc != None and k8resc.hasSpine():
            if hasK8RescSpine:
                index = k8resc.getSpineIndexByFilename(fname)
                if index != None:
                    ref = k8resc.getSpineIdref(index)
                    k8resc.setSpineStatus(index, True)

            if dir != '':
                data.append('<item id="' + ref + '" media-type="' + media + '" href="' + dir + '/' + fname +'" />\n')
            else:
                data.append('<item id="' + ref + '" media-type="' + media + '" href="' + fname +'" />\n')
            if ext in ['.xhtml', '.html']:
                spinerefs.append(ref)
            idcnt += 1

        for fname in self.imgnames:
            if fname != None:
                if self.used.get(fname,'not used') == 'not used':
                    continue
                name, ext = os.path.splitext(fname)
                ext = ext.lower()
                media = media_map.get(ext,ext[1:])
                if fname == self.covername:
                    ref = cover_id
                    # ref += '" properties="cover-image'
                else:
                    ref = self.createItemid('item{:d}'.format(idcnt), itemidset)

                if ext == '.ttf' or ext == '.otf':
                    data.append('<item id="' + ref + '" media-type="' + media + '" href="Fonts/' + fname +'" />\n')
                else:
                    data.append('<item id="' + ref + '" media-type="' + media + '" href="Images/' + fname +'" />\n')
                idcnt += 1

        if self.isNCX:
            if self.isK8:
                ncxname = 'toc.ncx'
            else:
                ncxname = self.files.getInputFileBasename() + '.ncx'
            data.append('<item id="ncx" media-type="application/x-dtbncx+xml" href="' + ncxname +'"></item>\n')
        data.append('</manifest>\n')
        return [data, spinerefs]

    # Building manifest for EPUB3 is separated due to many differences.
    def buildEpub3OPFManifest(self, has_ncx=True):
        # for EPUB3 manifest.
        files = self.files
        k8resc = self.k8resc
        cover_id = self.cover_id
        navname = self.navname
        hasK8RescSpine = k8resc != None and k8resc.hasSpine()

        data = []
        # build manifest
        data.append('<manifest>\n')
        media_map = {
                '.jpg'  : 'image/jpeg',
                '.jpeg' : 'image/jpeg',
                '.png'  : 'image/png',
                '.gif'  : 'image/gif',
                '.svg'  : 'image/svg+xml',
                '.xhtml': 'application/xhtml+xml',
                '.html' : 'text/x-oeb1-document', # obsoleted?
                '.pdf'  : 'application/pdf', # obsoleted?
                '.ttf'  : 'application/x-font-ttf',
                '.otf'  : 'application/x-font-opentype', # replaced?
                #'.otf' : 'application/vnd.ms-opentype', # [OpenType] OpenType fonts
                #'.woff' : 'application/font-woff', # [WOFF] WOFF fonts
                #'.smil' : 'application/smil+xml', # [MediaOverlays301] EPUB Media Overlay documents
                #'.pls' : 'application/pls+xml', # [PLS] Text-to-Speech (TTS) Pronunciation lexicons
                #'.mp3'  : 'audio/mpeg',
                #'.mp4'  : 'audio/mp4',
                #'.js'   : 'text/javascript',
                '.css'  : 'text/css'
                }
        spinerefs = []

        # Create an id set to prevent id confliction.
        itemidset = self.createItemidSet()

        idcnt = 0
        for [dir,fname] in self.filenames:
            name, ext = os.path.splitext(fname)
            ext = ext.lower()
            media = media_map.get(ext)
            ref = self.createItemid('item{:d}'.format(idcnt), itemidset)
            properties = ''
            #if k8resc != None and k8resc.hasSpine():
            if hasK8RescSpine:
                index = k8resc.getSpineIndexByFilename(fname)
                if index != None:
                    ref = k8resc.getSpineIdref(index)
                    k8resc.setSpineStatus(index, True)
            #if fname == navname:
            #    properties = 'properties="nav"'
            if dir != '':
                fpath = dir + '/' + fname
            else:
                fpath = fname
            data.append('<item id="{0:}" media-type="{1:}" href="{2:}" {3:}/>\n'.format(ref, media, fpath, properties))

            if ext in ['.xhtml', '.html']:
                #if fname != navname: # To prevent inserting nav.xhtml to spine.
                #    spinerefs.append(ref)
                spinerefs.append(ref)
            idcnt += 1

        for fname in self.imgnames:
            if fname != None:
                if self.used.get(fname,'not used') == 'not used':
                    continue
                name, ext = os.path.splitext(fname)
                ext = ext.lower()
                media = media_map.get(ext,ext[1:])
                properties = ''
                if fname == self.covername:
                    ref = cover_id
                    properties = 'properties="cover-image"'
                else:
                    ref = self.createItemid('item{:d}'.format(idcnt), itemidset)
                if ext == '.ttf' or ext == '.otf':
                    fpath = 'Fonts/' + fname
                else:
                    fpath = 'Images/' + fname
                data.append('<item id="{0:}" media-type="{1:}" href="{2:}" {3:}/>\n'.format(ref, media, fpath, properties))
                idcnt += 1

        if navname != None:
            media = media_map.get('.xhtml')
            ref = self.createItemid('nav', itemidset)
            properties = 'properties="nav"'
            textdir = os.path.relpath(files.k8text, files.k8oebps).replace('\\', '/') + '/'
            fpath = textdir + navname
            data.append('<item id="{0:}" media-type="{1:}" href="{2:}" {3:}/>\n'.format(ref, media, fpath, properties))

        if self.isNCX and has_ncx:
            if self.isK8:
                ncxname = 'toc.ncx'
            else:
                ncxname = self.files.getInputFileBasename() + '.ncx'
            data.append('<item id="ncx" media-type="application/x-dtbncx+xml" href="' + ncxname +'"></item>\n')
        data.append('</manifest>\n')
        return [data, spinerefs]


    def buildOPFSpine(self, spinerefs, has_ncx=True):
        k8resc = self.k8resc
        hasK8RescSpine = k8resc != None and k8resc.hasSpine()
        data = []
        # build spine
        spine_start_tag = '<spine'
        if self.page_progression_direction != None:
            spine_start_tag += ' page-progression-direction="{:s}"'.format(self.page_progression_direction)
        if self.isNCX and has_ncx:
            spine_start_tag += ' toc="ncx"'
        spine_start_tag += '>\n'
        data.append(spine_start_tag)
        #if k8resc != None and k8resc.hasSpine():
        if hasK8RescSpine:
            spine_ = k8resc.spine_toxml()
            data += spine_
        else:
            for entry in spinerefs:
                data.append('<itemref idref="' + entry + '"/>\n')
        data.append('</spine>\n')
        return data

    def buildOPFTours(self):
        data = []
        data.append('<tours>\n</tours>\n')
        return data

    def buildOPFGuide(self):
        data = []
        if not self.printReplica:
            metaguidetext = ''
            if not self.isK8:
                # get guide items from metadata (note starting offset previsouly processed)
                if self.starting_offset != None:
                    so = self.starting_offset
                    metaguidetext += '<reference type="text" href="'+self.filenames[0][1]+'#filepos'+so+'" />\n'
            data.append('<guide>\n' + metaguidetext + self.guidetext + '</guide>\n')
        return data


    def buildOPF(self, has_obfuscated_fonts=False):
        print "Build opf"
        #metadata = self.metadata
        #k8resc = self.k8resc
        data = []
        data.append('<?xml version="1.0" encoding="utf-8"?>\n')
        data.append('<package version="2.0" xmlns="http://www.idpf.org/2007/opf" unique-identifier="uid">\n')
        opf_metadata = self.buildOPFMetadata(has_obfuscated_fonts)
        #opf_metadata[0] ='<metadata xmlns:dc="http://purl.org/dc/elements/1.1/" xmlns:opf="http://www.idpf.org/2007/opf">\n'
        data += opf_metadata
        [opf_manifest, spinerefs] = self.buildOPFManifest()
        data += opf_manifest
        opf_spine = self.buildOPFSpine(spinerefs)
        data += opf_spine
        opf_tours = self.buildOPFTours()
        data += opf_tours
        opf_guide = self.buildOPFGuide()
        data += opf_guide
        data.append('</package>\n')
        return ''.join(data)

    def buildEpub3OPF(self, has_obfuscated_fonts=False):
        print "Build epub3 opf"
        #metadata = self.metadata
        #k8resc = self.k8resc
        has_ncx = EPUB3_WITH_NCX
        has_guide = EPUB3_WITH_GUIDE

        data = []
        data.append('<?xml version="1.0" encoding="utf-8"?>\n')
        data.append('<package version="3.0" xmlns="http://www.idpf.org/2007/opf" prefix="rendition: http://www.idpf.org/vocab/rendition/#" unique-identifier="uid">\n')

        #self.k8resc.metadata_outall = True
        opf_metadata = self.buildOPFMetadata(has_obfuscated_fonts)
        # Overwrite metadata start tag.
        #opf_metadata[0] = '<metadata xmlns:dc="http://purl.org/dc/elements/1.1/">\n') # XXX: kobo cannot process.
        opf_metadata[0] ='<metadata xmlns:dc="http://purl.org/dc/elements/1.1/" xmlns:opf="http://www.idpf.org/2007/opf">\n'
        newmetadata = self.convertAmazonMetadataToEpub3(opf_metadata)
        opf_metadata = newmetadata
        ## Delete obsoleted meta
        #i = taglist.find(opf_metadata, 'meta', 'name', 'cover')
        #opf_metadata = opf_metadata[:i] + opf_metadata[i+1:]
        data += opf_metadata

        [opf_manifest, spinerefs] = self.buildEpub3OPFManifest(has_ncx)
        data += opf_manifest

        opf_spine = self.buildOPFSpine(spinerefs, has_ncx)
        data += opf_spine
        #opf_tours = self.buildOPFTours()
        #data += opf_tours
        if has_guide:
            opf_guide = self.buildOPFGuide()
            data += opf_guide
        data.append('</package>\n')
        return ''.join(data)


    def writeOPF(self, has_obfuscated_fonts=False, epubver=2):
        # write out the metadata as an OEB 1.0 OPF file
        print "Write opf"
        if epubver >= 3 and self.isK8:
            data = self.buildEpub3OPF(has_obfuscated_fonts)
        else:
            data = self.buildOPF(has_obfuscated_fonts)

        if self.isK8:
            outopf = os.path.join(self.files.k8oebps,'content.opf')
        else:
            outopf = os.path.join(self.files.mobi7dir, self.files.getInputFileBasename() + '.opf')
        #open(pathof(outopf), 'wb').write("".join(data))
        open(pathof(outopf), 'wb').write(data)
        if self.isK8:
            return self.BookId


    def convertAmazonMetadataToEpub3(self, metadata):
        # Convert Amazon definded meta to epub3 meta.
        overwrite_rendition_properties = True

        insert_pos = taglist.find(metadata, '<!-- Begin: Tags will be ignored by kindlegen')
        if insert_pos == None:
            insert_pos = taglist.find(metadata, '<!-- The following meta tags are just for information and will be ignored by mobigen/kindlegen.')
        if insert_pos == None:
            insert_pos = len(metadata) - 1
        opf_metadata = metadata[:insert_pos]
        opf_metadata_tail = metadata[insert_pos:]

        isrc = taglist.find(opf_metadata, 'meta', 'name', 'fixed-layout')
        if isrc != None:
            fixedlayout = taglist.get_attrib(opf_metadata, isrc, 'content')
            content = {'true' : 'pre-paginated'}.get(fixedlayout.lower(), 'reflowable')
            idst = taglist.find(opf_metadata, 'meta', 'property', 'rendition:layout')
            if idst == None:
                newdata = '<meta property="rendition:layout">' + content + '<meta>\n'
                opf_metadata.append(newdata)
            elif overwrite_rendition_properties:
                newdata = taglist.set_content(opf_metadata, idst, content)
                opf_metadata[idst] = newdata

        isrc = taglist.find(opf_metadata, 'meta', 'name', 'orientation-lock')
        if isrc != None:
            orientation = taglist.get_attrib(opf_metadata, isrc, 'content')
            content = {'none' : 'auto'}.get(orientation.lower(), orientation)
            idst = taglist.find(opf_metadata, 'meta', 'property', 'rendition:orientation')
            if idst == None:
                newdata = '<meta property="rendition:orientation">' + content + '<meta>\n'
                opf_metadata.append(newdata)
            elif overwrite_rendition_properties:
                newdata = taglist.set_content(opf_metadata, idst, content)
                opf_metadata[idst] = newdata

        # Not certain whether corresponded or not.
        isrc = taglist.find(opf_metadata, 'meta', 'name', 'original-resolution')
        #if isrc != None:
        if False:
            resolution = taglist.get_attrib(opf_metadata, isrc, 'content')
            re_resolution = re.compile(r'(?P<width>\d+)\D+(?P<height>\d+)', re.I)
            mo_resolution = re_resolution.search(resolution)
            if mo_resolution != None:
                width = mo_resolution.group('width')
                height = mo_resolution.group('height')
                if int(width) > 0 and int(height) > 0:
                    viewport = 'width={}, height={}'.format(width, height)
            idst = taglist.find(opf_metadata, 'meta', 'property', 'rendition:viewport')
            if idst == None:
                newdata = '<meta property="rendition:viewport">' + viewport + '<meta>\n'
                opf_metadata.append(newdata)
            elif overwrite_rendition_properties:
                newdata = taglist.set_content(opf_metadata, idst, viewport)
                opf_metadata[idst] = newdata

        opf_metadata += opf_metadata_tail
        return opf_metadata
