#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys, os, re
#import xml.dom
#import xml.dom.minidom
#from path import pathof

class Metadata(object):
    """Class for metadata section.

    Data structure is the following:
    [0:element, 1:type_id, 2:tag, 3:attribs, 4:isEmpty, 5:start_tag, 6:content, 7:end_tag]
    """
    METADATA_START = 'start'
    METADATA_DC = 'dc'
    METADATA_META = 'meta'
    METADATA_LINK = 'link'
    METADATA_OTHER = 'other'
    METADATA_COMMENT = 'comment'
    METADATA_END = 'end'
    def __init__(self):
        self.data = None
        metadata_type = dict([ \
            [self.METADATA_START, 0], \
            [self.METADATA_DC, 1], \
            [self.METADATA_META, 2], \
            [self.METADATA_LINK, 3], \
            [self.METADATA_OTHER, 4], \
            [self.METADATA_COMMENT, -2], \
            [self.METADATA_END, -1]])
        metadata_type_inv = dict(zip(metadata_type.values(), metadata_type.keys()))
        self.metadata_type = metadata_type
        self.metadata_type_inv = metadata_type_inv

        self.tag_types = [self.METADATA_DC,
                          self.METADATA_META,
                          self.METADATA_LINK]

        self.re_metadata = re.compile(r'(<metadata[^>]*>)(.*?)(</metadata>)', re.I|re.S)
        self.re_attrib = re.compile(r'\s*(?P<attrib>\S+)\s*=\s*"(?P<value>[^"]*)"', re.I)
        self.re_element = re.compile(r'''
                (?P<comment><!--.*?-->)
            |
                (?P<start_tag><(?P<tag>\S+).*?((?P<empty>/>)|>))
                (?(empty)|(?P<content>[^<]*)(?P<end_tag></(?P=tag)>))
            ''', re.X|re.I|re.S)

        re_pattern = ''
        tag_types = self.tag_types
        for tag_type in tag_types[:-1]:
            re_pattern += '(?P<{}><{})|'.format(tag_type, tag_type)
        else:
            re_pattern += '(?P<{}><{})'.format(tag_types[-1], tag_types[-1])
        self.re_tag_type = re.compile(re_pattern, re.I)


    def process(self, src):
        """Import metadata from src.

        """
        re_metadata = self.re_metadata
        re_element = self.re_element
        re_attrib = self.re_attrib
        re_tag_type = self.re_tag_type

        mo_meta = re_metadata.search(src)
        if mo_meta != None:
            data = []
            #[0:element, 1:type_id, 2:tag, 3:attribs, 4:isEmpty, 5:start_tag, 6:content, 7:end_tag]
            data.append([mo_meta.group(1), self.getTypeId(self.METADATA_START),
                         None, {}, None, None, None, None])

            elements = mo_meta.group(2)
            pos = 0
            mo_element = re_element.search(elements, pos)
            while mo_element != None:
                if mo_element.group('comment') != None:
                    comment = mo_element.group()
                    data.append([mo_element.group(), self.getTypeId(self.METADATA_COMMENT),
                                 None, {}, None, None, None, None])

                elif mo_element.group('start_tag') != None:
                    start_tag = mo_element.group('start_tag')
                    mo_type = re_tag_type.match(start_tag)
                    if mo_type == None:
                        type_id = self.getTypeId(self.METADATA_OTHER)
                    else:
                        for tag in self.tag_types:
                            if mo_type.group(tag) != None:
                                type_id = self.getTypeId(tag)
                                break
                            else:
                                type_id = self.getTypeId(self.METADATA_OTHER)

                    tag_name = mo_element.group('tag')
                    content = mo_element.group('content')
                    end_tag = mo_element.group('end_tag')
                    attribs = dict(re_attrib.findall(start_tag))
                    isEmpty = mo_element.group('empty') != None

                    #[0:element, 1:type_id, 2:tag, 3:attribs, 4:isEmpty, 5:start_tag, 6:content, 7:end_tag]
                    data.append([mo_element.group(), type_id,
                                 tag_name, attribs, isEmpty,
                                 start_tag, content, end_tag])
                pos = mo_element.end()
                mo_element = re_element.search(elements, pos)

            data.append([mo_meta.group(3), self.getTypeId(self.METADATA_END),
                         None, {}, None, None, None, None])
            self.data = data


    def toxml(self):
        if self.data == None:
            return []
        metadata_ = []
        for [element, typeid, tag, attribs, isEmpty, start, content, end] \
                in self.data[1:-1]:
            if 'refines' in attribs:
                continue
            metadata_.append(element + '\n')
        return metadata_

    def getTypeId(self, type_):
        return self.metadata_type.get(type_)

    def getType(self, type_id):
        return self.metadata_type_inv.get(type_id)

    def getNumberOfElements(self):
        if self.data == None:
            return 0
        else:
            return len(self.data)

    def getElement(self, index):
        """Return a sturctured metadata.

        data structure is:
        [0:element, 1:type_id, 2:tag, 3:attribs, 4:isEmpty, 5:start_tag, 6:content, 7:end_tag]
        """
        return self.data[index]

    def getElements(self, indices=None):
        """Return sturctured metadatas.

        data structure is:
        [0:element, 1:type_id, 2:tag, 3:attribs, 4:isEmpty, 5:start_tag, 6:content, 7:end_tag]
        """
        if indices == None:
            return self.data
        else: #elif isinstance(indices, list):
            return [self.data[i] for i in indices]


class Spine(object):
    """Class for spine section.

    Data structure is the following:
    [0:itemref, 1:skelid, 2:itemid, 3:isvalid, 4:filename]
    """
    def __init__(self):
        self.data = None #[itemref, skelid, itemid, isvalid, filename]
        self.skelid_dict = None
        self.filename_dict = None

    def process(self, src):
        """Import spine from src.

        """
        mo_spine = re.search(r'(<spine[^>]*>)(.*?)(</spine>)', src, re.I|re.S)
        if mo_spine != None:
            data = []
            data.append([mo_spine.group(1), None, None, True, None])

            # process itemrefs
            data_ = re.sub(r'<!--.*?-->', '', mo_spine.group(2), 0, re.S)
            itemrefs = re.findall(r'<[^>]*>', data_)
            re_idref = re.compile(r'(.*?)\s*idref="([^"]*)"(.*)', re.I)
            re_skelid = re.compile(r'(.*?)\s*skelid="([^"]*)"(.*)', re.I)
            for itemref in itemrefs:
                mo_idref = re_idref.search(itemref)
                if mo_idref != None:
                    striped_itemref = mo_idref.group(1) + mo_idref.group(3)
                    itemid = mo_idref.group(2)
                else:
                    striped_itemref = itemref
                    print 'Warning: no itemid in <itemref /> in the spine of RESC.'
                    break

                mo_skelid = re_skelid.search(striped_itemref)
                if mo_skelid != None:
                    striped_itemref = mo_skelid.group(1) + mo_skelid.group(3)
                    if mo_skelid.group(2).isdigit():
                        skelid = int(mo_skelid.group(2))
                    else:
                        skelid = -1
                else:
                    skelid = -1

                data.append([striped_itemref, skelid, itemid, False, None])
            else:
                data.append(['</spine>', None, None, True, None])
                #pairs = [[data[i][1], i] for i in range(1, len(spine)-1)]
                #skelid_dict = dict(pairs)
                self.data = data
                self.createSkelidDict()
        return

    def create(self, itemidbase, num):
        """Create a dummy spine.

        """
        data = [['<spine toc="ncx">', None, None, True, None]]
        for i in range(num):
            data.append(['<itemref/>', i, '{:s}{:d}'.format(itemidbase, i), False, None])
        data.append(['</spine>', None, None, True, None])
        self.data = data
        self.createSkelidDict()

    def getItemidList(self):
        itemidlist =  zip(*self.data)[2][1:-1]
        return itemidlist

    def toxml(self):
        data = self.data
        spine_ = []
        if data != None:
            re_itemref = re.compile(r'<itemref(.*?)/>', re.I)
            for [itemref, skelid, itemid, isvalid, filename] in data[1:-1]:
                mo_itemref = re_itemref.search(itemref)
                if isvalid and mo_itemref != None:
                    elm = '<itemref idref="{:s}"{:s}/>'.format(itemid, mo_itemref.group(1))
                    spine_.append(elm + '\n')
        return spine_

    def hasData(self):
        return self.data != None

    def getStartIndex(self):
        return 1

    def getEndIndex(self):
        return len(self.data) - 1

    def getIndexBySkelid(self, skelid):
        """Return corresponding itemref index to skelnum.

        """
        if self.skelid_dict != None:
            #[itemref, skelid, itemid, isvalid, filename]
            index = self.skelid_dict.get(skelid)
        else:
            index = None
        return index

    def getIndexByFilename(self, filename):
        filename_dict = self.filename_dict
        if filename == None:
            return None
        elif filename_dict == None:
            return None
        else:
            return filename_dict.get(filename)

    def getFilename(self, i):
        if i != None:
            #[itemref, skelid, itemid, isvalid, filename]
            return self.data[i][4]
        else:
            return None

    def setFilename(self, i, filename):
        if i != None:
            #[itemref, skelid, itemid, isvalid, filename]
            self.data[i][4] = filename

    def getSkelid(self, i):
        #[itemref, skelid, itemid, isvalid, filename]
        return self.data[i][1]

    def setSkelid(self, i, skelid):
        #[itemref, skelid, itemid, isvalid, filename]
        self.data[i][1] = skelid

    def getIdref(self, i):
        #[itemref, skelid, itemid, isvalid, filename]
        return self.data[i][2]

    def setIdref(self, i, ref):
        #[itemref, skelid, itemid, isvalid, filename]
        self.data[i][2] = ref
        self.data[i][3] = True

    def setStatus(self, i, valid):
        self.data[i][3] = valid

    def setAttribute(self, i, name, content):
        itemref = self.data[i][0]
        pa_attrib = r'''(?P<tag><itemref)(?:
            ((?P<head>.*?)(?P<name>{:s})\s*=\s*"(?P<content>.*?)"(?P<tail>.*))
            |(?P<nomatch>.*?/>))'''.format(name)
        mo_attrib = re.search(pa_attrib, itemref, re.I + re.X)
        if mo_attrib != None:
            if mo_attrib.group('content') != None:
                new = mo_attrib.group('tag') + mo_attrib.group('head') \
                    + '{:s}="{:s}"'.format(name, content) \
                    + mo_attrib.group('tail')
            else:
                new = mo_attrib.group('tag') \
                    + ' {:s}="{:s}"'.format(name, content) \
                    + mo_attrib.group('nomatch')
            self.data[i][0] = new

    def insert(self, i, itemid, skelid=-1, filename=None):
        newdata = self.data[:i] \
            + [['<itemref/>', skelid, itemid, False, filename]] \
            + self.data[i:]
        self.data = newdata

    def createSkelidDict(self):
        data = self.data
        if data != None:
            pairs = [[data[i][1], i] for i in range(1, len(data)-1)]
            skelid_dict = dict(pairs)
            self.skelid_dict = skelid_dict

    def createFilenameDict(self):
        data = self.data
        if data != None:
            pairs = [[data[i][4], i] for i in range(1, len(data)-1)]
            filename_dict = dict(pairs)
            self.filename_dict = filename_dict


class K8RESCProcessor(object):
    """RESC section processor, using re module.

    """
    def __init__(self, resc):
        self.cover_id = None
        self.xml_header = None
        self.metadata = Metadata()
        self.spine = Spine()

        if resc == None or len(resc) != 3:
            return
        [version, type_, data] = resc
        self.version = version
        self.type = type_
        self.data = data

        mo_xml = re.search(r'<\?xml[^>]*>', data, re.I)
        if mo_xml != None:
            self.xml_header = mo_xml.group()

        self.metadata.process(data)
        self.spine.process(data)

        # Find cover in metadata
        metadata = self.metadata
        typeid_meta = metadata.getTypeId(metadata.METADATA_META)
        num = metadata.getNumberOfElements()
        for [element, typeid, tag, attribs, isEmpty, start, content, end] \
                in metadata.getElements(range(1, num-1)):
            if typeid == typeid_meta:
                name = attribs.get('name')
                content = attribs.get('content')
                if name != None and name.lower() == 'cover':
                    self.cover_id = content
                    break

    def metadata_toxml(self):
        return self.metadata.toxml()

    def spine_toxml(self):
        return self.spine.toxml()

    def hasSpine(self):
        return self.spine.hasData()

    def createSpine(self, itemidbase, num):
        self.spine.create(itemidbase, num)

    def getSpineItemidList(self):
        return self.spine.getItemidList()

    def getSpineStartIndex(self):
        return self.spine.getStartIndex()

    def getSpineEndIndex(self):
        return self.spine.getEndIndex()

    def getSpineIndexBySkelid(self, skelid):
        return self.spine.getIndexBySkelid(skelid)

    def getSpineIndexByFilename(self, filename):
        return self.spine.getIndexByFilename(filename)

    def getFilenameFromSpine(self, i):
        return self.spine.getFilename(i)

    def setFilenameToSpine(self, i, filename):
        return self.spine.setFilename(i, filename)

    def getSpineSkelid(self, i):
        return self.spine.getSkelid(i)

    def setSpineSkelid(self, i, skelid):
        self.spine.setSkelid(i, skelid)

    def getSpineIdref(self, i):
        return self.spine.getIdref(i)

    def setSpineIdref(self, i, ref):
        self.spine.setIdref(i, ref)

    def setSpineStatus(self, i, valid):
        self.spine.setStatus(i, valid)

    def setSpineAttribute(self, i, name, content):
        self.spine.setAttribute(i, name, content)

    def insertSpine(self, i, itemid, skelid=-1, filename=None):
        self.spine.insert(i, itemid, skelid, filename)

    def createSkelidToSpineIndexDict(self):
        self.spine.createSkelidDict()

    def createFilenameToSpineIndexDict(self):
        self.spine.createFilenameDict()
