#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, David Forrester <davidfor@internode.on.net>'
__docformat__ = 'restructuredtext en'

import time, os

from contextlib import closing

from calibre.utils.ipc.server import Server
from calibre.utils.ipc.job import ParallelJob
from calibre.utils.logging import Log
from calibre.constants import DEBUG

from calibre_plugins.koboutilities.action import (MIMETYPE_KOBO, KEPUB_FETCH_QUERY, 
                                                  EPUB_FETCH_QUERY, EPUB_FETCH_QUERY_NORATING,
                                                  KEPUB_FETCH_QUERY_NORATING, BOOKMARK_SEPARATOR, convert_kobo_date)
import calibre_plugins.koboutilities.config as cfg
#from calibre_plugins.koboutilities.common_utils import debug_print

BASE_TIME = None
def debug_print(*args):
    global BASE_TIME
    if BASE_TIME is None:
        BASE_TIME = time.time()
#    if DEBUG:
    print('DEBUG: %6.1f'%(time.time()-BASE_TIME), *args)

def do_koboutilitiesa(books_to_scan, options, cpus, notification=lambda x,y:x):
    '''
    Master job, to launch child jobs to modify each ePub
    '''
    server = Server(pool_size=cpus)

    print("do_koboutilities - options=%s" % (options))
    # Queue all the jobs

    args = ['calibre_plugins.koboutilities.jobs', 'do_koboutilities_single',
            (books_to_scan, options)]
    print("do_koboutilities - args=%s" % (args))
    job = ParallelJob('arbitrary', str(book_id), done=None, args=args)
    job._book_id    = book_id
    job._title      = title
    job._authors    = authors
    job._contentIDs = contentIDs
    server.add_job(job)

    for book_id, contentIDs, title, authors in books_to_scan:
        print("do_koboutilities - book_id=%s, title=%s, authors=%s" % (book_id, title, authors))
        args = ['calibre_plugins.koboutilities.jobs', 'do_koboutilities_single',
                (book_id, contentIDs, options)]
        print("do_koboutilities - args=%s" % (args))
        job = ParallelJob('arbitrary', str(book_id), done=None, args=args)
        job._book_id    = book_id
        job._title      = title
        job._authors    = authors
        job._contentIDs = contentIDs
        server.add_job(job)

    # This server is an arbitrary_n job, so there is a notifier available.
    # Set the % complete to a small number to avoid the 'unavailable' indicator
    notification(0.01, 'Storing reading locations')

    # dequeue the job results as they arrive, saving the results
    total = len(books_to_scan)
    count = 0
    stored_locations = dict()
    while True:
        job = server.changed_jobs_queue.get()
        # A job can 'change' when it is not finished, for example if it
        # produces a notification. Ignore these.
        job.update()
        if not job.is_finished:
            continue
        # A job really finished. Get the information.
        stored_location = job.result
        book_id = job._book_id
        stored_locations[book_id] = stored_location
        count += 1
        notification(float(count)/total, 'Storing locations')
        # Add this job's output to the current log
        print('Logfile for book ID %d (%s / %s)'%(book_id, job._title, job._authors))
        print(job.details)
        print("\tstored_location=", stored_location)
        if count >= total:
            # All done!
            break

    server.close()
    # return the map as the job result
    return stored_locations


def do_store_locations(books_to_scan, options, cpus, notification=lambda x,y:x):
    '''
    Master job, to launch child jobs to modify each ePub
    '''
    debug_print("do_koboutilities - start")
    server = Server(pool_size=cpus)

    debug_print("do_koboutilities - options=%s" % (options))
    # Queue all the jobs

    args = ['calibre_plugins.koboutilities.jobs', 'do_koboutilities_all',
            (books_to_scan, options)]
#    debug_print("do_koboutilities - args=%s" % (args))
    debug_print("do_koboutilities - len(books_to_scan)=%d" % (len(books_to_scan)))
    job = ParallelJob('arbitrary', "Store locations", done=None, args=args)
    server.add_job(job)

    # This server is an arbitrary_n job, so there is a notifier available.
    # Set the % complete to a small number to avoid the 'unavailable' indicator
    notification(0.01, 'Reading device database')

    # dequeue the job results as they arrive, saving the results
    total = 1
    count = 0
    stored_locations = dict()
    while True:
        job = server.changed_jobs_queue.get()
        # A job can 'change' when it is not finished, for example if it
        # produces a notification. Ignore these.
        job.update()
        if not job.is_finished:
            debug_print("do_koboutilities - Job not finished")
            continue
        debug_print("do_koboutilities - Job finished")
        # A job really finished. Get the information.
        stored_locations = job.result
#        book_id = job._book_id
#        stored_locations[book_id] = stored_location
        count += 1
        notification(float(count)/total, 'Storing locations')
        # Add this job's output to the current log
        #debug_print("Stored_location=", stored_locations)
        number_bookmarks = len(stored_locations) if stored_locations else 0
        debug_print("Stored_location count=%d" % number_bookmarks)
        debug_print(job.details)
        if count >= total:
            # All done!
            break

    server.close()
    debug_print("do_koboutilities - finished")
    # return the map as the job result
    return stored_locations, options


def do_koboutilities_single(book_id, contentIDs, options):
    '''
    Child job, to store location for this book
    '''
    return _store_current_bookmark(Log(), book_id, contentIDs, options)


def do_koboutilities_all(books, options):
    '''
    Child job, to store location for all the books
    '''
    return _store_bookmarks(Log(), books, options)


def _store_current_bookmark(log, book_id, contentIDs, options):

    count_books = 0
    result      = None

    import sqlite3 as sqlite
    with closing(sqlite.connect(options["device_database_path"])) as connection:
        # return bytestrings if the content cannot the decoded as unicode
        connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

        cursor = connection.cursor()
        count_books += 1
        
        for contentID in contentIDs:
            log("store_current_bookmark - contentId='%s'" % (contentID))
            fetch_values = (contentID,)
            if contentID.endswith(".kepub.epub"):
                if options['supports_ratings']:
                    fetch_query = KEPUB_FETCH_QUERY
                else:
                    fetch_query = KEPUB_FETCH_QUERY_NORATING
            else:
                if options['supports_ratings']:
                    fetch_query = EPUB_FETCH_QUERY
                else:
                    fetch_query = EPUB_FETCH_QUERY_NORATING
            cursor.execute(fetch_query, fetch_values)
            result = cursor.fetchone()

        connection.commit()
        cursor.close()
    
    return result


def _store_bookmarks(log, books, options):

    debug_print("_store_bookmarks - start")
    count_books      = 0
    stored_locations = dict()
    clear_if_unread          = options[cfg.KEY_CLEAR_IF_UNREAD]
    store_if_more_recent     = options[cfg.KEY_STORE_IF_MORE_RECENT]
    do_not_store_if_reopened = options[cfg.KEY_DO_NOT_STORE_IF_REOPENED]

    import sqlite3 as sqlite
    with closing(sqlite.connect(options["device_database_path"])) as connection:
        # return bytestrings if the content cannot the decoded as unicode
        connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")
#        connection.row_factory  = sqlite.Row

        cursor = connection.cursor()
        count_books += 1
        
        debug_print("_store_bookmarks - about to start book loop")
        for book_id, contentIDs, title, authors, current_chapterid, current_percentRead, current_rating, current_last_read in books:
            result = None
            debug_print("_store_bookmarks - Current book: %s - %s" %(title, authors))
            debug_print("_store_bookmarks - contentIds='%s'" % (contentIDs))
            for contentID in contentIDs:
#                log("_store_bookmarks - contentId='%s'" % (contentID))
                debug_print("_store_bookmarks - contentId='%s'" % (contentID))
                fetch_values = (contentID,)
                if contentID.endswith(".kepub.epub"):
                    fetch_query = KEPUB_FETCH_QUERY if options['supports_ratings'] else KEPUB_FETCH_QUERY_NORATING
                else:
                    fetch_query = EPUB_FETCH_QUERY if options['supports_ratings'] else EPUB_FETCH_QUERY_NORATING
                cursor.execute(fetch_query, fetch_values)
                result = cursor.fetchone()

            if not result:
                continue

            debug_print("_store_bookmarks - result=", result)
            if result[7] == MIMETYPE_KOBO:
                kobo_chapteridbookmarked = result[0]
                kobo_adobe_location      = None
            else:
                debug_print("_store_bookmarks -result[0]=", result[0])
                kobo_chapteridbookmarked = result[0][len(contentID) + 1:] if result[0] else None
                kobo_adobe_location      = result[1]
            if kobo_chapteridbookmarked and kobo_adobe_location:
                new_chapterid = kobo_chapteridbookmarked + BOOKMARK_SEPARATOR + kobo_adobe_location
            elif kobo_chapteridbookmarked:
                new_chapterid = kobo_chapteridbookmarked
            else:
                new_chapterid = None
            kobo_percentRead = None
            if result[2] == 1: # or (result[2] == 0 and result[3] > 0):
                kobo_percentRead = result[3]
            elif result[2] == 2:
                kobo_percentRead = 100
                
            if result[8]:
                kobo_rating = result[8] * 2
            else:
                kobo_rating = 0
            
            last_read = None
            if result[5]:
                debug_print("_store_bookmarks - result[5]=", result[5])
                last_read = convert_kobo_date(result[5])
                debug_print("_store_bookmarks - last_read=", last_read)

            reading_position_changed = False
            if result[2] == 0 and clear_if_unread:
                reading_position_changed = True
                new_chapterid            = None
                kobo_percentRead         = 0
                last_read                = None
            elif result[2] > 0:
                debug_print("_store_bookmarks - current_chapterid != new_chapterid=", current_chapterid != new_chapterid)
                reading_position_changed = current_chapterid != new_chapterid
#                debug_print("_store_bookmarks - reading_position_changed=", reading_position_changed)
                debug_print("_store_bookmarks - current_percentRead != kobo_percentRead=", current_percentRead != kobo_percentRead)
                reading_position_changed |= current_percentRead != kobo_percentRead
    #            debug_print("_store_bookmarks - reading_position_changed=", reading_position_changed)
                debug_print("_store_bookmarks - current_rating =", current_rating)
                debug_print("_store_bookmarks - kobo_rating    =", kobo_rating)
                debug_print("_store_bookmarks - current_rating != kobo_rating=", current_rating != kobo_rating)
    #            debug_print("_store_bookmarks - current_rating != kobo_rating and not (current_rating is None and kobo_rating == 0)=", current_rating != kobo_rating and not (current_rating is None and kobo_rating == 0))
                debug_print("_store_bookmarks - current_rating != kobo_rating and kobo_rating > 0=", current_rating != kobo_rating and kobo_rating > 0)
    #            reading_position_changed |= current_rating != kobo_rating and not (current_rating is None and kobo_rating == 0)
                reading_position_changed |= current_rating != kobo_rating and kobo_rating > 0
    #            debug_print("_store_bookmarks - reading_position_changed=", reading_position_changed)
                debug_print("_store_bookmarks - current_last_read=", current_last_read)
                debug_print("_store_bookmarks - last_read        =", last_read)
                debug_print("_store_bookmarks - current_last_read != last_read=", current_last_read != last_read)
                reading_position_changed |= current_last_read != last_read
                debug_print("_store_bookmarks - reading_position_changed=", reading_position_changed)
                if store_if_more_recent:
                    if current_last_read and last_read:
                        debug_print("_store_bookmarks - store_if_more_recent - current_last_read < last_read=", current_last_read < last_read)
                        reading_position_changed &= current_last_read < last_read
                    elif last_read:
                        reading_position_changed &= True
                if do_not_store_if_reopened:
                    debug_print("_store_bookmarks - do_not_store_if_reopened - current_percentRead=", current_percentRead)
                    reading_position_changed &= current_percentRead < 100

            if reading_position_changed:
                debug_print("_store_bookmarks - position changed for: %s - %s" %(title, authors))
                stored_locations[book_id] = result

        debug_print("_store_bookmarks - finished book loop")
        connection.commit()
        cursor.close()
    
    debug_print("_store_bookmarks - finished")
    return stored_locations


def do_clean_images_dir(options, cpus, notification=lambda x,y:x):
    main_image_path      = options['main_image_path']
    sd_image_path        = options['sd_image_path']
    device_database_path = options["device_database_path"]

    notification(1/7, 'Getting ImageIDs from main images directory')
    debug_print("Getting ImageIDs from main images directory - Path is: '%s'" % (main_image_path))
    imageids_files_main = _get_file_imageIds(main_image_path)


    notification(2/7, 'Getting ImageIDs from SD card images directory')
    debug_print("Getting ImageIDs from SD images directory - Path is: '%s'" % (sd_image_path))
    imageids_files_sd   = _get_file_imageIds(sd_image_path)
    
    notification(3/7, 'Getting ImageIDs from device database.')
    debug_print("Getting ImageIDs from device database.")
    imageids_db = _get_imageId_set(device_database_path)
#        debug_print("clean_images_dir - len(imageids_db)=%d imageids_db=%s" % (len(imageids_db), imageids_db))

    notification(4/7, 'Checking/removing images from main images directory')
    extra_imageids_files_main = imageids_files_main - imageids_db
    debug_print("Checking/removing images from main images directory - Number of extra images: %d" % (len(extra_imageids_files_main)))
    extra_image_files_main = _remove_extra_files(extra_imageids_files_main, options['delete_extra_covers'], main_image_path)

    notification(5/7, 'Checking/removing images from SD card images directory')
    extra_imageids_files_sd   = imageids_files_sd   - imageids_db
    debug_print("Checking/removing images from SD card images directory - Number of extra images: %d" % (len(extra_imageids_files_sd)))
    extra_image_files_sd   = _remove_extra_files(extra_imageids_files_sd,   options['delete_extra_covers'], sd_image_path)

    extra_image_files                = {}
    extra_image_files['main_memory'] = extra_image_files_main
    extra_image_files['sd_card']     = extra_image_files_sd

    notification(7/7, 'Cleaning images directory - Done')

    return extra_image_files

def _get_file_imageIds(image_path):
    imageids_files = []
    if image_path:
        for path, dirs, files in os.walk(image_path):
#            debug_print("_get_file_imageIds - path=%s, dirs=%s" % (path, dirs))
            for filename in files:
                if filename.find(" - N3_") > 0:
#                    debug_print("check_covers - filename=%s" % (filename))
                    imageid = filename.split(" - N3_")[0]
                    imageids_files.append(imageid)

    imageids_files = set(imageids_files)
    return imageids_files

def _remove_extra_files(extra_imageids_files, delete_extra_covers, image_path):
    extra_image_files = []
    from glob import glob
    for imageId in extra_imageids_files:
        for filename in glob(os.path.join(image_path, imageId + '*')):
            debug_print("_remove_extra_files - filename=%s" % (filename))
            extra_image_files.append(os.path.basename(filename))
            if delete_extra_covers:
                os.unlink(filename)
    
    return extra_image_files

def _get_imageId_set(device_database_path):
    import sqlite3 as sqlite
    with closing(sqlite.connect(device_database_path)) as connection:
        connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

        imageId_query = 'SELECT DISTINCT ImageId '       \
                        'FROM content '         \
                        'WHERE ContentType = 6'
        cursor = connection.cursor()

        imageIDs = []
        cursor.execute(imageId_query)
        for i, row in enumerate(cursor):
            imageIDs.append(row[0])
#            debug_print("_get_imageid_set - row[0]='%s'" % (row[0]))
        connection.commit()

        cursor.close()

    return set(imageIDs)
