/*---------------------------------------------------------------------------

   rpng - simple PNG display program                               rpng-x.c

   This program decodes and displays PNG images, with gamma correction and
   optionally with a user-specified background color (in case the image has
   transparency).  It is very nearly the most basic PNG viewer possible.

  ---------------------------------------------------------------------------

      Based on : http://www.libpng.org/pub/png/book/sources.html

      Copyright (c) 1998-2008 Greg Roelofs.  All rights reserved.

      This software is provided "as is," without warranty of any kind,
      express or implied.  In no event shall the author or contributors
      be held liable for any damages arising in any way from the use of
      this software.

      The contents of this file are DUAL-LICENSED.  You may modify and/or
      redistribute this software according to the terms of one of the
      following two licenses (at your option):


      LICENSE 1 ("BSD-like with advertising clause"):

      Permission is granted to anyone to use this software for any purpose,
      including commercial applications, and to alter it and redistribute
      it freely, subject to the following restrictions:

      1. Redistributions of source code must retain the above copyright
         notice, disclaimer, and this list of conditions.
      2. Redistributions in binary form must reproduce the above copyright
         notice, disclaimer, and this list of conditions in the documenta-
         tion and/or other materials provided with the distribution.
      3. All advertising materials mentioning features or use of this
         software must display the following acknowledgment:

            This product includes software developed by Greg Roelofs
            and contributors for the book, "PNG: The Definitive Guide,"
            published by O'Reilly and Associates.


      LICENSE 2 (GNU GPL v2 or later):

      This program is free software; you can redistribute it and/or modify
      it under the terms of the GNU General Public License as published by
      the Free Software Foundation; either version 2 of the License, or
      (at your option) any later version.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program; if not, write to the Free Software Foundation,
      Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  ---------------------------------------------------------------------------*/

#define PROGNAME  "rpng-kobo"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

#include "readpng.h"   /* typedefs, common macros, readpng prototypes */


/* could just include png.h, but this macro is the only thing we need
 * (name and typedefs changed to local versions); note that side effects
 * only happen with alpha (which could easily be avoided with
 * "ush acopy = (alpha);") */

#define alpha_composite(composite, fg, alpha, bg) {               \
    ush temp = ((ush)(fg)*(ush)(alpha) +                          \
                (ush)(bg)*(ush)(255 - (ush)(alpha)) + (ush)128);  \
    (composite) = (uch)((temp + (temp >> 8)) >> 8);               \
}


static char *filename;
static FILE *infile;

static char *bgstr;
static uch bg_red=0, bg_green=0, bg_blue=0;

static double display_exponent;

static ulg image_width, image_height, image_rowbytes;
static int image_channels;
static uch *image_data;

static int have_nondefault_visual = FALSE;
static int have_colormap = FALSE;
static int have_window = FALSE;
static int have_gc = FALSE;

struct screen {
 int x;
 int y;
 int width;
 int height;
 int flag1;
 int flag2;
 char data[800*600];
};

int main(int argc, char **argv)
{
    char *p;
    int rc, alen, flen;
    int error = 0;
    int have_bg = FALSE;
    double LUT_exponent;               /* just the lookup table */
    double CRT_exponent = 2.2;         /* just the monitor */
    double default_display_exponent;   /* whole display system */

    int xoffset,yoffset,x,y,padding;
    struct screen update;
    int fb0;

    filename = (char *)NULL;


    /* First set the default value for our display-system exponent, i.e.,
     * the product of the CRT exponent and the exponent corresponding to
     * the frame-buffer's lookup table (LUT), if any.  This is not an
     * exhaustive list of LUT values (e.g., OpenStep has a lot of weird
     * ones), but it should cover 99% of the current possibilities. */

    LUT_exponent = 1.0;   /* assume no LUT:  most PCs */

    /* the defaults above give 1.0, 1.3, 1.5 and 2.2, respectively: */
    default_display_exponent = LUT_exponent * CRT_exponent;


    /* If the user has set the SCREEN_GAMMA environment variable as suggested
     * (somewhat imprecisely) in the libpng documentation, use that; otherwise
     * use the default value we just calculated.  Either way, the user may
     * override this via a command-line option. */

    if ((p = getenv("SCREEN_GAMMA")) != NULL)
        display_exponent = atof(p);
    else
        display_exponent = default_display_exponent;


    /* Now parse the command line for options and the PNG filename. */

    filename=argv[1];

    if(argc>2)
      {
       xoffset=atoi(argv[2]);
       yoffset=atoi(argv[3]);
      }
    else
      {
       xoffset=0;
       yoffset=0;
      }

    if (!(infile = fopen(filename, "rb"))) {
        fprintf(stderr, PROGNAME ":  can't open PNG file [%s]\n", filename);
        ++error;
    } else {
        if ((rc = readpng_init(infile, &image_width, &image_height)) != 0) {
            switch (rc) {
                case 1:
                    fprintf(stderr, PROGNAME
                      ":  [%s] is not a PNG file: incorrect signature\n",
                      filename);
                    break;
                case 2:
                    fprintf(stderr, PROGNAME
                      ":  [%s] has bad IHDR (libpng longjmp)\n", filename);
                    break;
                case 4:
                    fprintf(stderr, PROGNAME ":  insufficient memory\n");
                    break;
                default:
                    fprintf(stderr, PROGNAME
                      ":  unknown readpng_init() error\n");
                    break;
            }
            ++error;
        }
        if (error)
            fclose(infile);
    }


    if (error) {
        fprintf(stderr, PROGNAME ":  aborting.\n");
        exit(2);
    }

    /* if the user didn't specify a background color on the command line,
     * check for one in the PNG file--if not, the initialized values of 0
     * (black) will be used */

    bg_red   = (uch)255;
    bg_green = (uch)255;
    bg_blue  = (uch)255;

    /* decode the image, all at once */

    image_data = readpng_get_image(display_exponent, &image_channels,
      &image_rowbytes);

    /* done with PNG file, so clean up to minimize memory usage (but do NOT
     * nuke image_data!) */

    readpng_cleanup(FALSE);
    fclose(infile);

    if (!image_data) {
        fprintf(stderr, PROGNAME ":  unable to decode PNG image\n");
        exit(3);
    }

    /* display image (composite with background if requested) */

    fb0=open("/dev/fb0", O_RDWR);

    update.x=xoffset;
    update.y=yoffset;
    update.width=image_width;
    update.height=image_height;
    update.flag1=2;
    update.flag2=2;

    if((xoffset+image_width)>600)
       update.width=600-xoffset;

    if((yoffset+image_height)>800)
       update.height=800-yoffset;

    padding=update.x%2;

    for(y=0;y<update.height;y++)
      {
       uch *src = image_data + y*image_rowbytes;

       for(x=0;x<update.width;x++)
         {
          unsigned char red,green,blue,grey;

          if (image_channels == 3)
            {
             red   = *src++;
             green = *src++;
             blue  = *src++;
            }
          else /* if (image_channels == 4) */
            {
             unsigned char r,g,b,a;

             r = *src++;
             g = *src++;
             b = *src++;
             a = *src++;

             if (a == 255) {
                 red   = r;
                 green = g;
                 blue  = b;
             } else if (a == 0) {
                 red   = bg_red;
                 green = bg_green;
                 blue  = bg_blue;
             } else {
                 /* this macro (from png.h) composites the foreground
                  * and background values and puts the result into the
                  * first argument */
                 alpha_composite(red,   r, a, bg_red);
                 alpha_composite(green, g, a, bg_green);
                 alpha_composite(blue,  b, a, bg_blue);
             }
            }

          grey=((int)red+(int)green+(int)blue)/3;

          update.data[y*(update.width+padding)+x+padding*(y%2)]=grey;
         }
      }

    ioctl(fb0,0x4539,&update);

    return 0;
}
