from __future__ import (unicode_literals, division, absolute_import, print_function)

__license__   = 'GPL v3'
__copyright__ = '2013, Jellby <jellby@yahoo.com>'
__docformat__ = 'restructuredtext en'

from PyQt4.Qt import QDialog, QVBoxLayout, QPushButton, QMessageBox, QLabel, QDialogButtonBox
from calibre_plugins.prince_pdf.config import prefs
from calibre_plugins.prince_pdf.convert import ConvertDialog
from calibre_plugins.prince_pdf.log_box import LogDialog

# Main dialog of the plugin
class PrincePDFDialog(QDialog):
    prince_log = ''

    # GUI definition
    def __init__(self, gui, icon, do_user_config):
        QDialog.__init__(self, gui)
        self.icon = icon
        self.gui = gui
        self.do_user_config = do_user_config
        self.db = gui.current_db

        self.setWindowTitle('Prince PDF')
        self.setWindowIcon(icon)

        self.l = QVBoxLayout()
        self.setLayout(self.l)

        self.convert_to_PDF_button = QPushButton('Convert to &PDF', self)
        self.convert_to_PDF_button.clicked.connect(self.convert_to_PDF)
        self.convert_to_PDF_button.setDefault(True)
        self.l.addWidget(self.convert_to_PDF_button)

        self.view_log = QPushButton('&View log', self)
        self.view_log.clicked.connect(self.show_log)
        self.l.addWidget(self.view_log)
        self.view_log.setEnabled(False)

        self.conf_button = QPushButton('Con&figure this plugin', self)
        self.conf_button.clicked.connect(self.config)
        self.l.addWidget(self.conf_button)

        self.buttons = QDialogButtonBox(QDialogButtonBox.Close | QDialogButtonBox.Help)
        self.l.addWidget(self.buttons)
        self.buttons.rejected.connect(self.reject)
        self.buttons.helpRequested.connect(self.about)

        self.resize(self.sizeHint())

    def about(self):
        '''
        Display a short help message
        '''
        text = get_resources('help.txt')
        QMessageBox.about(self, 'About the Prince PDF Plugin', text.decode('utf-8'))

    def convert_to_PDF(self):
        '''
        Unpack and convert the currently selected book to PDF
        '''
        from calibre.gui2 import error_dialog

        # Get currently selected books
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, 'Cannot convert to PDF', 'No books selected', show=True)
        elif len(rows) > 1:
            return error_dialog(self.gui, 'Cannot convert to PDF', 'Multiple books not supported', show=True)

        # Map the rows to book ids
        # (some day it may work with multiple books)
        ids = list(map(self.gui.library_view.model().id, rows))
        for book_id in ids:
            fmts = self.db.formats(book_id, index_is_id=True)
            if not fmts: continue
            fmts = map(lambda x:x.lower(), fmts.split(','))
            # Process only the first format matching the 'formats' configuration option
            for fmt in prefs['formats']:
                fmt = fmt.lower()
                if (not fmt in fmts): continue
                book_file = self.db.format(book_id, fmt, index_is_id=True, as_path=True)

                # This is the actual code:
                print('===========')
                # Unpack the book and call the conversion dialog
                (opf, oeb) = self.unpack(book_file, fmt)
                if (opf == None or oeb == None):
                   return error_dialog(self.gui, 'Cannot convert to PDF', 'Format not supported: %s' % fmt, show=True)
                convert_dialog = ConvertDialog(opf, oeb, self.icon)
                pdf_file = ''
                if (convert_dialog.exec_()):
                    pdf_file = convert_dialog.pdf_file
                self.prince_log = convert_dialog.prince_log
                # After the dialog returs, pdf_file has the output file path,
                # and prince_log has the Prince console output
                print(pdf_file)
                # If there is any log, enable the View log button
                if (self.prince_log):
                    self.view_log.setEnabled(True)
                else:
                    self.view_log.setEnabled(False)
                # If the conversion failed, pdf_file will be None,
                # if the user cancelled the dialog, pdf_file will be ''
                if (pdf_file == None):
                    error_dialog(self.gui, 'Could not convert to PDF', 'The conversion failed. Check the log and make sure Prince is correctly installed.', show=True)
                print('===========')
                return
        # No matching format in the book
        return error_dialog(self.gui, 'Cannot convert to PDF', 'No supported format available', show=True)

    def show_log(self):
        '''
        Display the Prince log dialog
        '''
        msg = LogDialog(self.prince_log, self.icon)
        msg.exec_()

    def config(self):
        '''
        Display the configuration dialog
        '''
        self.do_user_config(parent=self)

    def unpack(self, book_file, fmt):
        '''
        Unpack the book in a temporary directory
        '''
        from calibre.ptempfile import PersistentTemporaryDirectory
        from calibre.ebooks.tweak import get_tools
        from calibre.ebooks.oeb.base import OEBBook
        from calibre.ebooks.oeb.reader import OEBReader
        from calibre.utils.logging import default_log
        from calibre_plugins.prince_pdf.dummy_preprocessor import dummy_preprocessor

        print('Unpacking book...')
        tdir = PersistentTemporaryDirectory('_unpack')
        exploder = get_tools(fmt)[0]
        if (exploder == None): return (None, None)
        opf = exploder(book_file, tdir)
        html_preprocessor = dummy_preprocessor()
        css_preprocessor = dummy_preprocessor()
        oeb = OEBBook(default_log, html_preprocessor, css_preprocessor)
        OEBReader()(oeb, opf)
        return (opf, oeb)
