from __future__ import (unicode_literals, division, absolute_import, print_function)

__license__   = 'GPL v3'
__copyright__ = '2013, Jellby <jellby@yahoo.com>'
__docformat__ = 'restructuredtext en'

from PyQt4.Qt import Qt, QDialog, QHBoxLayout, QVBoxLayout, QPushButton, QTextEdit, QTabWidget, QDialogButtonBox
from calibre_plugins.prince_pdf.config import prefs

# This dialog is created after the input book has been unpacked
class ConvertDialog(QDialog):

    pdf_file = 'epub2pdf.pdf'
    prince_log = ''
    prince_file = ''
    prince_css = ''

    # GUI definition
    def __init__(self, opf, oeb, icon):
        self.opf = opf
        self.oeb = oeb
        # The unpacked book needs to be parsed before, to read the contents
        # of the prince-style file, if it exists
        self.parse()

        QDialog.__init__(self)

        self.setWindowTitle('Convert to PDF with Prince')
        self.setWindowIcon(icon)

        self.l = QVBoxLayout()
        self.setLayout(self.l)

        self.css = QTabWidget()
        self.css.setMinimumWidth(500)
        self.l.addWidget(self.css)

        self.css1 = QTextEdit()
        self.css1.setLineWrapMode(QTextEdit.NoWrap)
        self.css1.setPlainText(prefs['custom_CSS'])
        self.css1.setToolTip("<qt>This stylesheet can be modified<br/>The default can be configured</qt>")
        i = self.css.addTab(self.css1, "C&ustom CSS")
        self.css.setTabToolTip(i, "<qt>Custom CSS stylesheet to be used for this conversion</qt>")

        if (self.prince_css):
            self.css2 = QTextEdit()
            self.css2.setLineWrapMode(QTextEdit.NoWrap)
            self.css2.setPlainText(self.prince_css)
            self.css2.setReadOnly(True)
            self.css2.setToolTip("<qt>This stylesheet cannot be modified</qt>")
            i = self.css.addTab(self.css2, "&Book CSS")
            self.css.setTabToolTip(i, "<qt>Book-specific CSS stylesheet included in the ebook file</qt>")

        self.ll = QHBoxLayout()
        self.l.addLayout(self.ll)

        if (prefs['show_CSS']):
            self.toggle = QPushButton('&Hide stylesheets', self)
        else:
            self.toggle = QPushButton('&Show stylesheets', self)
        self.toggle.clicked.connect(self.toggle_tabs)

        self.convert = QPushButton('Con&vert', self)
        self.convert.setDefault(True)

        self.buttons = QDialogButtonBox(QDialogButtonBox.Cancel)
        self.buttons.addButton(self.toggle, QDialogButtonBox.ResetRole)
        self.buttons.addButton(self.convert, QDialogButtonBox.AcceptRole)
        self.l.addWidget(self.buttons)
        self.buttons.accepted.connect(self.prince_convert)
        self.buttons.rejected.connect(self.reject)

        if (not prefs['show_CSS']):
            self.css.hide()
        self.resize(self.sizeHint())

    def toggle_tabs(self):
        '''
        Enable/disable the CSS tabs, and store the setting
        '''
        if (self.css.isVisible()):
            self.css.hide()
            self.toggle.setText('&Show stylesheets')
            self.resize(self.sizeHint())
        else:
            self.css.show()
            self.toggle.setText('&Hide stylesheets')
            self.resize(self.sizeHint())
        prefs['show_CSS'] = self.css.isVisible()

    def parse(self):
        '''
        Parse the unpacked OPF file to find and read the prince-style file
        '''
        from os.path import dirname, join
        from lxml import etree
        import codecs

        print('Parsing book...')
        opf_dir = dirname(self.opf)
        root = etree.parse(self.opf).getroot()
        metadata = root.find('{*}metadata')
        for meta in metadata.findall("{*}meta[@name='prince-style']"):
            prince_id = meta.get('content')
            for item in self.oeb.manifest:
                if (item.id == prince_id):
                    self.prince_file = item.href
                    break
        if (self.prince_file):
            fl = codecs.open(join(opf_dir, self.prince_file), "rb", "utf-8")
            self.prince_css = fl.read()
            fl.close()

    def prince_convert(self):
        '''
        Call the actual Prince command to convert to PDF
        '''
        from os.path import dirname, join
        from subprocess import Popen, PIPE
        from calibre.ptempfile import PersistentTemporaryFile
        from calibre.gui2 import error_dialog

        # All files are relative to the OPF location
        opf_dir = dirname(self.opf)

        # Create a temporary CSS file with the box contents
        custom_CSS = PersistentTemporaryFile()
        custom_CSS.write(unicode(self.css1.toPlainText()))
        custom_CSS.close()
        # Build the command line
        command = [prefs['prince_exe'], '-v']
        if self.prince_file:
            command.append('-s')
            command.append(self.prince_file)
        command.append('-s')
        command.append(custom_CSS.name)
        command.append('-o')
        command.append(self.pdf_file)
        for item in self.oeb.spine:
            command.append(item.href)

        # Run the command and return the path to the PDF file
        print('Converting book...')
        try:
            convert = Popen(command, stderr=PIPE, cwd=opf_dir)
            self.prince_log = convert.communicate()[1]
            self.pdf_file = join(opf_dir, self.pdf_file)
        except OSError:
            self.pdf_file = None
        self.accept()
