from __future__ import (unicode_literals, division, absolute_import, print_function)

__license__   = 'GPL v3'
__copyright__ = '2013, Jellby <jellby@yahoo.com>'
__docformat__ = 'restructuredtext en'

from PyQt4.Qt import Qt, QWidget, QHBoxLayout, QVBoxLayout, QLabel, QLineEdit, QTextEdit, QPushButton, QFileDialog, QCheckBox
from calibre.utils.config import JSONConfig

prefs = JSONConfig('plugins/prince_pdf')

# Set defaults
prefs.defaults['prince_exe'] = 'prince'
prefs.defaults['formats'] = ['epub','azw3','htmlz']
prefs.defaults['show_CSS'] = False
prefs.defaults['custom_CSS'] = '''/*
    @font-face {
        font-family: serif;
        src: local("Droid Serif")
    }
    */

    @page {
      size: 9cm 12cm;
      margin: 5mm 1mm 1mm 1mm !important;
      @top-left {
        font-size: 50%;
        font-style: italic;
        border-bottom: solid thin black;
        margin-bottom: 1mm;
        content: string(booktitle);
      }
      @top-center {
        font-size: 50%;
        font-style: italic;
        border-bottom: solid thin black;
        margin-bottom: 1mm;
        content: string(chaptertitle);
      }
      @top-right {
        font-size: 50%;
        border-bottom: solid thin black;
        margin-bottom: 1mm;
        content: counter(page) "/" counter(pages);
      }
    }

    @page:first {
      margin: 1mm 1mm 1mm 1mm;
      @top-left {
        border-width: 0;
        margin: 0;
        content: normal;
      }
      @top-center {
        border-width: 0;
        margin: 0;
        content: normal;
      }
      @top-right {
        border-width: 0;
        margin: 0;
        content: normal;
      }
    }

    body {
      font-size: 10pt;
      font-family: serif;
      text-align: justify;
      prince-image-resolution: 166dpi;
      hyphens: auto;
    }'''

class ConfigWidget(QWidget):

    # GUI definition
    def __init__(self):
        QWidget.__init__(self)
        self.l = QVBoxLayout()
        self.setLayout(self.l)

        self.ll = QHBoxLayout()
        self.l.addLayout(self.ll)

        self.label_exe = QLabel('&Prince executable:')
        self.ll.addWidget(self.label_exe)

        self.exe = QLineEdit(self)
        self.exe.setText(prefs['prince_exe'])
        self.exe.setToolTip("<qt>Executable for the Prince program</qt>")
        self.ll.addWidget(self.exe)
        self.label_exe.setBuddy(self.exe)

        self.browse = QPushButton('&Browse...', self)
        self.browse.clicked.connect(self.select_exe)
        self.ll.addWidget(self.browse)

        self.lll = QHBoxLayout()
        self.l.addLayout(self.lll)

        self.label_fmts = QLabel('Preferred &formats:')
        self.lll.addWidget(self.label_fmts)

        self.fmts = QLineEdit(self)
        self.fmts.setText(','.join(prefs['formats']))
        self.fmts.setToolTip("<qt>Comma-separated list of preferred formats to use as source</qt>")
        self.lll.addWidget(self.fmts)
        self.label_fmts.setBuddy(self.fmts)

        self.label_css = QLabel('&Custom CSS:')
        self.l.addWidget(self.label_css)

        self.css = QTextEdit()
        self.css.setLineWrapMode(QTextEdit.NoWrap)
        self.css.setText(prefs['custom_CSS'])
        self.css.setToolTip("<qt>Custom stylesheet that will be applied to all Prince PDF conversions</qt>")
        self.l.addWidget(self.css)
        self.label_css.setBuddy(self.css)

        self.show_css = QCheckBox('&Show CSS in the Convert dialog')
        self.show_css.setToolTip("<qt>Show by default the stylesheets in the Convert dialog</qt>")
        self.show_css.setChecked(prefs['show_CSS'])
        self.l.addWidget(self.show_css)

        self.defaults = QPushButton('&Restore defaults')
        self.defaults.clicked.connect(self.restore_defaults)
        self.l.addWidget(self.defaults, alignment=Qt.AlignLeft)

        self.resize(self.sizeHint())

    def select_exe(self):
        '''
        Create a dialog to select the Prince executable
        '''
        dialog = QFileDialog()
        dialog.setFileMode(QFileDialog.ExistingFile)
        filename = dialog.getOpenFileName(self, "Select Prince executable", "", "")
        if filename:
            self.exe.setText(filename)

    def restore_defaults(self):
        '''
        Restore the default settings
        '''
        self.exe.setText(prefs.defaults['prince_exe'])
        self.fmts.setText(','.join(prefs.defaults['formats']))
        self.css.setPlainText(prefs.defaults['custom_CSS'])
        self.show_css.setChecked(prefs.defaults['show_CSS'])

    def save_settings(self):
        '''
        Save the current settings
        '''
        prefs['prince_exe'] = unicode(self.exe.text())
        prefs['formats'] = unicode(self.fmts.text()).split(',')
        prefs['custom_CSS'] = unicode(self.css.toPlainText())
        prefs['show_CSS'] = self.show_css.isChecked()
