#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

class Unbuffered:
    def __init__(self, stream):
        self.stream = stream
    def write(self, data):
        self.stream.write(data)
        self.stream.flush()
    def __getattr__(self, attr):
        return getattr(self.stream, attr)

import sys
sys.stdout=Unbuffered(sys.stdout)
import os
import getopt
import struct
import binascii

class DualMetaEditException(Exception):
    pass

# important  pdb header offsets
unique_id_seed = 68
number_of_pdb_records = 76

# important palmdoc header offsets
book_length = 4
book_record_count = 8
first_pdb_record = 78

# important rec0 offsets
length_of_book = 4
mobi_header_base = 16
mobi_header_length = 20
mobi_type = 24
mobi_version = 36
first_non_text = 80
title_offset = 84
first_image_record = 108
first_content_index = 192
last_content_index = 194
kf8_last_content_index = 192 # for KF8 mobi headers
fcis_index = 200
flis_index = 208
srcs_index = 224
srcs_count = 228
primary_index = 244
datp_index = 256
huffoff = 112
hufftbloff = 120

def getint(datain,ofs,sz='L'):
    i, = struct.unpack_from('>'+sz,datain,ofs)
    return i

def writeint(datain,ofs,n,len='L'):
    if len=='L':
        return datain[:ofs]+struct.pack('>L',n)+datain[ofs+4:]
    else:
        return datain[:ofs]+struct.pack('>H',n)+datain[ofs+2:]

def getsecaddr(datain,secno):
    nsec = getint(datain,number_of_pdb_records,'H')
    assert secno>=0 & secno<nsec,'secno %d out of range (nsec=%d)'%(secno,nsec)
    secstart = getint(datain,first_pdb_record+secno*8)
    if secno == nsec-1:
        secend = len(datain)
    else:
        secend = getint(datain,first_pdb_record+(secno+1)*8)
    return secstart,secend

def readsection(datain,secno):
    secstart, secend = getsecaddr(datain,secno)
    return datain[secstart:secend]

# overwrite - should always be same length
def replacesection(datain,secno,secdata): 
    secstart,secend = getsecaddr(datain,secno)
    datalst = []
    datalst.append(datain[0:secstart])
    datalst.append(secdata)
    datalst.append(datain[secend:])
    dataout = "".join(datalst)
    return dataout

def get_exth_params(rec0):
    rlen = len(rec0)
    ebase = mobi_header_base + getint(rec0,mobi_header_length)
    elen = getint(rec0,ebase+4)
    enum = getint(rec0,ebase+8)
    return ebase,elen,enum,rlen

def add_exth(rec0,exth_num,exth_bytes):
    ebase,elen,enum,rlen = get_exth_params(rec0)
    newrecsize = 8+len(exth_bytes)
    newrec0 = rec0[0:ebase+4]+struct.pack('>L',elen+newrecsize)+struct.pack('>L',enum+1)+\
              struct.pack('>L',exth_num)+struct.pack('>L',newrecsize)+exth_bytes+rec0[ebase+12:]
    newrec0 = writeint(newrec0,title_offset,getint(newrec0,title_offset)+newrecsize)
    # keep constant record length by removing newrecsize bytes from end
    mtmp = newrec0[-newrecsize:]
    print "adding enth so trimming ", mtmp.encode('hex')
    newrec0 = newrec0[0:rlen]
    return newrec0

def read_exth(rec0,exth_num):
    exth_values = []
    ebase,elen,enum,rlen = get_exth_params(rec0)
    ebase = ebase+12
    while enum>0:
        exth_id = getint(rec0,ebase)
        if exth_id == exth_num:
            # We might have multiple exths, so build a list.
            exth_values.append(rec0[ebase+8:ebase+getint(rec0,ebase+4)])
        enum = enum-1
        ebase = ebase+getint(rec0,ebase+4)
    return exth_values

def del_exth(rec0,exth_num):
    ebase,elen,enum,rlen = get_exth_params(rec0)
    ebase_idx = ebase+12
    enum_idx = 0
    while enum_idx < enum:
        exth_id = getint(rec0,ebase_idx)
        exth_size = getint(rec0,ebase_idx+4)
        if exth_id == exth_num:
            newrec0 = rec0
            newrec0 = writeint(newrec0,title_offset,getint(newrec0,title_offset)-exth_size)
            newrec0 = newrec0[:ebase_idx]+newrec0[ebase_idx+exth_size:]
            newrec0 = newrec0[0:ebase+4]+struct.pack('>L',elen-exth_size)+struct.pack('>L',enum-1)+newrec0[ebase+12:]
            newrec0 = newrec0 + '\0'*(exth_size)
            print "adding in null bytes of length ", exth_size
            if rlen != len(newrec0):
                print "error size mismatch in del_exth"
            return newrec0
        enum_idx += 1
        ebase_idx = ebase_idx+exth_size
    return rec0


class dualmobimetafix:

    def __init__(self, infile, asin):
        self.datain = open(infile, 'rb').read()
        self.datain_rec0 = readsection(self.datain,0)

        # in the first mobi header
        # add 501 to "EBOK", add 113 as asin, add 504 as asin
        rec0 = self.datain_rec0
        rec0 = del_exth(rec0, 501)
        rec0 = del_exth(rec0, 113)
        rec0 = del_exth(rec0, 504)
        rec0 = add_exth(rec0, 501, "EBOK")
        rec0 = add_exth(rec0, 113, asin)
        rec0 = add_exth(rec0, 504, asin)
        self.datain = replacesection(self.datain, 0, rec0)

        ver = getint(self.datain_rec0,mobi_version)
        self.combo = (ver!=8)
        if not self.combo:
            return

        exth121 = read_exth(self.datain_rec0,121)
        if len(exth121) == 0:
            self.combo = False
            return
        else:
            # only pay attention to first exth121
            # (there should only be one)
            datain_kf8, = struct.unpack_from('>L',exth121[0],0)
            if datain_kf8 == 0xffffffff:
                self.combo = False
                return
        self.datain_kfrec0 =readsection(self.datain,datain_kf8)

        # in the second header
        # add 501 to "EBOK", add 113 as asin, add 504 as asin
        rec0 = self.datain_kfrec0
        rec0 = del_exth(rec0, 501)
        rec0 = del_exth(rec0, 113)
        rec0 = del_exth(rec0, 504)
        rec0 = add_exth(rec0, 501, "EBOK")
        rec0 = add_exth(rec0, 113, asin)
        rec0 = add_exth(rec0, 504, asin)
        self.datain = replacesection(self.datain, datain_kf8, rec0)

    def getresult(self):
        return self.datain



def usage(progname):
    print ""
    print "Description:"
    print "   Simple Program to add EBOK and ASIN Info to Meta on Dual Mobis"
    print "  "
    print "Usage:"
    print "  %s -h asin infile.mobi outfile.mobi" % progname
    print "  "
    print "Options:"
    print "    -h           print this help message"


def main(argv=sys.argv):
    print "DualMetaFixer v001"
    progname = os.path.basename(argv[0])
    try:
        opts, args = getopt.getopt(sys.argv[1:], "h")
    except getopt.GetoptError, err:
        print str(err)
        usage(progname)
        sys.exit(2)

    if len(args) != 3:
        usage(progname)
        sys.exit(2)

    for o, a in opts:
        if o == "-h":
            usage(progname)
            sys.exit(0)

    asin = args[0]
    infile = args[1]
    infileext = os.path.splitext(infile)[1].upper()
    outfile = args[2]
    print infile, infileext
    if infileext not in ['.MOBI', '.PRC', '.AZW', '.AZW3','.AZW4']:
        print "Error: first parameter must be a Kindle/Mobipocket ebook."
        return 1

    try:
        # make sure it is really a mobi ebook
        mobidata = file(infile, 'rb').read(100)
        palmheader = mobidata[0:78]
        ident = palmheader[0x3C:0x3C+8]
        if ident != 'BOOKMOBI':
            raise dumpHeaderException('invalid file format')

        dmf = dualmobimetafix(infile, asin)
        file(outfile,'wb').write(dmf.getresult())

    except Exception, e:
        print "Error: %s" % e
        return 1

    return 0


if __name__ == '__main__':
    sys.stdout=Unbuffered(sys.stdout)
    sys.exit(main())
