#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2008  Iñigo Serna
# Time-stamp: <2008-03-03 00:44:16 inigo>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import calendar

from reportlab.pdfgen import canvas
from reportlab.lib.pagesizes import A4
from reportlab.lib.colors import lightcoral, black, grey, navy


######################################################################
##### some variables
YEAR = 2008
OUTPUT = 'agenda-%d.pdf' % YEAR
SIZE = A4
# SIZE = (124, 152) # iLiad with toolbar
# SIZE = (124, 164) # iLiad fullscreen
FIRST_WEEK_DAY = 0 # monday

# globals
w, h, f = 0, 0, 0  # page width, page height, scale factor
wds = [d[:2] for d in calendar.day_abbr]
cal = calendar.Calendar(FIRST_WEEK_DAY)


######################################################################
##### CANVAS
def prepare_canvas(year):
    global w, h, f
    c = canvas.Canvas(OUTPUT, pagesize=SIZE, pageCompression=1)
    c.setTitle('Agenda %d' % year)
    c.setSubject('Agenda %d' % year)
    c.setAuthor(u'Iñigo Serna')
    c.setKeywords(('agenda', str(year)))
    w, h = SIZE
    f = w / A4[0]
    return c


######################################################################
##### COMMON CONTENTS
def add_pagenumber(c):
    c.setFont('Times-Roman', 8*f)
    c.setFillColor(grey)
    c.drawCentredString(w/2, 2*f, '- Page %d. -' % c.getPageNumber())


def add_copyright(c):
    c.setFont('Times-Roman', 8*f)
    c.setFillColor(grey)
    c.drawRightString(w-2*f, 2*f, u'(C) 2008, Iñigo Serna')


######################################################################
##### YEAR VIEW
def year_view(c, year, links=True):
    # page header
    c.setFont('Helvetica-Bold', 22*f)
    c.setFillColor(lightcoral)
    c.drawString(10*f, h-35*f, '%d' % year)
    c.setStrokeColor(lightcoral)
    c.setLineWidth(4*f)
    c.line(10*f, h-45*f, w-10*f, h-45*f)

    # calc values
    margin = 15*f
    spacing = 20*f
    w2 = (w-2*margin-2*spacing) / 3.0
    w3 = w2 / 7.0
    xpos = (margin, w/2-w2/2, w-margin-w2)
    h2 = h - 80*f
    ypos = (h-80*f, h-220*f, h-360*f, h-500*f)

    # months
    for m in xrange(12):
        y = ypos[int(m/3)]
        # month header
        x = xpos[m%3] + w2/2
        c.setFont('Times-Bold', 17*f)
        c.setFillColor(lightcoral)
        c.drawCentredString(x, y, calendar.month_name[m+1].upper())
        if links:
            c.linkAbsolute('Month %d/%d' % (m+1, year),
                           'month_%d/%d' % (m+1, year),
                           (x-100*f, y-120*f, x+100*f, y+20*f))
        # day of week
        x = xpos[m%3]
        y -= 20*f
        c.setFont('Helvetica-Bold', 11*f)
        c.setFillColor(black)
        for i, d in enumerate(cal.iterweekdays()):
            c.drawCentredString(x+i*w3+w3/2, y, calendar.day_abbr[d])
        # days
        c.setFont('Helvetica', 12*f)
        for week in cal.monthdays2calendar(year, m+1):
            y -= 15*f
            for i, (day, dw) in enumerate(week):
                if day == 0:
                    continue
                color = grey if dw < 5 else navy
                c.setFillColor(color)
                c.drawRightString(x+i*w3+w3/1.2, y, '%2d' % day)

    # index
    if links:
        # FIXME: beautify this
        c.setFont('Times-Roman', 20*f)
        c.setFillColor(black)
        c.drawString(35*f, 15*f, 'Agenda')
        c.linkAbsolute('Agenda Week 1/%d' % year, 'agenda_week_1/%d' % year,
                       (30*f, 10*f, 100*f, 35*f))
        c.drawString(35*f, 45*f, 'Planner %d' % year)
        c.linkAbsolute('Planner 1/%d' % year, 'planner_1/%d' % year,
                       (30*f, 40*f, 100*f, 65*f))
        c.drawString(35*f, 75*f, 'Months view')
        c.linkAbsolute('Months %d' % year, 'month_1/%d' % year,
                       (30*f, 70*f, 150*f, 95*f))
        c.drawString(35*f, 105*f, 'Months2 view')
        c.linkAbsolute('Months2 %d' % year, 'month2_1/%d' % year,
                       (30*f, 100*f, 150*f, 125*f))

    # write page
    c.bookmarkPage('index')
    c.bookmarkPage('year_%d' % year)
    add_pagenumber(c)
    add_copyright(c)
    c.showPage()


def minical_year(c, year, y0):
    # header
    c.setFont('Helvetica-Bold', 18*f)
    c.setFillColor(lightcoral)
    c.setStrokeColor(lightcoral)
    c.setLineWidth(2*f)
    c.drawString(10*f, y0, '%d' % year)
    c.line(10*f, y0-5*f, w-10*f, y0-5*f)

    # calc values
    margin = 10*f
    spacing = 15*f
    w2 = (w-2*margin-3*spacing) / 4.0
    w3 = w2 / 7.0
    xpos = (margin, margin+w2+spacing, margin+2*w2+2*spacing, margin+3*w2+3*spacing)
    ypos = (y0-20*f, y0-105*f, y0-190*f)

    # months
    for m in xrange(12):
        y = ypos[int(m/4)]
        # month header
        c.setFont('Times-Bold', 13*f)
        c.setFillColor(lightcoral)
        c.drawCentredString(xpos[m%4]+w2/2, y, calendar.month_name[m+1].upper())
        # day of week
        x = xpos[m%4]
        y -= 10*f
        c.setFont('Helvetica-Bold', 8*f)
        c.setFillColor(black)
        for i, d in enumerate(cal.iterweekdays()):
            c.drawCentredString(x+i*w3+w3/2, y, calendar.day_abbr[d][:2])
        # days
        c.setFont('Helvetica', 9*f)
        for week in cal.monthdays2calendar(year, m+1):
            y -= 10*f
            for i, (day, dw) in enumerate(week):
                if day == 0:
                    continue
                color = grey if dw < 5 else navy
                c.setFillColor(color)
                c.drawRightString(x+i*w3+w3/1.2, y, '%2d' % day)


def year_3view(c, year):
    minical_year(c, year-1, h-20*f)
    minical_year(c, year, 2*h/3.0-20*f)
    minical_year(c, year+1, h/3.0-20*f)
    add_pagenumber(c)
    add_copyright(c)
    c.showPage()


######################################################################
##### MONTH VIEW
def render_minical_month(c, year, month, xpos, ypos, w3, h3, y):
    c.setFont('Helvetica-Bold', 13*f)
    c.setFillColor(black)
    c.setLineWidth(1*f)
    for i, d in enumerate(cal.iterweekdays()):
        c.drawCentredString(xpos[i]+w3/2, y, calendar.day_name[d])
    for j, week in enumerate(cal.monthdays2calendar(year, month)):
        weeknum=1 # FIXME: fix link
        for i, (d, dw) in enumerate(week):
            if d == 0:
                continue
            c.setFillColorRGB(0.8, 0.8, 0.8)
            fill = 1 if dw > 4 else 0
            c.rect(xpos[i], ypos[j], w3, h3, fill=fill)
            c.linkRect('Agenda Week %d/%d' % (weeknum, year),
                       'agenda_week_%d/%d' % (weeknum, year),
                       (xpos[i], ypos[j], xpos[i]+w3, ypos[j]+h3))
            c.setFillColor(navy)
            c.drawRightString(xpos[i]+w3-5*f, ypos[j]+h3-15*f, str(d))


# month view: 1 month landscape
def months_1month_landscape(c, year, month):
    # rotate page
    c.saveState()
    hh, ww = w, h
    c.rotate(90)

    # page header
    c.setFont('Helvetica-Bold', 22*f)
    c.setFillColor(lightcoral)
    c.drawCentredString(ww/2, -30*f, '%s %d' % (calendar.month_name[month], year))
    c.linkRect('Index', 'index', (ww/2-100*f, 0, ww/2+100*f, -50*f))

    # calc values
    cal_rowsnum = len(cal.monthdayscalendar(year, month))
    ww_margin = 25*f
    hh_margin_top = 65*f
    hh_margin_bottom = 15*f
    ww2 = (ww-2*ww_margin) / 7.0
    hh2 = (hh-hh_margin_top-hh_margin_bottom) / float(cal_rowsnum)
    xpos = [ww_margin+i*ww2 for i in range(7)]
    ypos = [-(hh_margin_top+(i+1)*hh2) for i in range(cal_rowsnum)]

    # contents
    render_minical_month(c, year, month, xpos, ypos, ww2, hh2, -55*f)

    # write page
    c.restoreState()
    c.bookmarkPage('month_%d/%d' % (month, year))
    add_pagenumber(c)
    add_copyright(c)
    c.showPage()


def minical_month(c, year, month, y0):
    # page header
    c.setFont('Helvetica-Bold', 22*f)
    c.setFillColor(lightcoral)
    c.drawCentredString(w/2, y0-25*f, '%s %d' % (calendar.month_name[month], year))
    c.linkRect('Index', 'index', (w/2-100*f, y0, w/2+100*f, y0-30*f))

    # calc values
    cal_rowsnum = len(cal.monthdayscalendar(year, month))
    w_margin = 15*f
    h_margin_top = 60*f
    h_margin_bottom = 10*f
    w2 = (w-2*w_margin) / 7.0
    h2 = (h/2-h_margin_top-h_margin_bottom) / (cal_rowsnum+1.0)
    xpos = [w_margin+i*w2 for i in range(7)]
    ypos = [y0-h_margin_top-(i+1)*h2 for i in range(cal_rowsnum)]

    # contents
    render_minical_month(c, year, month, xpos, ypos, w2, h2, y0-50*f)


# month view: 2 months vertical
def months_2month_vertical(c, year, month):
    minical_month(c, year, month, h)
    minical_month(c, year, month+1, h/2)
    c.bookmarkPage('month2_%d/%d' % (month, year))
    c.bookmarkPage('month2_%d/%d' % (month+1, year))
    add_pagenumber(c)
    add_copyright(c)
    c.showPage()


# month view
def months_view(c, year, landscape=False):
    if landscape:
        for m in range(12):
            months_1month_landscape(c, year, m+1)
    else:
        for m in range(6):
            months_2month_vertical(c, year, m*2+1)


######################################################################
##### PLANNER VIEW
def planner_3months(c, year, first_month, links):
    # calc values
    d1, dmax1 = calendar.monthrange(year, first_month)
    d2, dmax2 = calendar.monthrange(year, first_month+1)
    d3, dmax3 = calendar.monthrange(year, first_month+2)
    w2 = (w-60*f) / 3.0
    step = (h-40*f) / 32.0

    # headers
    c.setFont('Helvetica-Bold', 20*f)
    c.setFillColor(lightcoral)
    c.drawCentredString(w/2, h-24*f, 'Planning %d' % year)
    c.linkAbsolute('Index', 'index', (w/2-80*f, h-30*f, w/2+80*f, h-5*f))
    c.setFont('Times-Bold', 15*f)
    c.setFillColor(navy)
    c.drawString(35*f, h-46*f, '%s %d' % (calendar.month_name[first_month], year))
    if links:
        c.linkAbsolute('Months %d/%d' % (first_month, year),
                       'month_%d/%d' % (first_month, year),
                       (30*f, h-55*f, 30*f+w2, h-55*f+step))
    c.drawString(35*f+w2, h-46*f, '%s %d' % (calendar.month_name[first_month+1], year))
    if links:
        c.linkAbsolute('Months %d/%d' % (first_month+1, year),
                       'month_%d/%d' % (first_month+1, year),
                       (30*f+w2, h-55*f, 30*f+2*w2, h-55*f+step))
    c.drawString(35*f+2*w2, h-46*f, '%s %d' % (calendar.month_name[first_month+2], year))
    if links:
        c.linkAbsolute('Months %d/%d' % (first_month+2, year),
                       'month_%d/%d' % (first_month+2, year),
                       (30*f+2*w2, h-55*f, 30*f+3*w2, h-55*f+step))

    # contents
    c.setFont('Times-Roman', 13*f)
    c.setStrokeColor(black)
    c.setLineWidth(1*f)
    for i in range(32):
        y = h - 30*f - (i+1)*step
        c.line(5*f, y+step, w-5*f, y+step)
        dd1, dd2, dd3 = (d1+i-1) % 7, (d2+i-1) % 7, (d3+i-1) % 7
        fill1 = 1 if dd1 > 4 and 1 <= i <= dmax1 else 0
        fill2 = 1 if dd2 > 4 and 1 <= i <= dmax2 else 0
        fill3 = 1 if dd3 > 4 and 1 <= i <= dmax3 else 0
        c.setFillColorRGB(0.8, 0.8, 0.8)
        c.rect(30*f, y, w2, step, fill=fill1)
        c.rect(30*f+w2, y, w2, step, fill=fill2)
        c.rect(30*f+2*w2, y, w2, step, fill=fill3)
        if i != 0:
            c.setFillColor(navy)
            c.drawRightString(25*f, y+7*f, str(i))
            c.drawRightString(w-10*f, y+7*f, str(i))
            c.setFillColor(grey)
            if i <= dmax1:
                c.drawString(35*f, y+7*f, wds[dd1])
            if i <= dmax2:
                c.drawString(35*f+w2, y+7*f, wds[dd2])
            if i <= dmax3:
                c.drawString(35*f+2*w2, y+7*f, wds[dd3])

    # create lines
    c.rect(5*f, 10*f, w-10*f, h-15*f)
    c.line(55*f, 10*f, 55*f, h-30*f-step)
    c.line(55*f+w2, 10*f, 55*f+w2, h-30*f-step)
    c.line(55*f+2*w2, 10*f, 55*f+2*w2, h-30*f-step)

    # write page
    c.bookmarkPage('planner_%d/%d' % (first_month, year))
    add_pagenumber(c)
    add_copyright(c)
    c.showPage()


def planner_view(c, year, links=True):
    planner_3months(c, year, 1, links)
    planner_3months(c, year, 4, links)
    planner_3months(c, year, 7, links)
    planner_3months(c, year, 10, links)


######################################################################
##### AGENDA
def __get_prevweek(year, month):
    if month == 1:
        y = year-1; m = 12
    else:
        y = year; m = month-1
    prevweek = cal.monthdayscalendar(y, m)[-1]
    return [(d, 0) for d in prevweek], m, y


def __get_nextweek(year, month):
    if month < 12:
        y = year; m = month+1
    else:
        y = year+1; m = 1
    nextweek = cal.monthdayscalendar(y, m)[0]
    return [(d, 0) for d in nextweek], m, y


def nanocal_month(c, year, month, x, y, w2, current=False):
    c.setStrokeColorRGB(0.8, 0.8, 0.8)
    c.setFillColorRGB(0.8, 0.8, 0.8)
    if current:
        h2 = len(cal.monthdayscalendar(year, month)) + 1
        c.rect(x, y+7*f, w2, -h2*7.5*f, fill=1)
    c.linkRect('Month %d/%d' % (month, year), 'month_%d/%d' % (month, year),
                   (x, y+7*f, x+w2, y-45*f))
    # month header
    c.setFont('Courier-Bold', 6*f)
    c.setFillColor(grey)
    c.setLineWidth(1*f)
    c.setStrokeColor(grey)
    c.drawCentredString(x+w2/2, y, calendar.month_name[month])
    c.line(x, y-2*f, x+w2, y-2*f)
    # days
    c.setFont('Courier', 6*f)
    for week in cal.monthdays2calendar(year, month):
        y = y - 7*f
        for i, (d, dw) in enumerate(week):
            if d == 0:
                continue
            color = grey if dw < 5 else navy
            c.setFillColor(color)
            c.drawString(x+i*w2/8.+2*f, y, '%2d' % d)


def minical_halfyear(c, year, x, y, ww2, current_month, initial=1):
    ww3 = (ww2-5*4*f) / 6.0
    xpos = [x+(ww3+4*f)*i for i in range(6)]
    for m in range(6):
        current = True if current_month == m+initial else False
        nanocal_month(c, year, m+initial, xpos[m], y, ww3, current)


def draw_agenda_wholeweek_landscape(c, year, monthnum, ww, hh, ww_margin, ww2,
                                    m1, y1, m2, y2, weeknum, week, xpos):
    # rotate page
    c.saveState()
    c.rotate(90)

    # page header
    c.setStrokeColor(lightcoral)
    c.setFont('Helvetica', 12*f)
    c.setFillColor(lightcoral)
    c.drawString(ww_margin, -15*f, '%s %d' % (calendar.month_name[m1], y1))
    c.linkRect('Index', 'index', (0, 0, 100*f, -20*f))
    c.drawString(ww/2+10*f, -15*f, 'Week: %d' % weeknum)
    c.linkRect('Index', 'index', (ww/2+5*f, 0, ww/2+70*f, -20*f))
    c.drawRightString(ww-ww_margin, -15*f,
                      '%s %d' % (calendar.month_name[m2], y2))
    c.linkRect('Index', 'index', (ww-100*f, 0, ww, -20*f))

    # main division lines
    c.setLineWidth(2*f)
    c.line(ww_margin, -20*f, ww/2-ww_margin, -20*f)
    c.line(ww/2+ww_margin, -20*f, ww-ww_margin, -20*f)
    c.line(ww_margin, -100*f, ww/2-ww_margin, -100*f)
    c.line(ww/2+ww_margin, -100*f, ww-ww_margin, -100*f)
    c.line(ww_margin, -(hh-140*f), ww/2-ww_margin, -(hh-140*f))
    c.line(ww/2+ww_margin, -(hh-140*f), ww-ww_margin, -(hh-140*f))
    c.line(ww_margin, -(hh-65*f), ww/2-ww_margin, -(hh-65*f))
    c.line(ww/2+ww_margin, -(hh-65*f), ww-ww_margin, -(hh-65*f))
    c.setStrokeColor(navy)
    c.line(ww_margin, -60*f, ww/2-ww_margin, -60*f)
    c.line(ww/2+ww_margin, -60*f, ww-ww_margin, -60*f)

    # lines above diary
    c.setLineWidth(1*f)
    c.setStrokeColorRGB(0.8, 0.8, 0.8)
    step = 40*f / 3.0
    y = -60*f - step
    for i in range(2):
        c.line(ww_margin, y-i*step, ww/2-ww_margin, y-i*step)
        c.line(ww/2+ww_margin, y-i*step, ww-ww_margin, y-i*step)

    # lines in diary
    step = (hh-240*f) / 27.0
    c.setFillColor(navy)
    c.setFont('Times-Roman', 8*f)
    for i in range(27-1):
        y = -100*f-step*(i+1)
        c.line(ww_margin, y, ww/2-ww_margin, y)
        if i <= 13:
            c.line(ww/2+ww_margin, y, ww-ww_margin, y)
        else:
            c.line(ww/2+ww_margin, y, xpos[4]+ww2, y)
        if i%2 == 0:
            for j in range(6):
                if j == 5 and i > 13:
                    break
                c.drawRightString(xpos[j]+8*f, y+3*f, str(i/2+8))
    for i in range(9): # complete sunday
        y = -100*f - step*(14+4+i)
        c.line(xpos[5], y, ww-ww_margin, y)

    # lines above minicalendars
    step = 75*f / 5.0
    y = -(hh-140*f) - step
    for i in range(4):
        c.line(ww_margin, y-i*step, ww/2-ww_margin, y-i*step)
        c.line(ww/2+ww_margin, y-i*step, ww-ww_margin, y-i*step)

    # day
    c.setFillColor(navy)
    for i, (d, dw) in enumerate(week):
        c.setFont('Times-Roman', 40*f)
        c.drawString(xpos[i], -55*f, str(d))
        c.setFont('Times-Roman', 15*f)
        c.drawString(xpos[i]+45*f, -55*f, calendar.day_name[dw])
    y = -100*f - step*(14+1)
    c.setLineWidth(2*f)
    c.setStrokeColor(navy)
    c.line(xpos[5], y, ww-ww_margin, y)
    c.setFont('Times-Roman', 40*f)
    c.setFillColor(navy)
    c.drawString(xpos[5], y+5*f, str(week[6][0]))
    c.setFont('Times-Roman', 15*f)
    c.drawString(xpos[5]+45*f, y+5*f, calendar.day_name[week[6][1]])

    # page bottom: minicalendars
    ww2 = ww/2 - 2*ww_margin
    y = -hh + 50*f
    minical_halfyear(c, year, ww_margin, y, ww2, monthnum+1, 1)
    minical_halfyear(c, year, ww/2+ww_margin, y, ww2, monthnum+1, 7)

    # write page
    c.restoreState()
    c.bookmarkPage('agenda_week_%d/%d' % (weeknum, year))
    add_pagenumber(c)
    add_copyright(c)
    c.showPage()


def agenda_wholeweek_landscape(c, year):
    wholeyear = [cal.monthdays2calendar(year, m+1) for m in range(12)]
    weeknum = 0

    # calc values
    hh, ww = w, h
    ww_margin = 10*f
    ww_spacing = 10*f
    ww2 = (ww-2*ww_margin-5*ww_spacing) / 6.0
    xpos = [ww_margin+i*(ww2+ww_spacing) for i in range(7)]

    # show week
    skipnext = False
    for monthnum, month in enumerate(wholeyear):
        m1, y1 = m2, y2 = monthnum+1, year
        for week in month:
            if skipnext:
                skipnext = False
                continue
            weeknum += 1
            if weeknum == 1 and week[0][0] == 0: # beginning of year
                nw, m1, y1 = __get_prevweek(year, monthnum+1)
                week = [(week[i][0]+nw[i][0], week[i][1]+nw[i][1]) for i in range(7)]
            else:
                m1, y1 = monthnum+1, year
            if week[6][0] == 0: # end of month
                nw, m2, y2 = __get_nextweek(year, monthnum+1)
                week = [(week[i][0]+nw[i][0], week[i][1]+nw[i][1]) for i in range(7)]
                skipnext = True
            draw_agenda_wholeweek_landscape(c, year, monthnum, ww, hh, ww_margin, ww2,
                                            m1, y1, m2, y2, weeknum, week, xpos)


def agenda_view(c, year):
    agenda_wholeweek_landscape(c, year)


######################################################################
##### main
def main(year):
    c = prepare_canvas(year)
#     cover_page(year)
    year_view(c, year)
    year_3view(c, year)
    planner_view(c, year)
    months_view(c, year)
    months_view(c, year, landscape=True)
    agenda_view(c, year)
    planner_view(c, year+1, links=False)
#     back_page()
    c.save()


if __name__ == '__main__':
    main(YEAR)


######################################################################
##### TODO:
"""
- create cover and back pages
- modify author and copyright with program name and url
- create index
- Links:
  . year view:
    + month => month_view DONE
    + links DONE
  . month view:
    + title => year_view DONE
    + day => agenda_view (week, day)
  . planner view:
    + title => year_view DONE
    + month header => month_view DONE
    + day => agenda_view (week, day)
  . agenda:
    + "week" => year_view DONE
    + mini month cal => month_view DONE
- Configure options: default and command line
  . activate file compression
  . .pdf paper size: 124x152
  . first day of week
  . select what contents to create:
    + year/s
    + months vertical
    + months landscape
    + planner
    + agenda whole week landscape per page
    + agenda half week vertical per page
    + agenda day vertical per page
- populate agenda with .ics file contents
- i18n, l10n
- make links more visible

- Requirements: Python v2.5, ReportLab
"""
######################################################################
