#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy
from PyQt4.Qt import (Qt, QWidget, QGridLayout, QLabel, QPushButton, QUrl,
                      QGroupBox, QComboBox, QVBoxLayout, QCheckBox,
                      QLineEdit, QHBoxLayout)

from calibre.gui2 import open_url
from calibre.utils.config import JSONConfig

from calibre_plugins.koboutilities.common_utils import (get_library_uuid, CustomColumnComboBox,
                                     debug_print, KeyboardConfigDialog, KeyComboBox, ImageTitleLayout)


PREFS_NAMESPACE = 'KoboUtilitiesPlugin'
PREFS_KEY_SETTINGS = 'settings'

KEY_SCHEMA_VERSION = 'SchemaVersion'
DEFAULT_SCHEMA_VERSION = 0.1

STORE_LIBRARIES = 'libraries'
KEY_CURRENT_LOCATION_CUSTOM_COLUMN = 'currentReadingLocationColumn'
KEY_PERCENT_READ_CUSTOM_COLUMN     = 'precentReadColumn'
KEY_RATING_CUSTOM_COLUMN           = 'ratingColumn'
KEY_LAST_READ_CUSTOM_COLUMN        = 'lastReadColumn'
DEFAULT_LIBRARY_VALUES = {
                          KEY_CURRENT_LOCATION_CUSTOM_COLUMN: '',
                          KEY_PERCENT_READ_CUSTOM_COLUMN:     '',
                          KEY_RATING_CUSTOM_COLUMN:           None,
                          KEY_LAST_READ_CUSTOM_COLUMN:        None,
                         }

BOOKMARK_OPTIONS_STORE_NAME     = 'BookmarkOptions'
METADATA_OPTIONS_STORE_NAME     = 'MetadataOptions'
READING_OPTIONS_STORE_NAME      = 'ReadingOptions'
COMMON_OPTIONS_STORE_NAME       = 'commonOptionsStore'
DISMISSTILES_OPTIONS_STORE_NAME = 'dismissTilesOptionsStore'

KEY_STORE_BOOKMARK          = 'storeBookmarks'
KEY_DATE_TO_NOW             = 'setDateToNow'
KEY_SET_RATING              = 'setRating'
KEY_CLEAR_IF_UNREAD         = 'clearIfUnread'
KEY_BACKGROUND_JOB          = 'backgroundJob'
KEY_SET_TITLE               = 'title'
KEY_SET_AUTHOR              = 'author'
KEY_SET_DESCRIPTION         = 'description'
KEY_SET_PUBLISHER           = 'publisher'
KEY_SET_RATING              = 'rating'
KEY_SET_SERIES              = 'series'
KEY_SET_READING_STATUS      = 'setRreadingStatus'
KEY_READING_STATUS          = 'readingStatus'
KEY_SET_PUBLISHED_DATE      = 'published_date'
KEY_SET_ISBN                = 'isbn'
KEY_SET_NOT_INTERESTED      = 'mark_not_interested'
KEY_SET_LANGUAGE            = 'language'
KEY_RESET_POSITION          = 'resetPosition'
KEY_TILE_OPTIONS            = 'tileOptions'
KEY_CHANGE_DISMISS_TRIGGER  = 'changeDismissTrigger'
KEY_CREATE_DISMISS_TRIGGER  = 'createDismissTrigger'
KEY_DELETE_DISMISS_TRIGGER  = 'deleteDismissTrigger'
KEY_CREATE_ANALYTICSEVENTS_TRIGGER  = 'createAnalyticsEventsTrigger'
KEY_DELETE_ANALYTICSEVENTS_TRIGGER  = 'deleteAnalyticsEventsTrigger'
KEY_TILE_RECENT_NEW         = 'tileRecentBooksNew'
KEY_TILE_RECENT_FINISHED    = 'tileRecentBooksFinished'

KEY_READING_FONT_FAMILY     = 'readingFontFamily'
KEY_READING_ALIGNMENT       = 'readingAlignment'
KEY_READING_FONT_SIZE       = 'readingFontSize'
KEY_READING_LINE_HEIGHT     = 'readingLineHeight'
KEY_READING_LEFT_MARGIN     = 'readingLeftMargin'
KEY_READING_RIGHT_MARGIN    = 'readingRightMargin'

KEY_BUTTON_ACTION_DEVICE    = 'buttonActionDevice'
KEY_BUTTON_ACTION_LIBRARY   = 'buttonActionLibrary'

BOOKMARK_OPTIONS_DEFAULTS = {
                KEY_STORE_BOOKMARK:  True,
                KEY_READING_STATUS:  True,
                KEY_DATE_TO_NOW:     True, 
                KEY_SET_RATING:      True, 
                KEY_CLEAR_IF_UNREAD: False, 
                KEY_BACKGROUND_JOB:  False
                }
METADATA_OPTIONS_DEFAULTS = {
                KEY_SET_TITLE:          False,
                KEY_SET_AUTHOR:         False,
                KEY_SET_DESCRIPTION:    False,
                KEY_SET_PUBLISHER:      False,
                KEY_SET_RATING:         False,
                KEY_SET_SERIES:         False,
                KEY_SET_READING_STATUS: False,
                KEY_READING_STATUS:     -1,
                KEY_SET_PUBLISHED_DATE: False,
                KEY_SET_ISBN:           False,
                KEY_SET_NOT_INTERESTED: False,
                KEY_SET_LANGUAGE:       False,
                KEY_RESET_POSITION:     False
                }
READING_OPTIONS_DEFAULTS = {
                KEY_READING_FONT_FAMILY:  'Georgia',
                KEY_READING_ALIGNMENT:    'Off',
                KEY_READING_FONT_SIZE:    22,
                KEY_READING_LINE_HEIGHT:  1.3,
                KEY_READING_LEFT_MARGIN:  3,
                KEY_READING_RIGHT_MARGIN: 3,
                }
COMMON_OPTIONS_DEFAULTS = {
                KEY_BUTTON_ACTION_DEVICE:   '',
                KEY_BUTTON_ACTION_LIBRARY:  '',
                }
DISMISSTILES_OPTIONSS_DEFAULTS = {
                KEY_TILE_OPTIONS:{},
                KEY_TILE_RECENT_NEW:      False,
                KEY_TILE_RECENT_FINISHED: False
                }

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Kobo Utilities')

# Set defaults
plugin_prefs.defaults[BOOKMARK_OPTIONS_STORE_NAME]      = BOOKMARK_OPTIONS_DEFAULTS
plugin_prefs.defaults[METADATA_OPTIONS_STORE_NAME]      = METADATA_OPTIONS_DEFAULTS
plugin_prefs.defaults[READING_OPTIONS_STORE_NAME]       = READING_OPTIONS_DEFAULTS
plugin_prefs.defaults[COMMON_OPTIONS_STORE_NAME]        = COMMON_OPTIONS_DEFAULTS
plugin_prefs.defaults[DISMISSTILES_OPTIONS_STORE_NAME]  = DISMISSTILES_OPTIONSS_DEFAULTS
plugin_prefs.defaults[STORE_LIBRARIES]                  = {}


def get_plugin_pref(store_name, option):
    c = plugin_prefs[store_name]
    default_value = plugin_prefs.defaults[store_name][option]
    return c.get(option, default_value)

def get_plugin_prefs(store_name):
    c = plugin_prefs[store_name]
    return c

def migrate_library_config_if_required(db, library_config):
    schema_version = library_config.get(KEY_SCHEMA_VERSION, 0)
    if schema_version == DEFAULT_SCHEMA_VERSION:
        return
    # We have changes to be made - mark schema as updated
    library_config[KEY_SCHEMA_VERSION] = DEFAULT_SCHEMA_VERSION

    # Any migration code in future will exist in here.
    if schema_version < 0.1:
        if 'customColumn' in library_config:
            print('Migrating Count Pages plugin custom column for pages to new schema')
            library_config[KEY_PAGES_CUSTOM_COLUMN] = library_config['customColumn']
            del library_config['customColumn']
        store_prefs = plugin_prefs[STORE_NAME]
        if KEY_PAGES_ALGORITHM not in library_config:
            print('Migrating Count Pages plugin algorithm for pages to new schema')
            library_config[KEY_PAGES_ALGORITHM] = store_prefs.get('algorithm', 0)
            # Unfortunately cannot delete since user may have other libraries
        if 'algorithmWords' in store_prefs:
            print('Deleting Count Pages plugin word algorithm')
            del store_prefs['algorithmWords']
            plugin_prefs[STORE_NAME] = store_prefs

    set_library_config(db, library_config)


def get_library_config(db):
    library_id = get_library_uuid(db)
    library_config = None
    # Check whether this is a configuration needing to be migrated from json into database
    if 'libraries' in plugin_prefs:
        libraries = plugin_prefs['libraries']
        if library_id in libraries:
            # We will migrate this below
            library_config = libraries[library_id]
            # Cleanup from json file so we don't ever do this again
            del libraries[library_id]
            if len(libraries) == 0:
                # We have migrated the last library for this user
                del plugin_prefs['libraries']
            else:
                plugin_prefs['libraries'] = libraries

    if library_config is None:
        library_config = db.prefs.get_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS,
                                                 copy.deepcopy(DEFAULT_LIBRARY_VALUES))
#    migrate_library_config_if_required(db, library_config)
    return library_config


def set_library_config(db, library_config):
    db.prefs.set_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS, library_config)


class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        layout = QVBoxLayout(self)
        self.setLayout(layout)
        self.help_anchor = "configuration"
        
#        heading_layout = QHBoxLayout()
#        layout.addLayout(heading_layout)
        title_layout = ImageTitleLayout(self, 'images/icon.png', 'Kobo Utilities Options')
        layout.addLayout(title_layout)

#        c = plugin_prefs[STORE_NAME]
        library_config = get_library_config(self.plugin_action.gui.current_db)

        custom_column_group = QGroupBox('Custom Columns', self)
        layout.addWidget(custom_column_group )
        options_layout = QGridLayout()
        custom_column_group.setLayout(options_layout)

        avail_text_columns   = self.get_text_custom_columns()
        avail_number_columns = self.get_number_custom_columns()
        avail_rating_columns = self.get_rating_custom_columns()
        avail_date_columns   = self.get_date_custom_columns()
#        debug_print("ConfigWidget:__init__ - avail_rating_columns=", avail_rating_columns)
#        debug_print("ConfigWidget:__init__ - default columns=", self.plugin_action.gui.library_view.model().orig_headers)
        current_Location_column = library_config.get(KEY_CURRENT_LOCATION_CUSTOM_COLUMN, DEFAULT_LIBRARY_VALUES[KEY_CURRENT_LOCATION_CUSTOM_COLUMN])
        precent_read_column     = library_config.get(KEY_PERCENT_READ_CUSTOM_COLUMN, DEFAULT_LIBRARY_VALUES[KEY_PERCENT_READ_CUSTOM_COLUMN])
        rating_column           = library_config.get(KEY_RATING_CUSTOM_COLUMN, DEFAULT_LIBRARY_VALUES[KEY_RATING_CUSTOM_COLUMN])
        last_read_column        = library_config.get(KEY_LAST_READ_CUSTOM_COLUMN, DEFAULT_LIBRARY_VALUES[KEY_LAST_READ_CUSTOM_COLUMN])
#        debug_print("ConfigWidget:__init__ - current_Location_column=%s, precent_read_column=%s, rating_column=%s" % (current_Location_column, precent_read_column, rating_column))

        current_Location_label = QLabel('Current Reading Location Column:', self)
        current_Location_label.setToolTip("'Select a custom column to store the current reading location. The column type must be 'text'. Leave this blank if you do not want to store or restore the current reading location.")
        self.current_Location_combo = CustomColumnComboBox(self, avail_text_columns, current_Location_column)
        current_Location_label.setBuddy(self.current_Location_combo)
        options_layout.addWidget(current_Location_label, 0, 0, 1, 1)
        options_layout.addWidget(self.current_Location_combo, 0, 1, 1, 1)
        
        percent_read_label = QLabel('Percent Read Column:', self)
        percent_read_label.setToolTip("Column used to store the current percent read. The column type must be a 'integer'. Leave this blank if you do not want to store or restore the percentage read.")
        self.percent_read_combo = CustomColumnComboBox(self, avail_number_columns, precent_read_column)
        percent_read_label.setBuddy(self.percent_read_combo)
        options_layout.addWidget(percent_read_label, 2, 0, 1, 1)
        options_layout.addWidget(self.percent_read_combo, 2, 1, 1, 1)

        rating_label = QLabel('Rating Column:', self)
        rating_label.setToolTip("Column used to store the rating. The column type must be a 'integer'. Leave this blank if you do not want to store or restore the rating.")
        self.rating_combo = CustomColumnComboBox(self, avail_rating_columns, rating_column)
        rating_label.setBuddy(self.rating_combo)
        options_layout.addWidget(rating_label, 3, 0, 1, 1)
        options_layout.addWidget(self.rating_combo, 3, 1, 1, 1)

        last_read_label = QLabel('Last Read Column:', self)
        last_read_label.setToolTip("Column used to store when the book was last read. The column type must be a 'Date'. Leave this blank if you do not want to store the last read timestamp.")
        self.last_read_combo = CustomColumnComboBox(self, avail_date_columns, last_read_column)
        last_read_label.setBuddy(self.last_read_combo)
        options_layout.addWidget(last_read_label, 4, 0, 1, 1)
        options_layout.addWidget(self.last_read_combo, 4, 1, 1, 1)

        other_options_group = QGroupBox('Other Options', self)
        layout.addWidget(other_options_group )
        options_layout = QGridLayout()
        other_options_group.setLayout(options_layout)

        library_default_label = QLabel('&Library Button default:', self)
        library_default_label.setToolTip('If plugin is placed as a toolbar button, choose a default action when clicked on')
        self.library_default_combo = KeyComboBox(self, self.plugin_action.library_actions_map, unicode(get_plugin_pref(COMMON_OPTIONS_STORE_NAME, KEY_BUTTON_ACTION_LIBRARY)))
        library_default_label.setBuddy(self.library_default_combo)
        options_layout.addWidget(library_default_label, 0, 0, 1, 1)
        options_layout.addWidget(self.library_default_combo, 0, 1, 1, 2)

        device_default_label = QLabel('&Device Button default:', self)
        device_default_label.setToolTip('If plugin is placed as a toolbar button, choose a default action when clicked on')
        self.device_default_combo = KeyComboBox(self, self.plugin_action.device_actions_map, unicode(get_plugin_pref(COMMON_OPTIONS_STORE_NAME, KEY_BUTTON_ACTION_DEVICE)))
        device_default_label.setBuddy(self.device_default_combo)
        options_layout.addWidget(device_default_label, 1, 0, 1, 1)
        options_layout.addWidget(self.device_default_combo, 1, 1, 1, 2)

        keyboard_shortcuts_button = QPushButton('Keyboard shortcuts...', self)
        keyboard_shortcuts_button.setToolTip('Edit the keyboard shortcuts associated with this plugin')
        keyboard_shortcuts_button.clicked.connect(self.edit_shortcuts)
        layout.addWidget(keyboard_shortcuts_button)
        layout.addStretch(1)


    def save_settings(self):
        new_prefs = {}
        new_prefs[KEY_BUTTON_ACTION_DEVICE]     = unicode(self.device_default_combo.currentText())
        new_prefs[KEY_BUTTON_ACTION_LIBRARY]    = unicode(self.library_default_combo.currentText())
        plugin_prefs[COMMON_OPTIONS_STORE_NAME] = new_prefs

        db = self.plugin_action.gui.current_db
        library_config = get_library_config(db)
        library_config[KEY_CURRENT_LOCATION_CUSTOM_COLUMN] = self.current_Location_combo.get_selected_column()
        library_config[KEY_PERCENT_READ_CUSTOM_COLUMN]     = self.percent_read_combo.get_selected_column()
        library_config[KEY_RATING_CUSTOM_COLUMN]           = self.rating_combo.get_selected_column()
        library_config[KEY_LAST_READ_CUSTOM_COLUMN]        = self.last_read_combo.get_selected_column()
        set_library_config(db, library_config)

    def get_number_custom_columns(self):
        column_types = ['float','int']
        return self.get_custom_columns(column_types)

    def get_rating_custom_columns(self):
        column_types = ['rating','int']
        custom_columns = self.get_custom_columns(column_types)
        ratings_column_name = self.plugin_action.gui.library_view.model().orig_headers['rating']
        custom_columns['rating'] = {'name': ratings_column_name}
        return custom_columns

    def get_text_custom_columns(self):
        column_types = ['text']
        return self.get_custom_columns(column_types)

    def get_date_custom_columns(self):
        column_types = ['datetime']
        return self.get_custom_columns(column_types)

    def get_custom_columns(self, column_types):
        custom_columns = self.plugin_action.gui.library_view.model().custom_columns
        available_columns = {}
        for key, column in custom_columns.iteritems():
            typ = column['datatype']
            if typ in column_types and not column['is_multiple']:
                available_columns[key] = column
        return available_columns

    def help_link_activated(self, url):
        self.plugin_action.show_help(anchor="configuration")

    def edit_shortcuts(self):
        d = KeyboardConfigDialog(self.plugin_action.gui, self.plugin_action.action_spec[0])
        if d.exec_() == d.Accepted:
            self.plugin_action.gui.keyboard.finalize()
