#!/usr/bin/env python
# coding: utf-8

from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__ = 'GPL v3'
__copyright__ = '2013, Greg Riker <griker@hotmail.com>'
__docformat__ = 'restructuredtext en'

import glob, os, re, subprocess, sys

from collections import OrderedDict
from datetime import datetime
from lxml import etree
from time import sleep

from PyQt4.Qt import QModelIndex

from calibre.constants import islinux, isosx, iswindows
from calibre.ebooks.metadata import MetaInformation
from calibre.gui2 import error_dialog
from calibre.utils.zipfile import ZipFile

from calibre.utils.config import JSONConfig
prefs = JSONConfig('plugins/annotations')

try:
    import appscript
    appscript
except:
    # appscript fails to load on 10.4
    appscript = None


class ClassNotImplementedException(Exception):
    ''' '''
    pass


class DatabaseNotFoundException(Exception):
    ''' '''
    pass


class NotImplementedException(Exception):
    pass


class UnknownAnnotationTypeException(Exception):
    ''' '''
    pass


class ReaderApp(object):

    BOOKS_DB_TEMPLATE = "{0}_{1}_books"
    ANNOTATIONS_DB_TEMPLATE = "{0}_{1}_annotations"

    reader_app_classes = None
    MAX_ELEMENT_DEPTH = 6
    SUPPORTS_EXPORTING = False
    SUPPORTS_FETCHING = False

    NSTimeIntervalSince1970 = 978307200.0


    def __init__(self, parent):
        """
        Basic initialization
        """
        self.log = parent.opts.log
        self.log_location = parent.opts.log_location
        self.log_invocation = parent.opts.log_invocation
        self.opts = parent.opts
        self.parent = parent

        # News clippings
        self.collect_news_clippings = JSONConfig('plugins/annotations').get('cfg_news_clippings_checkbox', False)
        self.news_clippings_destination = JSONConfig('plugins/annotations').get('cfg_news_clippings_lineEdit', None)
        self.news_clippings_cid = None
        if self.collect_news_clippings and self.news_clippings_destination:
            self.news_clippings_cid = self.get_clippings_cid(self.news_clippings_destination)

    def close(self):
        """
        Perform device-specific shutdown
        """
        pass

    def commit(self):
        self.opts.db.commit()

    def open(self):
        """
        Perform device-specific initialization required for file system access
        """
        pass

    ''' Database operations '''
    def add_to_annotations_db(self, annotations_db, annotation_mi):
        self.opts.db.add_to_annotations_db(annotations_db, annotation_mi)

    def add_to_books_db(self, books_db, book_mi):
        self.opts.db.add_to_books_db(books_db, book_mi)

    def create_annotations_table(self, cached_db):
        self.opts.db.create_annotations_table(cached_db)

    def create_books_table(self, cached_db):
        """
        The <app>_books_<device> table contains a list of books installed on the device.
        book_id is the unique id that the app uses to reference the book in its
        associated annotations table.
        """
        self.opts.db.create_books_table(cached_db)

    @staticmethod
    def generate_annotations_db_name(reader_app, device_name):
        return ReaderApp.ANNOTATIONS_DB_TEMPLATE.format(re.sub(' ', '_', reader_app), re.sub(' ', '_', device_name))

    @staticmethod
    def generate_books_db_name(reader_app, device_name):
        return ReaderApp.BOOKS_DB_TEMPLATE.format(re.sub(' ', '_', reader_app), re.sub(' ', '_', device_name))

    def get_books(self, books_db):
        return self.opts.db.get_books(books_db)

    def get_clippings_cid(self, title):
        '''
        Find or create cid for title
        '''
        cid = None
        try:
            cid = list(self.parent.opts.gui.current_db.data.parse('title:"%s" and tag:Clippings' % title))[0]
        except:
            mi = MetaInformation(title, authors = ['Various'])
            mi.tags = ['Clippings']
            cid = self.parent.opts.gui.current_db.create_book_entry(mi, cover=None,
                add_duplicates=False, force_id=None)
        return cid

    @staticmethod
    def get_exporting_app_classes():
        """
        Utility method to find all subclasses supporting exported annotations
        having a parse_exported_highlights() method
        {app_class_name: app_name}
        """
        exporting_apps = OrderedDict()
        racs = ReaderApp.get_reader_app_classes()
        sorted_racs = sorted(racs, key=unicode.lower)
        for app_name in sorted_racs:
            kls = racs[app_name]
            if getattr(kls, 'SUPPORTS_EXPORTING', False):
                exporting_apps[kls] = app_name
        return exporting_apps

    def get_genres(self, books_db, book_id):
        return self.opts.db.get_genres(books_db, book_id)

    @staticmethod
    def get_reader_app_classes():
        """
        Utility method to find all subclasses of ourselves
        {app_name:cls}
        """
        if ReaderApp.reader_app_classes is None:
            known_reader_app_classes = {}
            for c in ReaderApp._iter_subclasses(ReaderApp):
                # Skip the abstract classes
                if c.app_name == '<placeholder>':
                    continue
                known_reader_app_classes[c.app_name] = c
            ReaderApp.reader_app_classes = known_reader_app_classes
        return ReaderApp.reader_app_classes

    def get_title(self, books_db, book_id):
        return self.opts.db.get_title(books_db, book_id)

    def update_book_last_annotation(self, books_db, timestamp, book_id):
        self.opts.db.update_book_last_annotation(books_db, timestamp, book_id)

    def update_timestamp(self, cached_db):
        self.opts.db.update_timestamp(cached_db)

    ''' Helpers '''
    def _cache_is_current(self, dependent_file, cached_db):
        """
        cached_db: an entry in the timestamps table
        Return True if:
            dependent_file is newer than cached content in db
        Return False if:
            dependent_file is older than cached content in db
            cached_db does not exist
        """
        cached_timestamp = self.opts.db.get('''SELECT timestamp
                                               FROM timestamps
                                               WHERE db="{0}"'''.format(cached_db), all=False)
        current_timestamp = unicode(datetime.fromtimestamp(os.path.getmtime(dependent_file)))

        if False and self.opts.verbose:
            self.log_location(cached_timestamp > current_timestamp)
            if False:
                if os.path.exists(dependent_file):
                    self.log(" current_timestamp: %s" % repr(current_timestamp))
                else:
                    self.log(" '%s' does not exist" % dependent_file)
                self.log("  cached_timestamp: %s" % repr(cached_timestamp))

        return cached_timestamp > current_timestamp

    def _get_epub_toc(self, cid=None, path=None, prepend_title=None):
        '''
        Given a calibre id, return the epub TOC indexed by section
        If cid, use copy in library, else use path to copy on device
        '''
        toc = None
        if cid is not None:
            mi = self.opts.gui.current_db.get_metadata(cid, index_is_id=True)
            toc = None
            if 'EPUB' in mi.formats:
                fpath = self.opts.gui.current_db.format(cid, 'EPUB', index_is_id=True, as_path=True)
            else:
                return toc
        elif path is not None:
            fpath = os.path.join(self.mount_point, path)
        else:
            return toc

        # iBooks stores books unzipped
        # Marvin stores books zipped
        # Need spine, ncx_tree to construct toc

        if os.path.isdir(fpath):
            # Find the OPF in the unzipped ePub
            with open(os.path.join(fpath, 'META-INF', 'container.xml')) as cf:
                container = etree.parse(cf)
                opf_file = container.xpath('.//*[local-name()="rootfile"]')[0].get('full-path')
                oebps = opf_file.rpartition('/')[0]
            with open(os.path.join(fpath, opf_file)) as opf:
                opf_tree = etree.parse(opf)
                spine = opf_tree.xpath('.//*[local-name()="spine"]')[0]
                ncx_fs = spine.get('toc')
                manifest = opf_tree.xpath('.//*[local-name()="manifest"]')[0]
                ncx_file = manifest.find('.//*[@id="%s"]' % ncx_fs).get('href')
            with open(os.path.join(fpath, oebps, ncx_file)) as ncxf:
                ncx_tree = etree.parse(ncxf)
            #self.log(etree.tostring(ncx_tree, pretty_print=True))

        else:
            # Find the OPF file in the zipped ePub
            try:
                with open(fpath, 'rb') as zfo:
                    zf = ZipFile(fpath, 'r')
                    container = etree.fromstring(zf.read('META-INF/container.xml'))
                    opf_tree = etree.fromstring(zf.read(container.xpath('.//*[local-name()="rootfile"]')[0].get('full-path')))

                    spine = opf_tree.xpath('.//*[local-name()="spine"]')[0]
                    ncx_fs = spine.get('toc')
                    manifest = opf_tree.xpath('.//*[local-name()="manifest"]')[0]
                    ncx = manifest.find('.//*[@id="%s"]' % ncx_fs).get('href')

                    # Find the ncx file
                    fnames = zf.namelist()
                    _ncx = [x for x in fnames if ncx in x][0]
                    ncx_tree = etree.fromstring(zf.read(_ncx))
            except:
                import traceback
                self.log_location()
                self.log(" unable to unzip '%s'" % fpath)
                self.log(traceback.format_exc())
                return toc

        # fpath points to epub (zipped or unzipped dir)
        # spine, ncx_tree populated
        try:
            toc = OrderedDict()
            # 1. capture idrefs from spine
            for i, el in enumerate(spine):
                toc[str(i)] = el.get('idref')

            # 2. Resolve <spine> idrefs to <manifest> hrefs
            for el in toc:
                toc[el] = manifest.find('.//*[@id="%s"]' % toc[el]).get('href')

            # 3. Build a dict of src:toc_entry
            src_map = OrderedDict()
            navMap = ncx_tree.xpath('.//*[local-name()="navMap"]')[0]
            for navPoint in navMap:
                # Get the first-level entry
                src = re.sub(r'#.*$', '', navPoint.xpath('.//*[local-name()="content"]')[0].get('src'))
                toc_entry = navPoint.xpath('.//*[local-name()="text"]')[0].text
                src_map[src] = toc_entry

                # Get any nested navPoints
                nested_navPts = navPoint.xpath('.//*[local-name()="navPoint"]')
                for nnp in nested_navPts:
                    src = re.sub(r'#.*$', '', nnp.xpath('.//*[local-name()="content"]')[0].get('src'))
                    toc_entry = nnp.xpath('.//*[local-name()="text"]')[0].text
                    src_map[src] = toc_entry

            # Resolve src paths to toc_entry
            for section in toc:
                if toc[section] in src_map:
                    if prepend_title:
                        toc[section] = "%s &middot; %s" % (prepend_title,  src_map[toc[section]])
                    else:
                        toc[section] = src_map[toc[section]]
                else:
                    toc[section] = None

            # 5. Fill in the gaps
            current_toc_entry = None
            for section in toc:
                if toc[section] is None:
                    toc[section] = current_toc_entry
                else:
                    current_toc_entry = toc[section]
        except:
            import traceback
            self.log_invocation("reader_app_support:_get_epub_toc()")
            self.log("error parsing '%s'" % fpath)
            self.log(traceback.format_exc())
            self.log_invocation("end traceback")

        return toc

    @staticmethod
    def _iter_subclasses(cls, _seen=None):
        if not isinstance(cls, type):
            raise TypeError('itersubclasses must be called with '
                            'new-style classes, not %.100r' % cls)
        if _seen is None:
            _seen = set()
        try:
            subs = cls.__subclasses__()
        except TypeError:  # fails only when cls is type
            subs = cls.__subclasses__(cls)
        for sub in subs:
            if sub not in _seen:
                _seen.add(sub)
                yield sub
                for sub in ReaderApp._iter_subclasses(sub, _seen):
                    yield sub


class ExportingReader(ReaderApp):
    annotations_subpath = None
    app_folder = None
    app_name = '<placeholder>'
    books_subpath = None
    SUPPORTS_EXPORTING = True
    exporting_reader_classes = None

    def __init__(self, parent):
        ReaderApp.__init__(self, parent)
        self.active_annotations = {}
        self.annotations_db = None
        self.app_name_ = re.sub(' ', '_', self.app_name)
        self.books_db = None
        self.installed_books = []
        self.mount_point = None
        if hasattr(self.opts, 'mount_point'):
            self.mount_point = self.opts.mount_point


class iOSReaderApp(ReaderApp):
    """
    Generic class for iOS reader apps using iExplorer for file access
    """
    # Reader-specific characteristics defined in subclass
    app_folder = None
    app_name = '<placeholder>'
    books_subpath = None
    HIGHLIGHT_COLORS = []
    metadata_subpath = None
    reader_app_classes = None

    def __init__(self, parent):
        ReaderApp.__init__(self, parent)
        self.active_annotations = {}
        self.annotations_db = None
        self.app_name_ = re.sub(' ', '_', self.app_name)
        self.books_db = None
        self.installed_books = []
        self.mount_point = None
        if hasattr(parent.opts, 'mount_point'):
            self.mount_point = parent.opts.mount_point

    ''' Utilities '''

    @staticmethod
    def get_reader_app_classes():
        """
        Utility method to find all subclasses of ourselves
        {app_name:cls}
        """
        if iOSReaderApp.reader_app_classes is None:
            known_reader_app_classes = {}
            for c in iOSReaderApp._iter_subclasses(iOSReaderApp):
                known_reader_app_classes[c.app_name] = c
            iOSReaderApp.reader_app_classes = known_reader_app_classes
        return iOSReaderApp.reader_app_classes

    @staticmethod
    def get_sqlite_app_classes():
        """
        Utility method to find all subclasses supporting fetching annotations from
        sqlite databases managed by app. These subclasses have a get_installed_books()
        method.
        {app_class_name: app_name}
        """
        sqlite_apps = OrderedDict()
        racs = iOSReaderApp.get_reader_app_classes()
        sorted_racs = sorted(racs, key=unicode.lower)
        for app_name in sorted_racs:
            kls = racs[app_name]
            if getattr(kls, 'SUPPORTS_FETCHING', False):
                sqlite_apps[kls] = app_name
        return sqlite_apps

    ''' Helpers '''
    def _get_cached_books(self, cached_db):
        """
        Return a list of installed book_ids for cached_db
        """
        cached_books = []
        columns = self.opts.db.get('''PRAGMA table_info({0})'''.format(cached_db))
        cols = {}
        for column in columns:
            cols[column[1]] = column[0]

        rows = self.opts.db.get('''SELECT book_id FROM {0}'''.format(cached_db))
        for row in rows:
            cached_books.append(row[cols['book_id']])
        return cached_books

    @staticmethod
    def _iter_subclasses(cls, _seen=None):
        if not isinstance(cls, type):
            raise TypeError('itersubclasses must be called with '
                            'new-style classes, not %.100r' % cls)
        if _seen is None:
            _seen = set()
        try:
            subs = cls.__subclasses__()
        except TypeError:  # fails only when cls is type
            subs = cls.__subclasses__(cls)
        for sub in subs:
            if sub not in _seen:
                _seen.add(sub)
                yield sub
                for sub in iOSReaderApp._iter_subclasses(sub, _seen):
                    yield sub

    def _validate_database_path(self, db_path, delay=0.5):
        valid_path = None
        tries = TOTAL_ATTEMPTS = 10
        while tries:
            try:
                valid_path = glob.glob(db_path)[0]
                if self.opts.verbose and tries < TOTAL_ATTEMPTS:
                    sys.stdout.write('\n')
                break
            except:
                sleep(delay)
                tries -= 1
                self.log(" waiting for '%s' %s" % (os.path.basename(db_path), '.' * (TOTAL_ATTEMPTS - tries)))
        else:
            self.close()
            raise DatabaseNotFoundException(db_path)

        return valid_path


class iOSMounter():
    """
    Generic class for iOS reader apps using iExplorer for file access
    """
    if isosx:
        INTERFACE_APP = {'path': '/Applications/iExplorer.app',
                           'fs': 'osxfuse'}
    elif iswindows:
        INTERFACE_APP = {'path': 'iExplorer.exe',
                           'fs': None}
    elif islinux:
        INTERFACE_APP = {'path': None}

    INTERFACE_APP['name'] = None
    if INTERFACE_APP['path'] is not None:
        INTERFACE_APP['name'] = os.path.basename(INTERFACE_APP['path']).rpartition('.')[0]

    def __init__(self, opts):
        self.opts = opts
        self.log = opts.log
        self.log_location = opts.log_location
        self.mount_point = None
        self.already_running = False
        self.app_path = self._get_app_path()
        self.installed_reader_apps = {}
        self.exporting_apps = {}

    def _get_app_path(self):
        ans = None
        if isosx:
            try:
                cn = str(appscript.app(self.INTERFACE_APP['name']))
                match = re.search(r"app\(u('.*')\)", cn)
                if match:
                    ans = match.group(1)
            except:
                pass

        elif iswindows:
            ans = prefs.get('cfg_path_to_ie_lineEdit', None)
            if ans == '':
                ans = None

        return ans

    def _get_installed_reader_apps(self):
        """
        Discover installed reader apps on connected iDevice
        {app_name: app_class}
        """
        reader_apps = {}
        known_reader_apps = iOSReaderApp.get_reader_app_classes()
        for kls in known_reader_apps.values():
            app_folder = getattr(kls, 'app_folder', None)
            if app_folder and os.path.exists(os.path.join(self.mount_point, app_folder)):
                reader_apps[kls.app_name] = kls
        return reader_apps

    def _get_reader_app_paths(self):
        """
        Return a list of app paths on iDevice.
        Used to determine which mounted drive is the iDevice
        """
        reader_app_paths = []
        known_reader_apps = iOSReaderApp.get_reader_app_classes()
        for kls in known_reader_apps.values():
            reader_app_paths.append(getattr(kls, 'app_folder', None))
        return reader_app_paths

    def mount(self, delay=0.5):
        '''
        Try to mount the iDevice if supported reader apps installed
        '''

        from time import sleep

        self.log_location(self.INTERFACE_APP['name'])

        if prefs.get('cfg_disable_iexplorer_radioButton', True):
            self.log_location("app disabled")
        elif not self.app_path:
            self.log_location("app not installed")
        else:
            tries = 10

            if isosx:
                while True:
                    running_apps = appscript.app('System Events')
                    if not self.INTERFACE_APP['name'] in running_apps.processes.name():
                        self.opts.pb.set_label("Mounting iDevice")
                        self.opts.pb.set_value(tries)
                        self.opts.pb.set_maximum(tries + 2)
                        self.opts.pb.show()

                        self.log.info(" launching %s" % self.app_path)
                        cmd = 'open %s -g -j' % self.app_path
                        os.system(cmd)
                        # Kill the UI, as we only need the mounted volume
                        #cmd = 'killall -KILL %s' % self.IOS_FILESYSTEM_APP['name']
                        #retcode = os.system(cmd)
                        sleep(1)
                    else:
                        self.log.info(" %s already running" % self.INTERFACE_APP['name'])
                        self.already_running = True

                    # Get the mounted volumes

                    while tries:
                        proc = subprocess.Popen('mount', shell=True,
                                                 stdout=subprocess.PIPE,
                                                 stderr=subprocess.STDOUT)
                        while True:
                            try:
                                mounted_raw = proc.communicate()[0]
                                mounted = mounted_raw.split('\n')
                                break
                            except:
                                if proc.returncode is None:
                                    continue

                        for item in mounted:
                            # Look for folders mounted by iExplorer
                            if "%s@%s" % (self.INTERFACE_APP['name'], self.INTERFACE_APP['fs']) in item:
                                mp = re.search(r"on (.+?) \(", item)
                                mount_point = mp.group(1)

                                drive_ready = False
                                reader_app_paths = self._get_reader_app_paths()
                                app_tries = 5
                                while not drive_ready and app_tries:
                                    for app in reader_app_paths:
                                        #self.log("probing %s" % os.path.join(mount_point, app))
                                        if app and os.path.exists(os.path.join(mount_point, app)):
                                            drive_ready = True
                                            self.mount_point = mount_point
                                            break
                                        else:
                                            #self.log("waiting for %s to become readable" % app)
                                            sleep(delay)
                                            app_tries -= 1
                                            if not app_tries:
                                                self.log_location("no supported reader apps found on iDevice")
                                                if not self.already_running:
                                                    self.mount_point = mount_point
                                                    self.unmount()
                                tries = 0
                                break
                        else:
                            self.opts.pb.increment()
                            tries -= 1
                            if tries:
                                sleep(delay)
                    break

            elif iswindows:
                import win32api, win32con, win32gui
                toplist = []
                winlist = []

                def enum_callback(hwnd, results):
                    winlist.append((hwnd, win32gui.GetWindowText(hwnd)))

                def find_active_drives(delay=0.5, reverse=True):
                    drives = win32api.GetLogicalDriveStrings()
                    drives = drives.split('\000')[:-1]
                    if reverse:
                        drives.reverse()
                    return drives

                def find_running_app():
                    import subprocess
                    cmd = 'WMIC PROCESS get Caption,Commandline,Processid'
                    proc = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE)
                    pid = None
                    path = None
                    for line in proc.stdout:
                        match = re.match(r'iExplorer.*(\".*\")\s+(\S*)', line)
                        if match:
                            path = match.group(1)
                            pid = match.group(2)
                            break
                    return pid, path

                def hide_iExplorer_window(app_name, delay=1.0):
                    # http://nullege.com/codes/search/win32con.SW_MINIMIZE
                    while True:
                        win32gui.EnumWindows(enum_callback, toplist)
                        target = [(hwnd, title) for hwnd, title in winlist if app_name.lower() in title.lower()]
                        if target:
                            target = target[0]
                            win32gui.ShowWindow(target[0], win32con.SW_MINIMIZE)
                            break
                        else:
                            sleep(delay)

                def probe_for_iDevice():
                    """
                    Discover which drive has supported reader_apps
                    Return mount_point
                    """
                    reader_app_paths = self._get_reader_app_paths()
                    mounted_drives = find_active_drives(reverse=True)
                    mount_point = None
                    for md in mounted_drives:
                        for app in reader_app_paths:
                            if os.path.exists(os.path.join(md, app)):
                                mount_point = md
                                break
                        if mount_point:
                            break
                    return mount_point

                def wait_for_iDevice(previous_drives, delay=0.5):
                    current_drives = previous_drives
                    while current_drives == previous_drives:
                        current_drives = find_active_drives()
                        sleep(delay)
                    mount_point = list(set(current_drives).difference(set(previous_drives)))[0]
                    reader_app_paths = self._get_reader_app_paths()
                    drive_ready = False
                    app_tries = 5
                    while not drive_ready and app_tries:
                        for app in reader_app_paths:
                            #self.log("probing %s" % os.path.join(mount_point, app))
                            if os.path.exists(os.path.join(mount_point, app)):
                                drive_ready = True
                                break
                            else:
                                #self.log("waiting for %s to become readable" % app)
                                sleep(delay)
                                app_tries -= 1
                    if not app_tries:
                        self.log_location("no supported reader apps found on iDevice")
                        self.mount_point = mount_point
                        self.unmount()
                        mount_point = None
                    return mount_point

                windows_app_path = prefs.get('cfg_path_to_ie_lineEdit', None)
                if (windows_app_path):
                    pid, path = find_running_app()
                    if pid is None:
                        # Launch iExplorer
                        self.log(" launching %s" % windows_app_path)
                        if False:
                            self.opts.pb.set_label("Mounting iDevice")
                            self.opts.pb.set_value(tries)
                            self.opts.pb.set_maximum(tries + 2)
                            self.opts.pb.show()

                        try:
                            current_drives = find_active_drives()
                            p = subprocess.Popen(windows_app_path)
                            hide_iExplorer_window(self.INTERFACE_APP['name'])
                            self.mount_point = wait_for_iDevice(current_drives)
                        except:
                            self.log(" unable to launch %s" % windows_app_path)
                    else:
                        #iExplorer already running - find iDevice
                        self.log.info(" %s already running" % self.INTERFACE_APP['name'])
                        self.already_running = True
                        self.mount_point = probe_for_iDevice()
                else:
                    self.log(" no path initialized for iExplorer")

        if self.mount_point:
            self.installed_reader_apps = self._get_installed_reader_apps()
        self.opts.pb.hide()
        return self.mount_point

    def unmount(self):
        if self.mount_point:
            self.log_location(self.INTERFACE_APP['name'])

            if isosx:
                cmd = 'killall -KILL %s' % self.INTERFACE_APP['name']
                os.system(cmd)
                if self.mount_point:
                    #cmd = "umount '%s'" % self.mounted_iDevice
                    cmd = "diskutil unmount force '%s'" % self.mount_point
                    os.system(cmd)
                    self.log(" %s unmounted" % self.mount_point)
                    self.mount_point = None
            elif iswindows:
                os.system("taskkill /im %s.exe /f" % self.INTERFACE_APP['name'])
                self.log(" %s unmounted" % self.mount_point)
                self.mount_point = None


class USBReader(ReaderApp):
    annotations_subpath = None
    app_folder = None
    app_name = '<placeholder>'
    books_subpath = None
    SUPPORTS_FETCHING = True
    usb_reader_classes = None

    def __init__(self, parent):
        ReaderApp.__init__(self, parent)
        self.active_annotations = {}
        self.annotations_db = None
        self.app_name_ = re.sub(' ', '_', self.app_name)
        self.books_db = None
        self.installed_books = []
        self.mount_point = None
        if hasattr(self.opts, 'mount_point'):
            self.mount_point = self.opts.mount_point

    def get_path_map(self):
        '''
        Models gui2.actions.annotate FetchAnnotationsAction():fetch_annotations()
        Generate a path_map from selected ids

        {id:{'path':'<storage>/author/title.bookmark', 'fmts':['epub']} ...}
        {53: {'path': '/<storage>/Townshend, Pete/Who I Am_ A Memoir - Pete Townshend.bookmark',
              'fmts': [u'epub', u'mobi']}}
        '''

        def get_ids_from_selected_rows():
            rows = self.opts.gui.library_view.selectionModel().selectedRows()
            if not rows or len(rows) < 2:
                rows = xrange(self.opts.gui.library_view.model().rowCount(QModelIndex()))
            ids = map(self.opts.gui.library_view.model().id, rows)
            return ids

        def get_formats(id):
            formats = db.formats(id, index_is_id=True)
            fmts = []
            if formats:
                for format in formats.split(','):
                    fmts.append(format.lower())
            return fmts

        def get_device_path_from_id(id_):
            paths = []
            for x in ('memory', 'card_a', 'card_b'):
                x = getattr(self.opts.gui, x + '_view').model()
                paths += x.paths_for_db_ids(set([id_]), as_map=True)[id_]
            return paths[0].path if paths else None

        def generate_annotation_paths(ids, db):
            # Generate path templates
            # Individual storage mount points scanned/resolved in driver.get_annotations()
            path_map = {}
            for id in ids:
                path = get_device_path_from_id(id)
                mi = db.get_metadata(id, index_is_id=True)
                a_path = self.device.create_annotations_path(mi, device_path=path)
                path_map[id] = dict(path=a_path, fmts=get_formats(id))
            return path_map

        # Entry point
        db = self.opts.gui.library_view.model().db

        # Get the list of ids in the library
        ids = get_ids_from_selected_rows()
        if not ids:
            return error_dialog(self.opts.gui, _('No books selected'),
                    _('No books selected to fetch annotations from'),
                    show=True)

        # Map ids to paths
        path_map = generate_annotation_paths(ids, db)
        return path_map

    def get_storage(self):
        storage = []
        if self.device._main_prefix:
            storage.append(os.path.join(self.device._main_prefix, self.device.EBOOK_DIR_MAIN))
        if self.device._card_a_prefix:
            storage.append(os.path.join(self.device._card_a_prefix, self.device.EBOOK_DIR_CARD_A))
        if self.device._card_b_prefix:
            storage.append(os.path.join(self.device._card_b_prefix, self.device.EBOOK_DIR_CARD_B))
        return storage

    @staticmethod
    def get_usb_reader_classes():
        """
        Utility method to find all subclasses of ourselves supporting fetching
        {app_name:cls}
        """
        if USBReader.usb_reader_classes is None:
            known_usb_reader_classes = {}
            for c in USBReader._iter_subclasses(USBReader):
                if c.SUPPORTS_FETCHING:
                    known_usb_reader_classes[c.app_name] = c
            USBReader.usb_reader_classes = known_usb_reader_classes
        return USBReader.usb_reader_classes

    @staticmethod
    def _iter_subclasses(cls, _seen=None):
        if not isinstance(cls, type):
            raise TypeError('itersubclasses must be called with '
                            'new-style classes, not %.100r' % cls)
        if _seen is None:
            _seen = set()
        try:
            subs = cls.__subclasses__()
        except TypeError:  # fails only when cls is type
            subs = cls.__subclasses__(cls)
        for sub in subs:
            if sub not in _seen:
                _seen.add(sub)
                yield sub
                for sub in USBReader._iter_subclasses(sub, _seen):
                    yield sub
