// Copyright (C) 2010, 2011, 2012 GlavSoft LLC.
// All rights reserved.
//
//-------------------------------------------------------------------------
// This file is part of the TightVNC software.  Please visit our Web site:
//
//                       http://www.tightvnc.com/
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//-------------------------------------------------------------------------
//

package com.glavsoft.drawing;

import com.glavsoft.exceptions.TransportException;
import com.glavsoft.rfb.encoding.PixelFormat;
import com.glavsoft.transport.Reader;

public class ColorDecoder {
	protected byte redShift;
	protected byte greenShift;
	protected byte blueShift;
	public short redMax;
	public short greenMax;
	public short blueMax;
	private final int bytesPerPixel;
	private final int bytesPerPixelSignificant;
	private final byte[] buff;

	private int startShift;
	private int startShiftCompact;
	private int addShiftItem;
	private final boolean isTightSpecific;

	public ColorDecoder(PixelFormat pf) {
		redShift = pf.redShift;
		greenShift = pf.greenShift;
		blueShift = pf.blueShift;
		redMax = pf.redMax;
		greenMax = pf.greenMax;
		blueMax = pf.blueMax;
		bytesPerPixel = pf.bitsPerPixel / 8;
		bytesPerPixelSignificant = 24 == pf.depth && 32 == pf.bitsPerPixel ? 3 : bytesPerPixel;
		buff = new byte[bytesPerPixel];
		if (0 == pf.bigEndianFlag) {
			startShift = 0;
			startShiftCompact = 0;
			addShiftItem = 8;
		} else {
			startShift = pf.bitsPerPixel - 8;
			startShiftCompact = Math.max(0, pf.depth - 8);
			addShiftItem = -8;
		}
		isTightSpecific = 4==bytesPerPixel && 3==bytesPerPixelSignificant &&
				255 == redMax && 255 == greenMax && 255 == blueMax;
	}

	protected int readColor(Reader reader) throws TransportException {
		return getColor(reader.readBytes(buff, 0, bytesPerPixel), 0);
	}

	protected int readCompactColor(Reader reader) throws TransportException {
		return getCompactColor(reader.readBytes(buff, 0, bytesPerPixelSignificant), 0);
	}

	protected int readTightColor(Reader reader) throws TransportException {
		return getTightColor(reader.readBytes(buff, 0, bytesPerPixelSignificant), 0);
	}

	protected int convertColor(int rawColor) {
		return  255 * (rawColor >> redShift & redMax) / redMax << 16 |
				255 * (rawColor >> greenShift & greenMax) / greenMax << 8 |
				255 * (rawColor >> blueShift & blueMax) / blueMax;
	}

	public void fillRawComponents(byte[] comp, byte[] bytes, int offset) {
		int rawColor = getRawTightColor(bytes, offset);
		comp[0] = (byte) (rawColor >> redShift & redMax);
		comp[1] = (byte) (rawColor >> greenShift & greenMax);
		comp[2] = (byte) (rawColor >> blueShift & blueMax);
	}

	protected int getTightColor(byte[] bytes, int offset) {
		return convertColor(getRawTightColor(bytes, offset));
	}

	private int getRawTightColor(byte[] bytes, int offset) {
		if (isTightSpecific)
			return (bytes[offset++] & 0xff)<<16 |
					(bytes[offset++] & 0xff)<<8 |
					bytes[offset++] & 0xff;
		else
			return getRawColor(bytes, offset);
	}

	protected int getColor(byte[] bytes, int offset) {
		return convertColor(getRawColor(bytes, offset));
	}

	private int getRawColor(byte[] bytes, int offset) {
		int shift = startShift;
		int item = addShiftItem;
		int rawColor = (bytes[offset++] & 0xff)<<shift;
		for (int i=1; i<bytesPerPixel; ++i) {
			rawColor |= (bytes[offset++] & 0xff)<<(shift+=item);
		}
		return rawColor;
	}

	protected int getCompactColor(byte[] bytes, int offset) {
		int shift = startShiftCompact;
		int item = addShiftItem;
		int rawColor = (bytes[offset++] & 0xff)<<shift;
		for (int i=1; i<bytesPerPixelSignificant; ++i) {
			rawColor |= (bytes[offset++] & 0xff)<<(shift+=item);
		}
		return convertColor(rawColor);
	}

}