#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, David Forrester <davidfor@internode.on.net>'
__docformat__ = 'restructuredtext en'

import os, threading, time, shutil
from datetime import datetime
from contextlib import closing
from collections import OrderedDict

from PyQt4.Qt import QUrl
from PyQt4.Qt import (Qt, QApplication, QMenu, QToolButton, QStandardItemModel, QStandardItem, QUrl, QModelIndex, QFileDialog)

from calibre import strftime
from calibre.gui2 import error_dialog, info_dialog, open_url, question_dialog, FileDialog
from calibre.gui2.actions import InterfaceAction
from calibre.ptempfile import PersistentTemporaryDirectory, remove_dir
from calibre.gui2.dialogs.message_box import ViewLog
from calibre.gui2.library.views import DeviceBooksView
from calibre.utils.icu import sort_key
from calibre.utils.config import config_dir

from calibre.devices.kobo.driver import KOBOTOUCH
from calibre.devices.kobo.books import Book
from calibre.devices.usbms.driver import USBMS

from calibre_plugins.koboutilities.dialogs import (ReaderOptionsDialog, CoverUploadOptionsDialog, RemoveCoverOptionsDialog, AboutDialog, 
                                        UpdateMetadataOptionsDialog, ChangeReadingStatusOptionsDialog, ShowBooksNotInDeviceDatabaseDialog, 
                                        ManageSeriesDeviceDialog, BookmarkOptionsDialog, BackupAnnotationsOptionsDialog, DismissTilesOptionsDialog)
from calibre_plugins.koboutilities.common_utils import (set_plugin_icon_resources, get_icon,
                                         create_menu_action_unique, get_library_uuid, debug_print)
from calibre_plugins.koboutilities.book import SeriesBook
import calibre_plugins.koboutilities.config as cfg

PLUGIN_ICONS = ['images/icon.png', 'images/logo_kobo.png', 'images/manage_series.png', 'images/lock.png', 'images/lock32.png',
                'images/lock_delete.png', 'images/lock_open.png', 'images/sort.png',
                'images/ms_ff.png']

MIMETYPE_KOBO = 'application/x-kobo-epub+zip'

BOOKMARK_SEPARATOR = '|@ @|'       # Spaces are included to allow wrapping in the details panel

class KoboUtilitiesAction(InterfaceAction):

    name = 'KoboUtilities'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = (_('KoboUtilities'), None, _('Utilites to use with Kobo ereaders'), ())
    action_type = 'current'
    
    CONTENTTYPE = 6

    def genesis(self):
        base = self.interface_action_base_plugin
        self.version = base.name+" v%d.%d.%d"%base.version

        self.menu = QMenu(self.gui)
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)
        self.old_actions_unique_map = {}
        self.device_actions_map     = {}
        self.library_actions_map    = {}
        
        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.toolbar_button_clicked)
        self.menu.aboutToShow.connect(self.about_to_show_menu)
        self.menus_lock = threading.RLock()

    def initialization_complete(self):
        # otherwise configured hot keys won't work until the menu's
        # been displayed once.
        self.rebuild_menus()

    def about_to_show_menu(self):
        self.rebuild_menus()

    def rebuild_menus(self):
        with self.menus_lock:
            # Show the config dialog
            # The config dialog can also be shown from within
            # Preferences->Plugins, which is why the do_user_config
            # method is defined on the base plugin class
            do_user_config = self.interface_action_base_plugin.do_user_config
            self.menu.clear()
            self.actions_unique_map = {}

            self.device     = self.get_device()
            haveKoboTouch   = self.device is not None
            self.supports_series = haveKoboTouch and "supports_series" in dir(self.device) and self.device.supports_series()

            self.set_reader_fonts_action = self.create_menu_item_ex(self.menu, '&Set Reader Font for Selected Books', 
                                                          unique_name='Set Reader Font for Selected Books',
                                                          shortcut_name='Set Reader Font for Selected Books',
                                                          triggered=self.set_reader_fonts,
                                                          enabled=haveKoboTouch, 
                                                          is_library_action=True, 
                                                          is_device_action=True)
            self.library_actions_map['Set Reader Font for Selected Books'] = self.set_reader_fonts_action
            self.device_actions_map['Set Reader Font for Selected Books']  = self.set_reader_fonts_action

            self.remove_reader_fonts_action = self.create_menu_item_ex(self.menu, '&Remove Reader Font for Selected Books', 
                                                          unique_name='Remove Reader Font for Selected Books',
                                                          shortcut_name='Remove Reader Font for Selected Books',
                                                          triggered=self.remove_reader_fonts,
                                                          enabled=haveKoboTouch, 
                                                          is_library_action=True, 
                                                          is_device_action=True)
            self.library_actions_map['Remove Reader Font for Selected Books'] = self.remove_reader_fonts_action
            self.device_actions_map['Remove Reader Font for Selected Books']  = self.remove_reader_fonts_action

            self.dismiss_tiles_action = self.create_menu_item_ex(self.menu, '&Dismiss tiles from new home screen', 
                                                          unique_name='Dismiss tiles from home screen',
                                                          shortcut_name='Dismiss tiles new home screen',
                                                          triggered=self.dismiss_tiles,
                                                          enabled=haveKoboTouch, 
                                                          is_library_action=True, 
                                                          is_device_action=True)
            self.library_actions_map['Dismiss tiles new home screen'] = self.dismiss_tiles_action
            self.device_actions_map['Dismiss tiles new home screen']  = self.dismiss_tiles_action

            self.menu.addSeparator()
            self.update_metadata_action = self.create_menu_item_ex(self.menu, 'Update &metadata in device library', 
                                                          unique_name='Update metadata in device library',
                                                          shortcut_name='Update metadata in device library',
                                                          triggered=self.update_metadata,
                                                          enabled=not self.isDeviceView() and haveKoboTouch, 
                                                          is_library_action=True)

            self.change_reading_status_action = self.create_menu_item_ex(self.menu, '&Change Reading Status in device library', 
                                                          unique_name='Change Reading Status in device library',
                                                          shortcut_name='Change Reading Status in device library',
                                                          triggered=self.change_reading_status,
                                                          enabled=self.isDeviceView() and haveKoboTouch, 
                                                          is_device_action=True)

#            self.mark_not_interested_action = self.create_menu_item_ex(self.menu, '&Mark as "Not Interested"', 
#                                                          unique_name='Mark as "Not Interested"',
#                                                          shortcut_name='Mark as "Not Interested"',
#                                                          triggered=self.mark_not_interested,
#                                                          enabled=self.isDeviceView() and haveKoboTouch, 
#                                                          is_device_action=True)

            self.manage_series_on_device_action = self.create_menu_item_ex(self.menu, '&Manage Series Information in device library', 
                                                          unique_name='Manage Series Information in device library',
                                                          shortcut_name='Manage Series Information in device library',
                                                          triggered=self.manage_series_on_device,
                                                          enabled=self.isDeviceView() and haveKoboTouch and self.supports_series, 
                                                          is_device_action=True)

            self.handle_bookmarks_action = self.create_menu_item_ex(self.menu, '&Store/Restore current bookmark', 
                                                          unique_name='Store/Restore current bookmark',
                                                          shortcut_name='Store/Restore current bookmark',
                                                          triggered=self.handle_bookmarks,
                                                          enabled=not self.isDeviceView() and haveKoboTouch, 
                                                          is_library_action=True)

#            self.store_current_bookmark_action = self.create_menu_item_ex(self.menu, '&Store current bookmark', 
#                                                          unique_name='Store current bookmark',
#                                                          shortcut_name='Store current bookmark',
#                                                          triggered=self.store_current_bookmark)
#
#            self.restore_current_bookmark_action = self.create_menu_item_ex(self.menu, '&Restore current bookmark', 
#                                                          unique_name='Restore current bookmark',
#                                                          shortcut_name='Restore current bookmark',
#                                                          triggered=self.restore_current_bookmark)

            self.menu.addSeparator()
            self.upload_covers_action = self.create_menu_item_ex(self.menu, '&Upload covers for Selected Books', 
                                                          unique_name='Upload/covers for Selected Books',
                                                          shortcut_name='Upload covers for Selected Books',
                                                          triggered=self.upload_covers,
                                                          enabled=not self.isDeviceView() and haveKoboTouch, 
                                                          is_library_action=True)
            self.remove_covers_action = self.create_menu_item_ex(self.menu, '&Remove covers for Selected Books', 
                                                          unique_name='Remove covers for Selected Books',
                                                          shortcut_name='Remove covers for Selected Books',
                                                          triggered=self.remove_covers,
                                                          enabled=haveKoboTouch, 
                                                          is_library_action=True, 
                                                          is_device_action=True)

            self.menu.addSeparator()
            self.getAnnotationForSelected_action = self.create_menu_item_ex(self.menu, 'Copy annotation for Selected Book', image='bookmarks.png',
                                                            unique_name='Copy annotation for Selected Book',
                                                            shortcut_name='Copy annotation for Selected Book',
                                                            triggered=self.getAnnotationForSelected,
                                                            enabled=not self.isDeviceView() and haveKoboTouch, 
                                                            is_library_action=True)
            self.backup_annotation_files_action = self.create_menu_item_ex(self.menu, 'Backup Annotation File',
                                                            unique_name='Backup Annotation File',
                                                            shortcut_name='Backup Annotation File',
                                                            triggered=self.backup_annotation_files,
                                                            enabled=not self.isDeviceView() and haveKoboTouch, 
                                                            is_library_action=True)

            self.menu.addSeparator()
            self.show_books_not_in_database_action = self.create_menu_item_ex(self.menu, 'Show books not in the device database',
                                                            unique_name='Show books not in the device database',
                                                            shortcut_name='Show books not in the device database',
                                                            triggered=self.show_books_not_in_database,
                                                            enabled=self.isDeviceView() and haveKoboTouch, 
                                                            is_device_action=True)

            self.refresh_device_books_action = self.create_menu_item_ex(self.menu, 'Refresh the list of books on the device',
                                                            unique_name='Refresh the list of books on the device',
                                                            shortcut_name='Refresh the list of books on the device',
                                                            triggered=self.refresh_device_books,
                                                            enabled=haveKoboTouch, 
                                                            is_library_action=True, 
                                                            is_device_action=True)
            self.databaseMenu = self.menu.addMenu('Database')
            self.check_device_database_action = self.create_menu_item_ex(self.databaseMenu, 'Check the device database',
                                                            unique_name='Check the device database',
                                                            shortcut_name='Check the device database',
                                                            triggered=self.check_device_database,
                                                            enabled=haveKoboTouch, 
                                                            is_library_action=True, 
                                                            is_device_action=True)
            self.backup_device_database_action = self.create_menu_item_ex(self.databaseMenu, 'Backup device database',
                                                            unique_name='Backup device database',
                                                            shortcut_name='Backup device database',
                                                            triggered=self.backup_device_database,
                                                            enabled=haveKoboTouch, 
                                                            is_library_action=True, 
                                                            is_device_action=True)

#            self.menu.addSeparator()
#            self.get_list_action = self.create_menu_item_ex(self.menu, 'Update TOC for Selected Book',
#                                                            unique_name='Update TOC for Selected Book',
#                                                            shortcut_name='Update TOC for Selected Book',
#                                                            triggered=self.updateTOCForSelected)


            self.menu.addSeparator()
#            self.config_action = create_menu_action_unique(self, self.menu, _('&Customize plugin')+'...', shortcut=False,
#                                                           image= 'config.png',
#                                                           triggered=self.show_configuration)
            self.config_action = self.create_menu_item_ex(self.menu, _('&Customize plugin')+'...', shortcut=False,
                                                            unique_name='Customize plugin',
                                                            shortcut_name='Customize plugin',
                                                            image= 'config.png',
                                                            triggered=self.show_configuration,
                                                            enabled=True,  
                                                            is_library_action=True, 
                                                            is_device_action=True)
            
            self.about_action = create_menu_action_unique(self, self.menu, '&About Plugin', shortcut=False,
                                                           image= 'images/icon.png',
                                                           unique_name='About KoboUtilities',
                                                           shortcut_name='About KoboUtilities',
                                                           triggered=self.about)

            create_menu_action_unique(self, self.menu, _('&Help'), 'help.png',
                                  shortcut=False, triggered=self.show_help1)

#            self.store_current_bookmark_action.setEnabled( not self.isDeviceView() and haveKoboTouch)
#            self.restore_current_bookmark_action.setEnabled( not self.isDeviceView() and haveKoboTouch)
            # self.get_list_action.setEnabled( len(self.gui.library_view.get_selected_ids()) > 0 )

            # Before we finalize, make sure we delete any actions for menus that are no longer displayed
            for menu_id, unique_name in self.old_actions_unique_map.iteritems():
                if menu_id not in self.actions_unique_map:
                    self.gui.keyboard.unregister_shortcut(unique_name)
            self.old_actions_unique_map = self.actions_unique_map
            self.gui.keyboard.finalize()            

    def about(self):
        # Get the about text from a file inside the plugin zip file
        # The get_resources function is a builtin function defined for all your
        # plugin code. It loads files from the plugin zip file. It returns
        # the bytes from the specified file.
        #
        # Note that if you are loading more than one file, for performance, you
        # should pass a list of names to get_resources. In this case,
        # get_resources will return a dictionary mapping names to bytes. Names that
        # are not found in the zip file will not be in the returned dictionary.
        
        text = get_resources('about.txt')
        AboutDialog(self.gui, self.qaction.icon(), self.version + text).exec_()
        
    def create_menu_item_ex(self, parent_menu, menu_text, image=None, tooltip=None,
                           shortcut=None, triggered=None, is_checked=None, shortcut_name=None,
                           unique_name=None, enabled=False, is_library_action=False, is_device_action=False):
        ac = create_menu_action_unique(self, parent_menu, menu_text, image, tooltip,
                                       shortcut, triggered, is_checked, shortcut_name, unique_name)
        self.actions_unique_map[ac.calibre_shortcut_unique_name] = ac.calibre_shortcut_unique_name
        ac.setEnabled(enabled)

        if is_library_action:
            self.library_actions_map[shortcut_name] = ac
        if is_device_action:
            self.device_actions_map[shortcut_name] = ac
        return ac

    def toolbar_button_clicked(self):

        if len(self.gui.current_view().selectionModel().selectedRows()) >= 0:
            self.device     = self.get_device()
            haveKoboTouch   = self.device is not None
            self.supports_series = haveKoboTouch and "supports_series" in dir(self.device) and self.device.supports_series()

            if self.isDeviceView():
                if self.supports_series:
                    button_action = cfg.get_plugin_pref(cfg.COMMON_OPTIONS_STORE_NAME, cfg.KEY_BUTTON_ACTION_DEVICE)
                    if button_action == '':
                        self.show_configuration()
                    else:
                        self.device_actions_map[button_action].trigger()
#                    self.manage_series_on_device()
#                    self.show_books_not_in_database()
#                    self.mark_not_interested()
                else:
                    self.change_reading_status()
            else:
                button_action = cfg.get_plugin_pref(cfg.COMMON_OPTIONS_STORE_NAME, cfg.KEY_BUTTON_ACTION_LIBRARY)
                if button_action == '':
                    self.show_configuration()
                else:
                    self.library_actions_map[button_action].trigger()
#                self.library_actions_map.values()[0].trigger()
#                self.handle_bookmarks()
#                self.upload_covers()
#                self.update_metadata()
#                self.set_reader_fonts_action.trigger()
#                self.backup_annotation_files()
#                self.show_configuration()

    def isDeviceView(self):
        view = self.gui.current_view()
        return isinstance(view, DeviceBooksView)

    def _get_contentIDs_for_selected(self):
        view = self.gui.current_view()
        if self.isDeviceView():
            rows = view.selectionModel().selectedRows()
            books = [view.model().db[view.model().map[r.row()]] for r in rows]
            contentIDs = [book.contentID for book in books]
#            debug_print("_get_contentIDs_for_selected - book.ImageID=", book.ImageID)
        else:
            book_ids = view.get_selected_ids()
            contentIDs = self.get_contentIDs_for_books(book_ids)
            
        return contentIDs

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def set_reader_fonts(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot set reader font settings.'),
                                _('No device connected.'),
                                show=True)
        self.device_path = self.get_device_path()
        self.singleSelected = len(self.gui.current_view().selectionModel().selectedRows()) == 1

        contentIDs = self._get_contentIDs_for_selected()

        debug_print('set_reader_fonts - contentIDs', contentIDs)

        #print("update books:%s"%books)

        if len(contentIDs) == 0:
            return

        if len(contentIDs) == 1:
            self.single_contentID = contentIDs[0]

        dlg = ReaderOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.prefs

        updated_fonts, added_fonts, deleted_fonts, count_books = self._set_reader_fonts(contentIDs)
        result_message = "Change summary:\n\tFont settings updated=%d\n\tFont settings added=%d\n\tTotal books=%d" % (updated_fonts, added_fonts, count_books)
        info_dialog(self.gui, _('Kobo Utilities - Device library updated'),
                    result_message,
                    show=True)


    def remove_reader_fonts(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot remove reader font settings'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        contentIDs = self._get_contentIDs_for_selected()
        
        if len(contentIDs) == 0:
            return

        mb = question_dialog(self.gui, 'Remove Reader settings', 'Do you want to remove the reader settings for the selected books?', show_copy_button=False)
        if not mb:
            return

        
        updated_fonts, added_fonts, deleted_fonts, count_books = self._set_reader_fonts(contentIDs, delete=True)
        result_message = "Change summary:\n\tFont settings deleted=%d" % (deleted_fonts)
        info_dialog(self.gui, _('Kobo Utilities - Device library updated'),
                    result_message,
                    show=True)

    def update_metadata(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot update metadata in device library.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return
        debug_print("update_metadata - selectedIDs:", selectedIDs)
        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
        for book in books:
            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            #                    contentID = self.device.contentid_from_path(path, self.CONTENTTYPE)
            debug_print("update_metadata - device_book_paths:", device_book_paths)
            book.paths = device_book_paths
            book.contentIDs = [self.device.contentid_from_path(path, self.CONTENTTYPE) for path in device_book_paths]
            book.series_index_string = None
        
        dlg = UpdateMetadataOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.new_prefs

        updated_books, unchanged_books, not_on_device_books, count_books = self._update_metadata(books)
        result_message = "Update summary:\n\tBooks updated=%d\n\tUnchanged books=%d\n\tBooks not on device=%d\n\tTotal books=%d" % (updated_books, unchanged_books, not_on_device_books, count_books)
        info_dialog(self.gui, _('Kobo Utilities - Device library updated'),
                    result_message,
                    show=True)


    def dismiss_tiles(self):
        #debug_print("dismiss_tiles - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot mark tiles to be dismissed.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        
        dlg = DismissTilesOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options
        debug_print("dismiss_tiles - self.options=", self.options)
        result = self._dismiss_tiles()
        result_message = "Update successful" if result == 1 else "Update unsuccessful"
        info_dialog(self.gui, _('Kobo Utilities - Dismiss Tiles from Home Screen'),
                    result_message,
                    show=True)


    def handle_bookmarks(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot store or restore current reading position.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return

        dlg = BookmarkOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options

        if self.options['storeBookmarks']:
            self.store_current_bookmark()
        else:
            self.restore_current_bookmark()


    def store_current_bookmark(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot update metadata in device library.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return
        debug_print("store_current_bookmark - selectedIDs:", selectedIDs)
        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
        for book in books:
            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            #                    contentID = self.device.contentid_from_path(path, self.CONTENTTYPE)
            debug_print("store_current_bookmark - device_book_paths:", device_book_paths)
            book.paths = device_book_paths
            book.contentIDs = [self.device.contentid_from_path(path, self.CONTENTTYPE) for path in device_book_paths]
        
        books_with_bookmark, books_without_bookmark, count_books = self._store_current_bookmark(books)
        result_message = "Update summary:\n\tBookmarks retrieved=%d\n\tBooks with no bookmarks=%d\n\tTotal books=%d" % (books_with_bookmark, books_without_bookmark, count_books)
        info_dialog(self.gui, _('Kobo Utilities - Library updated'),
                    result_message,
                    show=True)

    def restore_current_bookmark(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot set bookmark in device library.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return
        debug_print("restore_current_bookmark - selectedIDs:", selectedIDs)
#        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
#        for book in books:
#            device_book_path = self.get_device_path_from_id(book.calibre_id)
#            if device_book_path is None:
#                book.contentID = None
#            else:
#                book.contentID = self.device.contentid_from_path(device_book_path, self.CONTENTTYPE)
        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
        for book in books:
            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            #                    contentID = self.device.contentid_from_path(path, self.CONTENTTYPE)
            debug_print("store_current_bookmark - device_book_paths:", device_book_paths)
            book.paths = device_book_paths
            book.contentIDs = [self.device.contentid_from_path(path, self.CONTENTTYPE) for path in device_book_paths]
        
        updated_books, not_on_device_books, count_books = self._restore_current_bookmark(books)
        result_message = "Update summary:\n\tBooks updated=%d\n\tBooks not on device=%d\n\tTotal books=%d" % (updated_books, not_on_device_books, count_books)
        info_dialog(self.gui, _('Kobo Utilities - Device library updated'),
                    result_message,
                    show=True)

    def backup_device_database(self):
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot backup the device database.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        debug_print("backup_device_database")

        fd = FileDialog(parent=self.gui, name='Kobo Utilities plugin:choose backup destination', 
                        title='Choose Backup Destination',
                        filters=[('SQLite database', ['sqlite'])], 
                        add_all_files_filter=False,
                        mode=QFileDialog.AnyFile
                        )
        if not fd.accepted:
                return
        backup_file = fd.get_files()[0]

        if not backup_file:
            return

        debug_print("backup_device_database - backup file selected=", backup_file)
        source_file = self.device.normalize_path(self.device_path +'.kobo/KoboReader.sqlite')
        shutil.copyfile(source_file, backup_file)

    def backup_annotation_files(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return

        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot backup annotation files from device.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return

        dlg = BackupAnnotationsOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return

        dest_path = dlg.dest_path()
        debug_print("backup_annotation_files - selectedIDs:", selectedIDs)
        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
        for book in books:
            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            #                    contentID = self.device.contentid_from_path(path, self.CONTENTTYPE)
            debug_print("backup_annotation_files - device_book_paths:", device_book_paths)
            book.paths = device_book_paths
            book.contentIDs = [self.device.contentid_from_path(path, self.CONTENTTYPE) for path in device_book_paths]

        debug_print("backup_annotation_files - dest_path=", dest_path)
        annotations_found, no_annotations, kepubs, count_books = self._backup_annotation_files(books, dest_path)
        result_message = "Annotations backup summary:\n\tBooks with annotations=%d\n\tBooks without annotations=%d\n\tKobo epubs=%d\n\tTotal books=%d" % (annotations_found, no_annotations, kepubs, count_books)
        info_dialog(self.gui, _('Kobo Utilities - Annotations backup'),
                    result_message,
                    show=True)

    def refresh_device_books(self):
        self.gui.device_detected(True, KOBOTOUCH)

    def change_reading_status(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot change reading status in device library.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        books = self._get_books_for_selected()
        
        if len(books) == 0:
            return
        for book in books:
#            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            debug_print("change_reading_status - book:", book)
            book.contentIDs = [book.contentID]
        debug_print("change_reading_status - books:", books)

        dlg = ChangeReadingStatusOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options

        updated_books, unchanged_books, not_on_device_books, count_books = self._update_metadata(books)
        result_message = "Update summary:\n\tBooks updated=%d\n\tUnchanged books=%d\n\tBooks not on device=%d\n\tTotal books=%d" % (updated_books, unchanged_books, not_on_device_books, count_books)
        info_dialog(self.gui, _('Kobo Utilities - Device library updated'),
                    result_message,
                    show=True)


    def mark_not_interested(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot change reading status in device library.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        books = self._get_books_for_selected()
        
        if len(books) == 0:
            return
        recommendations = []
        for book in books:
#            device_book_paths = self.get_device_paths_from_id(book.calibre_id)
            if 'Recommendation' in book.device_collections:
                debug_print("mark_not_interested - book:", book)
                book.contentIDs = [book.contentID]
                recommendations.append(book)
                debug_print("mark_not_interested - book.device_collections:", book.device_collections)
        debug_print("mark_not_interested - recommendations:", recommendations)
        self.options = self.default_options()
        self.options['mark_not_interested'] = True

        updated_books, unchanged_books, not_on_device_books, count_books = self._update_metadata(recommendations)
        result_message = "Books marked as Not Interested:\n\tBooks updated=%d\n\tUnchanged books=%d\n\tTotal books=%d" % (updated_books, unchanged_books, count_books)
        info_dialog(self.gui, _('Kobo Utilities - Device library updated'),
                    result_message,
                    show=True)


    def show_books_not_in_database(self):

        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot list books not in device library.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        books = self._get_books_for_selected()
        
        if len(books) == 0:
            books = self.gui.current_view().model().db

        books_not_in_database = self._check_book_in_database(books)
#        for book in books:
#            debug_print("show_books_not_in_database - book.title='%s'" % book.title)
#            if not book.contentID:
#                books_not_in_database.append(book)
#            else:
#                debug_print("show_books_not_in_database - book.contentID='%s'" % book.contentID)

        dlg = ShowBooksNotInDeviceDatabaseDialog(self.gui, books_not_in_database)
        dlg.show()


    def check_device_database(self):

        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot list books not in device library.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()

        check_result = self._check_device_database()

        check_result = "Result of running 'PRAGMA integrity_check' on database on the Kobo device:\n\n" + check_result

        d = ViewLog(_("Kobo Utilities - Device Database Check"), check_result, parent=self.gui)
        d.setWindowIcon(self.qaction.icon())
        d.exec_()


    def default_options(self):
        options                        = {}
        options[cfg.KEY_SET_TITLE]          = False
        options[cfg.KEY_SET_AUTHOR]         = False
        options[cfg.KEY_SET_DESCRIPTION]    = False
        options[cfg.KEY_SET_PUBLISHER]      = False
        options[cfg.KEY_SET_PUBLISHED_DATE] = False
        options[cfg.KEY_SET_ISBN]           = False
        options[cfg.KEY_SET_LANGUAGE]       = False
        options[cfg.KEY_SET_SERIES]         = False
        options[cfg.KEY_SET_RATING]         = False
        options[cfg.KEY_SET_NOT_INTERESTED] = False
        options[cfg.KEY_SET_READING_STATUS] = False
        options[cfg.KEY_READING_STATUS]     = -1
        options[cfg.KEY_RESET_POSITION]     = False
        return options

    def manage_series_on_device(self):
        def digits(f):
            return len(str(f).split('.')[1].rstrip('0'))

        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return

        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot manage series in device library.'),
                    _('No device connected.'),
                    show=True)
        series_columns = self.get_series_columns()
        self.device_path = self.get_device_path()

        books = self._get_books_for_selected()
        debug_print("manage_series_on_device - books[0].__class__=", books[0].__class__)

        
        if len(books) == 0:
            return
        seriesBooks = [SeriesBook(book, series_columns) for book in books]
        seriesBooks = sorted(seriesBooks, key=lambda k: k.sort_key(sort_by_name=True))
        debug_print("manage_series_on_device - seriesBooks[0]._mi.__class__=", seriesBooks[0]._mi.__class__)
        debug_print("manage_series_on_device - seriesBooks[0]._mi.kobo_series=", seriesBooks[0]._mi.kobo_series)
        debug_print("manage_series_on_device - seriesBooks[0]._mi.kobo_series_number=", seriesBooks[0]._mi.kobo_series_number)
        debug_print("manage_series_on_device - books:", seriesBooks)

        library_db = self.gui.library_view.model().db
        all_series = library_db.all_series()
        all_series.sort(key=lambda x : sort_key(x[1]))

        d = ManageSeriesDeviceDialog(self.gui, self, seriesBooks, all_series, series_columns)
        d.exec_()
        if d.result() != d.Accepted:
            return
        
        debug_print("manage_series_on_device - done series management - books:", seriesBooks)

        self.options = self.default_options()
        books = []
        for seriesBook in seriesBooks:
            debug_print("manage_series_on_device - seriesBook._mi.contentID=", seriesBook._mi.contentID)
            if seriesBook.is_title_changed() or seriesBook.is_pubdate_changed() or seriesBook.is_series_changed():
                book = seriesBook._mi
                book.series_index_string = seriesBook.series_index_string()
                book.kobo_series_number  = seriesBook.series_index_string()
                book.kobo_series         = seriesBook.series_name()
                book._new_book           = True
                book.contentIDs          = [book.contentID]
                books.append(book)
                self.options['title']          = self.options['title'] or seriesBook.is_title_changed()
                self.options['series']         = self.options['series'] or seriesBook.is_series_changed()
                self.options['published_date'] = self.options['published_date'] or seriesBook.is_pubdate_changed()
                debug_print("manage_series_on_device - seriesBook._mi.__class__=", seriesBook._mi.__class__)
                debug_print("manage_series_on_device - seriesBook.is_pubdate_changed()=%s"%seriesBook.is_pubdate_changed())
                debug_print("manage_series_on_device - book.kobo_series=", book.kobo_series)
                debug_print("manage_series_on_device - book.kobo_series_number=", book.kobo_series_number)
                debug_print("manage_series_on_device - book.series=", book.series)
                debug_print("manage_series_on_device - book.series_index=%s"%unicode(book.series_index))


        if self.options['title'] or self.options['series'] or self.options['published_date']:
            updated_books, unchanged_books, not_on_device_books, count_books = self._update_metadata(books)
            
            debug_print("manage_series_on_device - about to call sync_booklists")
    #        self.device.sync_booklists((self.gui.current_view().model().db, None, None))
            USBMS.sync_booklists(self.device, (self.gui.current_view().model().db, None, None))
            result_message = "Update summary:\n\tBooks updated=%d\n\tUnchanged books=%d\n\tBooks not on device=%d\n\tTotal books=%d" % (updated_books, unchanged_books, not_on_device_books, count_books)
        else:
            result_message = "No changes made to series information."
        info_dialog(self.gui, _('Kobo Utilities - Manage Series On Device'),
                    result_message,
                    show=True)


    def get_series_columns(self):
        custom_columns = self.gui.library_view.model().custom_columns
        series_columns = OrderedDict()
        for key, column in custom_columns.iteritems():
            typ = column['datatype']
            if typ == 'series':
                series_columns[key] = column
        return series_columns

    def get_selected_books(self, rows, series_columns):
        def digits(f):
            return len(str(f).split('.')[1].rstrip('0'))

        db = self.gui.library_view.model().db
        idxs = [row.row() for row in rows]
        books = []
        for idx in idxs:
            mi = db.get_metadata(idx)
            book = SeriesBook(mi, series_columns)
            books.append(book)
        # Sort books by the current series
        books = sorted(books, key=lambda k: k.sort_key())
        return books


    def upload_covers(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,
                                _('Cannot upload covers.'),
                                _('No device connected.'),
                                show=True)
        self.device_path = self.get_device_path()

        selectedIDs = self._get_selected_ids()
        
        if len(selectedIDs) == 0:
            return
        debug_print("upload_covers - selectedIDs:", selectedIDs)
        books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)
        
        dlg = CoverUploadOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options
        
        total_books, uploaded_covers, not_on_device_books = self._upload_covers(books)
        result_message = "Change summary:\n\tCovers uploaded=%d\n\tBooks not on device=%d\n\tTotal books=%d" % (uploaded_covers, not_on_device_books, total_books)
        info_dialog(self.gui, _('Kobo Utilities - Covers uploaded'),
                    result_message,
                    show=True)

    def remove_covers(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("remove_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui, _('Cannot remove covers.'),
                    _('No device connected.'),
                    show=True)
        self.device_path = self.get_device_path()
        debug_print("remove_covers - self.device_path", self.device_path)

#        contentIDs = self._get_contentIDs_for_selected()
        if self.gui.stack.currentIndex() == 0:
            selectedIDs = self._get_selected_ids()
            books = self._convert_calibre_ids_to_books(self.gui.current_view().model().db, selectedIDs)

        else:
            books = self._get_books_for_selected()

        
        if len(books) == 0:
            return

        dlg = RemoveCoverOptionsDialog(self.gui, self)
        dlg.exec_()
        if dlg.result() != dlg.Accepted:
            return
        self.options = dlg.options
        
        removed_covers, not_on_device_books, total_books = self._remove_covers(books)
        result_message = "Change summary:\n\tCovers removed=%d\n\tBooks not on device=%d\n\tTotal books=%d" % (removed_covers, not_on_device_books, total_books)
        info_dialog(self.gui, _('Kobo Utilities - Covers removed'),
                    result_message,
                    show=True)


    def getAnnotationForSelected(self):
        if len(self.gui.current_view().selectionModel().selectedRows()) == 0:
            return
        #debug_print("upload_covers - start")
        self.device = self.get_device()
        if self.device is None:
            return error_dialog(self.gui,
                                _('Cannot upload covers.'),
                                _('No device connected.'),
                                show=True)

        self._getAnnotationForSelected()


    def _get_selected_ids(self):
        rows = self.gui.current_view().selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return []
        return map(self.gui.current_view().model().id, rows)

    def get_contentIDs_for_books(self, ids):
        contentIDs= []
        for book_id in ids:
            device_book_path = self.get_device_path_from_id(book_id)
            debug_print('get_contentIDs_for_books - device_book_path', device_book_path)
            if device_book_path is None:
                continue
            contentID = self.device.contentid_from_path(device_book_path, self.CONTENTTYPE)
            debug_print('get_contentIDs_for_books - contentID', contentID)
            contentIDs.append(contentID)
        return contentIDs

    def _get_books_for_selected(self):
        view = self.gui.current_view()
        if self.isDeviceView():
            rows  = view.selectionModel().selectedRows()
            books = []
            for r in rows:
#                debug_print('_get_books_for_selected - r.row()', r.row())
                book = view.model().db[view.model().map[r.row()]]
                book.calibre_id = r.row()
                books.append(book)
            #books = [view.model().db[view.model().map[r.row()]] for r in rows]
        else:
            books = []
            
        return books

    def _convert_calibre_ids_to_books(self, db, ids):
        books = []
        for book_id in ids:
            book = self._convert_calibre_id_to_book(db, book_id)
#            debug_print('_convert_calibre_ids_to_books - book', book)
            books.append(book)
        return books

    def _convert_calibre_id_to_book(self, db, book_id):
        mi = db.get_metadata(book_id, index_is_id=True, get_cover=True)
#        debug_print('_convert_calibre_id_to_book - mi', mi)
#        debug_print('_convert_calibre_id_to_book - mi.application_id', mi.application_id)
#        debug_print('_convert_calibre_id_to_book - mi.in_library', mi.in_library)
        book = Book('', 'lpath', title=mi.title, other=mi)
#        book = {}
        book.calibre_id  = mi.id
#        book['title']       = mi.title
#        book['metadata']    = mi

        return book


    def get_device_path(self):
        debug_print('BEGIN Get Device Path')
    
        device_path = ''
        try:
            # If we're in test mode TEST_DEVICE is defined, use the predefined test directory
            #TEST_DEVICE = 'fakeKindleDir2'
            device_path = TEST_DEVICE
            debug_print('RUNNING IN TEST MODE')
        except:
            # Not in test mode, so confirm a device is connected
            device_connected = self.gui.library_view.model().device_connected
            try:
                device_connected = self.gui.library_view.model().device_connected
            except:
                debug_print('No device connected')
                device_connected = None
    
            # If there is a device connected, test if we can retrieve the mount point from Calibre
            if device_connected is not None:
                try:
                    # _main_prefix is not reset when device is ejected so must be sure device_connected above
                    device_path = self.gui.device_manager.connected_device._main_prefix
                    debug_print('Root path of device: %s' % device_path)
                except:
                    debug_print('A device appears to be connected, but device path not defined')
            else:
                debug_print('No device appears to be connected')
    
        debug_print('END Get Device Path')
        return device_path

    def get_device(self):
#        debug_print('BEGIN Get Device Path')

        connected_device = None
        try:
            connected_device = self.gui.device_manager.connected_device
        except:
            debug_print('No device connected')
            connected_device = None
    
        # If there is a device connected, test if we can retrieve the mount point from Calibre
        if connected_device is None or not isinstance(connected_device, KOBOTOUCH):
            debug_print('No Kobo Touch, Glo or Mini appears to be connected')
        else:
            debug_print('Have a Kobo device connected connected')
#        debug_print('END Get Device Path')
        return connected_device

    def get_device_path_from_id(self, book_id):
        paths = []
        for x in ('memory', 'card_a', 'card_b'):
            x = getattr(self.gui, x+'_view').model()
            paths += x.paths_for_db_ids(set([book_id]), as_map=True)[book_id]
        return paths[0].path if paths else None


    def get_device_paths_from_id(self, book_id):
        paths = []
        for x in ('memory', 'card_a', 'card_b'):
            x = getattr(self.gui, x+'_view').model()
            paths += x.paths_for_db_ids(set([book_id]), as_map=True)[book_id]
#        debug_print("get_device_paths_from_id - paths=", paths)
        return [r.path for r in paths]

    def get_contentIDs_from_id(self, book_id):
        debug_print("get_device_paths_from_id - book_id=", book_id)
        paths = []
        for x in ('memory', 'card_a', 'card_b'):
            debug_print("get_device_paths_from_id - x=", x)
            x = getattr(self.gui, x+'_view').model()
            debug_print("get_device_paths_from_id - x=", x)
            paths += x.paths_for_db_ids(set([book_id]), as_map=True)[book_id]
        debug_print("get_device_paths_from_id - paths=", paths)
        return [r.contentID for r in paths]

    def get_contentIDs_from_book(self, book):
        paths = []
        for x in ('memory', 'card_a', 'card_b'):
            x = getattr(self.gui, x+'_view').model()
            paths += x.paths_for_db_ids(set([book_id]), as_map=True)[book_id]
        debug_print("get_device_paths_from_id - paths=", paths)
        return [r.contentID for r in paths]


    def _queue_job(self, tdir, options, books_to_modify):
        if not books_to_modify:
            # All failed so cleanup our temp directory
            remove_dir(tdir)
            return

        func = 'arbitrary_n'
        cpus = self.gui.job_manager.server.pool_size
        args = ['calibre_plugins.koboutilities.jobs', 'do_koboutilities',
                (books_to_modify, options, cpus)]
        desc = 'Modify SOL Story'
        job = self.gui.job_manager.run_job(
                self.Dispatcher(self._modify_completed), func, args=args,
                    description=desc)
        job._tdir = tdir
        self.gui.status_bar.show_message('Modifying %d books'%len(books_to_modify))


    def _getAnnotationForSelected(self, *args):
        # Generate a path_map from selected ids
        def get_ids_from_selected_rows():
            rows = self.gui.library_view.selectionModel().selectedRows()
            if not rows or len(rows) < 1:
                rows = xrange(self.gui.library_view.model().rowCount(QModelIndex()))
            ids = map(self.gui.library_view.model().id, rows)
            return ids

        def get_formats(_id):
            formats = db.formats(_id, index_is_id=True)
            fmts = []
            if formats:
                for format in formats.split(','):
                    fmts.append(format.lower())
            return fmts

        def get_device_path_from_id(id_):
            paths = []
            for x in ('memory', 'card_a', 'card_b'):
                x = getattr(self.gui, x+'_view').model()
                paths += x.paths_for_db_ids(set([id_]), as_map=True)[id_]
            return paths[0].path if paths else None

        def generate_annotation_paths(ids, db, device):
            # Generate path templates
            # Individual storage mount points scanned/resolved in driver.get_annotations()
            path_map = {}
            for _id in ids:
                paths = self.get_device_paths_from_id(_id)
                debug_print("generate_annotation_paths - paths=", paths)
#                mi = db.get_metadata(_id, index_is_id=True)
#                a_path = device.create_annotations_path(mi, device_path=paths)
                if len(paths) > 0:
                    the_path = paths[0]
                    if len(paths) > 1:
                        if os.path.splitext(paths[0]) > 1: # No extension - is kepub
                            the_path = paths[1]
                    path_map[_id] = dict(path=the_path, fmts=get_formats(_id))
            return path_map

        annotationText = []

        if self.gui.current_view() is not self.gui.library_view:
            return error_dialog(self.gui, _('Use library only'),
                    _('User annotations generated from main library only'),
                    show=True)
        db = self.gui.library_view.model().db

        # Get the list of ids
        ids = get_ids_from_selected_rows()
        if not ids:
            return error_dialog(self.gui, _('No books selected'),
                    _('No books selected to fetch annotations from'),
                    show=True)

        debug_print("getAnnotationForSelected - ids=", ids)
        # Map ids to paths
        path_map = generate_annotation_paths(ids, db, self.device)
        debug_print("getAnnotationForSelected - path_map=", path_map)
        if len(path_map) == 0:
            return error_dialog(self.gui, _('No books on device selected'),
                    _('None of the books selected were on the device. Annotations can only be copied for books on the device.'),
                    show=True)

        from calibre.ebooks.BeautifulSoup import BeautifulSoup, Tag, NavigableString
        from calibre.ebooks.metadata import authors_to_string

        # Dispatch to devices.kindle.driver.get_annotations()
        debug_print("getAnnotationForSelected - path_maps=", path_map)
        bookmarked_books = self.device.get_annotations(path_map)
        debug_print("getAnnotationForSelected - bookmarked_books=", bookmarked_books)

        for i, id_ in enumerate(bookmarked_books):
            bm = self.device.UserAnnotation(bookmarked_books[id_][0], bookmarked_books[id_][1])

            mi = db.get_metadata(id_, index_is_id=True)

            user_notes_soup = self.device.generate_annotation_html(bm.value)
            spanTag = Tag(user_notes_soup, 'span')
            spanTag['style'] = 'font-weight:normal'
            spanTag.insert(0,NavigableString(
                _("<hr /><b>%(title)s</b> by <b>%(author)s</b>") % \
                            dict(title=mi.title,
                            #loc=last_read_location,
                            author=authors_to_string(mi.authors))))
            user_notes_soup.insert(0, spanTag)
            bookmark_html = unicode(user_notes_soup.prettify())
            debug_print(bookmark_html)
            annotationText.append(bookmark_html)

        d = ViewLog(_("Kobo Touch Annotation"), "\n".join(annotationText), parent=self.gui)
#        d = ViewLog(_("Kobo Touch Annotation"), unicode(j), parent=self.gui)
        d.setWindowIcon(self.qaction.icon())
#       d.setWindowIcon(get_icon('bookmarks.png'))
        d.exec_()


    def _upload_covers(self, books):

        uploaded_covers     = 0
        total_books         = 0
        not_on_device_books = len(books)

        kobo_kepub_dir = self.device.normalize_path('.kobo/kepub/')
        sd_kepub_dir   = self.device.normalize_path('koboExtStorage/kepub/')
        debug_print("_upload_covers - kobo_image_dir=", kobo_kepub_dir)

        for book in books:
            total_books += 1
#            debug_print("_upload_covers - book=", book)
#            debug_print("_upload_covers - thumbnail=", book.thumbnail)
            paths = self.get_device_paths_from_id(book.calibre_id)
            not_on_device_books -= 1 if len(paths) > 0 else 0
            for path in paths:
                debug_print("_upload_covers - path=", path)
                if (kobo_kepub_dir not in path and sd_kepub_dir not in path) or self.options['kepub_covers']:
                    self.device._upload_cover(path, '', book, path, self.options['blackandwhite'], keep_cover_aspect=self.options['keep_cover_aspect'])
                    uploaded_covers += 1

        return total_books, uploaded_covers, not_on_device_books


    def _remove_covers(self, books):
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device.normalize_path(
            self.device_path +'.kobo/KoboReader.sqlite'))) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            total_books         = 0
            removed_covers      = 0
            not_on_device_books = 0

            imageId_query = 'SELECT ImageId '       \
                            'FROM content '         \
                            'WHERE ContentType = ? '\
                            'AND ContentId = ?'
            cursor = connection.cursor()

            for book in books:
                debug_print("_remove_covers - book=", book)
                debug_print("_remove_covers - book.__class__=", book.__class__)
                debug_print("_remove_covers - book.contentID=", book.contentID)
                debug_print("_remove_covers - book.lpath=", book.lpath)
                debug_print("_remove_covers - book.path=", book.path)
                contentIDs = [book.contentID] if book.contentID is not None else self.get_contentIDs_from_id(book.calibre_id)
                debug_print("_remove_covers - contentIDs=", contentIDs)
                for contentID in contentIDs:
                    debug_print("_remove_covers - contentID=", contentID)
                    if 'file:///' not in contentID and not self.options['kepub_covers']:
                        continue
                    
                    if contentID.startswith("file:///mnt/sd/"):
                        path = self.device._card_a_prefix
                    else:
                        path = self.device._main_prefix

                    query_values = (self.CONTENTTYPE, contentID,)
                    cursor.execute(imageId_query, query_values)
                    result = cursor.fetchone()
                    if result is not None:
                        debug_print("_remove_covers - contentId='%s', imageId='%s'" % (contentID, result[0]))
                        self.device.delete_images(result[0], path)
                        removed_covers +=1
                    else:
                        debug_print("_remove_covers - no match for contentId='%s'" % (contentID,))
                        not_on_device_books += 1
                    connection.commit()
                    total_books += 1

            cursor.close()

        return removed_covers, not_on_device_books, total_books


    def _check_book_in_database(self, books):
        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device.normalize_path(
            self.device_path +'.kobo/KoboReader.sqlite'))) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            not_on_device_books = []

            imageId_query = 'SELECT 1 '       \
                            'FROM content '         \
                            'WHERE ContentType = ? '\
                            'AND ContentId = ?'
            cursor = connection.cursor()

            for book in books:
                if not book.contentID:
                    not_on_device_books.append(book)
                    continue

                query_values = (self.CONTENTTYPE, book.contentID,)
                cursor.execute(imageId_query, query_values)
                result = cursor.fetchone()
                if result is None:
                    debug_print("_check_book_in_database - no match for contentId='%s'" % (book.contentID,))
                    not_on_device_books.append(book)
#                else:
#                    debug_print("_check_book_in_database - match for contentId='%s', imageId='%s'" % (book.contentID, result[0]))
                connection.commit()

            cursor.close()

        return not_on_device_books


    def _check_device_database(self):
        import sqlite3 as sqlite
#        with closing(sqlite.connect(self.device.normalize_path(
#            'D:\Download\Books\Kobo\Test Books\prospero/KoboReader-paola.sqlite'))) as connection:
        with closing(sqlite.connect(self.device.normalize_path(
            self.device_path +'.kobo/KoboReader.sqlite'))) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            check_query = 'PRAGMA integrity_check'
            cursor = connection.cursor()

            check_result = ''
            cursor.execute(check_query)
            result = cursor.fetchall()
            if not result is None:
                for line in result:
                    check_result += '\n' + line[0]
                    debug_print("_check_device_database - result line=", line[0])
            else:
                check_result = "Execution of '%s' failed" % check_query

            connection.commit()

            cursor.close()

        return check_result


    def generate_metadata_query(self):
        debug_print("generate_metadata_query - self.supports_series=", self.supports_series)
        if self.supports_series:
            debug_print("generate_metadata_query - supports series is true")
            test_query = 'SELECT Title,          '\
                        '    Attribution,        '\
                        '    Description,        '\
                        '    Publisher,          '\
                        '    Series,             '\
                        '    SeriesNumber,       '\
                        '    ReadStatus,         '\
                        '    DateCreated,        '\
                        '    ISBN,               '\
                        '    FeedbackType,       '\
                        '    FeedbackTypeSynced, '\
                        '    Language, '          \
                        '    r.Rating, '          \
                        '    r.DateModified '     \
                        'FROM content c1 left outer join '                 \
                                'ratings r on c1.ContentID = r.ContentID ' \
                        'WHERE c1.ContentType = ? '  \
                        'AND c1.ContentId = ?'
        else:
            test_query = 'SELECT Title,           '\
                        '    Attribution,         '\
                        '    Description,         '\
                        '    Publisher,           '\
                        '    null as Series,      '\
                        '    null as SeriesNumber,'\
                        '    ReadStatus,          '\
                        '    DateCreated,         '\
                        '    ISBN,                '\
                        '    FeedbackType,        '\
                        '    FeedbackTypeSynced, ' \
                        '    Language, '           \
                        '    r.Rating, '           \
                        '    r.DateModified '      \
                        'FROM content c1 left outer join '                 \
                                'ratings r on c1.ContentID = r.ContentID ' \
                        'WHERE c1.ContentType = ? '  \
                        'AND c1.ContentId = ?'

        return test_query

    def _update_metadata(self, books):
        from calibre.ebooks.metadata import authors_to_string
        from calibre.utils.localization import canonicalize_lang, lang_as_iso639_1

        updated_books       = 0
        not_on_device_books = 0
        unchanged_books     = 0
        count_books         = 0

        library_db     = self.gui.current_db
        library_config = cfg.get_library_config(library_db)
        rating_column  = library_config.get(cfg.KEY_RATING_CUSTOM_COLUMN, cfg.DEFAULT_LIBRARY_VALUES[cfg.KEY_RATING_CUSTOM_COLUMN])

        rating_update = 'UPDATE ratings '           \
                            'SET Rating = ?, '      \
                                'DateModified = ? ' \
                            'WHERE ContentID  = ?'
        rating_insert = 'INSERT INTO ratings (' \
                            'Rating, '          \
                            'DateModified, '    \
                            'ContentID '        \
                            ')'                 \
                            'VALUES (?, ?, ?)'
        rating_delete = 'DELETE FROM ratings '     \
                            'WHERE ContentID = ?'

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            test_query = self.generate_metadata_query()
            cursor = connection.cursor()

            for book in books:
#                device_book_paths = self.get_device_paths_from_id(book.id)
                for contentID in book.contentIDs:
                    count_books += 1
#                    contentID = self.device.contentid_from_path(path, self.CONTENTTYPE)
                    query_values = (self.CONTENTTYPE, contentID,)
                    cursor.execute(test_query, query_values)
                    result = cursor.fetchone()
                    if result is not None:
                        debug_print("_update_metadata - found contentId='%s'" % (contentID))#, book)
                        debug_print("    result=", result)
                        #self.device.delete_images(result[0])
                        update_query  = 'UPDATE content SET '
                        update_values = []
                        set_clause    = ''
                        changes_found = False
                        rating_values = []
                        rating_change_query = None
    
                        if self.options[cfg.KEY_SET_TITLE] and not result[0] == book.title:
                            set_clause += ', Title  = ? '
                            update_values.append(book.title)
                        if self.options[cfg.KEY_SET_AUTHOR] and not result[1] == authors_to_string(book.authors):
                            set_clause += ', Attribution  = ? '
                            update_values.append(authors_to_string(book.authors))
                        if self.options[cfg.KEY_SET_DESCRIPTION]  and not result[2] == book.comments:
                            set_clause += ', Description = ? '
                            update_values.append(book.comments)
                        if self.options[cfg.KEY_SET_PUBLISHER]  and not result[3] == book.publisher:
                            set_clause += ', Publisher = ? '
                            update_values.append(book.publisher)
                        if self.options[cfg.KEY_SET_PUBLISHED_DATE]:
                            pubdate_string = strftime(self.device.TIMESTAMP_STRING, book.pubdate)
                            if not (result[7] == pubdate_string):
                                set_clause += ', DateCreated = ? '
                                debug_print("_update_metadata - self.convert_kobo_date(result[7])=", self.convert_kobo_date(result[7]))
                                debug_print("_update_metadata - self.convert_kobo_date(result[7]).__class__=", self.convert_kobo_date(result[7]).__class__)
                                debug_print("_update_metadata - book.pubdate  =", book.pubdate)
                                debug_print("_update_metadata - result[7]     =", result[7])
                                debug_print("_update_metadata - pubdate_string=", pubdate_string)
                                debug_print("_update_metadata - book.pubdate.__class__=", book.pubdate.__class__)
                                update_values.append(pubdate_string)
    
                        if self.options[cfg.KEY_SET_ISBN]  and not result[8] == book.isbn:
                            set_clause += ', ISBN = ? '
                            update_values.append(book.isbn)

                        if self.options[cfg.KEY_SET_LANGUAGE] and not result[11] == lang_as_iso639_1(book.language):
                            debug_print("_update_metadata - book.language =", book.language)
                            debug_print("_update_metadata - lang_as_iso639_1(book.language)=", lang_as_iso639_1(book.language))
                            debug_print("_update_metadata - canonicalize_lang(book.language)=", canonicalize_lang(book.language))
#                            set_clause += ', ISBN = ? '
#                            update_values.append(book.isbn)

                        if self.options[cfg.KEY_SET_NOT_INTERESTED] and not (result[9] == 2): # or result[10] == 1):
                            set_clause += ', FeedbackType = ? '
                            update_values.append(2)
#                            set_clause += ', FeedbackTypeSynced = ? '
#                            update_values.append(1)

                        debug_print("_update_metadata - self.options[cfg.KEY_SET_RATING]= ", self.options[cfg.KEY_SET_RATING])
                        if rating_column and self.options[cfg.KEY_SET_RATING]:
                            if rating_column == 'rating':
                                rating = book.rating
                            else:
                                rating = book.get_user_metadata(rating_column, True)['#value#']
                            rating = None if rating == 0 else rating / 2
                            debug_print("_update_metadata - rating=", rating, "result[12]=", result[12])
                            rating_values.append(rating)
                            rating_values.append(strftime(self.device.TIMESTAMP_STRING, time.gmtime()))
                            rating_values.append(contentID)
                            if not rating == result[12]:
                                if not rating:
                                    rating_change_query = rating_delete
                                    rating_values = (contentID, )
                                elif result[13] is None: # If the date modified column does not have a value, there is no rating column
                                    rating_change_query = rating_insert
                                else:
                                    rating_change_query = rating_update

                        if self.supports_series:
                            debug_print("_update_metadata - self.options['series']", self.options['series'])
                            debug_print("_update_metadata - book.series=", book.series, "book.series_index=", book.series_index)
                            debug_print("_update_metadata - result[4] ='%s' result[5] =%s" % (result[4], result[5]))
                            debug_print("_update_metadata - result[4] == book.series =", (result[4] == book.series))
                            series_index_str = ("%g" % book.series_index) if book.series_index is not None else None
                            debug_print('_update_metadata - result[5] == series_index_str =', (result[5] == series_index_str))
                            debug_print('_update_metadata - not (result[4] == book.series or result[5] == series_index_str) =', not (result[4] == book.series or result[5] == series_index_str))
                            if self.options['series'] and not (result[4] == book.series and (result[5] == book.series_index_string or result[5] == series_index_str)):
                                debug_print("_update_metadata - setting series")
                                set_clause += ', Series  = ? '
                                set_clause += ', SeriesNumber   = ? '
                                if book.series is None or book.series == '':
                                    update_values.append(None)
                                    update_values.append(None)
                                else:
                                    update_values.append(book.series)
    #                                update_values.append("%g" % book.series_index)
                                    if book.series_index_string is not None:
                                        update_values.append(book.series_index_string)
                                    elif book.series_index is None:
                                        update_values.append(None)
                                    else:
                                        update_values.append("%g" % book.series_index)
    
    #                    debug_print("_update_metadata - self.options['setRreadingStatus']", self.options['setRreadingStatus'])
    #                    debug_print("_update_metadata - self.options['readingStatus']", self.options['readingStatus'])
    #                    debug_print("_update_metadata - not (result[6] == self.options['readingStatus'])", not (result[6] == self.options['readingStatus']))
                        if self.options['setRreadingStatus'] and (not (result[6] == self.options['readingStatus']) or self.options['resetPosition']):
                            set_clause += ', ReadStatus  = ? '
                            update_values.append(self.options['readingStatus'])
                            if self.options['resetPosition']:
                                set_clause += ', DateLastRead = ?'
                                update_values.append(None)
                                set_clause += ', ChapterIDBookmarked = ?'
                                update_values.append(None)
                                set_clause += ', ___PercentRead = ?'
                                update_values.append(0)
                                set_clause += ', FirstTimeReading = ? '
                                update_values.append(self.options['readingStatus'] < 2)
    
                        if len(set_clause) > 0:
                            update_query += set_clause[1:]
                            changes_found = True

                        if not (changes_found or rating_change_query):
                            debug_print("_update_metadata - no changes found to selected metadata. No changes being made.")
                            unchanged_books += 1
                            continue
    
                        update_query += 'WHERE ContentID = ? AND ContentType = ?'
                        update_values.append(contentID)
                        update_values.append(self.CONTENTTYPE)
                        debug_print("_update_metadata - update_query=%s" % update_query)
                        debug_print("_update_metadata - update_values= ", update_values)
                        try:
                            if changes_found:
                                cursor.execute(update_query, update_values)

                            if rating_change_query:
                                debug_print("_update_metadata - rating_change_query=%s" % rating_change_query)
                                debug_print("_update_metadata - rating_values= ", rating_values)
                                cursor.execute(rating_change_query, rating_values)

                            updated_books += 1
                        except:
                            debug_print('    Database Exception:  Unable to set series info')
                            raise
                    else:
                        debug_print("_update_metadata - no match for title='%s' contentId='%s'" % (book.title, contentID))
                        not_on_device_books += 1
                    connection.commit()
            debug_print("Update summary: Books updated=%d, unchanged books=%d, not on device=%d Total=%d" % (updated_books, unchanged_books, not_on_device_books, count_books))

            cursor.close()
        
        return (updated_books, unchanged_books, not_on_device_books, count_books)


    def _store_current_bookmark(self, books):

        books_with_bookmark     = 0
        books_without_bookmark = 0
        count_books            = 0

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            epub_fetch_query = 'SELECT c1.ChapterIDBookmarked, ' \
                                    'c2.adobe_location, '      \
                                    'c1.ReadStatus, '          \
                                    'c1.___PercentRead, '      \
                                    'c1.Attribution, '         \
                                    'c1.DateLastRead, '        \
                                    'c1.Title, '               \
                                    'c1.MimeType, '            \
                                    'r.rating '                \
                                'FROM content c1 JOIN content c2 ON c1.ChapterIDBookmarked = c2.ContentID ' \
                                    'LEFT OUTER JOIN ratings r ON c1.ContentID = r.ContentID '  \
                                'WHERE c2.bookid = ?'

            kepub_fetch_query = 'SELECT c1.ChapterIDBookmarked, ' \
                                    'c1.adobe_location, '      \
                                    'c1.ReadStatus, '          \
                                    'c1.___PercentRead, '      \
                                    'c1.Attribution, '         \
                                    'c1.DateLastRead, '        \
                                    'c1.Title, '               \
                                    'c1.MimeType, '            \
                                    'r.rating '                \
                                'FROM content c1 LEFT OUTER JOIN ratings r ON c1.ContentID = r.ContentID '  \
                                'WHERE c1.contentID = ?'

            library_db = self.gui.current_db
            library_config = cfg.get_library_config(library_db)
            kobo_chapteridbookmarked_column = library_config.get(cfg.KEY_CURRENT_LOCATION_CUSTOM_COLUMN, cfg.DEFAULT_LIBRARY_VALUES[cfg.KEY_CURRENT_LOCATION_CUSTOM_COLUMN])
            kobo_percentRead_column         = library_config.get(cfg.KEY_PERCENT_READ_CUSTOM_COLUMN, cfg.DEFAULT_LIBRARY_VALUES[cfg.KEY_PERCENT_READ_CUSTOM_COLUMN])
            rating_column                   = library_config.get(cfg.KEY_RATING_CUSTOM_COLUMN, cfg.DEFAULT_LIBRARY_VALUES[cfg.KEY_RATING_CUSTOM_COLUMN])
            last_read_column                = library_config.get(cfg.KEY_LAST_READ_CUSTOM_COLUMN, cfg.DEFAULT_LIBRARY_VALUES[cfg.KEY_LAST_READ_CUSTOM_COLUMN])
            if kobo_chapteridbookmarked_column:
                debug_print("_store_current_bookmark - kobo_chapteridbookmarked_column=", kobo_chapteridbookmarked_column)
#                kobo_chapteridbookmarked_col = custom_cols[kobo_chapteridbookmarked_column]
#                debug_print("_store_current_bookmark - kobo_chapteridbookmarked_col=", kobo_chapteridbookmarked_col)
                kobo_chapteridbookmarked_col_label = library_db.field_metadata.key_to_label(kobo_chapteridbookmarked_column)
                debug_print("_store_current_bookmark - kobo_chapteridbookmarked_col_label=", kobo_chapteridbookmarked_col_label)
            if kobo_percentRead_column:
#                kobo_percentRead_col = custom_cols[kobo_percentRead_column]
                kobo_percentRead_col_label = library_db.field_metadata.key_to_label(kobo_percentRead_column)

            rating_col_label = library_db.field_metadata.key_to_label(rating_column) if rating_column else ''
            if last_read_column:
#                kobo_percentRead_col = custom_cols[kobo_percentRead_column]
                last_read_col_label = library_db.field_metadata.key_to_label(last_read_column)

            debug_print("_store_current_bookmark - kobo_chapteridbookmarked_column=", kobo_chapteridbookmarked_column)
            debug_print("_store_current_bookmark - kobo_percentRead_column=", kobo_percentRead_column) 
            debug_print("_store_current_bookmark - rating_column=", rating_column) 
            debug_print("_store_current_bookmark - rating_col_label=", rating_col_label) 
            debug_print("_store_current_bookmark - last_read_column=", last_read_column) 

#            id_map = {}
            cursor = connection.cursor()
            for book in books:
                count_books += 1
#                mi = Metadata(_('Unknown'))
                for contentID in book.contentIDs:
                    debug_print("_store_current_bookmark - contentId='%s'" % (contentID))
                    fetch_values = (contentID,)
                    if contentID.endswith(".kepub.epub"):
                        fetch_query = kepub_fetch_query
                    else:
                        fetch_query = epub_fetch_query
                    cursor.execute(fetch_query, fetch_values)
                    result = cursor.fetchone()
                    
                    kobo_chapteridbookmarked = None
                    kobo_adobe_location      = None
                    kobo_percentRead         = None
                    last_read                = None
                    if result is not None: # and not result[7] == MIMETYPE_KOBO:
                        debug_print("_store_current_bookmark - result=", result)
                        books_with_bookmark      += 1
                        if result[7] == MIMETYPE_KOBO:
                            kobo_chapteridbookmarked = result[0]
                            kobo_adobe_location      = None
                        else:
                            kobo_chapteridbookmarked = result[0][len(contentID) + 1:]
                            kobo_adobe_location      = result[1]
                        if result[2] == 1:
                            kobo_percentRead     = result[3]
                        elif result[2] == 2:
                            kobo_percentRead     = 100
                        if result[8]:
                            kobo_rating = result[8] * 2
                        else:
                            kobo_rating = 0
                        if result[5]:
                            debug_print("_store_current_bookmark - result[5]=", result[5])
                            last_read = self.convert_kobo_date(result[5])
                            debug_print("_store_current_bookmark - last_read=", last_read)
                    elif self.options[cfg.KEY_CLEAR_IF_UNREAD]:
                        books_with_bookmark      += 1
                        kobo_chapteridbookmarked = None
                        kobo_adobe_location      = None
                        kobo_percentRead         = None
                        last_read                = None
                        kobo_rating              = 0
                    else:
                        books_without_bookmark += 1
                        continue
                    
                    debug_print("_store_current_bookmark - kobo_chapteridbookmarked='%s'" % (kobo_chapteridbookmarked))
                    debug_print("_store_current_bookmark - kobo_adobe_location='%s'" % (kobo_adobe_location))
                    debug_print("_store_current_bookmark - kobo_percentRead=", kobo_percentRead)
                    if kobo_chapteridbookmarked_column:
                        if kobo_chapteridbookmarked and kobo_adobe_location:
    #                        kobo_chapteridbookmarked_col['#value#'] = kobo_chapteridbookmarked + '|||' + kobo_adobe_location
                            new_value = kobo_chapteridbookmarked + BOOKMARK_SEPARATOR + kobo_adobe_location
                        elif kobo_chapteridbookmarked:
    #                        kobo_chapteridbookmarked_col['#value#'] = kobo_chapteridbookmarked + '|||' + kobo_adobe_location
                            new_value = kobo_chapteridbookmarked
                        else:
    #                        kobo_chapteridbookmarked_col['#value#'] = None
                            kobo_percentRead = None
                            new_value        = None
                            debug_print("_store_current_bookmark - setting bookmark column to None")
    #                    mi.set_user_metadata(kobo_chapteridbookmarked_column, kobo_chapteridbookmarked_col)
                        library_db.set_custom(book.calibre_id, new_value, label=kobo_chapteridbookmarked_col_label, commit=False)

                    if kobo_percentRead_column:
    #                    kobo_percentRead_col['#value#'] = kobo_percentRead
                        debug_print("_store_current_bookmark - setting mi.kobo_percentRead=", kobo_percentRead)
                        library_db.set_custom(book.calibre_id, kobo_percentRead, label=kobo_percentRead_col_label, commit=False)
    #                    mi.set_user_metadata(kobo_percentRead_column, kobo_percentRead_col)

                    if rating_column and kobo_rating > 0:
    #                    kobo_percentRead_col['#value#'] = kobo_percentRead
                        debug_print("_store_current_bookmark - setting mi.rating_column=", rating_column)
                        if rating_column == 'rating':
                            library_db.set_rating(book.calibre_id, kobo_rating, commit=False)
                        else:
                            library_db.set_custom(book.calibre_id, kobo_rating, label=rating_col_label, commit=False)

                    if last_read_column:
    #                    last_read_col_label['#value#'] = last_read
                        debug_print("_store_current_bookmark - setting mi.last_read=", last_read)
                        library_db.set_custom(book.calibre_id, last_read, label=last_read_col_label, commit=False)
    #                    mi.set_user_metadata(last_read_column, last_read_col_label)

    #                debug_print("_store_current_bookmark - mi=", mi)
    #                id_map[book.calibre_id] = mi

            connection.commit()
            cursor.close()
            
#            edit_metadata_action = self.gui.iactions['Edit Metadata']
#            edit_metadata_action.apply_metadata_changes(id_map)
            library_db.commit()
        
        return (books_with_bookmark, books_without_bookmark, count_books)


    def _restore_current_bookmark(self, books):
        from calibre.ebooks.metadata import authors_to_string

        updated_books       = 0
        not_on_device_books = 0
        count_books         = 0

        library_db     = self.gui.current_db
        library_config = cfg.get_library_config(library_db)
        kobo_chapteridbookmarked_column = library_config.get(cfg.KEY_CURRENT_LOCATION_CUSTOM_COLUMN, cfg.DEFAULT_LIBRARY_VALUES[cfg.KEY_CURRENT_LOCATION_CUSTOM_COLUMN])
        kobo_percentRead_column         = library_config.get(cfg.KEY_PERCENT_READ_CUSTOM_COLUMN, cfg.DEFAULT_LIBRARY_VALUES[cfg.KEY_PERCENT_READ_CUSTOM_COLUMN])
        rating_column                   = library_config.get(cfg.KEY_RATING_CUSTOM_COLUMN, cfg.DEFAULT_LIBRARY_VALUES[cfg.KEY_RATING_CUSTOM_COLUMN])
        last_read_column                = library_config.get(cfg.KEY_LAST_READ_CUSTOM_COLUMN, cfg.DEFAULT_LIBRARY_VALUES[cfg.KEY_LAST_READ_CUSTOM_COLUMN])

        chapter_query = 'SELECT c1.ChapterIDBookmarked, ' \
                               'c1.ReadStatus, '          \
                               'c1.___PercentRead, '      \
                               'c1.Attribution, '         \
                               'c1.DateLastRead, '        \
                               'c1.Title, '               \
                               'c1.MimeType, '            \
                               'r.Rating, '               \
                               'r.DateModified '          \
                            'FROM content c1 left outer join '                 \
                                    'ratings r on c1.ContentID = r.ContentID ' \
                            'WHERE c1.ContentType = ? '   \
                            'AND c1.ContentId = ?'
        chapter_update  = 'UPDATE content '                \
                            'SET ChapterIDBookmarked = ? ' \
                            '  , FirstTimeReading = ? '    \
                            '  , ReadStatus = ? '          \
                            '  , ___PercentRead = ? '      \
                            '  , DateLastRead = ? '        \
                            'WHERE ContentType = ? '       \
                            'AND ContentID = ?'
        location_update  = 'UPDATE content '           \
                             'SET adobe_location = ? ' \
                             'WHERE ContentType = 9 '  \
                             'AND ContentID = ?'
        rating_update = 'UPDATE ratings '           \
                            'SET Rating = ?, '      \
                                'DateModified = ? ' \
                            'WHERE ContentID  = ?'
        rating_insert = 'INSERT INTO ratings (' \
                            'Rating, '          \
                            'DateModified, '    \
                            'ContentID '        \
                            ')'                 \
                            'VALUES (?, ?, ?)'
        rating_delete = 'DELETE FROM ratings '  \
                            'WHERE ContentID = ?'

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path()
#            'D:/Download/Books/CalibrePlugins/KoboTouchTOCUpdatePlugin/KoboReader.sqlite'))) as connection:
            )) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            cursor = connection.cursor()

            for book in books:
                count_books += 1
                for contentID in book.contentIDs:
                    chapter_values = (self.CONTENTTYPE, contentID,)
                    cursor.execute(chapter_query, chapter_values)
                    result = cursor.fetchone()
                    
                    if result is not None: # and not result[6] == MIMETYPE_KOBO:
                        debug_print("_restore_current_bookmark - result= ", result)
                        chapter_update          = 'UPDATE content SET '
                        chapter_set_clause      = ''
                        chapter_values          = []
                        location_update         = 'UPDATE content SET '
                        location_set_clause     = ''
                        location_values         = []
                        rating_change_query     = None
                        rating_values           = []
    
                        kobo_chapteridbookmarked = None
                        kobo_adobe_location      = None
                        kobo_percentRead         = None
    
                        if kobo_chapteridbookmarked_column:
                            reading_location_string  = book.get_user_metadata(kobo_chapteridbookmarked_column, True)['#value#']
                            debug_print("_restore_current_bookmark - reading_location_string=", reading_location_string)
                            if reading_location_string:
                                if result[6] == MIMETYPE_KOBO:
                                    kobo_chapteridbookmarked = reading_location_string
                                    kobo_adobe_location      = None
                                else:
                                    reading_location_parts   = reading_location_string.split(BOOKMARK_SEPARATOR)
                                    kobo_chapteridbookmarked = contentID + "#" + reading_location_parts[0]
                                    kobo_adobe_location      = reading_location_parts[1]
                            else:
                                kobo_chapteridbookmarked = None
                                kobo_adobe_location      = None
    
                            if reading_location_string:
                                chapter_values.append(kobo_chapteridbookmarked)
                                chapter_set_clause += ', ChapterIDBookmarked  = ? '
                                location_values.append(kobo_adobe_location)
                                location_set_clause += ', adobe_location  = ? '
                            else:
                                debug_print("_restore_current_bookmark - reading_location_string=", reading_location_string)

                        if kobo_percentRead_column:
                            kobo_percentRead = book.get_user_metadata(kobo_percentRead_column, True)['#value#']
                            kobo_percentRead = kobo_percentRead if kobo_percentRead else result[2]
                            chapter_values.append(kobo_percentRead)
                            chapter_set_clause += ', ___PercentRead  = ? '

                        if self.options[cfg.KEY_READING_STATUS]:
                            if kobo_percentRead:
                                debug_print("_restore_current_bookmark - chapter_values= ", chapter_values)
                                if kobo_percentRead == 100:
                                    chapter_values.append(2)
                                    debug_print("_restore_current_bookmark - chapter_values= ", chapter_values)
                                else:
                                    chapter_values.append(1)
                                    debug_print("_restore_current_bookmark - chapter_values= ", chapter_values)
                                chapter_set_clause += ', ReadStatus  = ? '
                                chapter_values.append('false')
                                chapter_set_clause += ', FirstTimeReading = ? '

                        last_read = None
                        if self.options[cfg.KEY_DATE_TO_NOW]:
                            chapter_values.append(strftime(self.device.TIMESTAMP_STRING, time.gmtime()))
                            chapter_set_clause += ', DateLastRead  = ? '
                        elif last_read_column:
                            last_read = book.get_user_metadata(last_read_column, True)['#value#']
                            if last_read is not None:
                                chapter_values.append(last_read.strftime(self.device.TIMESTAMP_STRING))
                                chapter_set_clause += ', DateLastRead  = ? '

                        debug_print("_restore_current_bookmark - self.options[cfg.KEY_SET_RATING]= ", self.options[cfg.KEY_SET_RATING])
                        rating = None
                        if rating_column and self.options[cfg.KEY_SET_RATING]:
                            if rating_column == 'rating':
                                rating = book.rating
                            else:
                                rating = book.get_user_metadata(rating_column, True)['#value#']
                            rating = None if not rating or rating == 0 else rating / 2
                            debug_print("_restore_current_bookmark - rating=%d", rating, " result[7]=%s", result[7])
                            rating_values.append(rating)
                            if last_read:
                                rating_values.append(last_read)
                            else:
                                rating_values.append(strftime(self.device.TIMESTAMP_STRING, time.gmtime()))
                            rating_values.append(contentID)
                            if not rating:
                                rating_change_query = rating_delete
                                rating_values = (contentID, )
                            elif result[8] is None: # If the date modified column does not have a value, there is no rating column
                                rating_change_query = rating_insert
                            else:
                                rating_change_query = rating_update

                        debug_print("_restore_current_bookmark - found contentId='%s'" % (contentID))
                        debug_print("_restore_current_bookmark - kobo_chapteridbookmarked=", kobo_chapteridbookmarked)
                        debug_print("_restore_current_bookmark - kobo_adobe_location=", kobo_adobe_location)
                        debug_print("_restore_current_bookmark - kobo_percentRead=", kobo_percentRead)
                        debug_print("_restore_current_bookmark - rating=", rating)
                        debug_print("_restore_current_bookmark - last_read=", last_read)
                        debug_print("    result=", result)
    
                        if len(chapter_set_clause) > 0:
                            chapter_update += chapter_set_clause[1:]
                            chapter_update += 'WHERE ContentID = ? AND ContentType = ?'
                            chapter_values.append(contentID)
                            chapter_values.append(self.CONTENTTYPE)
                        else:
                            debug_print("_restore_current_bookmark - no changes found to selected metadata. No changes being made.")
                            not_on_device_books += 1
                            continue
    
                        debug_print("_restore_current_bookmark - chapter_update=%s" % chapter_update)
                        debug_print("_restore_current_bookmark - chapter_values= ", chapter_values)
                        try:
                            cursor.execute(chapter_update, chapter_values)
                            if len(location_set_clause) > 0 and not result[6] == MIMETYPE_KOBO:
                                location_update += location_set_clause[1:]
                                location_update += ' WHERE ContentID = ? AND ContentType = ?'
                                location_values.append(kobo_chapteridbookmarked)
                                location_values.append(9)
                                debug_print("_restore_current_bookmark - location_update=%s" % location_update)
                                debug_print("_restore_current_bookmark - location_values= ", location_values)
                                cursor.execute(location_update, location_values)
                            if rating_change_query:
                                debug_print("_restore_current_bookmark - rating_change_query=%s" % rating_change_query)
                                debug_print("_restore_current_bookmark - rating_values= ", rating_values)
                                cursor.execute(rating_change_query, rating_values)
                                
                            updated_books += 1
                        except:
                            debug_print('    Database Exception:  Unable to set bookmark info.')
                            raise
                    else:
                        debug_print("_restore_current_bookmark - no match for title='%s' contentId='%s'" % (book.title, book.contentID))
                        not_on_device_books += 1
                    connection.commit()
            debug_print("_restore_current_bookmark - Update summary: Books updated=%d, not on device=%d, Total=%d" % (updated_books, not_on_device_books, count_books))

            cursor.close()
        
        return (updated_books, not_on_device_books, count_books)



    def fetch_book_fonts(self):

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device.normalize_path(
            self.device_path +'.kobo/KoboReader.sqlite'))) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            book_options = {}
            
            fetch_query = 'SELECT  '                   \
                            '"ReadingFontFamily", '    \
                            '"ReadingFontSize", '      \
                            '"ReadingAlignment", '     \
                            '"ReadingLineHeight", '    \
                            '"ReadingLeftMargin", '    \
                            '"ReadingRightMargin"  '   \
                            'FROM content_settings '   \
                            'WHERE ContentType = ? '   \
                            'AND ContentId = ?'
            fetch_values = (self.CONTENTTYPE, self.single_contentID,)

            cursor = connection.cursor()
            cursor.execute(fetch_query, fetch_values)
            result = cursor.fetchone()
            if result is not None:
                book_options['readingFontFamily']   = result[0]
                book_options['readingFontSize']     = result[1]
                book_options['readingAlignment']    = result[2].title()
                book_options['readingLineHeight']   = result[3]
                book_options['readingLeftMargin']   = result[4]
                book_options['readingRightMargin']  = result[5]
            connection.commit()

            cursor.close()
        
        return book_options




    def _set_reader_fonts(self, contentIDs, delete=False):

        updated_fonts  = 0
        added_fonts    = 0
        deleted_fonts  = 0
        count_books    = 0

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device.normalize_path(
            self.device_path +'.kobo/KoboReader.sqlite'))) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            test_query = 'SELECT 1 '                    \
                            'FROM content_settings '    \
                            'WHERE ContentType = ? '    \
                            'AND ContentId = ?'
            delete_query = 'DELETE '                    \
                            'FROM content_settings '    \
                            'WHERE ContentType = ? '    \
                            'AND ContentId = ?'

            if not delete:
                font_face       = self.options[cfg.KEY_READING_FONT_FAMILY]
                justification   = self.options[cfg.KEY_READING_ALIGNMENT].lower()
                justification   = '' if justification == 'off' else justification
                font_size       = self.options[cfg.KEY_READING_FONT_SIZE]
                line_spacing    = self.options[cfg.KEY_READING_LINE_HEIGHT]
                left_margins    = self.options[cfg.KEY_READING_LEFT_MARGIN]
                right_margins   = self.options[cfg.KEY_READING_RIGHT_MARGIN]
               
                add_query = 'INSERT INTO content_settings ( '   \
                                '"ContentType", '               \
                                '"DateModified", '              \
                                '"ReadingFontFamily", '         \
                                '"ReadingFontSize", '           \
                                '"ReadingAlignment", '          \
                                '"ReadingLineHeight", '         \
                                '"ReadingLeftMargin", '         \
                                '"ReadingRightMargin", '        \
                                '"ContentID" '                  \
                                ') '                            \
                            'VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)'
                add_values = (
                              self.CONTENTTYPE, 
                              time.strftime(self.device.TIMESTAMP_STRING, time.gmtime()), 
                              font_face, 
                              font_size, 
                              justification, 
                              line_spacing, 
                              left_margins, 
                              right_margins, 
                              )
                update_query = 'UPDATE content_settings '    \
                                'SET "DateModified" = ?, '   \
                                '"ReadingFontFamily" = ?, '  \
                                '"ReadingFontSize" = ?, '    \
                                '"ReadingAlignment" = ?, '   \
                                '"ReadingLineHeight" = ?, '  \
                                '"ReadingLeftMargin" = ?, '  \
                                '"ReadingRightMargin" = ? '  \
                                'WHERE ContentType = ?  '    \
                                'AND ContentId = ?'
                update_values = (
                                 time.strftime(self.device.TIMESTAMP_STRING, time.gmtime()), 
                                 font_face, 
                                 font_size, 
                                 justification, 
                                 line_spacing, 
                                 left_margins, 
                                 right_margins, 
                                 self.CONTENTTYPE, 
                                 )

            cursor = connection.cursor()

            for contentID in contentIDs:
                test_values = (self.CONTENTTYPE, contentID,)
                if delete:
                    cursor.execute(delete_query, test_values)
                    deleted_fonts += 1
                else:
                    cursor.execute(test_query, test_values)
                    result = cursor.fetchone()
                    if result is None:
                        cursor.execute(add_query, add_values + (contentID,))
                        added_fonts += 1
                    else:
                        cursor.execute(update_query, update_values + (contentID,))
                        updated_fonts += 1
                connection.commit()
                count_books += 1

            cursor.close()
        
        return updated_fonts, added_fonts, deleted_fonts, count_books

    def _dismiss_tiles(self):
        
        debug_print("_dismiss_tiles - self.options[cfg.KEY_TILE_OPTIONS]", self.options[cfg.KEY_TILE_OPTIONS])
        where_clause = ''
        for option in self.options[cfg.KEY_TILE_OPTIONS]:
            where_clause += ", '" + option + "'" if self.options[cfg.KEY_TILE_OPTIONS][option] else ''
        if len(where_clause) > 0:
            if self.options[cfg.KEY_CHANGE_DISMISS_TRIGGER] and self.options[cfg.KEY_CREATE_DISMISS_TRIGGER]:
                trigger_when_clause = "new.Type IN (" + where_clause[1:] + ")"
            where_clause = "WHERE type in (" + where_clause[1:] + ")"
        else:
            return 0
        
        if self.options[cfg.KEY_CHANGE_DISMISS_TRIGGER]:
            if self.options[cfg.KEY_CREATE_DISMISS_TRIGGER]:
                trigger_change_statement = "CREATE TRIGGER Activity_DismissTiles "  \
                                            "AFTER INSERT ON Activity "             \
                                            "FOR EACH ROW "                         \
                                            "WHEN ( " + trigger_when_clause + ") "  \
                                            "BEGIN "                                \
                                                "UPDATE Activity "                  \
                                                "SET Enabled    = 'false' "         \
                                                "WHERE rowid = new.rowid; "         \
                                            "END"
            trigger_delete_statement = "DROP TRIGGER IF EXISTS Activity_DismissTiles"

        import sqlite3 as sqlite
        with closing(sqlite.connect(self.device_database_path())) as connection:
            # return bytestrings if the content cannot the decoded as unicode
            connection.text_factory = lambda x: unicode(x, "utf-8", "ignore")

            update_query = 'UPDATE Activity '           \
                            "SET Enabled = 'false'"    \
                            + where_clause
            debug_print("_dismiss_tiles - update_query", update_query)

            cursor = connection.cursor()

            debug_print("KoboUtilities:_dismiss_tiles - update_query=", update_query)
            cursor.execute(update_query)
            
            if self.options[cfg.KEY_CHANGE_DISMISS_TRIGGER]:
                debug_print("KoboUtilities:_dismiss_tiles - trigger_delete_statement=", trigger_delete_statement)
                cursor.execute(trigger_delete_statement)
                if self.options[cfg.KEY_CREATE_DISMISS_TRIGGER]:
                    debug_print("KoboUtilities:_dismiss_tiles - trigger_change_statement=", trigger_change_statement)
                    cursor.execute(trigger_change_statement)

            connection.commit()

            cursor.close()
        
        return 1

    def _backup_annotation_files(self, books, dest_path):

        annotations_found = 0
        kepubs            = 0
        no_annotations    = 0
        count_books       = 0
#        source_file = self.device.normalize_path(self.device_path +'.kobo/KoboReader.sqlite')
#        shutil.copyfile(source_file, backup_file)
        debug_print("_backup_annotation_files - self.device_path='%s'" % (self.device_path))
        kepub_dir = self.device.normalize_path('.kobo/kepub/')
        annotations_dir = self.device.normalize_path(self.device_path + 'Digital Editions/Annotations/')
        annotations_ext = '.annot'

        for book in books:
            count_books += 1

            for book_path in book.paths:
#                    contentID = self.device.contentid_from_path(path, self.CONTENTTYPE)
                relative_path = book_path.replace(self.device_path, '')
                annotation_file = self.device.normalize_path(annotations_dir + relative_path + annotations_ext)
                debug_print("_backup_annotation_files - kepub title='%s' annotation_file='%s'" % (book.title, annotation_file))
                if relative_path.startswith(kepub_dir):
                    debug_print("_backup_annotation_files - kepub title='%s' book_path='%s'" % (book.title, book_path))
                    kepubs += 1
                elif os.path.exists(annotation_file):
                    debug_print("_backup_annotation_files - book_path='%s'" % (book_path))
                    backup_file = self.device.normalize_path(dest_path + '/'+ relative_path + annotations_ext)
                    debug_print("_backup_annotation_files - backup_file='%s'" % (backup_file))
                    d, p = os.path.splitdrive(backup_file)
                    debug_print("_backup_annotation_files - d='%s' p='%s'" % (d, p))
                    backup_path = os.path.dirname(unicode(backup_file))
                    try:
                        os.makedirs(backup_path)
                    except OSError:
                        debug_print("_backup_annotation_files - path exists: backup_path='%s'" % (backup_path))
                        pass
                    shutil.copyfile(annotation_file, backup_file)
                    annotations_found += 1
                else:
                    debug_print("_backup_annotation_files - book_path='%s'" % (book_path))
                    no_annotations += 1

        debug_print("Backup summary: annotations_found=%d, no_annotations=%d, kepubs=%d Total=%d" % (annotations_found, no_annotations, kepubs, count_books))

        return (annotations_found, no_annotations, kepubs, count_books)


    def convert_kobo_date(self, kobo_date):
        from calibre.utils.date import parse_date

        try:
            converted_date = datetime.strptime(kobo_date, "%Y-%m-%dT%H:%M:%S.%f")
            debug_print('convert_kobo_date - "%Y-%m-%dT%H:%M:%S.%f"')
        except:
            try:
                converted_date = datetime.strptime(kobo_date, "%Y-%m-%dT%H:%M:%S%+00:00")
                debug_print('convert_kobo_date - "%Y-%m-%dT%H:%M:%S+00:00"')
            except:
                try:
                    converted_date = datetime.strptime(kobo_date.split('+')[0], "%Y-%m-%dT%H:%M:%S")
                    debug_print('convert_kobo_date - "%Y-%m-%dT%H:%M:%S"')
                except:
                    try:
                        converted_date = datetime.strptime(kobo_date.split('+')[0], "%Y-%m-%d")
                        debug_print('convert_kobo_date - "%Y-%m-%d"')
                    except:
                        try:
                            converted_date = parse_date(kobo_date, assume_utc=True)
                            debug_print('convert_kobo_date - parse_date')
                        except:
                            try:
                                converted_date = time.gmtime(os.path.getctime(self.path))
                            except:
                                converted_date = time.gmtime()
        return converted_date


    def device_database_path(self):
        return self.device.normalize_path(self.device._main_prefix + '.kobo/KoboReader.sqlite')


    def show_help1(self):
        self.show_help()

    def show_help(self, anchor=''):
        debug_print("show_help - anchor=", anchor)
        # Extract on demand the help file resource
        def get_help_file_resource():
            # We will write the help file out every time, in case the user upgrades the plugin zip
            # and there is a later help file contained within it.
            HELP_FILE = 'KoboUtilities_Help.html'
            file_path = os.path.join(config_dir, 'plugins', HELP_FILE)
            file_data = self.load_resources(HELP_FILE)[HELP_FILE]
            with open(file_path,'w') as f:
                f.write(file_data)
            return file_path
        debug_print("show_help - anchor=", anchor)
        url = 'file:///' + get_help_file_resource()
        url = QUrl(url)
        if not (anchor or anchor == ''):
            url.setFragment(anchor)
        open_url(url)

