#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <dlfcn.h>
#include <pthread.h>
#include <errno.h>

#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/epoll.h>
#include <sys/types.h>

#include <linux/fb.h>
#include <linux/mxcfb.h>

#include <asm-generic/ioctl.h>


static void initialize() __attribute__((constructor));
static void cleanup() __attribute__((destructor));
int ioctl(int filp, unsigned long cmd, unsigned long arg);

static const char ctlPipe[] = "/tmp/invertScreen";
static const char logPath[] = "/tmp/screenInvertLog";
static int (*ioctl_orig)(int filp, unsigned long cmd, unsigned long arg) = NULL;
static pthread_t cmdReaderThread = 0;
static bool inversionActive = false;
static struct mxcfb_update_data fullUpdRegion, workaroundRegion;
static int fb0fd = 0;
static FILE *logFP;

static void forceUpdate()
{
    ioctl(fb0fd , MXCFB_SEND_UPDATE, (unsigned long int)&fullUpdRegion);
}

static void *cmdReader(void *arg) //reader thread
{
    char input;
    int fd = open(ctlPipe, O_NONBLOCK);
    
    int epollFD = epoll_create(1);
    struct epoll_event readEvent;
    readEvent.events = EPOLLIN;
    readEvent.data.fd = fd;
    epoll_ctl(epollFD, EPOLL_CTL_ADD, fd, &readEvent);
    
    while(1)
    {
        int err = epoll_wait(epollFD, &readEvent, 1, -1);
        if(err > 0)
        {
            int bytesRead = read(fd, &input, sizeof(input));
            
            if(bytesRead == 0) //writing application left the pipe -> reopen
            {
                close(fd);
                fd = open(ctlPipe, O_NONBLOCK);
                readEvent.data.fd = fd;
                epoll_ctl(epollFD, EPOLL_CTL_ADD, fd, &readEvent);
                continue;
            }
            
            switch(input)
            {
                case 't': //toggle
                    inversionActive = !inversionActive;
                    printf("ScreenInverter: Toggled\n");
                    forceUpdate();
                    break;
                case 'y': //yes
                    inversionActive = true;
                    printf("ScreenInverter: Inversion on\n");
                    forceUpdate();
                    break;
                case 'n': //no
                    inversionActive = false;
                    printf("ScreenInverter: Inversion off\n");
                    forceUpdate();
                    break;
                case 10: //ignore linefeed
                    break;
                default:
                    printf("ScreenInverter: Unknown command!\n");
                    break;
            }
            fflush(stdout);
        }
    }
    
    return NULL;
}

static void initialize()
{
    ioctl_orig = (int (*)(int filp, unsigned long cmd, unsigned long arg)) dlsym(RTLD_NEXT, "ioctl");
    
    char execPath[32];
    int end = readlink("/proc/self/exe", execPath, 31);
    execPath[end] = 0;
    
    if(strncmp(execPath, "/usr/local/Kobo/nickel", 31))
        return;
    
    if((logFP = freopen(logPath, "w" ,stdout))==NULL)
    {
        printf("ScreenInverter: Couldn't open logfile, logging to stdout");
    }
    
    printf("ScreenInverter: Hooked to nickel!\n");
    
    
    if ((fb0fd = open("/dev/fb0", O_RDWR)) == -1) 
        printf("ScreenInverter: Error opening /dev/fb0!");
    
    fullUpdRegion.update_marker = 999;
    fullUpdRegion.update_region.top = 0;
    fullUpdRegion.update_region.left = 0;
    fullUpdRegion.update_region.width = 1024;
    fullUpdRegion.update_region.height = 758; //full screen on the glo
    fullUpdRegion.waveform_mode = WAVEFORM_MODE_AUTO;
    fullUpdRegion.update_mode = UPDATE_MODE_PARTIAL;
    fullUpdRegion.temp = TEMP_USE_AMBIENT;
    fullUpdRegion.flags = 0;
    
    workaroundRegion.update_marker = 998;
    workaroundRegion.update_region.top = 0;
    workaroundRegion.update_region.left = 0;
    workaroundRegion.update_region.width = 1;
    workaroundRegion.update_region.height = 1; //1px in the top right(!) corner
    workaroundRegion.waveform_mode = WAVEFORM_MODE_AUTO;
    workaroundRegion.update_mode = UPDATE_MODE_PARTIAL;
    workaroundRegion.temp = TEMP_USE_AMBIENT;
    workaroundRegion.flags = 0;
    
    remove(ctlPipe); //just to be sure
    mkfifo(ctlPipe, 0600);
    fflush(stdout);
    pthread_create(&cmdReaderThread, NULL, cmdReader, NULL);
}

static void cleanup()
{
    if(cmdReaderThread)
    {
        pthread_cancel(cmdReaderThread);
        remove(ctlPipe);
        close(fb0fd);
        printf("ScreenInverter: Shut down!\n");
        fclose(logFP);
    }
}

int ioctl(int filp, unsigned long cmd, unsigned long arg)
{
    int ret;
    
    if(inversionActive & (cmd == MXCFB_SEND_UPDATE))
    {
        ioctl_orig(filp, MXCFB_SEND_UPDATE, (long unsigned)&workaroundRegion);
        //necessary because there's a bug in the driver (or i'm doing it wrong):
        //  i presume the device goes into some powersaving moden when usb und wifi are not used (great for debugging ;-) )
        //  it takes about 10sec after the last touch to enter this mode. after that it is necessary to issue a screenupdate 
        //  without inversion flag, otherwise it will ignore the inversion flag and draw normally (positive).
        //  so i just update a 1px region in the top-right corner, this costs no time and the pixel should be behind the bezel anyway.
        
        struct mxcfb_update_data *region = (struct mxcfb_update_data *)arg;
        region->flags ^= EPDC_FLAG_ENABLE_INVERSION;
        ret = ioctl_orig(filp, cmd, arg);
        region->flags ^= EPDC_FLAG_ENABLE_INVERSION; //not necessary for nickel, but other apps might not rewrite the request everytime
    }
    else
        ret = ioctl_orig(filp, cmd, arg);
    
    //if(!strncmp(nameBuf, "/dev/fb0", 255))
    //    printf("IOCTL! dev: %s[%d], cmd: %lu, arg: %lu, ret: %d\n", nameBuf, filp, _IOC_NR(cmd), arg, ret);
    //if(ret < 0) printf("IOCTLfailed: %s!\n", strerror(errno));
    //fflush(stdout);
    return ret;
}

/* Code to get the device file from the file descriptor "filp"

    char nameBuf[256];
    char linkPath[256];
    sprintf(linkPath, "/proc/self/fd/%d", filp);
    int end = readlink(linkPath, nameBuf, 255);
    nameBuf[end] = 0;
    if(!strncmp(nameBuf, "/dev/fb0", 255)) { modify request}
    
shouldn't be necessary as IOCTLs *should* be unique, and unambiguously identifiable by "cmd"
*/
