#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
from PyQt4.Qt import Qt, QApplication, QMenu, QToolButton
from calibre.gui2 import error_dialog
from calibre.gui2.actions import InterfaceAction

from calibre_plugins.temp_marker.common_utils import set_plugin_icon_resources, get_icon, \
                                                     create_menu_action_unique

PLUGIN_ICONS = ['images/temp_marker.png']

class TempMarkerAction(InterfaceAction):

    name = 'Temp Marker'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Temp Marker', None, None, None)
    popup_type = QToolButton.MenuButtonPopup
    action_type = 'current'
    dont_add_to = frozenset(['context-menu-device'])

    def genesis(self):
        self.is_library_selected = True
        self.marked_ids = set()
        self.menu = QMenu(self.gui)

        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        self.rebuild_menus()

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.toolbar_triggered)
        self.menu.aboutToShow.connect(self.about_to_show_menu)

    def library_changed(self, db):
        # We need to reapply keyboard shortcuts after switching libraries
        self.rebuild_menus()

    def location_selected(self, loc):
        self.is_library_selected = loc == 'library'

    def about_to_show_menu(self):
        self.rebuild_menus()

    def rebuild_menus(self):
        m = self.menu
        m.clear()
        create_menu_action_unique(self, m, 'Mark selected', 'plus.png',
                                  triggered=partial(self.mark_selected, add=True))
        create_menu_action_unique(self, m, 'Mark books by selected author(s)', 'plus.png',
                                  triggered=partial(self.mark_selected, add=True, same_author=True, same_series=False ))
        create_menu_action_unique(self, m, 'Mark books in selected series', 'plus.png',
                                  triggered=partial(self.mark_selected, add=True, same_author=False, same_series=True ))
        m.addSeparator()
        create_menu_action_unique(self, m, 'Show marked (%d)'%len(self.marked_ids),
                                  'search.png', shortcut_name='Show marked',
                                   unique_name='Show marked', triggered=self.show_marked)
        m.addSeparator()
        create_menu_action_unique(self, m, 'Unmark selected', 'minus.png',
                                  triggered=partial(self.mark_selected, add=False))
        create_menu_action_unique(self, m, 'Unmark books by selected author(s)', 'minus.png',
                                  triggered=partial(self.mark_selected, add=False, same_author=True, same_series=False ))
        create_menu_action_unique(self, m, 'Unmark books in selected series', 'minus.png',
                                  triggered=partial(self.mark_selected, add=False, same_author=False, same_series=True ))
        m.addSeparator()
        create_menu_action_unique(self, m, 'Clear all marked', 'clear_left.png',
                                  triggered=self.clear_all_marked)
        m.addSeparator()
        create_menu_action_unique(self, m, _('&Customize plugin')+'...', 'config.png',
                                  shortcut=False, triggered=self.show_configuration)
        self.gui.keyboard.finalize()

    def toolbar_triggered(self):
        mods = QApplication.keyboardModifiers()
        if mods & Qt.ControlModifier or mods & Qt.ShiftModifier:
            self.show_marked()
        else:
            self.mark_selected(add=True)

    def mark_selected(self, add=True, same_author=False, same_series=False):
        if not self.is_library_selected:
            return
        self.db = self.gui.current_db
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, 'No rows selected',
                                'You must select one of more books to perform this action.',
                                show=True)
        ids = set(self.gui.library_view.get_selected_ids())
        extraids = set()

        if same_author:
            # Get all the books for each author selected
            # First build a unique list of selected authors
            unique_authors = set()
            for book_id in ids:
                # Get the current metadata for this book from the db
                mi = self.db.get_metadata(book_id, index_is_id=True, get_cover=False)
                authors = mi.get('authors', None)
                if len(authors):
                    unique_authors |= set(authors)
            # Now find all the books for each author
            for author in unique_authors:
                search = 'authors:"=' + author + '"'
                author_book_ids = self.db.search_getting_ids(search, '')
                extraids |= set(author_book_ids)

        if same_series:
            # Get all the books in each series selected
            unique_series = set()
            for book_id in ids:
                # Get the current metadata for this book from the db
                mi = self.db.get_metadata(book_id, index_is_id=True, get_cover=False)
                if mi.series is not None:
                    unique_series.add(mi.series)
            # Now find all the books for each series
            for series in unique_series:
                search = 'series:"=' + series + '"'
                series_book_ids = self.db.search_getting_ids(search, '')
                extraids |= set(series_book_ids)

        # add the new books to the set
        ids |= extraids
        if add:
            self.marked_ids.update(ids)
        else:
            self.marked_ids.difference_update(ids)
        self.gui.library_view.model().db.set_marked_ids(self.marked_ids)

    def show_marked(self):
        self.gui.search.set_search_string('marked:true')

    def clear_all_marked(self):
        self.marked_ids.clear()
        self.gui.library_view.model().db.set_marked_ids([])
        if unicode(self.gui.search.text()).startswith('marked:'):
            self.gui.search.clear()

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
