//----------------------------------------------------
// screen2pgm - save framebuffer to portable greymap
// (C)opyright 2012 by NuPogodi, with MIT license:
// http://www.opensource.org/licenses/mit-license.php
//----------------------------------------------------
// Compiled with tcc (tiny c compiler)
// Tested on KDX (~0.80s), K3(~0.39s), K5(~0.02s)
//----------------------------------------------------
#include <stdio.h>		// printf
#include <stdlib.h>		// system
#include <string.h>		// strcat, strcpy
#include <fcntl.h>		// open, close, read, write
#include <time.h>		// time
#include <sys/ioctl.h>	// ioctl
#include <linux/fb.h>	// screeninfo
//------------ global definitions -----------
#define PACKAGE	"screen2pgm"
//------------- type definitions ------------
typedef unsigned char u8;
typedef unsigned int u32;
//----------- function prototypes -----------
void initialize();
char* getexename(char *, size_t);
void fb_convert();
void fb2pgm(char *);
void writelog(char *, char *, float);
//------------ global variables --------------
int FB;	// fb0 file descriptor
u32 WI;	// width / pixels
u32 HE;	// height / pixels
u8 BPP;	// bits per pixel
u32 WB;	// bytes in one horizontal line
u32 SZ;	// raw bytes in fb0
//============================================
void initialize()
{
	struct fb_var_screeninfo info;
	FB = open("/dev/fb0", O_RDONLY);
	ioctl(FB, FBIOGET_VSCREENINFO, &info);
	WI = info.xres_virtual;
	HE = info.yres;
	BPP = info.bits_per_pixel;
	WB = info.xres*BPP/8;
	SZ = WB * HE;
}
//--------------------------------------------
char* getexename(char* buf, size_t size)
{
	char linkname[64]; // /proc/<pid>/exe
	pid_t pid=getpid();
	if (snprintf(linkname, sizeof(linkname), "/proc/%i/exe", pid)<0) abort();
	int ret=readlink(linkname, buf, size);
	if ((ret==-1) || (ret>=size)) return NULL;
	buf[ret]=0;
	return buf;
}
//--------------- total convert --------------
void fb_convert()
{
	FILE *out;
	clock_t st = clock();
	time_t now = time(NULL);
	char name[15], filename[255], command[255], path2exe[255];
	if (getexename(path2exe, 255)==NULL) return;
	strftime(name, 15, "%Y%m%d%H%M%S", localtime(&now));
	strcpy(command, path2exe);
	sprintf(filename, "%s/%s.pgm", dirname(command), name);
	//-------- conversion -------
	fb2pgm(filename);
	//----------- end -----------
	close(FB);
	sprintf(command, "bzip2 %s", filename);
	system(command);
	sprintf(command, "%s.log", path2exe);
	writelog(command, filename, (float)(clock()-st)/CLOCKS_PER_SEC);
}
//------------------ save log ----------------
void writelog(char *s, char *c, float t)
{
	FILE *log = fopen(s, "w");
	fprintf(log, "W*H\t= %d*%d\nBPP\t= %d\nFILE\t= %s\nCPU\t= %.2f", WB/BPP*8, HE, BPP, c, t);
	fclose(log);
}
//--------------- convert to pgm -------------
void fb2pgm(char *filename)
{
	FILE *f = fopen(filename, "w+b");
	u32 i, j;
	u8 c, hi[256], lo[256], line[810]; // WB
	sprintf(hi, "P5\n# Created by %s\n%d %d\n%d\n", PACKAGE, WB/BPP*8, HE, (1<<BPP)-1);
	fwrite(&hi, strlen(hi), 1, f);
	// write image data
	if (BPP==8)	// 8bpp @ K4 (not tested), K5
		for (i=0; i<HE; i++) {
			read(FB, &line, WI);
			fwrite(&line, WB, 1, f);
			}
	else {		// 4bpp @ K2, K3, KDX > slow 'cause of the 4to8bpp conversion
		for (i=0; i<256; i++) {
			hi[i] =  (~i) & 0xF0;
			lo[i] = ((~i) & 0x0F) << 4;
			}
		for (i=0; i<HE; i++) {	// line-by-line
			read(FB, &line, WB);
			for (j=0; j<WB; j++) {
				fwrite(&hi[line[j]], 1, 1, f);
				fwrite(&lo[line[j]], 1, 1, f);
				}
			}
		}
	fclose(f);
}
//-------------------- main -----------------
int main (int argc, char **argv)
{
	initialize();
	fb_convert();
	return 0;
}
