//-----------------------------------------------------
// screen2bmp - to save 4bpp-framebuffer as bitmap
// (C)opyright 2012 by NuPogodi, with MIT license:
// http://www.opensource.org/licenses/mit-license.php
//-----------------------------------------------------
// Compiled with tcc (tiny c compiler)
// Tested on KDX (~0.04s), K3(~0.04s), K5(~0.01s)
//-----------------------------------------------------
#include <stdio.h>		// printf
#include <stdlib.h>		// system
#include <string.h>		// strcat, strcpy
#include <fcntl.h>		// open, close, read, write
#include <time.h>		// time
#include <sys/ioctl.h>	// ioctl
#include <linux/fb.h>	// screeninfo
//-------------- type definitions ------------
typedef unsigned char u8;
typedef unsigned int u32;
//------------ function prototypes -----------
void initialize();
char* getexename(char *, size_t);
void fb_convert();
void fb2bmp(char *);
void writelog(char *, char *, float);
//------------- global variables -------------
int FB;	// fb0 file descriptor
u32 WI;	// width / pixels
u32 HE;	// height / pixels
u8 BPP;	// bits per pixel
u32 WB;	// bytes in one horizontal line
u32 SZ;	// raw bytes in fb0
//============================================
void initialize()
{
	struct fb_var_screeninfo info;
	FB = open("/dev/fb0", O_RDONLY);
	ioctl(FB, FBIOGET_VSCREENINFO, &info);
	WI = info.xres_virtual;
	HE = info.yres;
	BPP = info.bits_per_pixel;
	WB = info.xres*BPP/8;
	SZ = WB*HE;
}
//--------------------------------------------
char* getexename(char* buf, size_t size)
{
	char linkname[64]; // /proc/<pid>/exe
	pid_t pid=getpid();
	if (snprintf(linkname, sizeof(linkname), "/proc/%i/exe", pid)<0) abort();
	int ret=readlink(linkname, buf, size);
	if ((ret==-1) || (ret>=size)) return NULL;
	buf[ret]=0;
	return buf;
}
//--------------- total convert --------------
void fb_convert()
{
	FILE *out;
	clock_t st = clock();
	time_t now = time(NULL);
	char name[15], filename[255], command[255], path2exe[255];
	if (getexename(path2exe, 255)==NULL) return;
	strftime(name, 15, "%Y%m%d%H%M%S", localtime(&now));
	strcpy(command, path2exe);
	sprintf(filename, "%s/%s.bmp", dirname(command), name);
	//-------- conversion -------
	fb2bmp(filename);
	//----------- end -----------
	close(FB);
	sprintf(command, "bzip2 %s", filename);
	system(command);
	sprintf(command, "%s.log", path2exe);
	writelog(command, filename, (float)(clock()-st)/CLOCKS_PER_SEC);
}
//----------------- save log -----------------
void writelog(char *s, char *c, float t)
{
	FILE *log = fopen(s, "w");
	fprintf(log, "W*H\t= %d*%d\nBPP\t= %d\nFILE\t= %s\nCPU\t= %.2f", WB*8/BPP, HE, BPP, c, t);
	fclose(log);
}
//--------------- convert to bmp -------------
void fb2bmp(char *filename) // for K2, K3, KDX
{
	FILE *f = fopen(filename, "w+b");
	int i;
	u8 j, l;
	u32 w=WB*8/BPP;
	u32 k=4*(1<<BPP)+54;
	u32 sh=WB*HE+k;
	u8 header[54] = { 
		0x42, 0x4D, (sh&0xFF), (sh&0xFF00)>>8, (sh&0xFF0000)>>16, sh>>24, 0, 0,
		0, 0, (k&0xFF), (k&0xFF00)>>8, 0, 0, 40, 0,
		0, 0, (w&0xFF), (w&0xFF00)>>8, 0, 0, (HE&0xFF), (HE&0xFF00)>>8,
		0, 0, 1, 0, BPP, 0, 0, 0,
		0, 0, (SZ&0xFF), (SZ&0xFF00)>>8, (SZ&0xFF0000)>>16, SZ>>24,
		0x87, 0x19, 0, 0, 0x87, 0x19, 0, 0,
		(1<<BPP)&0xFF, ((1<<BPP)&0xFF00)>>8};
	fwrite(&header, 54, 1, f);
	l = 0;
	if (BPP==4) for (i=15; i>=0; i--) {	// 4bpp-palette
			j = (i*16+i);
			fwrite(&j, 1, 1, f);
			fwrite(&j, 1, 1, f);
			fwrite(&j, 1, 1, f);
			fwrite(&l, 1, 1, f);
			}
	else for (i=0; i<256; i++) {	// 8bpp-palette
			fwrite(&i, 1, 1, f);
			fwrite(&i, 1, 1, f);
			fwrite(&i, 1, 1, f);
			fwrite(&l, 1, 1, f);
			}
	// write image data with vertical flip
	u8 line[810];
	u32 str;
	if (BPP==4) str=WB; else str=WI*BPP/8;
	for (i=HE-1; i>=0; i--) {
		lseek(FB, str*i, SEEK_SET);
		read(FB, &line, WB);
		fwrite(&line, WB, 1, f);
		}
	fclose(f);
}
//-------------------- main ------------------
int main (int argc, char **argv)
{
	initialize();
	fb_convert();
	return 0;
}
