//------------------------------------------------------------
// screen2tiff - to save 4bpp-framebuffer as uncompressed TIFF
// (C)opyright 2012 by NuPogodi, with MIT license:
// http://www.opensource.org/licenses/mit-license.php
//------------------------------------------------------------
// Compiled with tcc (tiny c compiler)
// Tested on KDX (~0.03s), K3(~0.02s), K5(~0.02s)
//------------------------------------------------------------
#include <stdio.h>		// printf
#include <stdlib.h>		// system
#include <string.h>		// strcat, strcpy
#include <fcntl.h>		// open, close, read, write
#include <time.h>		// time
#include <sys/ioctl.h>	// ioctl
#include <linux/fb.h>	// screeninfo
//-------------- type definitions ------------
typedef unsigned char u8;
typedef unsigned int u32;
//------------ function prototypes -----------
void initialize();
char* getexename(char *, size_t);
void fb_convert();
void fb2tiff(char *);
void writelog(char *, char *, float);
//------------- global variables -------------
int FB;	// fb0 file descriptor
u32 WI;	// width / pixels
u32 HE;	// height / pixels
u8 BPP;	// bits per pixel
u32 WB;	// bytes in one horizontal line
u32 SZ;	// raw bytes in fb0
//============================================
void initialize()
{
	struct fb_var_screeninfo info;
	FB = open("/dev/fb0", O_RDONLY);
	ioctl(FB, FBIOGET_VSCREENINFO, &info);
	WI = info.xres_virtual;
	HE = info.yres;
	BPP = info.bits_per_pixel;
	WB = info.xres*BPP/8;
	SZ = WB*HE;
}
//--------------------------------------------
char* getexename(char* buf, size_t size)
{
	char linkname[64]; // /proc/<pid>/exe
	pid_t pid=getpid();
	if (snprintf(linkname, sizeof(linkname), "/proc/%i/exe", pid)<0) abort();
	int ret=readlink(linkname, buf, size);
	if ((ret==-1) || (ret>=size)) return NULL;
	buf[ret]=0;
	return buf;
}
//--------------- total convert --------------
void fb_convert()
{
	FILE *out;
	clock_t st = clock();
	time_t now = time(NULL);
	char name[15], filename[255], command[255], path2exe[255];
	if (getexename(path2exe, 255)==NULL) return;
	strftime(name, 15, "%Y%m%d%H%M%S", localtime(&now));
	strcpy(command, path2exe);
	sprintf(filename, "%s/%s.tif", dirname(command), name);
	//-------- conversion -------
	fb2tiff(filename);
	//----------- end -----------
	close(FB);
	sprintf(command, "bzip2 %s", filename);
	system(command);
	sprintf(command, "%s.log", path2exe);
	writelog(command, filename, (float)(clock()-st)/CLOCKS_PER_SEC);
}
//----------------- save log -----------------
void writelog(char *s, char *c, float t)
{
	FILE *log = fopen(s, "w");
	fprintf(log, "W*H\t= %d*%d\nBPP\t= %d\nFILE\t= %s\nCPU\t= %.2f", WB*8/BPP, HE, BPP, c, t);
	fclose(log);
}
//-------------- convert to tiff -------------
void fb2tiff(char *filename)
{
	FILE *f = fopen(filename, "w+b");
	u32 p1, p2;
	u32 p3 = SZ+8;
	u8 header[8] = { 0x49, 0x49, 0x2A, 0, p3&0xFF, (p3&0xFF00)>>8, (p3&0xFF0000)>>16, p3>>24 };
	fwrite(&header, 8, 1, f);
	u32 block=WB;
	while (block<8000) block+=WB;
	if ((2*block-16000-WB)>=0) block-=WB;

	u8 line[810], offsets[500], bsizes[500];
	u32 l=0;
	u32 i=0;
	u32 j=8;
	u32 k=0;	// to count blocks
	p3 = block/WB;	// lines per block
	u32 w = WI*BPP/8;
	for (i=0; i<HE; i++) {
		read(FB, &line, w);
		fwrite(&line, WB, 1, f);
		l+=1;
		if (l==p3) {
			offsets[k] = j&0xFF; offsets[k+1] = (j&0xFF00)>>8; 
			offsets[k+2] = (j&0xFF0000)>>16; offsets[k+3] = j>>24;
			j+=block;
			bsizes[k] = block&0xFF; bsizes[k+1] = (block&0xFF00)>>8; 
			bsizes[k+2] = (block&0xFF0000)>>16; bsizes[k+3] = block>>24;
			k+=4;
			l=0;
			}
		}
	if (l>0) {
		offsets[k] = j&0xFF; offsets[k+1] = (j&0xFF00)>>8; 
		offsets[k+2] = (j&0xFF0000)>>16; offsets[k+3] = j>>24;
		j=l*WB;
		bsizes[k] = j&0xFF; bsizes[k+1] = (j&0xFF00)>>8; 
		bsizes[k+2] = (j&0xFF0000)>>16; bsizes[k+3] = j>>24;
		k+=4;
		}
	p1 = k/4;	// blocks
	p2 = SZ+206;	// points to...
	w = WB*8/BPP;
	// here should start footer
	u8 footer1[118] = { 	
			0x10,0,0,1,	3,0,1,0,	0,0,	w&0xFF, w>>8,
			0,0,1,1,	3,0,1,0,	0,0, 	HE&0xFF, HE>>8,
			0,0,2,1,	3,0,1,0,	0,0,	BPP&0xFF, BPP>>8,
			0,0,3,1,	3,0,1,0,	0,0,	1&0xFF, 0,	// uncompressed=1
			0,0,6,1,	3,0,1,0,	0,0,3,0,
			0,0,0x11,1,	4,0,
			p1&0xFF, (p1&0xFF00)>>8, (p1&0xFF0000)>>16, p1>>24, // blocks
			p2&0xFF, (p2&0xFF00)>>8, (p2&0xFF0000)>>16, p2>>24,
			0x12,1,3,0,	1,0,0,0,	1,0,0,0,	0x15,1,3,0,
			1,0,0,0,	1,0,0,0,	0x16,1,3,0,	1,0,0,0,
			p3&0xFF, (p3&0xFF00)>>8, (p3&0xFF0000)>>16, p3>>24, // lines per block
			0x17, 1, 4, 0,
			p1&0xFF, (p1&0xFF00)>>8, (p1&0xFF0000)>>16, p1>>24 };
	p1 = SZ+k+206;
	p2 = p1+k;
	p3 = p2+8;
	u8 produced[255];	// use less than 256 chars
	strcpy(produced, "screen2tiff");
	u8 footer2[60] = {
			p1&0xFF, (p1&0xFF00)>>8, (p1&0xFF0000)>>16, p1>>24, 
			0x1A,1,5,0,		1,0,0,0,
			p2&0xFF, (p2&0xFF00)>>8, (p2&0xFF0000)>>16, p2>>24,
			0x1B,1,5,0,		1,0,0,0,
			p3&0xFF, (p3&0xFF00)>>8, (p3&0xFF0000)>>16, p3>>24,
			0x1C,1,3,0,		1,0,0,0,	1,0,0,0,	0x28,1,3,0,
			1,0,0,0,		2,0,0,0,	0x31,1,2,0,
			(strlen(produced)+1)&0xFF, 0, 0, 0 };
	p2 += 16;
	p1 = p2+strlen(produced)+1;
	u32 pal = 3*(1<<BPP);	// length of palette
	u8 footer3[20] = {
			p2&0xFF, (p2&0xFF00)>>8, (p2&0xFF0000)>>16, p2>>24,
			0x40, 1, 3, 0, pal&0xFF, (pal&0xFF00)>>8, (pal&0xFF0000)>>16, pal>>24,
			p1&0xFF, (p1&0xFF00)>>8, (p1&0xFF0000)>>16, p1>>24,
			0, 0, 0, 0 };
	u32 dpi = 166; // use no more than 16 bit integer
	u8 footer4[16] = { 
			0, 0, dpi&0xFF, dpi>>8,
			0, 0, 1, 0,
			0, 0, dpi&0xFF, dpi>>8,
			0, 0, 1, 0 };
	fwrite(&footer1, 118, 1, f);
	fwrite(&footer2, 60, 1, f);
	fwrite(&footer3, 20, 1, f);
	fwrite(&offsets, k, 1, f);
	fwrite(&bsizes, k, 1, f);
	fwrite(&footer4, 16, 1, f);
	fwrite(&produced, strlen(produced), 1, f);
	fwrite("\0", 1, 1, f);
	int z;
	if (BPP==4) {	// 4bpp-palette
		for (j=0; j<3; j++)
		for (z=15; z>=0; z--) {
			i=z*16+z;
			fwrite(&i, 1, 1, f);
			fwrite(&i, 1, 1, f);
			}
		}
	else {		// 8bpp-palette
		for (j=0; j<3; j++)
		for (i=0; i<256; i++) {
			fwrite(&i, 1, 1, f);
			fwrite(&i, 1, 1, f);
			}
		}
	fclose(f);
}
//-------------------- main ------------------
int main (int argc, char **argv)
{
	initialize();
	fb_convert();
	return 0;
}
