#!/usr/bin/env bash

#
# Creates a file.2x1.pdf from the input file.pdf
# (Note: the "author" and "title" strings must be ASCII, no UTF-8 allowed)
#
# Needs ...
# awk : gawk
# sed : sed
# pdfinfo, pdftops : poppler-tools or poppler-utils or xpdf-tools or xpdf-utils
# pstops : psutils
# ps2pdf : ghostscript or ghostscript-library
# acroread (optional) : AdobeReader_enu or adobereader-enu or acroread
#

PDF2PS="pdftops"

if [[ ${#} -gt 0 && "${1}" = "-a" ]] ; then
    PDF2PS="acroread"
    shift
fi

if [ ${#} -lt 1 -o ${#} -gt 3 ] ; then
    echo -e "\nUsage: $0 [-a] file.pdf [author] [title]"
    echo -e "  -a     : utilise acroread (default: use pdftops)"
    echo -e "  author : file's author (default: just someone)"
    echo -e "  title  : file's title (default: the basename of the input file.pdf)"
    echo -e "Creates a file.2x1.pdf from the input file.pdf"
    echo -e "Note: needs awk, sed, pdfinfo, pdftops, pstops, ps2pdf, optionally acroread\n"
    exit 1
fi

PDF_1x1="${1}" # input PDF file

if [ ! -r "${PDF_1x1}" ] ; then
    echo -e "\n$0 : input file ${PDF_1x1} does NOT exist!\n"
    exit 2
fi

PDF_2x1="${PDF_1x1%.[pP][dD][fF]}.2x1.pdf" # output 2x1 "portrait" PDF file
PS_1x1="${PDF_1x1}.1x1.ps" # temporary 1x1 "portrait" PS file
PS_2x1="${PDF_1x1}.2x1.ps" # temporary 2x1 "portrait" PS file

AUTHOR="${2:-Silesius Anonymus}" # file's author

TITLE="${3}" # file's title

if [ -z "${TITLE}" ] ; then
    TITLE="${PDF_1x1}" # use the input PDF file name
    TITLE="${TITLE##*/}" # remove and "subdirectory names"
    TITLE="${TITLE%%.*}" # remove any "file extensions"
fi

# initial cleanup
rm -rf "${PS_1x1}" "${PS_2x1}" "${PDF_2x1}"

# calculate the new page size
TMP="`pdfinfo \"${PDF_1x1}\"|awk '/^Page size:/ {printf \"%d %d\",0.5+$3,0.5+$5;exit}' -`"
declare -i PTS_X=${TMP% *} # X size in [pts] units
declare -i PTS_Y=${TMP#* } # Y size in [pts] units
if [ ${PTS_X} -lt 72 -o ${PTS_Y} -lt 72 ] ; then PTS_X=420; PTS_Y=595; fi
(( PTS_X*=2, PTS_Y*=2 )) # make it 4 times bigger
# (( PTS_X*=2, PTS_Y*=2 )) # make it 16 times bigger

# convert and resize the original PDF file into a 1x1 "portrait" PS
if [ "${PDF2PS}" = "acroread" ] ; then
    acroread -toPostScript -saveVM -optimizeForSpeed -binary -markupsOn -expand -size ${PTS_X}x${PTS_Y} -pairs "${PDF_1x1}" "${PS_1x1}"
else
    pdftops -paperw ${PTS_X} -paperh ${PTS_Y} -expand "${PDF_1x1}" "${PS_1x1}"
fi

# add / modify the "%%Title:" and the "%%For:" strings in the 1x1 "portrait" PS
sed -i -e '{
    1,/^%%EndComments/ {
        /^%%Title:.*/ d
        /^%%For:.*/ d
        s/^%%EndComments/%%Title: '"${TITLE}"'\n%%For: '"${AUTHOR}"'\n&/
    }
}' "${PS_1x1}"

# convert the 1x1 "portrait" PS into a 2x1 "portrait" PS
# pstops -w${PTS_X} -h${PTS_Y} "2:0@0.5(0,0)+1@0.5(0.5w,0)" "${PS_1x1}" "${PS_2x1}" 2> /dev/null
pstops -w${PTS_X} -h${PTS_Y} "2:0@0.499(1,0)+1@0.499(0.5w,0)" "${PS_1x1}" "${PS_2x1}" 2> /dev/null

# convert the 2x1 "portrait" PS into a 2x1 "portrait" PDF
ps2pdf -dDEVICEWIDTHPOINTS=${PTS_X} -dDEVICEHEIGHTPOINTS=$(( (PTS_Y+1)/2 )) "${PS_2x1}" "${PDF_2x1}"

# final cleanup
rm -rf "${PS_1x1}" "${PS_2x1}"

# echo -e "Output file ${PDF_2x1} is ready!\n"

# end of file pdf2x1pdf.sh by Silesius Anonymus
