package de.pkloes.KindleStrip;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

public class Tools {
    final private static long MOBI_FORMAT_IDENTIFIER = 0x424f4f4b4d4f4249L;
    final private static int MOBI_SRCS_RECORD_IDENTIFIER = 0x53524353;
    final private static int MOBI_EXTH_RECORD_IDENTIFIER = 0x45585448;
    final private static int MOBI_EXTH_BOUNDARY_SECTION_IDENTIFIER = 0x79;

    public static void stripSRCS(File mobi, File stripped) throws IOException, InvalidMobiFileException {
        FileChannel fich = null;
        FileChannel foch = null;

        try {
            FileInputStream fin = new FileInputStream(mobi);
            fich = fin.getChannel();
            RandomAccessFile fout = new RandomAccessFile(stripped, "rw");
            foch = fout.getChannel();
            fout.setLength(0);

            ByteBuffer bul = ByteBuffer.allocateDirect(8);
            ByteBuffer bui = ByteBuffer.allocateDirect(4);

            fich.read(bul, 0x3c);

            if((getUInt(bul, 0) << 32 | getUInt(bul, 4)) != MOBI_FORMAT_IDENTIFIER) {
                throw new InvalidMobiFileException("Unsupported Mobi Format");
            }

            bul.clear();
            fich.read(bul, 76);
            int record_count = bul.getChar(0);
            long header_offset = getUInt(bul, 2);
            bul.clear();

            ByteBuffer data = ByteBuffer.allocateDirect((int) (header_offset + 0xe8));
            fich.read(data, 0);

            long srcs_record_number = getUInt(data, (int) (header_offset + 0xe0));
            long srcs_record_count  = getUInt(data, (int)(header_offset + 0xe4));

            if(srcs_record_number == 0xffffffffL || srcs_record_count == 0) {
                foch.transferFrom(fich, 0, fich.size());
                return;
            }

            long strip_start_offset = getUInt(data, (int)((srcs_record_number * 8) + 78));
            long strip_end_offset = getUInt(data, (int)(((srcs_record_number + srcs_record_count) * 8) + 78));
            long strip_length = strip_end_offset - strip_start_offset;

            fich.read(bul, strip_start_offset);
            if(getUInt(bul, 0) != MOBI_SRCS_RECORD_IDENTIFIER) {
                throw new InvalidMobiFileException("Broken SRCS Record");
            }
            bul.clear();

            int new_record_count = (int)(record_count - srcs_record_count);

            fich.transferTo(0, ((new_record_count) * 8) + 78, foch);
            putUnsignedInt(bui, 0, ((new_record_count) * 2) + 1);
            foch.write(bui, 68);
            bui.rewind();
            bui.putChar(0, (char) new_record_count);
            foch.write(bui, 76);

            long delta = srcs_record_count * -8;

            for(int x = 0; x <= srcs_record_number; x++) {
                bul.clear();
                putUnsignedInt(bul, 0, getUInt(data, (x * 8) + 78) + delta);
                putUnsignedInt(bul, 4, getUInt(data, (x * 8) + 82));
                foch.write(bul, (x * 8) + 78);
            }

            delta = delta - strip_length;

            for(int x = (int) (srcs_record_number + srcs_record_count); x < record_count; x++) {
                bul.clear();
                putUnsignedInt(bul, 0, getUInt(data, (x * 8) + 78) + delta);
                putUnsignedInt(bul, 4, (x - srcs_record_count) * 2);
                foch.write(bul, ((x - srcs_record_count) * 8) + 78);
            }

            bui.clear();
            foch.read(bui, 78);

            long first_record_offset = getUInt(bui, 0);

            foch.write(ByteBuffer.allocateDirect((int)(first_record_offset - foch.size())));
            fich.transferTo(header_offset, strip_start_offset - header_offset, foch);
            fich.transferTo(strip_start_offset + strip_length, fich.size(), foch);

            bul.clear();
            putUnsignedInt(bul, 0, 0xffffffffL);
            putUnsignedInt(bul, 4, 0);
            foch.write(bul, first_record_offset + 0xe0);

            bui.clear();
            foch.read(bui, first_record_offset + 0x14);

            long mobi_header_length = getUInt(bui, 0);

            bui.clear();
            foch.read(bui, first_record_offset + 0x80);

            long exth_flag = getUInt(bui, 0);
            long exth_start_offset = first_record_offset + mobi_header_length + 16;

            if((exth_flag & 0x40) == 0x40) {
                bui.clear();
                foch.read(bui, exth_start_offset);

                if(getUInt(bui, 0) == MOBI_EXTH_RECORD_IDENTIFIER) {
                    bui.clear();
                    foch.read(bui, exth_start_offset + 8);

                    long item_count = getUInt(bui, 0);
                    int pos = 12;

                    for(int x = 0; x < item_count; x++) {
                        bul.clear();
                        foch.read(bul, exth_start_offset + pos);

                        long item_type = getUInt(bul, 0);
                        long item_size = getUInt(bul, 4);

                        if(item_type == MOBI_EXTH_BOUNDARY_SECTION_IDENTIFIER) {
                            bui.clear();
                            foch.read(bui, exth_start_offset + pos + 8);

                            long boundary_pointer = getUInt(bui, 0);

                            if(srcs_record_number <= boundary_pointer) {
                                bui.clear();
                                putUnsignedInt(bui, 0, boundary_pointer - srcs_record_count);
                                foch.write(bui, first_record_offset + mobi_header_length + pos + 0x18);
                            }
                        }

                        pos += item_size;
                    }
                }
            }
        }
        finally {
            if(fich != null) fich.close();
            if(foch != null) foch.close();
        }
    }

    private static void putUnsignedInt(ByteBuffer bb, int position, long value) {
        bb.putInt(position, (int) (value & 0xffffffffL));
    }

    private static long getUInt(ByteBuffer b, int idx) {
        return ((long) b.getInt(idx) & 0xffffffffL);
    }
}
